/* -*- Mode: C; tab-width: 8; c-basic-offset: 8; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 1997, 1998, 1999, 2000 Free Software Foundation
 * Copyright (C) 2007 Josselin Mouette
 * All rights reserved.
 *
 * This file was originally part of the Gnome Library.
 *
 * The Viaticum library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The Viaticum Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the Viaticum Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#define _GNU_SOURCE
#include <stdlib.h>
#include <signal.h>
#include <dlfcn.h>
#include <string.h>
#include <unistd.h>
#include <stdint.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>

/* Pointers to external functions */
static void(*gdk_pointer_ungrab)(uint32_t);
static void(*gdk_keyboard_ungrab)(uint32_t);
static void **gdk_display;
static void(*XUngrabServer)(void *);
static void(*gdk_flush)();

/* Area to store the program name */
static char *program_name;

static void
viaticum_detect_libraries (void)
{
        gdk_pointer_ungrab = dlsym (RTLD_DEFAULT, "gdk_pointer_ungrab");
        gdk_keyboard_ungrab = dlsym (RTLD_DEFAULT, "gdk_keyboard_ungrab");
        gdk_display = dlsym (RTLD_DEFAULT, "gdk_display");
        gdk_flush = dlsym (RTLD_DEFAULT, "gdk_flush");
        XUngrabServer = dlsym (RTLD_DEFAULT, "XUngrabServer");
}

/* local definitions */
static void viaticum_signal_handle (int signum);

static void
viaticum_signal_setup (void)
{
        static struct sigaction *setptr;
        struct sigaction sa;

        memset(&sa, 0, sizeof(sa));
        setptr = &sa;
        
        sa.sa_handler = viaticum_signal_handle;
        sigaction(SIGSEGV, setptr, NULL);
        sigaction(SIGABRT, setptr, NULL);
        sigaction(SIGTRAP, setptr, NULL);
        sigaction(SIGFPE, setptr, NULL);
        sigaction(SIGBUS, setptr, NULL);
        sigaction(SIGILL, setptr, NULL);
}

__attribute__ ((constructor))
void viaticum_init (int argc, char **argv)
{
        if (! getenv ("VIATICUM_DISABLE")) {
                if (argc > 0) {
                        program_name = strdup (basename (argv[0]));
                } else {
                        program_name = "(unknown)";
                }
                viaticum_detect_libraries ();
                viaticum_signal_setup ();
        }
}

static void viaticum_signal_handle (int signum)
{
	static int in_segv = 0;
        struct sigaction sa;
	pid_t pid;
	
	memset(&sa, 0, sizeof(sa));
	in_segv++;

        if (in_segv > 2) {
                /* The fprintf() was segfaulting, we are just totally hosed */
                _exit(1);
        } else if (in_segv > 1) {
                /* dialog display isn't working out */
                fprintf(stderr, "Multiple error signals occurred; can't display error dialog.\n");
                _exit(1);
        }

        /* Make sure we release grabs */
        if (gdk_pointer_ungrab != NULL) {
                gdk_pointer_ungrab (0);
        }
        if (gdk_keyboard_ungrab != NULL) {
                gdk_keyboard_ungrab(0);
        }
        if (XUngrabServer != NULL && gdk_display != NULL) {
                XUngrabServer (*gdk_display);
        }
        if (gdk_flush != NULL) {
                gdk_flush();
        }
        
	pid = fork();

	if (pid < 0) {
		/* Eeeek! Can't show dialog */
                fprintf (stderr, "Signal %i received!\n"
				 "Cannot display crash dialog.\n", signum);

                /* Don't use app attributes here - a lot of things are probably hosed */
		if (getenv ("VIATICUM_DUMP_CORE")) {
                        /* Reset SIGABRT so we don't get called again. */
                        sa.sa_handler = SIG_DFL;
                        sigaction (SIGABRT, &sa, NULL);
	                abort ();
                }

		_exit(1);
	} else if (pid > 0) {
                /* Wait for user to see the dialog, then exit. */
                /* Why wait at all? Because we want to allow people to attach to the
		   process */
		int estatus;
		pid_t eret;

		eret = waitpid(pid, &estatus, 0);

                /* Don't use app attributes here - a lot of things are probably hosed */
		if(getenv("VIATICUM_DUMP_CORE")) {
                        /* Reset SIGABRT so we don't get called again. */
                        sa.sa_handler = SIG_DFL;
                        sigaction (SIGABRT, &sa, NULL);
	                abort ();
                }

		_exit(1);
	} else /* pid == 0 */ {
		char buf[32];

		snprintf(buf, sizeof(buf), "%d", signum);

		/* Child process */
		execl ("/usr/share/debreaper/debreaper", "debreaper",
		       program_name, buf, NULL);

                execlp ("debreaper", "debreaper",
			program_name, buf, NULL);

                _exit(99);
	}

	in_segv--;
}
