# debpartial_mirror - partial debian mirror package tool
# (c) 2004 Otavio Salvador <otavio@debian.org>, Nat Budin <natb@brandeis.edu>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import os.path
import ConfigParser
import sys
import string

class InvalidOption(Exception):
    """
    Exception called when a invalid option is found in configuration
    file.

    Attributes:
        section -- Where the invalid option was set;
        option -- The name of invalid option.
    """
    def __init__(self, section, option):
        Exception.__init__(self)
        self.section = section
        self.option = option

class RequiredOptionMissing(Exception):
    """
    Exception called when a required option in the config file is not
    present.

    Attributes:
        section -- Where the invalid option was set;
        option -- The name of invalid option.
    """
    def __init__(self, section, option):
        Exception.__init__(self)
        self.section = section
        self.option = option

class InvalidSection(Exception):
    """
    Exception called when a invalid section is found in configuration
    file.

    Attributes:
        section -- The wrong section name.
    """
    def __init__(self, section):
        Exception.__init__(self)
        self.section = section

class ConfigSection:
    _required = []
    _allowed = []
    _options_with_type = {}
    _allowed_in_filter_field = [
        'subsection',
        'priority',
        'name',
        'include-from',
        'exclude-from',
        'include-script',
        'exclude-script'
        ]
    _name = ""

    def __init__(self, config, section):
        self.config = config
        self.section = section

        # fix up self._allowed to include all required options too
        for item in self._required:
            if item not in self._allowed:
                self._allowed.append(item)

        self.confs = {}

        for item, value in self.config.items(self.section):
            value = self.__cast_option(item, value)
            if item not in self._allowed:
                rv = self.__parse_variable_options(item)
                if hasattr(rv, '__len__'):
                    allowed_key, variable, match = rv
                    if variable == '@BACKEND@':
                        if match not in self.config.sections():
                            logging.debug("[%s] is not the name of a backend" \
                                          "(found in [%s] option in [%s]" % 
                                          (match, item, self.section))
                            raise InvalidOption(self.section, item)
                    else:
                        print("You found a bug: [%s] matches unknown variable" \
                              "[%s]! Please report it." % (item, variable))
                        sys.exit(1)
                else:
                    # __parse_variable_options didn't return a sequence
                    logging.debug("[%s] is not allowed in a %s section" \
                                  "(it was found in [%s])."
                          % (item, self._name, self.section))
                    raise InvalidOption(self.section, item)
            self.confs[item] = value

        for item in self._required:
            if not self.confs.has_key(item):
                rv = self.__parse_variable_options(item)
                if not hasattr(rv, '__len__'):
                    logging.debug("Required option [%s] not found in" \
                                  "[%s] section." % (item, self.section))
                    raise RequiredOptionMissing(self.section, item)

    def __parse_variable_options(self, item):
        for allowed_key in self._allowed:
            if allowed_key.find('@') != -1:
                left_length = allowed_key.find('@')
                right_length = allowed_key[left_length+1:].find('@') + \
								left_length + 2
                if (allowed_key[:left_length] == item[:left_length]
                    and allowed_key[right_length:] == item[right_length:]):
                    # found it!
                    variable = allowed_key[left_length:right_length]
                    match = item[left_length:len(item)-(len(allowed_key)- \
								right_length)]
                    return allowed_key, variable, match
        return None

    def __cast_option(self, option, value):
        if option not in self._options_with_type:
            try:
                allowed_key, variable, match = \
				 self.__parse_variable_options(option)
                option = allowed_key
            except TypeError:
                pass
        if option in self._options_with_type:
            if self._options_with_type[option] == 'list':
                return value.split()
            elif self._options_with_type[option] == 'boolean':
                value = string.lower(value)
                if value in ('0', 'false', 'no', 'off'):
                    return False
                elif value in ('1', 'true', 'yes', 'on'):
                    return True
                else:
                    logging.debug("'%s' is not a valid boolean value " \
                                  "(found in option [%s] in section [%s])." \
                                  % (value, option, self.section))
                    raise InvalidOption(self.section, option)
            elif self._options_with_type[option] == 'filter':
                opts = value.split()
                ret = {}
                for opt in opts:
                    key, val = opt.split(':')
                    if key not in self._allowed_in_filter_field:
                        logging.debug("[%s] is not a filter field " \
                                      "(found in option [%s] in section [%s])."\
                                      % (key, option, self.section))
                        raise InvalidOption(self.section, option)
                    if key in ('include-from', 'exclude-from'):
                        if not os.path.exists(val):
                            logging.debug("[%s] doesn't exists" \
                                          "(found in option [%s]" \
                                          "in section [%s])." %
                                          (val, option, self.section))
                            print("Fatal error: [%s] doesn't exists. " \
                                  "Check your [%s] option in [%s] section." % 
                                  (val, option, self.section))
                            sys.exit(1)
                    if ret.has_key(key):
                        print("Fatal error: [%s] option has repeated entries. " \
                              "Check your [%s] option in [%s] section" %
                              (key, option, self.section))
                        sys.exit(1)
                        
                    ret[key] = val
                return ret
        else:
            return value

class ConfigGlobal(ConfigSection):
    _required = [
        'mirror_dir',
        'architectures',
        'components',
        'distributions',
        'get_suggests',
        'get_recommends',
        'get_provides',
        'get_sources',
        'get_packages',
        ]

    _allowed = [
        'debug',
        'display', # WARN: this keyword has not been implemented yet
        ]

    _options_with_type = {
        'architectures': 'list',
        'distributions': 'list',
        'get_provides': 'boolean',
        'get_recommends': 'boolean',
        'get_suggests': 'boolean',
        'get_sources': 'boolean',
        'get_packages': 'boolean',
        'components': 'list',
        }

    _name = "global"

class ConfigBackendMirror(ConfigSection):
    _allowed = [
        'server',
        'architectures',
        'components',
        'distributions',
        'filter',
        'files',
        'get_suggests',
        'get_recommends',
        'get_provides',
        'get_sources',
        'get_packages',
	'resolve_deps_using',
        'lock',
        ]

    _options_with_type = {
        'architectures': 'list',
        'distributions': 'list',
        'filter': 'filter',
        'files': 'list',
        'get_provides': 'boolean',
        'get_recommends': 'boolean',
        'get_suggests': 'boolean',
        'get_sources': 'boolean',
        'get_packages': 'boolean',
        'components': 'list',
        'resolve_deps_using': 'list',
        'lock': 'boolean',
        }

    _name = "mirror backend"

class ConfigBackendMerge(ConfigSection):
    _allowed = [
        'filter_@BACKEND@',
	'backends',
	'name',
        'sources_only',
        'origin',
        'label',
        'suite',
        'codename',
        'description'
        ]

    _options_with_type = {
        'backends': 'list',
        'filter_@BACKEND@': 'filter',
        }

    _name = "merge backend"

class Config(ConfigParser.ConfigParser):
    """
    Store the configurations used by our system.
    """

    def __init__(self, filename):
        ConfigParser.ConfigParser.__init__(self)
        self.read(filename)

        self.confs = {}
        self.section_objs = {}

        self.backends = {}

        for section in self.sections():
            section_type = self.__get_section_type(section)
            sectionObj = section_type(self, section)
            self.section_objs[section] = sectionObj
            self.confs[section] = sectionObj.confs
            self.confs[section]['name'] = section

        for section in self.section_objs.values():
            if not isinstance(section, ConfigGlobal):
                self.backends[section.section] = section

    def __get_section_type(self, section):
        # detect which config type this is
        if section == 'GLOBAL':
            return ConfigGlobal
        elif 'backends' in self.options(section):
            return ConfigBackendMerge
        elif 'server' in self.options(section):
            return ConfigBackendMirror
        else:
            logging.debug("Unknown section type in section [%s]." % (section))
            raise InvalidSection(section)

    def get_dependencies(self, backend):
        """
        Get the list of backends that the given backend depends on
        """
        if isinstance(self.get_backend(backend), ConfigBackendMirror):
            keyword = 'resolve_deps_using'
        elif isinstance(self.get_backend(backend), ConfigBackendMerge):
            keyword = "backends"
        else:
            raise InvalidSection(backend)
        try:
            dependencies = self.get_option(keyword, backend)
            return dependencies
        except InvalidOption:
            dependencies = []
            return dependencies

    def check_dependencies(self, backend):
        """
        Checks whether the given beckend depends on valid backend names
        """
        dependencies = self.get_dependencies(backend)
        for dependency in dependencies:
            if dependency not in self.backends.keys():
                raise InvalidSection(section)

    def get_backends(self):

        # Check backend dependencies
        for backend in self.backends.keys():
            self.check_dependencies(backend)

        # Sort backends
        unsorted = self.backends.values()
        sorted = []
        names  = []

        while not len(unsorted) == 0:
            backend = unsorted.pop(0)
            name = backend.section

            dependencies = self.get_dependencies(name)

            if len(dependencies) == 0:
                sorted.append(backend)
                names.append(name)
                continue

            deps_ok = True

            for dependency in dependencies:
                if dependency not in names:
                    deps_ok = False
                    break

            if deps_ok:
                sorted.append(backend)
                names.append(backend.section)
            else:
                unsorted.append(backend)

        mirrors = []
        merges  = []

        for backend in sorted:
            if isinstance(self.get_backend(backend.section), \
						  ConfigBackendMirror):
                mirrors.append(backend)
            elif isinstance(self.get_backend(backend.section), \
							ConfigBackendMerge):
                merges.append(backend)

        return (mirrors, merges)

    def get_backend(self, name):
        if name in self.section_objs:
            return self.section_objs[name]
        raise InvalidSection(name)

    def get_option(self, option, section='GLOBAL'):
	# specified, fall back to GLOBAL
	if self.confs[section].has_key(option):
	    return self.confs[section][option]
        else:
            if section != 'GLOBAL':
                logging.debug("[%s] is not present in section [%s]." \
                              "Fallback to global section." % (option, section))
                try:
                    return self.get_option(option, 'GLOBAL')
                except InvalidOption, msg:
                    logging.debug("[%s] is not present in section [%s]." %
								  (option, section))
                    raise InvalidOption(section, msg.option)
                except InvalidSection, msg:
                    raise InvalidSection(msg.section)
            else:
                logging.debug("[%s] is not present in section [%s]." %
							  (option, section))
                raise InvalidOption(section, option)

	    return self.confs[section][option]


    def dump(self):
        for section, options in self.confs.items():
            print '\n' + section
            for item, value in options.items():
                print "  %s = %s" % (item, self.get_option(item, section))

