package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gtk.event.*;
import java.io.*;
import java.util.*;
import org.gnu.glib.CustomEvents;

/**
 * Widget giving a report of watched litings and bug reports.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: WatchedReport.java,v 1.29 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class WatchedReport extends VBox implements ListingStatusListener, ListingListener{

	private ListingReport listingReport;				// Bug listing widget (top right)
	private BugReport bugReport;

	private WatchedList watchList;				// List of bugs/listings being followed
	
	private TreeView pkgView;					// View for watched items
	private ListStore listingStore;					// Associated store.

	private Hashtable iterList = new Hashtable();

	private org.gnu.gtk.Window parent;
	private BugWatcher bugWatcher;


	private DataColumnObject dataType = new DataColumnObject();
	private DataColumnString dataName = new DataColumnString();
	private DataColumnString dataNameDisplay = new DataColumnString();
	private DataColumnPixbuf dataTypeDesc = new DataColumnPixbuf();
	private DataColumnInt	dataFontWeight = new DataColumnInt();
	private DataColumnString 	dataCountAll = new DataColumnString();
	private DataColumnString 	dataCountRC = new DataColumnString();
	private DataColumnString 	dataCountIN = new DataColumnString();
	private DataColumnString 	dataCountMW = new DataColumnString();
	private DataColumnString 	dataCountFP = new DataColumnString();

	/*
	 * Pixmaps
	 */
	private static final org.gnu.gdk.Pixbuf MAINTAINER = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/maintainer.png" );
	private static final org.gnu.gdk.Pixbuf SUBMITTER = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/submitter.png" );
	private static final org.gnu.gdk.Pixbuf SOURCE = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/source.png" );
	private static final org.gnu.gdk.Pixbuf BINARY = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/binary.png" );
	private static final org.gnu.gdk.Pixbuf VIRTUAL = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/virtual.png" );
	private static final org.gnu.gdk.Pixbuf PSEUDO = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/pseudo.png" );
	private static final org.gnu.gdk.Pixbuf BUGREPORT = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/bugreport.png" );

	
	public WatchedReport(ListingReport listingReport, BugReport bugReport, org.gnu.gtk.Window parent, BugWatcher bugWatcher){
		super( false, 0);
		this.bugWatcher = bugWatcher;
		this.parent = parent;
		this.listingReport = listingReport;
		this.bugReport = bugReport;

		// make the gui
		makeGUI();
	}

	public void loadItems(){		
		// sort out the list of items
		if ( (new File(System.getProperty("user.home") +"/.debbug-java/bugwatcher/watched")).exists()){
			try{
				// Load the list of watched items
				watchList = WatchedList.load();
				setWatched(true);
			}catch(OptionalDataException e){
				e.printStackTrace();
				loadException("Unable to load list of watched items", "The file format of the watched list was not recognised. This could occur when you upgrade to a newer version of debbuggtk when the format has changed. You will have to recreate your watched list. sorry." );
			}catch(InvalidClassException e){
				e.printStackTrace();
				loadException("Unable to load list of watched items", "The file format of the watched list was not recognised. This could occur when you upgrade to a newer version of debbuggtk when the format has changed. You will have to recreate your watched list. sorry." );
			}catch(ClassNotFoundException e){
				e.printStackTrace();
				loadException("Unable to load list of watched items", "The file format of the watched list was not recognised. This could occur when you upgrade to a newer version of debbuggtk when the format has changed. You will have to recreate your watched list. sorry." );
			}catch(IOException e){
				e.printStackTrace();
				loadException("Error loading list of watched items", "An IO Error occured: "+e );
			}
		}else{
			// watch list doesn't exist
			setDefaultList();
			try{
				watchList.save();
			}catch(Exception e){
				// don't do anything - not being able to save defaults is not a
				// major problem.
				e.printStackTrace();
			}
		}
	}

	private void setDefaultList(){
			watchList = new WatchedList();
			watchList.add( new ListingStub( ListingType.SOURCE, "java-gnome" ) );
			watchList.add( new ListingStub( ListingType.MAINT, "mh@debian.org" ) );
			watchList.add( new ListingStub( ListingType.SOURCE, "debbuggtk" ) );
			watchList.add( new ListingStub( ListingType.SOURCE, "libdebbug-java" ) );
			setWatched(true);
	}
	
	/**
	 * An exception occured while loading the list - explain it.
	 */
	private void loadException( String title, String explain ){
		MessageDialog md = new MessageDialog(parent, DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), MessageType.ERROR, ButtonsType.CLOSE, 
				"<span weight=\"bold\" size=\"larger\">" + title + "</span>\n" +explain, true);
		md.run();
		md.destroy();
		setDefaultList();
	}
	
	private void makeGUI() {
		// listingStore - for the Watched items view. 
		listingStore = new ListStore( new DataColumn[] { 
			dataType, dataName, dataNameDisplay, dataTypeDesc, dataCountAll, dataCountRC, dataCountIN, dataCountMW, dataCountFP, dataFontWeight
		} );
		pkgView = new TreeView( listingStore );		// list for the watched items. 
		iterList = new Hashtable();
		pkgView.getSelection().setMode( SelectionMode.MULTIPLE );
		pkgView.setAlternateRowColor(true);
		pkgView.setEnableSearch(true);
		pkgView.setHeadersClickable(true);
		
		TreeViewColumn col1 = new TreeViewColumn();
		CellRendererPixbuf rend = new CellRendererPixbuf();
		col1.packStart(rend, false);
		col1.addAttributeMapping( rend, CellRendererPixbuf.Attribute.PIXBUF, dataTypeDesc);
//		col1.setTitle("Type");
		col1.setResizable(true);
		col1.setReorderable(true);
		pkgView.appendColumn(col1);
				
		makeTextColumn( "List", dataNameDisplay );
		makeTextColumn( "RC", dataCountRC );
		makeTextColumn( "IN", dataCountIN );
		makeTextColumn( "MW", dataCountMW );
		makeTextColumn( "FP", dataCountFP );
		makeTextColumn( "All", dataCountAll );

		pkgView.getSelection().addListener( watchSelectionChange ); 	// Loads listings/bug report when selected item changes.
		// make it look pretty...
		ScrolledWindow pkgWin = new ScrolledWindow( new Adjustment(0,0,0,0,0,0), new Adjustment(0,0,0,0,0,0));
		pkgWin.setPolicy( PolicyType.AUTOMATIC, PolicyType.AUTOMATIC);
		pkgWin.add(pkgView);

		setMinimumSize(200, 250);
		packStart(pkgWin);
	}
		
	private void makeTextColumn( String title, DataColumn data ){
		TreeViewColumn col = new TreeViewColumn();
		CellRendererText rend = new CellRendererText();
		col.packStart( rend, false );
		col.addAttributeMapping( rend, CellRendererText.Attribute.TEXT, data );
		col.addAttributeMapping( rend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		col.setTitle( title );
		col.setResizable( true );
		col.setReorderable( true );
		col.setSortColumn( data );
		pkgView.appendColumn( col );
	}

	public void removeSelected(){
					if (DebbugGtk.DEBUG) System.out.println("Remove item clicked");
					pkgView.getSelection().forEachSelected( new TreeSelectionForEach(){
						public void forEach(TreeModel model, TreePath path, TreeIter iter){
							if (DebbugGtk.DEBUG) System.out.println("DEBUG: removing " +listingStore.getValue(iter, dataName) + "     " + listingStore.getValue(iter, dataType).toString() );
							watchList.remove(  new ListingStub( (ListingType) listingStore.getValue(iter, dataType), listingStore.getValue(iter, dataName) ) );
							setWatched(false);
							try{
								watchList.save();
							}catch(Exception e){
								MessageDialog md = new MessageDialog(parent, DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), MessageType.ERROR, ButtonsType.CLOSE, 
									"Unable to save list of watched items.\n\n" + e, true);
								e.printStackTrace();
								md.run();
								md.destroy();
							}
						}
					});
					if (DebbugGtk.DEBUG) System.out.println("Remove item clicked - end");
	}
	/**
	 * Sets the list of watched items and selects the first item.
	 */
	private void setWatched( boolean preLoop){
		setWatched(  (ListingStub)  null, preLoop );
	}
	/**
	 * Sets the list of watched items from the WatchedList object.
	 * @param selItem The item to have selected
	 */
	private void setWatched(ListingStub selItem, boolean preLoop){		
		// TODO: remove listeners from old entries.
		listingStore.clear();
		iterList.clear();
		TreeIter iter;
		LinkedList items = (LinkedList) watchList.getItems().clone();
		TreeIter selIter = null;
		while (items.size() > 0 ){
			ListingStub item = (ListingStub) items.removeFirst();
			Listing listing = null;
			listing = BTS.getInstance().getListing( item.getType(), item.getName() );

			
			if (DebbugGtk.DEBUG) System.out.println("Adding WatchedList as a listener for listing");
			listing.addListener( (ListingListener) this );
			iter = listingStore.appendRow();
			iterList.put( item.getType().toString() + item.getName(), iter );
			String name = item.getName();
			listingStore.setValue(iter, dataType, item.getType());
			listingStore.setValue(iter, dataName, name  );
			listingStore.setValue(iter, dataNameDisplay, (listing.getType().equals(ListingType.BUG) ? "" : "~")+name);
			if (item.getType().equals( ListingType.SUBMITTER ) )
				listingStore.setValue( iter, dataTypeDesc, SUBMITTER );
			else if (item.getType().equals( ListingType.MAINT ) )
				listingStore.setValue( iter, dataTypeDesc, MAINTAINER );
			else if (item.getType().equals( ListingType.BINARY ) )
				listingStore.setValue( iter, dataTypeDesc, BINARY );
			else if (item.getType().equals( ListingType.SOURCE ) )
				listingStore.setValue( iter, dataTypeDesc, SOURCE );
			else if (item.getType().equals( ListingType.VIRTUAL ) )
				listingStore.setValue( iter, dataTypeDesc, VIRTUAL );
			else if (item.getType().equals( ListingType.PSEUDO ) )
				listingStore.setValue( iter, dataTypeDesc, PSEUDO);
			else if (item.getType().equals( ListingType.BUG ) )
				listingStore.setValue( iter, dataTypeDesc, BUGREPORT);
			if (preLoop)
				while( Gtk.eventsPending() ){ Gtk.mainIteration(); }

			setCounts( listing, iter );

			if (selItem != null && selItem.equals(item))	selIter = iter;
		}
		if (selIter != null) pkgView.getSelection().select(selIter);
	}

	private void setCounts( Listing listing, TreeIter iter ){
		if( iter == null) return;
		String unknown = listing.getType().equals(ListingType.BUG) ? "" : "?";

		if (listing == null || !listing.getComplete() ){
			listingStore.setValue(iter, dataCountAll, unknown);
			listingStore.setValue(iter, dataCountRC, unknown);
			listingStore.setValue(iter, dataCountIN, unknown);
			listingStore.setValue(iter, dataCountMW, unknown);
			listingStore.setValue(iter, dataCountFP, unknown);
			listingStore.setValue(iter, dataFontWeight, 400);
		}else{
			if (listing.getAllUnreadCount() > 0 ){
				listingStore.setValue(iter, dataFontWeight, 900 );
			}else{
				listingStore.setValue(iter, dataFontWeight,  400);
			}
			listingStore.setValue(iter, dataCountAll, listing.getAllCount() 
					+ ( ( listing.getAllUnreadCount() > 0 ) ?  "(" + listing.getAllUnreadCount() + ")" : ""   )  );
			listingStore.setValue(iter, dataCountRC, listing.getRCCount()
					+ ( ( listing.getRCUnreadCount() > 0 ) ?  "(" + listing.getRCUnreadCount() + ")" : ""   )  );
			listingStore.setValue(iter, dataCountIN, listing.getINCount() 
					+ ( ( listing.getINUnreadCount() > 0 ) ?  "(" + listing.getINUnreadCount() + ")" : ""   )  );
			listingStore.setValue(iter, dataCountMW, listing.getMWCount() 
					+ ( ( listing.getMWUnreadCount() > 0 ) ?  "(" + listing.getMWUnreadCount() + ")" : ""   )  );
			listingStore.setValue(iter, dataCountFP, listing.getFPCount() 
					+ ( ( listing.getFPUnreadCount() > 0 ) ?  "(" + listing.getFPUnreadCount() + ")" : ""   )  );
			boolean partial = ((!listing.getComplete()) && (!listing.getType().equals(ListingType.BUG)));
			listingStore.setValue(iter, dataNameDisplay,( partial ? "~" : "" ) + listing.getName() );
		}
	}
	
	private TreeSelectionListener watchSelectionChange = new TreeSelectionListener(){ 
			public void selectionChangedEvent(TreeSelectionEvent event){
				if (DebbugGtk.DEBUG) System.out.println("DEBUG: watchSelectionChangedEVENT");

				TreeSelectionForEach fe = new TreeSelectionForEach(){
					public void forEach(TreeModel model, TreePath path, TreeIter iter){
						String pkg = listingStore.getValue(iter, dataName);
						ListingType type = (ListingType) listingStore.getValue(iter, dataType);
						if (DebbugGtk.DEBUG) System.out.println("DEBUG: "+pkg);
						if (DebbugGtk.DEBUG) System.out.println("DEBUG: "+type.toString());
						
						if (type.equals(ListingType.BUG)){
							Bug bug = BTS.getInstance().getBug( new Integer( pkg ) );
							bugWatcher.listingSelectionChanged( bug );
						}else{
							listingReport.setPackage( new ListingStub( type, pkg ) );
						}
					}
				};				
				pkgView.getSelection().forEachSelected(fe);
			}
		};


	/**
	 * Creates a {@link ListingSelector} dialog to retrieve a new item to watch.
	 * Adds that item to the watched items list.
	 */
	public void watchNew(){
		ListingSelector selector = new ListingSelector(new ListingSelectorHandler(){
			public void listingSelected(String entry, ListingType type){
				ListingStub item = new ListingStub(type, entry);
				watchList.add( item );
				if (DebbugGtk.DEBUG) System.out.println("Added item to list");
				setWatched(item, false);				
				try{
					watchList.save();
				}catch( java.io.IOException e){
					System.out.println("ERROR SAVING WATCHED LIST "+e);
					e.printStackTrace();
					// FIXME
				}
			}
			public void newVirtual(){
				if (DebbugGtk.DEBUG) System.out.println("New Virtual");
				newVirt();
			}
		}, true);
		selector.setTitle("BugWatcher - Select item to watch");
		selector.showAll();
	}


	public void newVirt(){
		VirtualListingEditor selector = new VirtualListingEditor(new ListingSelectorHandler(){
			public void listingSelected(String entry, ListingType type){
				ListingStub item = new ListingStub(type, entry);
				watchList.add( item );
				if (DebbugGtk.DEBUG) System.out.println("Added item to list");
				setWatched(item, false);				
				try{
					watchList.save();
				}catch( java.io.IOException e){
					System.out.println("ERROR SAVING WATCHED LIST "+e);
					e.printStackTrace();
					// FIXME
				}
			}
			public void newVirtual(){
				newVirt();
			}
		});
		selector.showAll();
		
	}

	public void listingStatusChanged( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("ListingStatusChanged");
		TreeIter iter = (TreeIter) iterList.get( listing.getType().toString() + listing.getName() );
		setCounts( listing, iter );
	}

	public WatchedList getList(){
		return watchList;
	}

	///////// ListingListener methods ////////////
	public void bugCountsChanged( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("Watched Report - bugCountsChanged");
		 final Listing l = listing;
		 CustomEvents.addEvent( new Runnable(){
			 public void run(){
				if (DebbugGtk.DEBUG) System.out.println("Watched Report - bugCountsChanged - handling");
				 listingStatusChanged( l );
			 }});
	}
	public void listingUpdateStart( Listing listing ){}
	public void listingUpdateDone( Listing listing ){}
	public void downloadingListing( Listing listing ){}
	public void interpretingListing( Listing listing ){}
	public void downloadingListingBug( Listing listing, Integer bugNumber, int pending, int total){}
	public void loadingListingBug( Listing listing, Integer bugNumber, int pending, int total){}
	public void bugRemoved( Listing listing, Integer bugNumber ){}
	public void bugAdded( Listing listing, Bug bug ){}
	public void bugChanged( Listing listing, Bug bug ){}
	public boolean  listingException( Listing listing, Exception e ){ return true; }
	public void loadReportsDone( Listing listing ){ }

}
