package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gnome.*;
import org.gnu.gtk.event.*;
import java.util.LinkedList;
import java.util.regex.*;

/**
 * Container for displaying comments of bug reports, allowing the user o cycle
 * between reports.
 *
 * <p>There is much scope for improvement - noticing duplicate reports; showing
 * a tree of subjects; syntax highlighting; creating more links; ...
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: CommentViewer.java,v 1.28 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class CommentViewer extends VBox implements ToggleListener{
	
	private LinkedList comments;
	private int current = 1;

	private Bug bug;

	private Button first;
	private Button prev;
	private Button next;
	private Button last;

	private CheckButton viewFullSource;

	private TextView tv;
	private TextBuffer tb;
	
	
	/**
	 */
	public CommentViewer(Bug bug){
		super(false, 0);
		init();
		setBug(bug);
	}
	/**
	 */
	public CommentViewer(){
		super(false, 0);
		init();
	}

	private Adjustment adj = new Adjustment(0, 0, 100, 2, 10, 10);
	private Adjustment adj2 = new Adjustment(0, 0, 100, 2, 10, 10);
	private void init(){		

		tb = new TextBuffer();
		tv = new TextView( tb );
		tv.setEditable(false);
		ScrolledWindow tf = new ScrolledWindow( null, null );
		tf.setShadowType( ShadowType.IN );
		tf.setPolicy(PolicyType.AUTOMATIC, PolicyType.AUTOMATIC);
		tf.add(tv);
		
		TextTag footTag = tb.createTag("foot");
		footTag.setForeground("red");
		TextTag headTag = tb.createTag("head");
		headTag.setForeground("darkgreen");
		TextTag quote1 = tb.createTag("quote1");
		quote1.setForeground("blue");
		TextTag quote2 = tb.createTag("quote2");
		quote2.setForeground("lightblue");
		TextTag version = tb.createTag("version");
		version.setForeground("green");

		HBox hb = new HBox( false, 0);
		packStart(tf, true, true, 0);
		packStart( hb, false, false, 0);

		viewFullSource = new CheckButton("View Full Source", false);
		viewFullSource.setState(false);
		hb.packStart( viewFullSource, true, true, 0);

		viewFullSource.addListener( (ToggleListener) this );
		/*new ToggleListener(){
			public void toggleEvent( ToggleEvent event ){
				setText();
			}
		});
		*/
		
		HButtonBox bbox = new HButtonBox();
		bbox.setLayout( ButtonBoxStyle.END );
		hb.packStart( bbox, false, false, 0 );

		// todo: l10n
		first = new Button( GtkStockItem.GOTO_FIRST );
		prev = new Button( GtkStockItem.GO_BACK );
		next = new Button( GtkStockItem.GO_FORWARD );
		last = new Button( GtkStockItem.GOTO_LAST );

		first.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event){
				if (event.isOfType( ButtonEvent.Type.CLICK )){
					goFirst();
				}
			}
		});
		prev.addListener(  new ButtonListener(){
			public void buttonEvent( ButtonEvent event){
				if (event.isOfType( ButtonEvent.Type.CLICK )){
					goPrev();
				}
			}
		});
		next.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event){
				if (event.isOfType( ButtonEvent.Type.CLICK )){
					goNext();
				}
			}
		});
		last.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event){
				if (event.isOfType( ButtonEvent.Type.CLICK )){
					goLast();
				}
			}
		});
		bbox.packStart(first);
		bbox.packStart(prev);
		bbox.packStart(next);
		bbox.packStart(last);

		first.setSensitive(false);
		prev.setSensitive(false);
		next.setSensitive(false);
		last.setSensitive(false);
	}
	
			public void toggleEvent( ToggleEvent event ){
				System.out.println("Toggle Event");
				setText();
			}
	/**
	 */
	public void setBug(Bug bug){
		this.bug = bug;
		current = 1;
		comments = bug.getComments();
		if (comments == null || comments.size() < 1){
			System.out.println("WARNING: Displaying bug with no comments");
			tb.setText("No comments found - perhaps there was an error reading the bug report from the server??");
			current = -1;
			comments = new LinkedList();
		}else{
			BugComment comment = (BugComment) comments.get(current - 1);
			setText( comment );
		}
		updateButtonVisibility();
	}

	private String headText = null;
	private String bodyText = null;

	private void setText(BugComment comment){
/*		String head = "Packages:\n";
		Object[] stubs = bug.getListings().toArray();
		for (int i = 0; i < stubs.length; i++){
			head += "\t" + ((ListingStub) stubs[i]).getName() + "\n";
		}
		head += "--\n";
*/		
//		System.out.println("About to set bug comment text");
//		System.out.println("About to set bug comment text "+ current+ " of "+comments.size());
//		System.out.println("About to set bug comment text "+ comment.getSource());
		if (viewFullSource.getState()){
			headText =  "Comment "+current+" of "+comments.size()+":" ;
			bodyText =  comment.getSource();
		}else{
			headText =  "Comment "+current+" of "+comments.size()+":" ;
			String header = comment.getHead();
			if (header.indexOf("\n", 5)>=0)
				headText += header.substring(0,  header.indexOf("\n", 5)+1 );
			if (header.indexOf("From: ") >=0)
				header = header.substring( header.indexOf("From: ") );
			if (header.indexOf("\n") > 0)
				headText += header.substring( 0, header.indexOf("\n") );
			bodyText = comment.getBody();
		}

		tb.deleteText( tb.getStartIter(), tb.getEndIter() );
		tb.setText("");
		tb.setText( headText + bodyText );
		TextIter headEnd = tb.getIter( headText.length() );
		tb.applyTag( "head", tb.getStartIter(), headEnd );
		if (bodyText.indexOf("\n--") > 0){
			TextIter footStart = tb.getIter( headText.length() + bodyText.indexOf("\n--") );
			tb.applyTag( "foot", footStart, tb.getEndIter() );
		}
		// TODO: Colour quotes

		/*
		 * 	Replace URLS with Gnome HRef widgets.
		 *
		 */
		if (!viewFullSource.getState()){
			Pattern http = Pattern.compile("((?:HT|F)TPS?://\\S+)[\\s\n]", Pattern.CASE_INSENSITIVE );
			Matcher m = http.matcher( bodyText );
			while ( m.find() ){
				int start = m.start();
				int end = m.end() -1 ;
				String ref = m.group(1);
				
				tb.deleteText( tb.getIter( start + headText.length() ),
							tb.getIter( end + headText.length() ) );
		
				TextChildAnchor child = new TextChildAnchor();
				tb.inserChildAnchor( tb.getIter( start + headText.length() ), child );
				HRef hr = new HRef( ref, ref );
				hr.setRelief( ReliefStyle.NONE );
				tv.addChild( hr, child );
				bodyText = bodyText.substring( 0, start ) + "+"+ bodyText.substring( end );
				hr.showAll();
				m = m.reset( bodyText );
			}

			Pattern bug = Pattern.compile("(#\\d{5,7})\\D");
			m = bug.matcher( bodyText );
			while ( m.find() ){
				int start = m.start();
				int end = m.end() -1 ;
				String ref = m.group(1);
				
				tb.deleteText( tb.getIter( start + headText.length() ),
							tb.getIter( end + headText.length() ) );
		
				TextChildAnchor child = new TextChildAnchor();
				tb.inserChildAnchor( tb.getIter( start + headText.length() ), child );
				Button button = new Button( ref );	
				final String number = ref.substring(1);
				button.addListener( new ButtonListener() {
					public void buttonEvent( ButtonEvent event ){
						if (event.isOfType(ButtonEvent.Type.CLICK)){
							new BugViewer(number );
						}
					}
				});
				tv.addChild( button, child );
				bodyText = bodyText.substring( 0, start ) + "+"+ bodyText.substring( end );
				button.showAll();
				m = m.reset( bodyText );
			}
			Pattern mailto = Pattern.compile("(\\S+@\\S+)\\s", Pattern.CASE_INSENSITIVE );
			m = mailto.matcher( bodyText );
			while ( m.find() ){
				int start = m.start();
				int end = m.end() -1 ;
				String ref = m.group(1);
				
				tb.deleteText( tb.getIter( start + headText.length() ),
							tb.getIter( end + headText.length() ) );
		
				TextChildAnchor child = new TextChildAnchor();
				tb.inserChildAnchor( tb.getIter( start + headText.length() ), child );
				HRef hr = new HRef( "mailto:" + ref, ref );
				hr.setRelief( ReliefStyle.NONE );
				tv.addChild( hr, child );
				bodyText = bodyText.substring( 0, start ) + "+"+ bodyText.substring( end );
				hr.showAll();
				m = m.reset( bodyText );
			}
		}
		}

		private void setText(){
			setText( (BugComment) comments.get(current - 1) );
		}
	

	private void goNext(){
		current++;
		setText( ((BugComment) comments.get(current - 1)));
		updateButtonVisibility(); 
	}
	private void goPrev(){
		current--;
		setText( ((BugComment) comments.get(current - 1)));
		updateButtonVisibility(); 
	}
	private void goFirst(){
		current = 1;
		setText( ((BugComment) comments.get(current - 1)));
		updateButtonVisibility(); 
	}
	private void goLast(){
		current = comments.size();
		setText( ((BugComment) comments.get(current - 1)));
		updateButtonVisibility(); 
	}


	/**
	 * Returns a quote to be used in reply emails. This is either the currently
	 * selected region, or the whole message.
	 */
	public String getQuote(){
		if (tb.getSelected()){
			return quoted( tb.getText( tb.getSelectionStart(), tb.getSelectionEnd(), false ));
		}else{
			String quote = ((BugComment) comments.get(current - 1) ).getBody();
			// remove sigs, etc
			int i =  quote.indexOf("\n--");
			if ( i > 0 )
				quote = quote.substring( 0, i );
			return quoted( quote );
		}
	}

	private String quoted(String quote){
		StringBuffer q = new StringBuffer( quote );
		q.insert( 0, ">" );
		for (int i = 0; i < q.length() - 1; i++)
			if (q.charAt(i) == '\n')
				q.insert(i+1, ">");
		return q.toString();
	}
	
	/**
	 * Updates the shaded status of the butons, depending on which comment is
	 * currontly being displayed, and how many are available for display.
	 */
	private void updateButtonVisibility(){
		first.setSensitive( current > 1 );
		prev.setSensitive( current > 1 );
		next.setSensitive( current < comments.size() );
		last.setSensitive( current < comments.size() );
	}
}
