package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gnome.*;
import org.gnu.glade.*;
import java.io.*;

/**
 * Application to give a listing of bugs in a package/src/package/pseudo
 * package/maintainer.
 *
 * <p>{@link com.tildemh.debbuggtk.BugViewer} will be called to display bugs
 * when the user double clicks on them.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugLister.java,v 1.23 2004/03/16 19:31:03 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugLister implements ListingReportListener{
	
	private ListingReport report;
	private App app;
	private AppBar appBar;

	private OptionMenu listingType;
	private OptionMenu includeClosed;
	
	
	/**
	 * Create a new window
	 */
	public BugLister(){
		makeWindow();
	}
	
	/**
	 * Create a new window
	 */
	public BugLister(Listing list){
		makeWindow();
		setListing( list );
	}

	private LibGlade glade;
	private void makeWindow(){
		DebbugGtk.newWindow();

		WindowIcon.setDefaultIcon(DebbugGtk.ICON);
		
		String filename = DebbugGtk.GLADE_XML_DIR+"buglister.glade";
		try {
			glade = new LibGlade(filename, this, null);
			app = (App) glade.getWidget("bugLister");
			app.addListener( DebbugGtk.lifeCycleListener );
			app.setTitle("BugLister "+DebbugGtk.VERSION);
			VBox vb = (VBox) glade.getWidget("mainVB");
			appBar = (AppBar) glade.getWidget("appBar");
			report = new ListingReport(BTS.getInstance(), this, app);
			vb.packStart( report, true, true, 0);
			vb.showAll();
			listingType = (OptionMenu) glade.getWidget("listingType");
			includeClosed = (OptionMenu) glade.getWidget("includeClosed");
			glade.getWidget("setListing").grabDefault();
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}
		app.showAll();
		
	}

	////////////////////////////////////////////////////////////////////////////////
	// Events defined within glade file
	//
	public void refresh(){
		report.update();
	}
	public void stop(){
		report.stopUpdate();
	}
	public void editCols(){
		report.editColumns();
	}
	public void quit(){
		app.hide();
		DebbugGtk.closeWindow();
	}
	public void about(){
		DebbugGtk.helpAbout();
	}
	public void bugWatcher(){
		new BugWatcher();
	}
	public void bugViewer(){
		new BugViewer();
	}
	public void bugReporter(){
		new BugReporter();
	}
	public void newWindow(){
		new BugLister();
	}
	public void setListing(){
		String name = ((org.gnu.gtk.Entry) glade.getWidget("listingEntry")).getText();
		ListingType type = ListingType.SOURCE;
		switch (listingType.getHistory()){
					case -1: return; 
					case 0: type = ListingType.SOURCE; break;
					case 1: type = ListingType.BINARY; break;
					case 2: type = ListingType.PSEUDO; break;
					case 3: type = ListingType.MAINT; break;
					case 4: type = ListingType.SUBMITTER; break;
					default: throw new RuntimeException("Unknown optionMenu selection in ListingSelector: "+ listingType.getHistory());
		}
		type.setIncludeClosed( includeClosed.getHistory() == 0 );
		
		ListingStub stub = new ListingStub( type, name );
		
		Listing listing = BTS.getInstance().getListing( stub );
		System.out.println(listing.toString());

		setListing( listing );
	}
	////////////////////////////////////////////////////////////////////////////////	

	public void setListing( Listing list ){
		report.setPackage(list);
		app.setTitle(list.getName() + " ("+list.getType()+") - BugLister "+DebbugGtk.VERSION);
	}
		public void listingSelectionChanged(com.tildemh.debbug.Bug bug){
			System.out.println("Selected: xx");
			System.out.println("Selected: "+bug.getNumber());
			System.out.println("Selected: "+bug.getTitle());
			System.out.println("Selected: "+bug);
			try{
				new BugViewer(String.valueOf(bug.getNumber()));
			}catch(Exception e){
				//fixme
				e.printStackTrace();
			}
		}


	public static void main(String[] args) throws Exception{
		Thread.currentThread().setPriority( Thread.MAX_PRIORITY );

		
		if (args.length==0){
			Program.initGnomeUI("debbuggtk", DebbugGtk.VERSION, args);
			new BugLister();
			Gtk.main();
			return;
		}
			
		Listing list = null;
		File f = new File(args[0]);
		if (f.exists()){
			// load from file
		
			StringBuffer htmlPage = new StringBuffer();
			FileReader reader = new FileReader(args[0]);
			int ch =0;
			while ((ch = reader.read()) != -1){
				htmlPage.append((char)ch);
			}
		
// FIXME			list = WWWListingParser.makeListing(htmlPage.toString());
		}else{
			// load from web
			 ListingType type = null;
			if (args[0].equals("source")){
				type = ListingType.SOURCE;
			}else if (args[0].equals("binary")){
				type = ListingType.BINARY;
			}else if (args[0].equals("maint")){
				type = ListingType.MAINT;
			}else if (args[0].equals("submitter")){
				type = ListingType.SUBMITTER;
			}
			list = (BTS.getInstance()).getListing( type, args[1] );
		}
		Program.initGnomeUI("debbuggtk", DebbugGtk.VERSION, args);
		new BugLister(list);
	    Gtk.main();

	}

		public void listingChanged(ListingStub listing){
			app.setTitle( listing.getName() +" ("+ listing.getType() +") - BugLister "+DebbugGtk.VERSION);
		}
		public void listingReportStateChanged( ListingState state){
			appBar.clearStack();
			appBar.pushStack( state.toString() );
			if (state.equals(ListingState.DLLISTING)){
				appBar.getProgressBar().pulse();
			}else{
				appBar.setProgressPercentage( (double) state.getCurrent() / (double) state.getMax() );
			}
			appBar.showAll();
		}
			public void help(){
		try{
			Help.display( "debbuggtk", "contents" );
		}catch( HelpException e ){
			e.printStackTrace();
			throw new RuntimeException( e.toString() );
		}
	}
}
