package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.glade.*;
import java.io.*;
import org.gnu.gtk.event.*;

/**
 * Form for editing bugs. This provides the functionality to add comments to
 * bugs and change various fields. The actual bts interaction is done by
 * generating an email, which may then be sent using the gnome standard mail
 * editor. (probably)
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugEdit.java,v 1.34 2004/03/20 16:00:57 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugEdit implements OptionMenuListener, ResponseEditorListener{
	

	private Bug bug;
	
	private TextBuffer commentBuffer;
	private CheckButton includeSysInfo;

	private ControlEdit controlEditControls;

	private RadioButton submitterAddr;
	private RadioButton normalAddr;
	private RadioButton maintOnlyAddr;
	private RadioButton doneAddr;
	private Notebook noteBook;

	private RadioButton controlSendNow;
	private RadioButton controlQueue;
	
	private Window window;
	private String quote;
	private LibGlade glade;

	private OptionMenu responsesMenu;
	
	/**
	 * Constructs a bug editor object. 
	 * @param bug The bug to be edited
	 */
	public BugEdit(Bug bug, String quote){
		this.bug = bug; 
		this.quote = quote;

		String filename = DebbugGtk.GLADE_XML_DIR+"bugedit.glade";
		LibGlade glade = null;
		try {
			glade = new LibGlade(filename, this, null);
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}

		window = (Window) glade.getWidget("bugEdit");
		includeSysInfo = (CheckButton) glade.getWidget("includeSysInfo");

		controlEditControls = new ControlEdit(bug);
		((VBox) glade.getWidget("controlPage")).packStart(controlEditControls);

		submitterAddr = (RadioButton) glade.getWidget("submitterAddr");
		normalAddr = (RadioButton) glade.getWidget("normalAddr");
		maintOnlyAddr = (RadioButton) glade.getWidget("maintOnlyAddr");
		doneAddr = (RadioButton) glade.getWidget("doneAddr");

		controlSendNow = (RadioButton) glade.getWidget("controlSendNow");
		controlQueue = (RadioButton) glade.getWidget("controlQueue");
		
		commentBuffer = ((TextView) glade.getWidget("textView")).getBuffer();
		setBug();

		responsesMenu = (OptionMenu) glade.getWidget("responsesMenu");

		fillOptionMenu( ResponseEditor.getInstance().getResponses() );
		
		responsesMenu.addListener( (OptionMenuListener) this );

		ResponseEditor.getInstance().addListener( (ResponseEditorListener) this );
	}

	private void fillOptionMenu( String[] responses ){
		Menu menu = new Menu();
		responsesMenu.setMenu( menu );
		for( int i = 0; i < responses.length; i++){
			menu.append( new MenuItem( responses[i], false ) );
		}
		menu.append( new SeparatorMenuItem() );
		menu.append( new ImageMenuItem( GtkStockItem.ADD.getString(), new AccelGroup() ) );
		menu.showAll();
	}

	/**
	 * Adds bug details to form
	 */
	private void setBug(){
		//TODO: If the bug has not been checked with the server for a long time,
		// make this very clear.
		commentBuffer.setText(quote);
		includeSysInfo.setState(true);
		submitterAddr.setState(true);
		window.setTitle("Editing Bug #"+bug.getNumber() + ": "+bug.getTitle());
	}

	/**
	 * Generates email to be sent to debian bug server
	 */
	public void generateEmail(){
		System.out.println("GenerateEmail");
				
		String mail = "";
		
		System.out.println("GenerateEmail");
		String control = controlEditControls.getControlMsg(); 
		System.out.println("GenerateEmail");
		boolean needControl = (control != null && !control.equals("")); // should we send to  control@bugs?
		System.out.println("GenerateEmail");
		if (controlQueue.getState()){
			ControlQueue.getInstance().addItem( control );
			needControl = false;
		}
		System.out.println("GenerateEmail");

		if (needControl)
			mail += control + "thanks\n\n";

		boolean haveComments = false;
		if (! commentBuffer.getText( commentBuffer.getStartIter(), commentBuffer.getEndIter(), true ).equals(quote) )
			haveComments = true;
		
		if (haveComments && includeSysInfo.getState())
			mail += "Followup-For: Bug #"+bug.getNumber() + "\n"
				+ "Package: "+ bug.getPackageName() + "\n"
				+ "Status: " + Packages.getStatus( bug.getPackageName() ) 
				+ "Version: " + Packages.getVersion( bug.getPackageName() )
				;

		if (haveComments){
			String comments = commentBuffer.getText( commentBuffer.getStartIter(), commentBuffer.getEndIter(), true );
			mail += Util.wordWrap( comments, 80 );
		}

		if (haveComments && includeSysInfo.getState()){
			mail += "\n" + Packages.getSystemInfo( bug.getPackageName() );
		}
			
		String recipients = "";
		if (haveComments)
		if (submitterAddr.getState()){
			recipients = bug.getNumber() + "-submitter@bugs.debian.org";
		}else if (	normalAddr.getState()){
			recipients = bug.getNumber() + "@bugs.debian.org";
		}else if ( maintOnlyAddr.getState() ){
			recipients = bug.getNumber() + "-quiet@bugs.debian.org";
		}else if ( doneAddr.getState() ){
			recipients = bug.getNumber() + "-done@bugs.debian.org";
		}

		mail += "\n";

		
		if (haveComments || needControl)
			org.gnu.gnome.Url.show("mailto:"+recipients
				+ "?subject=Bug #"+bug.getNumber() + ": "+bug.getTitle() 
				+ (needControl ? "&cc=control@bugs.debian.org" : "" ) 
				+ "&body="+mail
				+ "&X-debbuggtk-Version="+DebbugGtk.VERSION 
				);
		window.hide();
	}

	public void showAll(){
		window.showAll();
	}
	public void cancel(){
		window.hide();
	}

	public void optionMenuEvent( OptionMenuEvent event ){
		if (event.isOfType( OptionMenuEvent.Type.CHANGE ) ){
			ResponseEditor.getInstance().handleResponse( responsesMenu.getHistory(), commentBuffer );
		}
	}

	public void responsesChanged( String[] newResponses ){
		fillOptionMenu( newResponses );
	}

	public void clear(){
		commentBuffer.setText("");
	}
}
