package com.tildemh.debbug;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;

/**
 * Lazy writer for bug reports in cache. This will run in a separate thread to
 * the main application. This should improve performance as cache entries are
 * written often - when refreshing them and also when changing their read
 * status.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: CacheWriter.java,v 1.11 2004/01/08 14:47:18 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class CacheWriter implements Runnable{

	private String dir;					// Cache directory

	public CacheWriter(String dir){
		this.dir = dir;
	}
	
	private boolean noRead = false;		// determines whether cache reads are permitted
	private boolean noWrite = false;	// determines whether cache writes are permitted

	// list of things to be written.
	private LinkedList writeQueue = new LinkedList();

	/**
	 * Adds an item to be saved to disk
	 */
	public synchronized void addItem(Bug bug){
		writeQueue.add(bug);
		notifyAll();
	}

	/**
	 * Main loop
	 */
	public void run(){
		while(!Thread.interrupted()){
			while( writeQueue.size() > 0){
				// write some items to disk
				Bug b;
				synchronized(this){
					b = (Bug) writeQueue.removeFirst();
				}
				writeBug(b);
			}
			synchronized(this){
//				Thread.currentThread().setDaemon(false);
				while (writeQueue.size() <= 0){
					try{
						wait();
					}catch(InterruptedException e){
						e.printStackTrace();
					}
				}
//				Thread.currentThread().setDaemon(true);
			}
		}
	}

	/**
	 * writes a single bug report to disk
	 */
	private void writeBug(Bug bug){
//		if(DEBUG) System.out.println("DEBUG: writing file to "+dir + Integer.toString( bug.getNumber() ) +"*********************");
		File f = new File(dir + Integer.toString( bug.getNumber() ) );
		
		try{
			//ObjectOutputStream out = new ObjectOutputStream( new FileOutputStream(f) );
			//out.writeObject(bug);
			//out.flush();
			//out.close();
			BugFile.storeBug(bug, f );
		}catch(IOException e){
			System.out.println("IOError when writing bug report to "+f.toString() + " -  "+e.getMessage());
			e.printStackTrace();
		}
	}
}
