package com.tildemh.debbuggtk;


import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gtk.event.*;
import java.util.Hashtable;
import java.util.Date;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.LinkedList;
import org.gnu.glib.CustomEvents;

/**
 * Container for displaying Package reports.
 *
 * <p>This includes package data, status and listings of all bugs in the
 * package. 
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: ListingReport.java,v 1.99 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class ListingReport extends VBox implements ListingListener{

	private static final org.gnu.gdk.Pixbuf READ = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/read.png" );
	private static final org.gnu.gdk.Pixbuf UNREAD = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/unread.png" );
	private static final org.gnu.gdk.Pixbuf READ_DONE = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/read-done.png" );
	private static final org.gnu.gdk.Pixbuf UNREAD_DONE = new org.gnu.gdk.Pixbuf( "/usr/share/pixmaps/debbuggtk/unread-done.png" );
	private ListingState state = ListingState.IDLE;
	private Thread constructorThread = null;
	private Runnable constructorObject = null;

	private ListingReportListener listener = null;
	private ListingStatusListener statusListener = null;
	
	
	/**
	 * Time of last call to customEvents.
	 * We're trying to reduce the number of calls by queueing actions.
	 */
	private long customEventTime = 0;
	private LinkedList customEventAddQueue = new LinkedList();
	private LinkedList customEventRemoveQueue = new LinkedList();
	
	private ListStore store;
	private TreeView view;
	private TreeSelection selection;
	private TreeSelectionListener selListener;
	/*
	 * Map bug reports to iters
	 */
	private  Hashtable iterList  = new Hashtable();

	private ScrolledWindow sw;
	private volatile Listing listing = null;
	private Entry filter;
	private BugFilterer filterer = BugFilterer.ALL;
	private Button filterBtn; 
	private Button clearFilterBtn; 
	private Button attnFilterBtn; 
	private Button rcFilterBtn; 

	private Label noBugs = new Label("There are no bug reports to display");

	private CellRendererText descRend;

	private BTS bts;
	private Window parent;
	
	/**
	 */
	private TreeViewColumn colFirst = new TreeViewColumn();
	private TreeViewColumn colNumber = new TreeViewColumn();
	private TreeViewColumn colSeverity = new TreeViewColumn();
	private TreeViewColumn colDesc = new TreeViewColumn();
	private TreeViewColumn colCreated = new TreeViewColumn();
	private TreeViewColumn colComments = new TreeViewColumn();
	private TreeViewColumn colPatch = new TreeViewColumn();
	private TreeViewColumn colSecurity = new TreeViewColumn();
	private TreeViewColumn colMerged = new TreeViewColumn();
	//tags
	private TreeViewColumn colUpstream = new TreeViewColumn();
	private TreeViewColumn colFixed = new TreeViewColumn();
	private TreeViewColumn colPending = new TreeViewColumn();
	private TreeViewColumn colHelp = new TreeViewColumn();
	private TreeViewColumn colPotato = new TreeViewColumn();
	private TreeViewColumn colWoody = new TreeViewColumn();
	private TreeViewColumn colSarge = new TreeViewColumn();
	private TreeViewColumn colSid = new TreeViewColumn();
	private TreeViewColumn colExperimental = new TreeViewColumn();
	private TreeViewColumn colUnreproducible = new TreeViewColumn();

	private TreeViewColumn colD_I = new TreeViewColumn();
	private TreeViewColumn colConfirmed = new TreeViewColumn();
	private TreeViewColumn colIPV6 = new TreeViewColumn();
	private TreeViewColumn colLFS = new TreeViewColumn();
	private TreeViewColumn colWontfix = new TreeViewColumn();
	private TreeViewColumn colMoreinfo = new TreeViewColumn();

	private TreeViewColumn colIsForwarded = new TreeViewColumn();
	private TreeViewColumn colPackage = new TreeViewColumn();
	private TreeViewColumn colStatus = new TreeViewColumn();

	private LinkedList columns = new LinkedList();
	
	private int topColumn = -1; // index of the last column in the view.
	

	// DataColumns for storing data in the List model
	DataColumnString dataPackage			= new DataColumnString();
	DataColumnObject dataBugObject 		= new DataColumnObject();
	DataColumnString dataNumber 			= new DataColumnString();
	DataColumnString dataTitle 			= new DataColumnString();
	DataColumnString dataSeverity		= new DataColumnString();
	DataColumnString dataCreated			= new DataColumnString();
	DataColumnString dataTagList 		= new DataColumnString();
	DataColumnString dataIsForwarded		= new DataColumnString();
	DataColumnString dataPatch			= new DataColumnString();
	DataColumnString dataSecurity		= new DataColumnString();
	DataColumnString dataUpstream		= new DataColumnString();
	DataColumnString dataFixed			= new DataColumnString();
	DataColumnString dataPending			= new DataColumnString();
	DataColumnString dataHelp			= new DataColumnString();
	DataColumnString dataPotato			= new DataColumnString();
	DataColumnString dataWoody			= new DataColumnString();
	DataColumnString dataSarge			= new DataColumnString();
	DataColumnString dataSid				= new DataColumnString();
	DataColumnString dataExperimental	= new DataColumnString();
	DataColumnString dataD_I				= new DataColumnString();
	DataColumnString dataConfirmed		= new DataColumnString();
	DataColumnString dataIPV6			= new DataColumnString();
	DataColumnString dataLFS				= new DataColumnString();
	DataColumnString dataWontfix			= new DataColumnString();
	DataColumnString dataMoreinfo		= new DataColumnString();

	DataColumnString dataMerged			= new DataColumnString();
	DataColumnInt	dataFontWeight		= new DataColumnInt();
	DataColumnInt	dataComments		= new DataColumnInt();
	DataColumnString dataUnreproducible	= new DataColumnString();
	DataColumnPixbuf dataStatus			= new DataColumnPixbuf();
	DataColumnInt 	dataIntNum			= new DataColumnInt();
	DataColumnInt 	dataIntSeverity 	= new DataColumnInt();
	private void init(){

		makeView();

		HBox hb = new HBox(false, 0);
		filter = new Entry();
		hb.packStart( new Label( "Filter: "), false, false, 0);
		hb.packStart( filter, true, true, 0);

		filterBtn = new Button(GtkStockItem.APPLY);
		hb.packStart(filterBtn, false, false, 0);
		filterBtn.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event ){
				if (event.isOfType( ButtonEvent.Type.CLICK ) ){
					applyFilters();
				}
			}
		});
		filterBtn.setSensitive(false);
		
		clearFilterBtn = new Button(GtkStockItem.CLEAR);
		hb.packStart(clearFilterBtn, false, false, 0);
		clearFilterBtn.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event ){
				if (event.isOfType( ButtonEvent.Type.CLICK ) ){
					clearFilters();
				}
			}
		});
		clearFilterBtn.setSensitive(false);
		attnFilterBtn = new Button( "ATTN");
		hb.packStart(attnFilterBtn, false, false, 0);
		attnFilterBtn.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event ){
				if (event.isOfType( ButtonEvent.Type.CLICK ) ){
					filter( "ATTN" );
				}
			}
		});
		attnFilterBtn.setSensitive(false);

		rcFilterBtn = new Button("RC");
		hb.packStart(rcFilterBtn, false, false, 0);
		rcFilterBtn.addListener( new ButtonListener(){
			public void buttonEvent( ButtonEvent event ){
				if (event.isOfType( ButtonEvent.Type.CLICK ) ){
					filter( "RC" );
				}
			}
		});
		rcFilterBtn.setSensitive(false);

		packStart(hb, false, false, 0);

		packStart(sw);
		setState(ListingState.IDLE);
	}

	private void makeView( ){
		store = new ListStore( new DataColumn[] { 
			dataBugObject,
			dataNumber,
			dataTitle ,
			dataSeverity,
			dataCreated,
			dataTagList,
			dataIsForwarded	,
			dataPatch,
			dataSecurity,
			dataUpstream,
			dataFixed,
			dataPending,
			dataHelp,
			dataPotato,
			dataWoody,
			dataSarge,
			dataSid,
			dataExperimental,
			dataMerged,
			dataFontWeight,
			dataComments,
			dataUnreproducible,
			dataPackage,
			dataStatus,
			dataD_I,
			dataConfirmed,
			dataIPV6,
			dataLFS,
			dataMoreinfo,
			dataWontfix,
			dataIntNum,
			dataIntSeverity
		});

		view = new TreeView(store);
		view.setAlternateRowColor(true);
		view.setEnableSearch(true);
		view.setHeadersClickable(true);

		selection = view.getSelection();
		final ListingReportListener lst = listener;
		selListener = new TreeSelectionListener(){
			public void selectionChangedEvent(TreeSelectionEvent event){

				TreeSelectionForEach fe = new TreeSelectionForEach(){
					public void forEach(TreeModel model, TreePath path, TreeIter iter){
						Bug b = (Bug) store.getValue(iter, dataBugObject);
						lst.listingSelectionChanged(b);
					}
				};
				
				view.getSelection().forEachSelected(fe);
			}
		};
		selection.addListener( selListener );

		CellRendererPixbuf statusRend = new CellRendererPixbuf();
		colStatus.packStart(statusRend, true);
		colStatus.addAttributeMapping(statusRend, CellRendererPixbuf.Attribute.PIXBUF, dataStatus);
		colStatus.setResizable(true);
		colStatus.setReorderable(true);
		colStatus.setSortColumn( dataFontWeight );

		columns.add( new ListingColumn("Status", "", true, colStatus ) );
		
		
		colPackage.setTitle("Package");
		CellRendererText pkgRend = new CellRendererText();
		colPackage.packStart(pkgRend, true);
		colPackage.addAttributeMapping(pkgRend, CellRendererText.Attribute.TEXT, dataPackage);
		colPackage.setResizable(true);
		colPackage.setReorderable(true);
		colPackage.setSortColumn( dataPackage );
		//colPackage.setSizing( TreeViewColumnSizing.FIXED );
		columns.add( new ListingColumn("Package", "debbuggtk", false, colPackage ) );
		
		
		colNumber.setTitle("Bug #");
		CellRendererText numRend = new CellRendererText();
		colNumber.packStart(numRend, true);
		colNumber.addAttributeMapping(numRend, CellRendererText.Attribute.TEXT, dataNumber);
		colNumber.setResizable(true);
		colNumber.setReorderable(true);
		colNumber.setSortColumn( dataIntNum );
		columns.add( new ListingColumn("Bug Number", "123456", true, colNumber ) );
		
		colSeverity.setTitle("Severity");
		CellRendererText severityRend = new CellRendererText();
		colSeverity.packStart(severityRend, true);
		colSeverity.addAttributeMapping(severityRend, CellRendererText.Attribute.MARKUP, dataSeverity);
		colSeverity.setResizable(true);
		colSeverity.setReorderable(true);
		colSeverity.setSortColumn( dataIntSeverity );
		columns.add( new ListingColumn("Severity", "wishlist", true, colSeverity ) );
		
		
		colDesc.setTitle("Title");
		descRend = new CellRendererText();
		colDesc.packStart(descRend, true);
		colDesc.addAttributeMapping(descRend, CellRendererText.Attribute.TEXT, dataTitle);
		colDesc.setResizable(true);
		colDesc.setReorderable(true);
		colDesc.setSortColumn( dataTitle );
		view.setSearchColumn(colDesc);
		columns.add( new ListingColumn("Title", "Typo in man page", true, colDesc ) );
		
		colComments.setTitle("# comments");
		CellRendererText commentsRend = new CellRendererText();
		colComments.packStart(commentsRend, true);
		colComments.addAttributeMapping(commentsRend, CellRendererText.Attribute.TEXT, dataComments);
		colComments.setResizable(true);
		colComments.setReorderable(true);
		colComments.setSortColumn( dataComments );
		view.setSearchColumn(colComments);
		columns.add( new ListingColumn("Comment Count", "5", true, colComments ) );


		colCreated.setTitle("Created");
		CellRendererText createdRend = new CellRendererText();
		colCreated.packStart(createdRend, true);
		colCreated.addAttributeMapping(createdRend, CellRendererText.Attribute.TEXT, dataCreated);
		colCreated.setResizable(true);
		colCreated.setReorderable(true);
		colCreated.setSortColumn( dataCreated );
		view.setSearchColumn(colCreated);
		columns.add( new ListingColumn("Date Created", "23 Mar 2005", true, colCreated ) );
		
		makeTextColumn(colPatch, 	dataPatch, "Tag: Patch", "A",  true);
		makeTextColumn(colUpstream, dataUpstream, "Tag:Upstream", "U", true);
		makeTextColumn(colFixed , 	dataFixed, "Tag: Fixed", "", true);
		makeTextColumn(colPending , dataPending, "Tag: Pending", "P", true);
		makeTextColumn(colHelp , 	dataHelp, "Tag: Help", "H", true);
		makeTextColumn(colPotato , 	dataPotato, "Tag: Potato", "3", true);
		makeTextColumn(colWoody , 	dataWoody, "Tag: Woody", "2", true);
		makeTextColumn(colSarge , 	dataSarge, "Tag: Sarge", "1", true);
		makeTextColumn(colSid , 	dataSid, "Tag: Sid", "0", true);
		makeTextColumn(colExperimental , 	dataExperimental, "Tag: Experimental", "-1", true);
		makeTextColumn(colIsForwarded , 	dataIsForwarded, "Forwarded", "F", true);
		makeTextColumn(colUnreproducible , 	dataUnreproducible, "Tag: Unreproducible", "?", true);
		makeTextColumn( colSecurity, 		dataSecurity , "Tag: Security", "", true);
		makeTextColumn( colD_I, 			dataD_I, "Tag: Debian Installer", "", true );
		makeTextColumn( colConfirmed, 		dataConfirmed, "Tag: Confirmed", "", true );
		makeTextColumn( colIPV6, 			dataIPV6, "Tag: IPv6", "", true );
		makeTextColumn( colLFS, 			dataLFS, "Tag: LFS", "", true );
		makeTextColumn( colWontfix, 		dataWontfix, "Tag: WontFix", "X", true );
		makeTextColumn( colMoreinfo, 		dataMoreinfo, "Tag: MoreInfo", "", true );
		colMerged.setTitle("Merged");
		CellRendererText mergeRend = new CellRendererText();
		colMerged.packStart(mergeRend, true);
		colMerged.addAttributeMapping(mergeRend, CellRendererText.Attribute.TEXT, dataMerged);
		colMerged.setResizable(true);
		colMerged.setReorderable(true);
		colMerged.setSortColumn( dataMerged );
		view.setSearchColumn(colMerged);
		columns.add( new ListingColumn("Merged With", "123457 123459", true, colMerged ) );
		
		colDesc.addAttributeMapping( descRend, CellRendererText.Attribute.WEIGHT, 		dataFontWeight);
		colMerged.addAttributeMapping( mergeRend, CellRendererText.Attribute.WEIGHT, 	dataFontWeight);
		colSeverity.addAttributeMapping( severityRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		colNumber.addAttributeMapping( numRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		colCreated.addAttributeMapping( createdRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		colPackage.addAttributeMapping( pkgRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		colComments.addAttributeMapping( commentsRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		
		view.appendColumn( colFirst );
		colFirst.setVisible( false );
		for(int i = 0; i < columns.size(); i++){
			ListingColumn listingCol = (ListingColumn) columns.get(i);
			TreeViewColumn col = (listingCol).getColumn();
			view.appendColumn( col );
			col.setVisible( listingCol.getVisible() );
		}

		sw = new ScrolledWindow( new Adjustment(0,0,0,0,0,0), new Adjustment(0,0,0,0,0,0));
		sw.setPolicy( PolicyType.AUTOMATIC, PolicyType.AUTOMATIC );
		sw.add(view);
	}

	private void makeTextColumn( TreeViewColumn column, DataColumn data, String title, String example, boolean visible ){
		CellRendererText tagsRend = new CellRendererText();
		column.packStart(tagsRend, true);
		column.addAttributeMapping(tagsRend, CellRendererText.Attribute.TEXT, data);
		column.addAttributeMapping( tagsRend, CellRendererText.Attribute.WEIGHT, dataFontWeight);
		column.setResizable(true);
		column.setReorderable(true);
		column.setSortColumn( data);
		columns.add( new ListingColumn(title, example, visible, column ) );
	}

	private void setColumns(){
		TreeViewColumn prev = colFirst;
		for(int i = 0; i < columns.size(); i++){
			ListingColumn listingCol = (ListingColumn) columns.get(i);
			TreeViewColumn col = listingCol.getColumn();

			if (prev != null)
				view.moveColumn( col, prev );
			col.setVisible( listingCol.getVisible() );
			prev = col;
		}
	}

	/**
	 * Sets which columns should be displayed, and in what order.
	 */
	public void changeColumns( LinkedList columns ){
		this.columns = columns;
		setColumns();
	}

	public void showError(String txt){
		MessageDialog md = 
			new MessageDialog(parent, 
						DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), 
						MessageType.ERROR, 
						ButtonsType.CLOSE, 
						txt, true);
		md.run();
		md.destroy();
	}

	////////////////////////////////////////////////////////////////////////////
	// END OF GUI STUFF
	////////////////////////////////////////////////////////////////////////////

	public void editColumns(){
		ListingColumnEditor lc = new ListingColumnEditor( this, columns ); 
	}
	
	/**
	 * called when refresh is clicked
	 */
	public void update(){
		if (listing != null){
			Thread t = new Thread( new Runnable(){
				public void run(){
					listing.update();
				}
			});
			t.start();
		}
	}

	/**
	 * Sends a request for the update of thie list to stop
	 */
	public synchronized void stopUpdate(){
		if (constructorThread != null) constructorThread.interrupt();
	}

	private void clearDisplay(){
		if (constructorThread != null) 
			while (constructorThread.isAlive()){
				constructorThread.interrupt();
				try{
				wait(200);
				}catch (InterruptedException e){
					// do nothing
				}
			}
		if (listing != null){
			Object[] oldBugs = (listing.getBugs() == null) ? new Object[0] : listing.getBugs().keySet().toArray();
			for (int i = 0; i < oldBugs.length; i ++){
				if (iterList.containsKey( oldBugs[i] ) ){
					store.removeRow( (TreeIter) iterList.get(oldBugs[i]));
					iterList.remove( oldBugs[i]) ;
				}
			}
		}
		// just in case any remain...
		store.clear();
	}
	/**
	 * sets the package to show listings for
	 */
	public synchronized void setPackage(ListingStub listingStub){
		if (DebbugGtk.DEBUG) System.out.println("ListingReport.setPackage( "+listingStub+")");

		if (listing != null){

			if ( listingStub.equals((ListingStub) listing) ){
				// same listing as already loaded - don't load again
				return;
			}
			listing.removeListener( (ListingListener) this );

		}
		clearDisplay();

		listener.listingChanged( listingStub );
		listing = BTS.getInstance().getListing(  listingStub.getType(), listingStub.getName());
		
		// attach ourselves as a listener for this listing
		if (DebbugGtk.DEBUG) System.out.println("***Adding ListingReport as a listener000");
		listing.addListener( (ListingListener) this );

		// if listing has bugs in it, display them all
		if (listing.getComplete()){
				if (DebbugGtk.DEBUG) System.out.println("Setpackage - listing has bugreports.");

			constructorObject =  new Runnable() { public void run(){ listing.loadReports(); } };
		}else{
				if (DebbugGtk.DEBUG) System.out.println("Setpackage - listing is incomplete - downloading.");
			// if listing not downloaded, download it
			opNum++;
			constructorObject =  new Runnable() { public void run(){ listing.update(); } };
		}
		constructorThread = new Thread( constructorObject );
		constructorThread.setName( "ListingConstructor");
		constructorThread.setDaemon(true);
		constructorThread.start();
	if (DebbugGtk.DEBUG) System.out.println("000");
	}

	private void clearFilters(){
		filter.setText("");
		applyFilters();
	}
	private void filter( String filterText ){
		filter.setText( filterText );
		applyFilters();
		filter.setText( filterer.toString() );
	}

	// To be called from constructor thread!
	private void filterReports(){
		if (listing != null){
			Hashtable bugtable = listing.getBugs();
			Object[] bugs = (bugtable == null) ? new Object[0] : bugtable.keySet().toArray();
			for (int i = 0; i < bugs.length; i ++){
				Bug bug = BTS.getInstance().getBug( (Integer) bugs[i] );

				if (!iterList.containsKey( bugs[i] ) ){
					// bugAdded checks filter and only adds if valid.
					bugAdded( listing, bug );
				}
				if ( iterList.containsKey( bugs[i] ) 
						&& !filterer.acceptable(bug) ) {
					bugRemoved( listing, (Integer) bugs[i] );
				}
			}
		}
		flushCustomEventQueues();
	}
	
	private void applyFilters(){
		synchronized(constructorThread){
		try{
			if (constructorThread != null) 
			while (constructorThread.isAlive()){
				constructorThread.interrupt();
				try{
					constructorThread.wait(200);
					System.out.println("Constructor thread is still alive -- interrupted & waiting");
				}catch (InterruptedException e){
					// do nothing
				}
			}
				if (DebbugGtk.DEBUG) System.out.println("ApplyFilters");
			filterer = BugFilterer.makeFilterer(filter.getText());
				if (DebbugGtk.DEBUG) System.out.println("ApplyFilters");
			constructorObject =  new Runnable() { public void run(){ filterReports(); } };
			constructorThread = new Thread( constructorObject );
			constructorThread.setName( "ListingConstructor - filtering");
			constructorThread.setDaemon(true);
			constructorThread.start();
		}catch( BugFilterInvalid e ){
			MessageDialog md = new MessageDialog(parent, DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), MessageType.ERROR, ButtonsType.CLOSE, 
							"Invalid Filter: " + e.getMessage(), true);
			md.run();
			md.destroy();
		}
		}
	}

	private int opNum = 0;
	
	private 	DateFormat dateFormat = new SimpleDateFormat( "yyyy-MM-dd" );

	public void addRow( Bug bug, int opNum ){
		if (opNum < this.opNum) 
			return;
		if (DebbugGtk.DEBUG) System.out.println("DEBUG: Adding listing row: "+bug.getNumber() );
		TreeIter iter = store.appendRow();
		bug.setData( iter );
		iterList.put( new Integer( bug.getNumber() ), iter );
		store.setValue(iter, dataBugObject, (Object) bug);
		store.setValue(iter, dataNumber, Integer.toString(bug.getNumber()) );
		store.setValue(iter, dataPackage, bug.getPackageName() );
		store.setValue(iter, dataTitle, bug.getTitle() );
		String severity;
		Severity s = bug.getSeverity();
		if (s == null){
			System.out.println("ERROR severity == null");
			(new Exception()).printStackTrace();
			store.setValue(iter, dataSeverity, "Error - unknown" );
		}else{

			if( // RC bug
				s.equals( Severity.CRITICAL )
				|| s.equals( Severity.GRAVE )
				|| s.equals( Severity.SERIOUS )
				)
				severity = "<span foreground=\"#ff0000\">"+ s.toString() +"</span>";
			else
				severity = s.toString();

			store.setValue(iter, dataSeverity, severity );
		}
		store.setValue(iter, dataCreated, dateFormat.format(new Date( bug.getCreated() )) );
		
		// fixme: use better text
		store.setValue(iter, dataTagList, "tags unimplemented" ) ; // bug.getTagsList() );
		store.setValue(iter, dataIsForwarded, bug.getIsForwarded() ? "F" : "" );
		store.setValue(iter, dataPatch, bug.getTagPatch() ? "A" : "" );
		store.setValue(iter, dataSecurity, bug.getTagSecurity() ? "S" : "" );
		store.setValue(iter, dataUpstream, bug.getTagUpstream() ? "U" : "" );
		store.setValue(iter, dataFixed, bug.getTagFixed() ?  "N" : "" );
		store.setValue(iter, dataPending, bug.getTagPending() ? "P" : "" );
		store.setValue(iter, dataHelp, bug.getTagHelp() ? "H" : "" );
		store.setValue(iter, dataPotato, bug.getTagPotato() ? "3" : "" );
		store.setValue(iter, dataWoody, bug.getTagWoody() ? "2" : "" );
		store.setValue(iter, dataSarge, bug.getTagSarge() ? "1" : "" );
		store.setValue(iter, dataSid, bug.getTagSid() ? "0" : "" );
		store.setValue(iter, dataExperimental, bug.getTagExperimental() ? "-1" : "" );

		store.setValue( iter, dataD_I, bug.getTagD_I() ? "d-i" : "");
		store.setValue( iter, dataConfirmed , bug.getTagConfirmed () ? "Confirmed" : "");
		store.setValue( iter, dataIPV6, bug.getTagIPV6() ? "IPv6" : "");
		store.setValue( iter, dataLFS, bug.getTagLFS() ? "LFS" : "");
		store.setValue( iter, dataWontfix, bug.getTagWontfix() ? "X" : "");
		store.setValue( iter, dataMoreinfo, bug.getTagMoreinfo() ? "Moreinfo" : "");

		int[] merged = bug.getMerged();
		String merge = "";
		for(int i = 0; i < merged.length; i++){
			if (i > 0){
				merge += " "+merged[i];
			}else
				merge += merged[i];
		}
		store.setValue(iter, dataMerged, merge);
		Status status =  bug.getStatus();
		store.setValue(iter, dataFontWeight, (status.equals(Status.NEW) || status.equals(Status.UPDATED)) ? 900 : 400);
		store.setValue(iter, dataComments, bug.getCommentCount() );
		store.setValue(iter, dataUnreproducible, bug.getTagUnreproducible() ? "?" : "" );
		store.setValue(iter, dataIntNum, bug.getNumber() );
		store.setValue(iter, dataIntSeverity, bug.getSeverity().getValue() );
		if (status.equals(Status.NEW) || status.equals(Status.UPDATED)){
			if (bug.getIsDone())
				store.setValue(iter, dataStatus, UNREAD_DONE);
			else
				store.setValue(iter, dataStatus, UNREAD);
		}else{
			if (bug.getIsDone())
				store.setValue(iter, dataStatus, READ_DONE);
			else
				store.setValue(iter, dataStatus, READ);
		}
	}

	public ListingState getState(){
		return state;
	}

	public void setState( ListingState state ){
		this.state = state;
		filterBtn.setSensitive(true);
		clearFilterBtn.setSensitive(true);
		attnFilterBtn.setSensitive(true);
		rcFilterBtn.setSensitive(true);
		if (statusListener != null)
			statusListener.listingStatusChanged(listing);
		listener.listingReportStateChanged( state );
	}
		
	public ListingReport(BTS bts, Listing pkg, ListingReportListener listener, Window parent){
		super( false, 0 );
		this.bts = bts;
		this.parent = parent;
		this.listener = listener;
		init();
		setPackage(pkg);
	}
	
	public ListingReport( ListingReportListener listener, Window parent){
		super( false, 0 );
		this.parent = parent;
		this.bts = BTS.getInstance();
		this.listener = listener;
		init();
	}

	public ListingReport(BTS bts, ListingReportListener listener, Window parent){
		super( false, 0 );
		this.parent = parent;
		this.bts = bts;
		this.listener = listener;
		init();
	}

	public void setStatusListener( ListingStatusListener statusListener ){
		this.statusListener = statusListener;
	}

	public Listing getListing(){
		return listing;
	}

	
	/**
	 * Called whenever the numbers of bugs of each type (including number of read
	 * bugs) changes. If this is likely to happen a lot in a short period of
	 * time (e.g. when loading a listing from disk), it may not be called for
	 * every change.
	 */
	public void bugCountsChanged( Listing listing ){
		// we're not interested in this one
	}

	/**
	 * Called when the listing is about to contact the server to perform an
	 * update.
	 */
	public void listingUpdateStart( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("ListingReport - listingUpdateStart");
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( ListingState.UPDATING );
			}
		});
	}

	/**
	 * Called when the listing has completed it's interactions with the server.
	 */
	public void listingUpdateDone( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("ListingReport - listingUpdateDone");
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( ListingState.IDLE );
			}
		});
	}

	/**
	 * Called occasionally when downloading a package listing. This may be used
	 * as part of a progress screen. 
	 */
	public void downloadingListing( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("downloadinglisting event");
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( ListingState.DLLISTING );
			}
		});
	}
	public void interpretingListing( Listing listing ){
		if (DebbugGtk.DEBUG) System.out.println("downloadinglisting event");
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( ListingState.INTERPRET_LISTING );
			}
		});
	}

	/**
	 * Called at the start of te download of each bug report from the server as
	 * part of a listing update process.
	 * @param bugNumber The number of the bug being downloaded
	 * @param pending Number of bugs downloaded so far + 1
	 * @param total Number of bugs to be downloaded. Note that this is not the
	 * number of bugs in the listing, merely the number of bugs which need
	 * updating.
	 */
	public void downloadingListingBug( Listing listing, Integer bugNumber, int pending, int total){
		final ListingState s =  ListingState.REFRESHING;
		s.setMax( total );
		s.setBug( bugNumber.intValue() );
		s.setCurrent( pending );
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( s );
			}
		});
	}

	/**
	 * Called when a bug is removed from the listing (e.g. when the server says
	 * the bug has been fixed for a list of open reports)
	 */
	public void bugRemoved( Listing listing, Integer bugNumber ){
		
		if (iterList.containsKey(bugNumber)){
			
			synchronized( customEventRemoveQueue ){
				customEventRemoveQueue.addLast( bugNumber );
			}

			if (System.currentTimeMillis() - customEventTime > 100){
				flushCustomEventQueues();
			}
		}
	}

	/**
	 * Called when a bug is added to the listing for the first time.
	 */
	public void bugAdded( Listing listing, Bug bug ){
		if (DebbugGtk.DEBUG) System.out.println("Bug Added event - ListingReport");
		if (!filterer.acceptable(bug))
			return;
	
		synchronized( customEventAddQueue ){
			customEventAddQueue.addLast( bug );
		}

		if (System.currentTimeMillis() - customEventTime > 100){
			flushCustomEventQueues();
		}
	}
	
	private void flushCustomEventQueues(){
		final Object[] addListCopy;
		final Object[] removeListCopy;
		synchronized( customEventAddQueue ){
			addListCopy = customEventAddQueue.toArray();
			customEventAddQueue.clear();
		}
		synchronized( customEventRemoveQueue ){
			removeListCopy = customEventRemoveQueue.toArray();
			customEventRemoveQueue.clear();
		}
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				for (int i = 0; i < addListCopy.length; i++){
					bugAddedHandler( (Bug) addListCopy[i] );
				}
				for (int i = 0; i < removeListCopy.length; i++){
					bugRemovedHandler( (Integer) removeListCopy[i] );
				}
			}
		});
		customEventTime = System.currentTimeMillis();
	}
	
	public void loadReportsDone(com.tildemh.debbug.Listing listing){
		flushCustomEventQueues();
	}
	

	private void bugAddedHandler( Bug bug ){
		if ( !iterList.containsKey( new Integer( bug.getNumber() ) ) )
		addRow( bug, opNum );
	}

	private void bugRemovedHandler( Integer bugNum ){
		store.removeRow( (TreeIter) iterList.get(bugNum));
		iterList.remove( bugNum ) ;
	}

	/**
	 * Called when a modification is made to a bug in the listing (including
	 * change of read/unread status) (excluding addition or removal of bugs).
	 */
	public void bugChanged( Listing listing, Bug bug ){
		if (DebbugGtk.DEBUG) System.out.println("ListingReport - BugChanged event");
		Integer bugNum = new Integer( bug.getNumber() );
		// previously acceptable bug is no longer acceptable.
		if (!filterer.acceptable(bug) && iterList.containsKey(bugNum)){
			final Integer bugNumber = bugNum;
			CustomEvents.addEvent( new Runnable(){
				public void run(){
					store.removeRow( (TreeIter) iterList.get(bugNumber) );
				}
			});
			return;
		}
		
		// previously unacceptable bug is now acceptable
		if (filterer.acceptable(bug) && !iterList.containsKey(bugNum)){
			bugAdded(listing,bug);
			return;
		}
		
		// previously acceptable bug is still acceptable - record needs updating
		if (filterer.acceptable(bug) && iterList.containsKey(bugNum)){
			Status status =  bug.getStatus();
			Severity severity = bug.getSeverity();
			if ( status.equals(Status.NEW) || status.equals(Status.UPDATED) ){
				if (iterList.containsKey(bugNum)){
					store.setValue((TreeIter) iterList.get(bugNum) , dataFontWeight,  900);
					store.setValue((TreeIter) iterList.get(bugNum) , dataStatus,  UNREAD);
				}else{
					System.out.println("*** ERROR - Bug not in iterList");
					(new Exception()).printStackTrace();
				}
			}else{
				if (iterList.containsKey(bugNum)){
					store.setValue((TreeIter) iterList.get(bugNum) , dataFontWeight,  400);
					store.setValue((TreeIter) iterList.get(bugNum) , dataStatus,  READ);
				}else{
					System.out.println("*** ERROR - Bug not in iterList");
					(new Exception()).printStackTrace();
				}
			}
			return;
		}
		
		System.err.println("ListingReport.bugChanged - something's gone wrong");
		(new Exception()).printStackTrace();
	}

	/**
	 * Called when an exception occurred while updating the report. 
	 * @return true if you want the execution to continue, false otherwise.
	 */
	public boolean listingException( Listing listing, Exception e ){
		final Exception f = e;
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setState( ListingState.ERROR );
				showError( "listingException: "+f.toString() );
			}
		});
		return false;
	}

}

