/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * DdtManager.h
 *   Class that calls database and dns routines. This class is an
 *   abstraction layer that allows us to use any implementation of
 *   a database and dns backend.
 *
 * $Id: DdtManager.h,v 1.14 2003/04/24 04:44:56 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@dhis.net>
 * Copyright (c) 2000 Benoit Joly <benoit@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#ifndef DDTMANAGER_H
#define DDTMANAGER_H

//#include <map> // for 0.6

// general defines
#include "global.h"

#include "Logger.h"
#include "Dns.h"
#include "Db.h"

/*
struct runtimeInfo
{
    unsigned long addr;
    // etc ...
};
*/

class DdtManager
{
  public:
    /**
     * class constructor
     * @param log a pointer to an instance of a Logger class
     * @param db  a pointer to an instance of one of the database classes
     * @param dns a pointer to an instance of one of the dns classes
     */
    DdtManager(Logger *log, Db *db, Dns *dns);

    /**
     * class destructor
     */
    ~DdtManager(); 


    /**************  methods for user accounts  **************/

    /**
     * add a user account
     * @param account a pointer to a UserAccount structure
     * @return the user id for the account
     */
    int addUserAccount(UserAccount *account);

    /**
     * delete a user account
     * @param id the user id for the account to be delete
     */
    void delUserAccount(int id);

    /**
     * log a user on to the system
     * @param id the user id for the account that is logging on to the system
     */
    void logonUserAccount(int id);

    /**
     * log a user off of the system
     * @param id the user id for the account that is logging off of the system
     */
    void logoffUserAccount(int id);

    /**
     * update the IP address associated with a user account
     * @param id the user id for the account having its IP address updated
     * @param ip the IP address with which to update the account
     */
    void updateIPAddress(int id, unsigned long ip);

    /**
     * update the time stamp associated with a user account
     * @param id the user id for the account having its time stamp updated
     */
    void updateTimeStamp(int id);

    /**
     * Modify a field in a user accounts table.
     * @param account a pointer to a UserAccount structure
     */
    void modifyUserAccount(UserAccount *account);



    /****************** methods for dns records *****************/
    
    /**
     * add a DNS record and associate it with a user account
     * @param id the user id for the account owning this dns record
     * @param dname the 'left hand side' of the dns record
     * @param type the dns record type
     * @param data the 'right hand side' of the dns record
     */
    void addDnsRecord(int id, const char *dname, DnsRecordType type,
                      const char *data);

    /**
     * delete a DNS record and dissociate it from a user account
     * @param id the user id for the account owning this dns record
     * @param dname the 'left hand side' of the dns record
     * @param type the dns record type
     * @param data the 'right hand side' of the dns record
     */
    void delDnsRecord(int id, const char *dname, DnsRecordType type,
                      const char *data);

    /**
     * delete all the DNS records associated with a user account
     * @param id the user id for the account that is having its
     *           dns records deleted
     */
    void delDnsRecords(int id);


    /**
     * Print DNS records to stdout
     * @param id userAccountId of account to list dns records
     */
    void listDnsRecords(int id, std::vector<DnsRecord> &dnsRecordList);


    /***************    compatibility methods only    ****************/
    int markOnline(int id, unsigned int ip) 
        { updateIPAddress(id, ip); return 0; }

    int markOffline(int id) { logoffUserAccount(id); return 0; }

    int markAlive(int id) { updateTimeStamp(id); return 0; }

    bool getAcctInfo (int id, userAccount *account)
        {
            db->fetchAccountInfo(id, account);
            return 0;
        }

    bool fetchAccountInfo(int id, char *field, char *dest, int length)
        {
            UserAccount a;
            db->fetchAccountInfo(id, &a);

            if (strcasecmp(field, "adminPassword") == 0)
                strncpy(dest, a.adminPassword, length);
            else if (strcasecmp(field, "updatePassword") == 0)
                strncpy(dest, a.updatePassword, length);
            else if (strcasecmp(field, "contactName") == 0)
                strncpy(dest, a.contactName, length);
            else if (strcasecmp(field, "contactEmail") == 0)
                strncpy(dest, a.contactEmail, length);
            else if (strcasecmp(field, "arch") == 0)
                strncpy(dest, a.arch, length);
            else if (strcasecmp(field, "os") == 0)
                strncpy(dest, a.os, length);
            else if (strcasecmp(field, "fqdn") == 0)
                strncpy(dest, a.fqdn, length);
            else if (strcasecmp(field, "ipAddress") == 0)
                strncpy(dest, a.ipAddress, length);
            else
                return false;

            return true;
        }

    int findUserAccountIdFromFqdn(char *fqdn)
        { 
            return db->findUserAccountIdFromFqdn(fqdn);
        }

    int modUserAccount(int id, char *field, char *value)
        {
            db->modUserAccount(id, field, value);
            return 0;
        }

    int pruneActiveAccount(void (*func)(int, unsigned long, time_t)) 
        { return db->pruneActiveAccount(func); }

    unsigned long fetchAcctAddr(int uid) { return db->fetchAcctAddr(uid); }


  protected:
    Logger *log;
    Db *db;
    Dns *dns;

//    map<int, runtimeInfo> onlineAccounts; // for 0.6 only
};

#endif // DDTMANAGER_H
