/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * DbPsql.cc
 *   Class to handle postgresql database.
 *
 * $Id: DbPsql.cc,v 1.21 2003/04/24 04:44:56 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include <iostream>
#include <sstream>
#include <string>

#include <arpa/inet.h>

#include "Exception.h"
#include "DbPsql.h"

using std::ostringstream;

DbPsql::DbPsql()
{
    this->log = NULL;
    this->pg = NULL;
}

DbPsql::DbPsql(Logger *log, const char *dbname, const char *dbuser,
               const char *dbpass)
{
    this->log = log;

    ostringstream conninfo;
    ostringstream query;
    
    conninfo << "dbname=" << dbname
             << " user=" << dbuser
             << " password=" << dbpass << std::ends;
    
    if ((pg = new PgDatabase (conninfo.str().c_str())) == NULL)
    {
        throw DdtException("Can't allocate memory.");
    }

    if (pg->Status() == CONNECTION_BAD)
    {
        throw DbError("Failed to open database.");
    }

// bleh, that doesn't work, any clue why?  FIXME!
//    query << "SELECT * FROM useraccounts, dnsrecords LIMIT 1" << ends;
//    if (pg->ExecCommandOk(query.str().c_str()) == false)
//    {
//        throw DbError("Invalid database type");
//    }
}

DbPsql::~DbPsql()
{
    if (pg) delete pg;
}

/* add an account to the database */
int DbPsql::addUserAccount (UserAccount *account)
{
    ostringstream updateQuery;
    int id;
    
    if ((id = allocateUserAccountId())== -1)
    {
        throw DdtException("can't allocate a new account id");
    }

    updateQuery << "UPDATE userAccounts SET"
                << " adminPassword='" << account->adminPassword << "',"
                << " updatePassword='" << account->updatePassword << "',"
                << " contactName='" << account->contactName << "',"
                << " contactEmail='" << account->contactEmail << "',"
                << " arch='" << account->arch << "',"
                << " os='" << account->os << "',"
        // new accounts are OFFLINE
                << " hostStatus='" << OFFLINE << "',"
                << " fqdn='" << account->fqdn << "',"
                << " ipAddress='" << account->ipAddress << "'"
                << " WHERE userAccountId=" << id << std::ends;
             
    // fill database
    if (pg->ExecCommandOk (updateQuery.str().c_str()) == false)
    {
        ostringstream msg;
        
        msg << "DbPsql::addUserAccount(): pg->ExecCommandOk (" 
            << updateQuery.str().c_str() << ") " << pg->ErrorMessage() << std::ends;

        log->debug("%s", msg.str().c_str());
        delUserAccount(id);
        throw DbError(msg.str().c_str());
    }
    
    return id;
}

/* allocate a free useraccountid */
int DbPsql::allocateUserAccountId ()
{
    ostringstream query;
    ostringstream insertQuery;
    int id = -1;

    // find a free useraccountid
    query << "SELECT useraccountid FROM useraccounts u1 WHERE "
          << "NOT EXISTS(SELECT * FROM useraccounts u2 WHERE "
          << "u1.useraccountid+1=u2.useraccountid) "
          << "AND userAccountId>=1000 ORDER BY userAccountId LIMIT 1" << std::ends;

    if (pg->ExecTuplesOk(query.str().c_str()) == false)
    {
        log->debug("Psql::allocateUserAccountId (): pg->ExecTuplesOk (%s) "
                   "%s", query.str().c_str(), pg->ErrorMessage ());

        throw DbError("DbPsql::allocateUserAccountId(): select query "
                      "failed.");
    }

    if (pg->Tuples() == 0)
    {
        // no user accounts in the database... this is not an error
        // return value 1000 indicating first user account
        id = 1000;
    }
    else if (pg->Tuples () > 1 || pg->Fields () > 1)
    {
        log->debug ("Psql::allocateUserAccountId (): more than one tuple "
                    "or one field returned by query!");
        throw DbError("DbPsql::allocateUserAccountId(): weird response "
                      "from db");
    }
    else
    {
        id = atoi (pg->GetValue (0,0)) + 1;
    }

    // add new accountid into database
    insertQuery << "INSERT into userAccounts (userAccountId) values ("
                << id << ")" << std::ends;

    if (pg->ExecCommandOk (insertQuery.str().c_str()) == false)
    {
        log->debug ("DbPsql::allocateUserAccountId(): pg->ExecCommandOk "
                    "(%s) %s", insertQuery.str().c_str(), pg->ErrorMessage ());
        throw DbError("DbPsql::allocateUserAccountId(): insert for new "
                      "account failed");
    }

    return id;
}

// remove user account
void DbPsql::delUserAccount (int id)
{
    ostringstream query;
    query << "DELETE FROM userAccounts WHERE userAccountId=" << id << std::ends;
    
    if (pg->ExecCommandOk(query.str().c_str()) == false)
    {
        throw DbError("Psql::delUserAccount(): delete query failed");
    }
}

bool DbPsql::accountIdExists(int id)
{
    ostringstream request;

    request << "SELECT * from userAccounts where userAccountId=" 
            << id << std::ends;


    if (!pg->ExecTuplesOk (request.str().c_str()))
    {
        return false;
    }

    if (pg->Tuples () == 1)
    {
        return true;
    }

    return false;
}

int DbPsql::getNbOfDnsRecords(int accountId, DnsRecordType type)
{
    ostringstream query;

    // first check that 'id' doesn't have more than MAX records already
    // registered
    
    query << "SELECT * FROM dnsrecords WHERE type='" << type << "' AND "
          << "userAccountId='" << accountId << "'" << std::ends;

    if (pg->ExecTuplesOk (query.str().c_str()) == false)
    {
        log->debug("Psql::getNbOfDnsRecords(): error executing query");
        return -1;
    }

    return pg->Tuples();
}

int DbPsql::pruneActiveAccount(void (*func)(int, unsigned long, time_t))
{
    ostringstream query;

    typedef struct onlineAccount
    {
        int accountId;
        unsigned long addr;
        time_t db_time;
    } onlineAccount;

    int i;

    query << "SELECT userAccountId, ipAddress, date_part('epoch', lastAccess) "
          << "FROM userAccounts WHERE ipAddress <> '169.254.0.0' AND "
          << "hostStatus != 3" << std::ends;
    // FIXME: FIXME: FIXME: this should be a define for the static status 

    // If there are any tuples, the Fqdn is not valid.
    if (pg->ExecTuplesOk(query.str().c_str()))
    {
        if (pg->Tuples() == 0)
        {
            log->debug("no host qualify");
            return 0;
        }
        
        onlineAccount accts[pg->Tuples()];

        log->debug("%d hosts qualifies.", 
                   sizeof (accts) / sizeof (onlineAccount));
        
        // fill the structure with online hosts
        for (i = 0; i < pg->Tuples(); i++)
        {
            accts[i].accountId = atoi(pg->GetValue(i, 0));
            accts[i].addr = inet_addr(pg->GetValue(i,1));
            accts[i].db_time = atol(pg->GetValue(i, 2));
            log->debug("(id %d) (addr %d) (stamp %d)", accts[i].accountId,
                       accts[i].addr, accts[i].db_time);
        }
        
        for (unsigned int i = 0; 
             i < (sizeof(accts)/sizeof(onlineAccount)); i++)
        {
            if (func != NULL)
            {
                func(accts[i].accountId, accts[i].addr, accts[i].db_time);
            }
        }
        return 0;
    }
    return -1;
}

void DbPsql::modUserAccount(int id, char *field, char *value)
{
    ostringstream query;

    query << "UPDATE userAccounts SET " << field << "='" << value << "' "
          << "WHERE userAccountId='" << id << "'" << std::ends;
    
    if (pg->ExecCommandOk(query.str().c_str()) == false)
    {
        throw DbError("Psql::modUserAccount (): query failed");
    }
}

void DbPsql::modUserAccount(int id, char *field, int value)
{
    ostringstream query;

    query << "UPDATE userAccounts SET " << field << "='" << value << "' "
          << "WHERE userAccountId='" << id << "'" << std::ends;
    
    if (pg->ExecCommandOk(query.str().c_str()) == false)
    {
        throw DbError("Psql::modUserAccount (): query failed");
    }
}


void DbPsql::addDnsRecord(int accountId, const char *dname,
                          DnsRecordType type, const char *data)
{
    ostringstream query;
    ostringstream insert;
    
    log->debug("Adding DNS record.");

    // check if the accountId exists
    if (accountIdExists(accountId) == false)
    {
        log->debug("DbPsql::addDnsRecord: account doesn't exists");
        throw DbError("DbPsql::addDnsRecord: account doesn't exists");
    }

    int nbDnsRecords = getNbOfDnsRecords(accountId, type);
    
    // FIXME: max MX/CNAME/NS shouldn't be harcoded in the database
    //        driver. that's very very bad!
    switch (type)
    {
    case CNAME:
        if (nbDnsRecords >= 5)
        {
            std::cout << "Too many CNAMEs" << std::endl;
            log->debug("DbPsql::addDnsRecord (): Nb of CNAME allowed is 5");
            throw DdtException("DbPsql::addDnsRecord (): Nb of CNAME "
                               "allowed is 5");
        }
        
        // check if the requested record is available
        query << "SELECT * FROM userAccounts u, dnsRecords d WHERE "
              << "u.fqdn='" << dname << "' OR d.dname='" << dname
              << "' AND d.type=" << CNAME << std::ends;

        if (pg->ExecTuplesOk(query.str().c_str()) == false)
        {
            log->debug("DbPsql::addDnsRecord(): pg->ExecCommandOk "
                       "(%s) %s", query.str().c_str(), pg->ErrorMessage());
            throw DbError("DbPsql::addDnsRecord(): select query failed");
        }

        if (pg->Tuples() > 0)
        {
            log->debug("dname %s not available!", dname);
            throw DdtException("dname not available");
        }
        break;

    case MX:
        if (nbDnsRecords >= 2)
        {
            std::cout << "Too many MXs" << std::endl;
            log->debug ("DbPsql::addDnsRecord(): Nb of MX allowed is 2");
            throw DdtException("DbPsql::addDnsRecord(): Nb of MX "
                               "allowed is 2");
        }
        break;

    default:
        log->debug("DbPsql::addDnsRecord(): DnsRecordType unknown");
        throw DdtException("DbPsql::addDnsRecord(): DnsRecordType unknown");
        break;
    }
                
    // update the database
    insert << "INSERT INTO dnsRecords (useraccountid, dname, type, data) "
           << "VALUES ('" << accountId << "', '" << dname << "', '"
           << type << "', '" << data << "')" << std::ends;
    
    if (pg->ExecCommandOk(insert.str().c_str()) == false)
    {
        log->debug("DbPsql::addDnsRecord(): insert failed: %s",
                   pg->ErrorMessage());
        throw DbError("DbPsql::addDnsRecord(): insert failed");
    }
}


/* remove all dns records for an host */
void DbPsql::delDnsRecords(int id)
{
    log->debug("Removing all DNS records.");
  
    ostringstream query;

    typedef struct recs
    {
        const char dname[256];
        DnsRecordType type;
        const char data[256];
    };

    query << "SELECT * FROM dnsRecords WHERE userAccountId='"
          << id << "'" << std::ends;

    if (pg->ExecTuplesOk(query.str().c_str()) == false)
    {
        log->debug("DbPsql::delDnsRecords(): no dns records for this host");
    }

    ostringstream str;
    int nbTuples = pg->Tuples();
    recs r[nbTuples];
    str << "DbPsql::delDnsRecords (): nb of dns records for host " << id
        << " = " << nbTuples << std::ends;
    log->debug("%s", str.str().c_str());


    /* we need to first build a list and then loop through it because
     * GetValue won't work properly after running delDnsRecord() which
     * does some psql queries too ... !lart benoit */
    for (int recNo = 0; recNo < nbTuples; recNo++)
    {
        strncpy(const_cast<char *>(r[recNo].dname),
                pg->GetValue(recNo, "dname"), sizeof(r[recNo].dname));
        r[recNo].type = (DnsRecordType) atoi(pg->GetValue(recNo, "type"));
        strncpy(const_cast<char *>(r[recNo].data),
                pg->GetValue(recNo, "data"), sizeof(r[recNo].data));
    }

    for (int recNo = 0; recNo < nbTuples; recNo++)
    {
        delDnsRecord(id, r[recNo].dname, r[recNo].type, r[recNo].data);
    }
}

void DbPsql::delDnsRecord(int id, const char *dname, DnsRecordType type,
                          const char *data)
{
    log->debug("Removing DNS record.");

    ostringstream query;
    ostringstream str;

    str << "dname: " << dname << " data: " << data << " type: "
        << type << std::ends;

    log->debug("%s", str.str().c_str());

    // update the database
    query << "DELETE FROM dnsRecords WHERE "
          << "userAccountId='" << id << "' "
          << "AND dname='" << dname << "' "
          << "AND type='" << type << "' "
          << "AND data='" << data << "'" << std::ends;

    if (pg->ExecCommandOk (query.str().c_str()) == false)
    {
        log->debug("DbPsql::delDnsRecord(): pg->ExecCommandOk (%s) %s",
                   query.str().c_str(), pg->ErrorMessage());
        throw DbError("DbPsql::delDnsRecord(): error while executing "
                      "delete query");
    }
    return;
}

/* do we need such a function? its darn ugly. */
void DbPsql::listDnsRecords(int id, std::vector<DnsRecord> &dnsRecordList)
{
    ostringstream query;

    query << "SELECT * FROM dnsRecords WHERE userAccountId='"
          << id << "'" << std::ends;

    if (pg->ExecTuplesOk(query.str().c_str()) == false)
    {
        log->debug("DbPsql::delDnsRecords(): no dns records for this host");
    }

    ostringstream str;
    int nbTuples = pg->Tuples();
    str << "DbPsql::delDnsRecords (): nb of dns records for host " << id
        << " = " << nbTuples << std::ends;
    log->debug("%s", str.str().c_str());


    DnsRecord record;
    
    for (int recNo = 0; recNo < nbTuples; recNo++)
    {

        record.dnsRecordId = atoi(pg->GetValue(recNo, "dnsRecordId"));
        record.userAccountId = atoi(pg->GetValue(recNo, "userAccountId"));
        record.type = atoi(pg->GetValue(recNo, "type"));
        strncpy(record.dname, pg->GetValue(recNo, "dname"), sizeof(record.dname));
        strncpy(record.data, pg->GetValue(recNo, "data"), sizeof(record.data));

        dnsRecordList.push_back(record);
    }
}

bool DbPsql::fetchAccountInfo (int id, UserAccount *account)
{
    ostringstream query;

    query << "SELECT userAccountId, adminPassword, updatePassword, "
          << "contactName, contactEmail, arch, os, hostStatus, fqdn, "
          << "ipAddress from userAccounts WHERE userAccountId=" << id << std::ends;

    if (!pg->ExecTuplesOk (query.str().c_str()) || pg->Tuples() != 1)
    {
        log->debug("DbPsql::fetchAccountInfo(): unknown accountId");
        return false;
    }

    int size;
    account->userAccountId = atoi(pg->GetValue(0,0));
    
    size = sizeof(account->adminPassword);
    strncpy(account->adminPassword, (pg->GetValue(0,1)), size);
    
    size = sizeof(account->updatePassword);
    strncpy(account->updatePassword, (pg->GetValue(0,2)), size);
    
    size = sizeof(account->contactName);
    strncpy(account->contactName, (pg->GetValue(0,3)), size);
    
    size = sizeof(account->contactEmail);
    strncpy(account->contactEmail, (pg->GetValue(0,4)), size);
    
    size = sizeof(account->arch);
    strncpy(account->arch, (pg->GetValue(0,5)), size);
    
    size = sizeof(account->os);
    strncpy(account->os, (pg->GetValue(0,6)), size);
    
    account->hostStatus = atoi(pg->GetValue(0,7));
    
    size = sizeof(account->fqdn);
    strncpy(account->fqdn, (pg->GetValue(0,8)), size);
    
    size = sizeof(account->ipAddress);
    strncpy(account->ipAddress, pg->GetValue(0,9), size);

    return true;
}

int DbPsql::findUserAccountIdFromFqdn(const char *fqdn)
{
    int id = 0;

    ostringstream query;
    query << "SELECT userAccountId FROM userAccounts WHERE fqdn='"
          << fqdn << "'" << std::ends;
    
    if (pg->ExecTuplesOk(query.str().c_str()) == false)
    {
        log->debug("DbPsql::findAccountIdFromFqdn(): query failed");
        throw DbError("DbPsql::findAccountIdFromFqdn(): query failed");
    }

    if (pg->Tuples() == pg->Fields() == 1)
    {
        id = atoi(pg->GetValue(0,0));
        return id;
    }
    return -1;
}

unsigned long DbPsql::fetchAcctAddr(int id)
{
    ostringstream query;
    char addr[16];

    query << "SELECT ipAddress from userAccounts WHERE userAccountId="
          << id << std::ends;
    
    if(pg->ExecTuplesOk(query.str().c_str()) == false)
    {
        log->debug("DbPsql::fetchAcctAddr(): query failed");
        throw DbError("DbPsql::fetchAcctAddr(): query failed");
    }

    if (pg->Tuples() == pg->Fields() == 1)
    {
        strncpy(addr, (pg->GetValue(0,0)), sizeof(addr));
        return inet_addr(addr);
    }
    else
    {
        log->debug("DbPsql::fetchAcctAddr(): accountId not found");
        throw DbError("DbPsql::fetchAcctAddr(): accountId not found");
    }
}

void DbPsql::getActiveAccounts()
{
    // this method will return a PgDbDbResultSet constructed from a
    // select query
}
