/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * xf_io.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: xf_io.c,v 2.15 2003/05/31 16:31:28 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <glib.h>

#include "network.h"
#include "xf_io.h"
#include "cnx_if_detect.h"
#include "ged_if.h"
#include "g_string_ref.h"
#include "gvar.h"

#define EXPECTED_DATA_LEN 700

static void xfio_append_new_cmd_to_glist_incoming(XF_IO *xfio, char *string);

static const GString_ref *const xfio_peek_first_string_of_glist_outgoing(XF_IO *xfio);
static GString_ref *xfio_take_first_string_of_glist_outgoing(XF_IO *xfio);

/****************************************************/
/* array of XF_IO *                                 */
/* for speed, this array is sorted on the socket fd */
/****************************************************/
static GPtrArray *xf_io_array=NULL;

/*******************/
/* global counters */
/*******************/
static guint64 gl_up_count=0;			/* uploaded bytes */
static guint64 gl_down_count=0;		/* download bytes */


static int comp_xf_io(const void *a, const void *b)
{
	return (*((XF_IO**)a))->sock_fd - (*((XF_IO**)b))->sock_fd;
}

/* ------------------------------------------------------------------------------------ */
/* --------------------- XF_IO creation/destruction functions ------------------------- */
/* ------------------------------------------------------------------------------------ */
/*************************************************************************************/
/* create a new XF_IO. The XF_IO entry is automatically registered in the IO handler */
/*************************************************************************************/
/* if remote_addr ==NULL, the function computes it */
/***************************************************/
XF_IO *create_xfio(int sock_fd, struct in_addr *remote_addr)
{
	XF_IO *nw;

	nw=malloc(sizeof(XF_IO));
	if(nw==NULL)
	{
		fprintf(stderr,"create_xfio: out of memory.\n");
		return nw;
	}

	/* initialize const values */
	nw->sock_fd=sock_fd;
	nw->start_time=time(NULL);
	if(remote_addr!=NULL)
		nw->user_ip=*remote_addr;
	else
	{
		get_remote_if_ip(sock_fd,&(nw->user_ip));
	}

	nw->if_type=identify_if_of_cnx(sock_fd);

	/* initialize variable values */
	nw->incoming_commands=NULL;
	nw->in_partial=NULL;
	nw->outgoing_commands=NULL;
	XFIO_SET_CLOSE_STATUS(nw,CNX_OPENED);

	/* initialize counters */
	nw->in_data=0;
	nw->out_data=0;

	g_ptr_array_add(xf_io_array,nw);

	/* sort to speed up future scan */
	qsort(xf_io_array->pdata,xf_io_array->len,sizeof(void*), comp_xf_io);
	return nw;
}

static void free_a_gstring(gpointer data, gpointer user_data)
{
	g_string_free((GString *)data,TRUE);
}

static void free_a_gstring_ref(gpointer data, gpointer user_data)
{
	g_string_ref_free((GString_ref *)data);
}

/*************************************************************************************/
/* delete a XF_IO. The XF_IO entry is automatically unregistered from the IO handler */
/* and the socket is closed.                                                         */
/*************************************************************************************/
void delete_xfio(XF_IO *to_del)
{
	if(g_ptr_array_remove(xf_io_array,to_del)==FALSE)
	{
		fprintf(stderr,"delete_xfio: error, XF_IO not found, ignored\n");
		return;
	}

	close(to_del->sock_fd);

	/* flush incoming commands */
	g_list_foreach(to_del->incoming_commands,free_a_gstring,NULL);
	g_list_free(to_del->incoming_commands);

	if(to_del->in_partial)
		g_string_free(to_del->in_partial,TRUE);

	/* flush outgoing commands */
	g_list_foreach(to_del->outgoing_commands,free_a_gstring_ref,NULL);
	g_list_free(to_del->outgoing_commands);

	free(to_del);
}

/* ------------------------------------------------------------------------------------ */
/* ---------------------------- XF_IO private functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */
/***********************************************************/
/* read the XF_IO socket, append it to the partial content */
/* and split it by row into the incoming_commands          */
/***********************************************************/
static void xfio_process_incoming_cnx_data(XF_IO *xfio)
{
	char buf[8192];
	int ret;
	char *t;

	ret=recv(xfio->sock_fd, buf,sizeof(buf)-1,MSG_NOSIGNAL);
	if(ret==-1)
	{
		if((errno!=EAGAIN)&&(errno!=EINTR))
		{
			perror("xfio_process_incoming_cnx_data");
			XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
		}
		return;
	}

	if(ret==0)
	{
		/* connection closed */
		XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
		return;
	}

	buf[ret]='\0';

	if(xfio->in_partial==NULL)
		xfio->in_partial=g_string_new(buf);
	else
		g_string_append(xfio->in_partial,buf);

	xfio->in_data += ret;
	gl_down_count += ret;

	/* split incoming string along | */
	t=strchr(xfio->in_partial->str,'|');
	while(t!=NULL)
	{
		GString *nw;

		/* future new content of in_partial */
		/* = old content w/o the first string ending w/ a | */
		if (xfio->in_partial->len == (t-xfio->in_partial->str+1)) /*add in 0.1.4.b2*/
			nw =NULL;
		else
			nw=g_string_new(t+1);

		/* new string to add to incoming_commands */
		/* = first string ending w/ a | */
		xfio->in_partial=g_string_truncate(xfio->in_partial,(t+1)-xfio->in_partial->str);

		xfio_append_new_cmd_to_glist_incoming(xfio,xfio->in_partial->str);
		g_string_free(xfio->in_partial,TRUE);

		xfio->in_partial=nw;
		if (nw != NULL) /*add in 0.1.4.b2*/
			t=strchr(xfio->in_partial->str,'|');
		else
			t = NULL;
	}

	/* protection against incoming enormous single instruction */
	if((xfio->in_partial!=NULL)&&(xfio->in_partial->len>gl_buf_size))
	{
		unsigned int hip;

		hip=ntohl(xfio->user_ip.s_addr);
		printf("loflood: %hhu.%hhu.%hhu.%hhu %lu %Lu/%Lu\n",
							(unsigned char)(hip>>24)&0xff,
							(unsigned char)(hip>>16)&0xff,
							(unsigned char)(hip>>8)&0xff,
							(unsigned char)hip&0xff,
							(long unsigned int)(gl_cur_time - xfio->start_time),
							xfio->out_data,
							xfio->in_data
							);
		shutdown(xfio->sock_fd, SHUT_RD);
		XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
	}
	return;
}

/******************************************/
/* send outgoing_commands into the socket */
/******************************************/
static void xfio_process_outgoing_cnx_data(XF_IO *xfio)
{
	GString_ref *str;

	const GString_ref *peeked_str; /* dchub 0.2.5 */
	char out_data[EXPECTED_DATA_LEN + 1]; /* dchub 0.2.5 */ /* add +1 for the '\0' add in debug*/
	int data_len = 0; /* dchub 0.2.5 */
	int use_buf = 0; /* dchub 0.2.5 */
	
	int err;
#ifdef WIN32
		int errlen = sizeof(err);
#else
	#ifdef __APPLE__
		int errlen = sizeof(err);
	#else
		socklen_t errlen = sizeof(err);
	#endif
#endif
	int sres;

	/* pack several outgoing messages into a biggest block */
	str=xfio_take_first_string_of_glist_outgoing(xfio);
	if(str==NULL)
		return;			/* having nothing to send is not a fatal error */

	data_len = str->data->len;
	while (data_len < EXPECTED_DATA_LEN)
	{
		peeked_str = xfio_peek_first_string_of_glist_outgoing(xfio);
		if (peeked_str == NULL)
			break;
		if ((data_len + peeked_str->data->len) < EXPECTED_DATA_LEN)
		{
			if (use_buf == 0)
			{
				use_buf = 1;
				memcpy(out_data, str->data->str, str->data->len);
				g_string_ref_free(str);	
			}
			str=xfio_take_first_string_of_glist_outgoing(xfio);
			memcpy(out_data + data_len, str->data->str, str->data->len);
			data_len += str->data->len;
			g_string_ref_free(str);
			str=NULL;
		}
		else
			break; /* The buffer is too small, data will be send next time */
	}

	/* send the message in one part, it should not be broken into serveral parts */
	if (getsockopt(xfio->sock_fd , SOL_SOCKET, SO_ERROR , &err , &errlen) != 0 ||
		err == ETIMEDOUT || err == ECONNREFUSED || err == EHOSTDOWN ||
		err == EHOSTUNREACH)
	{
		/* connexion error -> shutdown the socket */
		shutdown(xfio->sock_fd,SHUT_RDWR);
		XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
	}
	else
	{
		retry_send:
		if (use_buf)
			sres=send(xfio->sock_fd,out_data,data_len,MSG_NOSIGNAL); /* send data using local buffer */
		else
			sres=send(xfio->sock_fd,str->data->str,data_len,MSG_NOSIGNAL); /* send a message */
		if (sres!=data_len)
		{
			if(sres==-1)
			{
				switch(errno)
				{
					case EAGAIN:
					case EINTR:	/* non fatal errors */
						goto retry_send;

					default:	/* other errors are fatal -> shutdown the socket */
						shutdown(xfio->sock_fd,SHUT_RDWR);
						XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
						break;
				}
			}
		}

		/* update counters */
		xfio->out_data += data_len;
		gl_up_count += data_len;
	}

	if (use_buf == 0)
		g_string_ref_free(str);
	return;
}

	
/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO incoming functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */
/************************************************************/
/* same as peek_first_string_of_glist except the GString is */
/* also removed from the GList. You must free it yourself   */
/* when you don't need it anymore.                          */
/************************************************************/
/* output: the pointer or NULL */
/*******************************/
GString *xfio_take_first_string_of_glist_incoming(XF_IO *xfio)
{
	GString *st;

	if(xfio->incoming_commands==NULL)
		st=NULL;
	else
	{
		st=xfio->incoming_commands->data;
		xfio->incoming_commands=g_list_remove(xfio->incoming_commands,st);
	}
	return st;
}

/********************************************************/
/* append a copy of the given string to the given glist */
/********************************************************/
static void xfio_append_new_cmd_to_glist_incoming(XF_IO *xfio, char *string)
{
	GString *s;

	s=g_string_new(string);
	xfio->incoming_commands=g_list_append(xfio->incoming_commands,s);
}

/* ------------------------------------------------------------------------------------ */
/* --------------------------- XF_IO outgoing functions ------------------------------- */
/* ------------------------------------------------------------------------------------ */
/**************************************************************/
/* return the pointer of the first GString of the given GList */
/**************************************************************/
/* output: the pointer or NULL. the pointer and the GString */
/*         content should not be modified                   */
/************************************************************/
static const GString_ref *const xfio_peek_first_string_of_glist_outgoing(XF_IO *xfio)
{
	const GString_ref *st;

	if(xfio->outgoing_commands==NULL)
		st=NULL;
	else
		st=xfio->outgoing_commands->data;
	return st;
}

/************************************************************/
/* same as peek_first_string_of_glist except the GString is */
/* also removed from the GList. You must free it yourself   */
/* when you don't need it anymore.                          */
/************************************************************/
/* output: the pointer or NULL */
/*******************************/
static GString_ref *xfio_take_first_string_of_glist_outgoing(XF_IO *xfio)
{
	GString_ref *st;

	if(xfio->outgoing_commands==NULL)
		st=NULL;
	else
	{
		st=xfio->outgoing_commands->data;
		xfio->outgoing_commands=g_list_remove(xfio->outgoing_commands,st);
	}
	return st;
}

/********************************************************/
/* append a copy of the given string to the given glist */
/********************************************************/
void xfio_append_new_cmd_to_glist_outgoing(XF_IO *xfio, GString_ref *string)
{
	string->count++;
	xfio->outgoing_commands=g_list_append(xfio->outgoing_commands,string);
}

/* ------------------------------------------------------------------------------------ */
/* ----------------------------- XF_IO misc functions --------------------------------- */
/* ------------------------------------------------------------------------------------ */
/************************/
/* return I/O statistic */
/************************/
void xfio_get_io_stats(guint64 *incoming_bytes, guint64 *outgoing_bytes)
{
	*incoming_bytes=gl_down_count;
	*outgoing_bytes=gl_up_count;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------- GED handler for cluster connections ----------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************/
/* function called at end */
/**************************/
static int xf_io_ged_exit(const GED_CALLBACKS *ged)
{
   return 0;
}

/****************************/
/* add FD event to wait for */
/****************************/
static void xf_io_ged_add_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array)
{
	/* for each XF_IO, add an entry in the pollfd_array */
	/* if the outgoing command is empty, the event if POLLIN */
	/* else, it is POLLIN|POLLOUT */
	int i;
	XF_IO *xfio;
	struct pollfd nw;

#ifdef DEBUG
	fprintf(stderr,"xf_io_ged_add_fd: xf_io len: %d\n",xf_io_array->len);
#endif
	/* for speed, we must ALWAYS add an entry for each fd to monitor */
	for(i=0;i<xf_io_array->len;i++)
	{
		xfio=g_ptr_array_index(xf_io_array,i);

		nw.fd=xfio->sock_fd;

		switch(xfio->closed)
		{
			case CNX_OPENED:
									nw.events=POLLIN;
									if(xfio->outgoing_commands)
										nw.events|=POLLOUT;
									break;

			case CNX_CLOSED:	continue;		/* this entry will disappear at the next loop, don't poll it */

			case CNX_WILL_DIE:
									if(xfio->outgoing_commands)
									{
										if((gl_cur_time-xfio->closed_date)>10)	/* max 10 seconds to die */
											goto game_over;
										nw.events=POLLOUT;
									}
									else
									{
										game_over:
										XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);		/* the queue is finally empty */
										shutdown(xfio->sock_fd,SHUT_RD);	/* just to be sure nothing will come accidentally */
										nw.events=POLLIN;		/* this entry will disappear at the next loop */
									}
									break;
		}

		nw.revents=0;
		
#ifdef DEBUG
		fprintf(stderr,"xfio [%d]=%d (status: %d, outq: %p) (events=%d)\n",i,xfio->sock_fd,xfio->closed,xfio->outgoing_commands,nw.events);
#endif
		g_array_append_val(struct_pollfd_array,nw);
	}
}

/*******************************/
/* check for the arrived event */
/*******************************/
static void xf_io_ged_scan_fd(const GED_CALLBACKS *ged, GArray *struct_pollfd_array, int low_range, int high_range)
{
	/* foreach ours fd, handle reception of data and splitting into incoming_commands */
	/* and sending of outgoing_commands */
	XF_IO *xfio;
	struct pollfd *nw;

	/* both arrays are sorted on sock_fd */
	int xfio_minus;
	int i;

#ifdef DEBUG
	/* debug code */
	fprintf(stderr,"xf_io_ged_scan_fd: xf_io len: %d  range=[%d:%d[ => %d\n",xf_io_array->len,low_range,high_range,high_range-low_range);
	for(i=low_range;i<high_range;i++)
	{
		fprintf(stderr,"pollfd [%d]=%d (revents=%d)\n",i,(g_array_index(struct_pollfd_array,struct pollfd,i)).fd,(g_array_index(struct_pollfd_array,struct pollfd,i)).revents);
	}
	for(i=0;i<xf_io_array->len;i++)
	{
		fprintf(stderr,"xfio [%d]=%d\n",i,((XF_IO*)g_ptr_array_index(xf_io_array,i))->sock_fd);
	}
#endif

	xfio_minus=0;
	i=low_range;
	while(i<high_range)
	{
		int idx;

		nw=&(g_array_index(struct_pollfd_array,struct pollfd,i));
		
		/* now, we must find the xfio with the same sock_fd */
		/* it is pretty easy, its index (idx) is >=xfio_minus, <xf_io_array->len and sock_fd[idx] <=nw->sock_fd */
		idx=xfio_minus;
		while(idx<xf_io_array->len)
		{
			xfio=g_ptr_array_index(xf_io_array,idx);
			if(xfio->sock_fd==nw->fd)
			{
				/* we have found it */
				xfio_minus=idx+1;		/* the next time, start after this idx */
				if(nw->revents&POLLIN)
				{
					/* process incoming data */
					xfio_process_incoming_cnx_data(xfio);
				}

				if(nw->revents&POLLOUT)
				{
					/* process outgoing data */
					xfio_process_outgoing_cnx_data(xfio);
				}

				if(nw->revents&(POLLERR|POLLHUP|POLLNVAL))
				{
					/* discard the connection if any error occurs */
					XFIO_SET_CLOSE_STATUS(xfio,CNX_CLOSED);
				}
				break;
			}

			if(xfio->sock_fd>nw->fd)
			{	/* the fd does not exists, we are already above what we search for */
				xfio_minus=i;		/* the next time, start here */
				break;
			}
			idx++;
		}
		if(idx==xf_io_array->len)	/* end of array reached */
			break;						/* all other fd will be unavailable */
		
		i++;
	}
}

static GED_CALLBACKS xf_io_ged=
					{
						"XF_IO ged",
						xf_io_ged_exit,
						xf_io_ged_add_fd,
						xf_io_ged_scan_fd,
						NULL,
						NULL,
						NULL,
						NULL,
					};

/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *xf_io_handler_init(void)
{
	xf_io_array=g_ptr_array_new();
   return &xf_io_ged;
}

