/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * user_cnx_lst.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: user_cnx_lst.c,v 2.41 2003/05/31 16:31:28 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <openssl/md4.h>
#include <glib.h>

#include "gvar.h"
#include "ged_if.h"
#include "xf_io.h"
#include "user_cnx_lst.h"
#include "macro.h"
#include "process_ucmd.h"
#include "global_user_if.h"
#include "timed_out_string.h"
#include "tos_key.h"
#include "md5.h"

/* hash table of connection */
/* the key is the nickname (stored inside the value) */
/* the value is a LOCAL_USER_CNX_ENTRY * */
/* there is no possible nickname conflict */
static GHashTable *user_cnx_entry_list=NULL;

#ifndef HAVE_NO_MUTEX
HL_MUTEX user_cnx_entry_lock = HL_MUTEX_INIT;
#endif

/* -------------------------------------------------------------------------- */
/* -------------------- creation/destruction functions ---------------------- */
/* -------------------------------------------------------------------------- */

/**********************/
/* create a new entry */
/******************************************************************************************/
/* input: xfio to used [the pointer is stolen, the caller must ignore it after this call] */
/*        nickname to use (must be !=NULL)                                                */
/*        shared_size: size of the shared data of this user                               */
/*        umail, udesc, ucliver: can be NULL                                              */
/******************************************************************************************/
/* output: address of the newly created and inserted LOCAL_USER_CNX_ENTRY * */
/****************************************************************************/
LOCAL_USER_CNX_ENTRY *uc_create_entry(XF_IO *xfio, char *nickname, guint64 shared_size, char *umail, char *udesc, char *ucliver,
							unsigned int user_flag, unsigned int ext_flag, int privilege_mask)
{
	LOCAL_USER_CNX_ENTRY *luce;

	luce=malloc(sizeof(LOCAL_USER_CNX_ENTRY));
	if(luce==NULL)
		return luce;		/* error */

	luce->user_xfio=xfio;
	luce->user_cnx_type="???";
	luce->user_nick=g_string_new(nickname);
	luce->shared_size=shared_size;

	if(umail!=NULL)
		luce->user_mail=g_string_new(umail);
	else
		luce->user_mail=g_string_new("");

	if(udesc!=NULL)
		luce->user_description=g_string_new(udesc);
	else
		luce->user_description=g_string_new("");

	luce->user_flag=user_flag|1;		/* the lowest bit is always set */
	luce->ext_flag=ext_flag;
	luce->privilege=privilege_mask;
	if(ucliver!=NULL)
		luce->client_version=g_string_new(ucliver);
	else
		luce->client_version=g_string_new("");

	if(luce->privilege)	/* op users do not need timeout sytem */
		luce->last_time=gl_cur_time + 5 * 356 * 36000;
	else
		luce->last_time=gl_cur_time;
	luce->last_hcall=gl_cur_time;
	luce->last_msg=0;
	luce->msg_count=0;
	
	luce->sr_count=0;
	luce->in_cnx_count=0;
	luce->out_cnx_count=0;

	luce->ext_display_flag=0;

	HL_LOCK_WRITE(user_cnx_entry_lock);
	g_hash_table_insert(user_cnx_entry_list,luce->user_nick->str,luce);
	HL_UNLOCK_WRITE(user_cnx_entry_lock);

	return luce;
}

/*******************************************************************/
/* remove an entry from the cnx_entry_list and free its ressources */
/*******************************************************************************/
/* this function can be used as value destruction function in g_hash_table_new */
/*******************************************************************************/
void uc_destroy_entry(LOCAL_USER_CNX_ENTRY *luce)
{
	if(luce==NULL)
		return;

	if(luce->user_xfio)
		delete_xfio(luce->user_xfio);

	if(luce->user_nick)
		g_string_free(luce->user_nick,TRUE);
	if(luce->user_mail)
		g_string_free(luce->user_mail,TRUE);
	if(luce->user_description)
		g_string_free(luce->user_description,TRUE);
	if(luce->client_version)
		g_string_free(luce->client_version,TRUE);
}


/* -------------------------------------------------------------------------- */
/* ------------------------- processing functions --------------------------- */
/* -------------------------------------------------------------------------- */

/***********************************************************************/
/* terminate a user connection. After this call, the luce is destroyed */
/*******************************************************************************/
/* input: with_remove== TRUE, the luce is removed from the user_cnx_entry_list */
/*                   == FALSE, the pointer of the destroyed luce is kept in the*/
/*                user_cnx_entry_list (useful for g_hash_table_foreach_remove) */
/*******************************************************************************/
static void end_user_connection(LOCAL_USER_CNX_ENTRY *luce, gboolean with_remove)
{
	if( (luce->user_nick->len<strlen("$to_delete")) ||
	    (strcmp(luce->user_nick->str+luce->user_nick->len-strlen("$to_delete"),"$to_delete")) )	/* test if the nickname does not end with $to_delete */
	{
		/* don't send the $Quit when a user ends with $to_delete */
		glus_do_quit(luce->user_nick->str);

		SEND_EVT_TO("quit",luce->user_nick->str,0);
	}

	/* printf("cnx over1.\n"); */
	{
		unsigned int hip;

		hip=ntohl(luce->user_xfio->user_ip.s_addr);
		printf("cnxover1: <%s> \t%hhu.%hhu.%hhu.%hhu %lu %d %Lu/%Lu:%u/%u\n",
						luce->user_nick->str,
	 					(unsigned char)(hip>>24)&0xff,
		 				(unsigned char)(hip>>16)&0xff,
		 				(unsigned char)(hip>>8)&0xff,
		 				(unsigned char)hip&0xff,
		 				(long unsigned int)(gl_cur_time - luce->user_xfio->start_time),
						luce->sr_count,
		 				luce->user_xfio->out_data,
		 				luce->user_xfio->in_data,
		 				luce->out_cnx_count,
		 				luce->in_cnx_count
						);
	}
	if(with_remove)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		HL_LOCK_WRITE(user_cnx_entry_lock);
		g_hash_table_remove(user_cnx_entry_list,luce->user_nick->str);
		HL_UNLOCK_WRITE(user_cnx_entry_lock);
		HL_LOCK_READ(user_cnx_entry_lock);
	}
	uc_destroy_entry(luce);
	return;
}

/**************************************/
/* main handler of the user cnx entry */
/*****************************************************/
/* this function handles incoming requests from user */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= NULL (unused, just required for the GHfunc)             */
/*****************************************************************************/
/* NOTE: the array has LOCK_READ set */
/**********************************************************/
/* output: TRUE= the entry is removed from the hash table */
/*         FALSE= the entry remains in the hash table     */
/**********************************************************/
static gboolean process_user_cnx_entry(gpointer key, gpointer value, gpointer user_data)
{
	LOCAL_USER_CNX_ENTRY *luce=value;

	/* check if the IO connection is closed */
	if(XFIO_IS_CLOSED(luce->user_xfio))
	{
		/* yes -> end of this user */
		end_user_connection(luce,FALSE);		/* don't delete luce ptr, it is handle after */
		return TRUE;		/* delete key/value pair */
	}

	/* check if something has to be done */
	if(!XFIO_INPUT_AVAILABLE(luce->user_xfio))
		return FALSE;		/* nope -> keep key/value pair */

	luce->last_time=gl_cur_time;		/* update user last action time */

	if(process_one_command(luce)<0)
	{
		end_user_connection(luce,FALSE);		/* don't delete luce ptr, it is handle after */
		return TRUE;		/* delete key/value pair */
	}

	return FALSE;		/* keep key/value pair */
}

/*****************************/
/* timeout handling function */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GINT_TO_POINTER(max_idle)                               */
/*****************************************************************************/
/* NOTE: the array has LOCK_READ set */
/**********************************************************/
/* output: TRUE= the entry is removed from the hash table */
/*         FALSE= the entry remains in the hash table     */
/**********************************************************/
static gboolean uc_close_long_time_idle(gpointer key, gpointer value, gpointer user_data)
{
	int max_idle=GPOINTER_TO_INT(user_data);
	LOCAL_USER_CNX_ENTRY *luce=value;

	if(luce->privilege==0)
	{
		/* simple users need system timeout, other don't */
		if ((gl_cur_time - luce->last_time) > max_idle)
		{
			printf("user %s leaves: time out.\n", luce->user_nick->str);
			end_user_connection(luce,FALSE);		/* don't delete luce ptr, it is handle after */
			return TRUE;		/* luce was destroyed -> delete key/value pair */
		}
	}

	return FALSE;		/* keep key/value pair */
}

/******************************************************/
/* send a gstring_ref to a user having the given name */
/******************************************************/
/* output: 0=ok else user not found */
/************************************/
static int send_gstring_ref_to_named_user(const char *nickname, GString_ref *msg)
{
	LOCAL_USER_CNX_ENTRY *luce;

	HL_LOCK_READ(user_cnx_entry_lock);
	luce=user_cnx_entry_get_by_nickname(nickname);
	if(luce==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		return 1;		/* user not found */
	}
	
	luce->last_time=gl_cur_time;		/* update user last action time */
	xfio_append_new_cmd_to_glist_outgoing(luce->user_xfio,msg);
	HL_UNLOCK_READ(user_cnx_entry_lock);
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GLUS handlers ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************************************************************/
/* send a message to a user having the given name and required user_flag and ext_flag */
/**************************************************************************************/
/* output: 0=ok else user not found */
/************************************/
static int uc_send_to_named_user(GLUS_PARAM *glus_param)
{
	return send_gstring_ref_to_named_user(glus_param->nickname,glus_param->msg);
}

/****************************/
/* send a message to a user */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GLUS_PARAM *                                            */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_send_to_all_users_one(gpointer key, gpointer value, gpointer user_data)
{
	GLUS_PARAM *glus_param=user_data;
	LOCAL_USER_CNX_ENTRY *luce=value;

	if(glus_param->nickname!=NULL)
	{
		if(!strcmp(glus_param->nickname,luce->user_nick->str))
			return;
	}

	luce->last_time=gl_cur_time;		/* update user last action time */
	if( ((luce->user_flag & glus_param->uflag_msk) == glus_param->wanted_uflag) &&
	    ((luce->ext_flag & glus_param->eflag_msk) == glus_param->wanted_eflag) )
	{
		xfio_append_new_cmd_to_glist_outgoing(luce->user_xfio,glus_param->msg);
	}
}

/**********************************************************************/
/* send a message to all users having required user_flag and ext_flag */
/**********************************************************************/
static void uc_send_to_all_users(GLUS_PARAM *glus_param)
{
	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_send_to_all_users_one,glus_param);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
/* output: NULL= not found */
/***************************/
static GLUS_USER_INFO *uc_glus_get_user_info(const char *nickname)
{
	LOCAL_USER_CNX_ENTRY *luce;
	GLUS_USER_INFO *gui;

	HL_LOCK_READ(user_cnx_entry_lock);
	luce=user_cnx_entry_get_by_nickname(nickname);
	if(luce==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		return NULL;		/* user not found */
	}

	gui=malloc(sizeof(GLUS_USER_INFO));
	if(gui==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		fprintf(stderr,"uc_glus_get_user_info: out of memory\n");
		return NULL;
	}

	gui->cnx_start_time=luce->user_xfio->start_time;
	gui->user_cnx_type=luce->user_cnx_type;
	gui->user_nick=g_string_new(luce->user_nick->str);
	gui->shared_size=luce->shared_size;
	if(luce->user_description)
		gui->user_description=g_string_new(luce->user_description->str);
	else
		gui->user_description=g_string_new("");
	if(luce->user_mail)
		gui->user_mail=g_string_new(luce->user_mail->str);
	else
		gui->user_mail=g_string_new("");
	gui->user_flag=luce->user_flag;
	gui->ext_flag=luce->ext_flag;
	gui->privilege=luce->privilege;
	if(luce->client_version)
		gui->client_version=g_string_new(luce->client_version->str);
	else
		gui->client_version=g_string_new("");

	gui->user_ip=luce->user_xfio->user_ip;

	HL_UNLOCK_READ(user_cnx_entry_lock);
	return gui;
}

/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
/* output: NULL= not found */
/***************************/
static GLUS_USER_INFO *uc_glus_get_user_info_from_luce(LOCAL_USER_CNX_ENTRY *luce)
{
	GLUS_USER_INFO *gui;

	HL_LOCK_READ(user_cnx_entry_lock);
	gui=malloc(sizeof(GLUS_USER_INFO));
	if(gui==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		fprintf(stderr,"uc_glus_get_user_info: out of memory\n");
		return NULL;
	}

	gui->user_cnx_type=luce->user_cnx_type;
	gui->user_nick=g_string_new(luce->user_nick->str);
	gui->shared_size=luce->shared_size;
	if(luce->user_description)
		gui->user_description=g_string_new(luce->user_description->str);
	else
		gui->user_description=g_string_new("");
	if(luce->user_mail)
		gui->user_mail=g_string_new(luce->user_mail->str);
	else
		gui->user_mail=g_string_new("");
	gui->user_flag=luce->user_flag;
	gui->ext_flag=luce->ext_flag;
	gui->privilege=luce->privilege;
	if(luce->client_version)
		gui->client_version=g_string_new(luce->client_version->str);
	else
		gui->client_version=g_string_new("");

	gui->user_ip=luce->user_xfio->user_ip;

	HL_UNLOCK_READ(user_cnx_entry_lock);
	return gui;
}

/************************************/
/* add a user to user (and op) list */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GString **  ([0]=user nick list, [1]=op nick list)      */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_get_nick_lists_one(gpointer key, gpointer value, gpointer user_data)
{
	GString **lists=user_data;
	LOCAL_USER_CNX_ENTRY *luce=value;

	/* test if the nickname does not end with $to_delete */
	if( (luce->user_nick->len<strlen("$to_delete")) ||
	    (strcmp(luce->user_nick->str+luce->user_nick->len-strlen("$to_delete"),"$to_delete")) )
	{
		/* put the user in the lists only if it is not scheduled for deletion */

		g_string_append(lists[0],luce->user_nick->str);
		g_string_append(lists[0],"$$");
	
		if(luce->privilege & HAVE_OP_KEY)
		{
			g_string_append(lists[1],luce->user_nick->str);
			g_string_append(lists[1],"$$");
		}
	}
}

/************************************************************************/
/* add the nicks to the provided strings (each nick is '$$' terminated) */
/************************************************************************/
/* list[0]= user nick list */
/* list[1]= op nick list   */
/***************************/
static void uc_glus_get_nick_lists(GString **lists)
{
	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_get_nick_lists_one,lists);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

/*****************************************/
/* add a user share size to the variable */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= guint64 * (current share size)                          */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_add_hub_users_stat_one(gpointer key, gpointer value, gpointer user_data)
{
	guint64 *share_size=user_data;
	LOCAL_USER_CNX_ENTRY *luce=value;

	(*share_size)+=luce->shared_size;
}

/*************************************************/
/* add number of users and share size of the hub */
/*************************************************/
static void uc_add_hub_users_stat(guint64 *shared_size, unsigned int *nb_users)
{
	HL_LOCK_READ(user_cnx_entry_lock);
	(*nb_users)+=g_hash_table_size(user_cnx_entry_list);
	g_hash_table_foreach(user_cnx_entry_list,uc_add_hub_users_stat_one,shared_size);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

/*****************************************/
/* kick a user having the given nickname */
/*****************************************/
/* output: 0=ok else user not found */
/************************************/
static int uc_kick_named_user(const char *kicking_nickname, const char *nickname)
{
	LOCAL_USER_CNX_ENTRY *luce; /* the kicked user */
	LOCAL_USER_CNX_ENTRY *kuce; /* the kicking user */
	/* int temp_ban; */
	int bot_kick;
	GString *str;

	HL_LOCK_READ(user_cnx_entry_lock);
	luce=user_cnx_entry_get_by_nickname(nickname);
	kuce=user_cnx_entry_get_by_nickname(kicking_nickname);
	if(luce==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		return 1;		/* user not found */
	}
	
	/* retrieve the temp ban duration when a kick occurs */
	G_LOCK(conf_file);
	/* if(!e_db_int_get(conf_file,"KICKTEMPBAN",&temp_ban)) */
		/* temp_ban=0; */
	if(!e_db_int_get(conf_file,"DISPLAY_BOT_KICK",&bot_kick))
		bot_kick=1;
	G_UNLOCK(conf_file);

	/* don't prevent anyone from getting disconnected at all */
	/* the disconnecting/kicking decision should be made before */
	/* executing such a command */

	/* if (luce->ext_flag & SHILD_MODE) */
	/* { */
		/* HL_UNLOCK_READ(user_cnx_entry_lock); */
		/* return 0; */	/* user found, nothing to do */
	/* } */

	/* The ban was done in a separate 'do_banip'-funktion-call before */

	/* if (temp_ban != 0) */
	/* { */
		/* add_tos_entry_v1_uniq(TBAN_TOSKEY,temp_ban*60,(void*)(&(luce->user_xfio->user_ip)),sizeof(struct in_addr),NULL,0); */
	/* } */

	XFIO_SET_CLOSE_STATUS(luce->user_xfio,CNX_WILL_DIE);	/* flush queued message but only if possible */
	shutdown(luce->user_xfio->sock_fd,SHUT_RD);		/* we don't want to receive more data from this connexion */

	str=g_string_new("");
	if(kicking_nickname!=NULL)
	{
		g_string_sprintf(str,"<Hub-Security> The user %s was kicked by %s. IP: %s|",
		 	nickname,kicking_nickname,inet_ntoa(luce->user_xfio->user_ip));
	}
	else
	{
		g_string_sprintf(str,"<Hub-Security> The user %s was kicked. IP: %s|",
			 nickname,inet_ntoa(luce->user_xfio->user_ip));
	}
	/* message only to the kicked user ??? I don't think so ;) */
	/* send_stolen_gstr_to_luce(luce,str);	*/	/* don't free str */
	if(kuce!=NULL)
	{
		if(bot_kick==0 && (kuce->privilege & BOT_PRIV))
		{
			g_string_free(str,TRUE);
		}
		else
			GLUS_SEND_TO_EVERYONE(str);      /* don't free str */
	}
	else
	{
		if(bot_kick==0)
		{
			g_string_free(str,TRUE);
		}
		else
			GLUS_SEND_TO_EVERYONE(str);      /* don't free str */
	}

	HL_UNLOCK_READ(user_cnx_entry_lock);
	return 0;
}

/***********************************************/
/* disconnect a user having the given nickname */
/***********************************************/
/* output: 0=ok else user not found */
/************************************/
static int uc_disconnect_named_user(const char *nickname)
{
	LOCAL_USER_CNX_ENTRY *luce;

	HL_LOCK_READ(user_cnx_entry_lock);
	luce=user_cnx_entry_get_by_nickname(nickname);
	if(luce==NULL)
	{
		HL_UNLOCK_READ(user_cnx_entry_lock);
		return 1;		/* user not found */
	}
	
	/* if (luce->ext_flag & SHILD_MODE) */
	/* { */
		/* HL_UNLOCK_READ(user_cnx_entry_lock); */
		/* return 0; */	/* user found, nothing to do */
	/* } */

	XFIO_SET_CLOSE_STATUS(luce->user_xfio,CNX_WILL_DIE);	/* flush queued message but only if possible */
	shutdown(luce->user_xfio->sock_fd,SHUT_RD);		/* we don't want to receive more data from this connexion */

	HL_UNLOCK_READ(user_cnx_entry_lock);
	return 0;
}

/**************************************************************/
/* function called after a user was disconnected from the hub */
/* the function is not here to destroy a user itself, it is   */
/* here to notify other I/O function this user has left       */
/**************************************************************/
static void uc_user_is_disconnected(const char *nickname)
{
}

/* -------------------------------------------------------------------------- */
/*************************************************/
/* add the glus user info of a user to the array */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GPtrArray of GLUS_USER_INFO                             */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_get_user_info_one(gpointer key, gpointer value, gpointer user_data)
{
	GPtrArray *array_of_gui_ptr=user_data;
	LOCAL_USER_CNX_ENTRY *luce=value;

	g_ptr_array_add(array_of_gui_ptr,uc_glus_get_user_info_from_luce(luce));
}

/******************************************************/
/* get all users. (add GLUS_USER_INFO * to the array) */
/******************************************************/
void uc_glus_get_users_info(GPtrArray *array_of_gui_ptr)
{
	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_get_user_info_one,array_of_gui_ptr);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

/* -------------------------------------------------------------------------- */
typedef struct
{
	GPtrArray *array_of_gui_ptr;
	struct in_addr ip;
} GETUIPARAM;

/*****************************************/
/* add a user share size to the variable */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= guint64 * (current share size)                          */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_get_user_info_by_ip_one(gpointer key, gpointer value, gpointer user_data)
{
	GETUIPARAM *guip=user_data;
	LOCAL_USER_CNX_ENTRY *luce=value;

	if(!memcmp(&(luce->user_xfio->user_ip),&(guip->ip),sizeof(struct in_addr)))
	{
		g_ptr_array_add(guip->array_of_gui_ptr,uc_glus_get_user_info_from_luce(luce));
	}
}

/**************************************************************************/
/* get all users having the given IP. (add GLUS_USER_INFO * to the array) */
/**************************************************************************/
void uc_glus_get_users_info_by_ip (GPtrArray *array_of_gui_ptr, struct in_addr ip)
{
	GETUIPARAM guip;

	guip.array_of_gui_ptr=array_of_gui_ptr;
	guip.ip=ip;

	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_get_user_info_by_ip_one,&guip);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

/* -------------------------------------------------------------------------- */
/***********************************************/
/* send the given string_ref to all local user */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GString_ref (string to send)                            */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_send_to_all_local_users_a_strref(gpointer key, gpointer value, gpointer user_data)
{
   GString_ref *ref=user_data;
   LOCAL_USER_CNX_ENTRY *luce=value;

	send_str_to_luce(luce,ref);
}

/*****************************************************/
/* function used to say a "$Hello" on the connection */
/*****************************************************/
static void uc_user_do_hello(const char *nickname)
{
	GString *str;
	GString_ref *ref;

	str=g_string_new("");
	g_string_sprintf(str,"$Hello %s|",nickname);
	ref=conv_gstring_to_gstring_ref(str);		/* don't free str */

	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_send_to_all_local_users_a_strref,ref);
	HL_UNLOCK_READ(user_cnx_entry_lock);

	g_string_ref_free(ref);
}

/****************************************************/
/* function used to say a "$Quit" on the connection */
/****************************************************/
static void uc_user_do_quit(const char *nickname)
{
	GString *str;
	GString_ref *ref;

	str=g_string_new("");
	g_string_sprintf(str,"$Quit %s|",nickname);
	ref=conv_gstring_to_gstring_ref(str);		/* don't free str */

	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_send_to_all_local_users_a_strref,ref);
	HL_UNLOCK_READ(user_cnx_entry_lock);

	g_string_ref_free(ref);
}

/******************************************************/
/* function used to say a "$MyINFO" on the connection */
/******************************************************/
static void uc_user_do_my_info(const char *nickname, const char *description, const char *user_cnx_type, const unsigned int user_flag, const char *user_mail, const guint64 shared_size, const unsigned int ext_flag, const unsigned int privilege, const char *client_version)
{
	GString *str;
	GString_ref *ref;

	str=g_string_new("");
	g_string_sprintf(str,"$MyINFO $ALL %s %s$ $%s%c$%s$%Lu$|",
                              nickname,description,user_cnx_type,
                              (char)user_flag,
                              user_mail,
                              shared_size);

	ref=conv_gstring_to_gstring_ref(str);		/* don't free str */

	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_send_to_all_local_users_a_strref,ref);
	HL_UNLOCK_READ(user_cnx_entry_lock);

	g_string_ref_free(ref);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
typedef struct
{
	guint16 edflag_mask;
	guint16 edflag_wanted;

	GString_ref *dont_have_ref;	/* string to send if the wanted flag is not set */
	GString_ref *have_ref;			/* string to send if the wanted flag is set */
} ED_SEND_STRING;

/*************************************************************************************************/
/* send one of the 2 given string_ref to all local user according to their extended_display_flag */
/*************************************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= ED_SEND_STRING (string to send)                         */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void uc_send_to_all_local_users_a_conditionnal_strref(gpointer key, gpointer value, gpointer user_data)
{
   LOCAL_USER_CNX_ENTRY *luce=value;
	ED_SEND_STRING *ess=user_data;

	if((luce->ext_display_flag&ess->edflag_mask)!=ess->edflag_wanted)
	{
		send_str_to_luce(luce,ess->dont_have_ref);
	}
	else
	{
		send_str_to_luce(luce,ess->have_ref);
	}
}

/****************************************************/
/* process a $XSearch and send it on the connection */
/****************************************************/
static void uc_user_do_xsearch(const char *return_path, const guint8 *file_crc, const guint32 file_length, const char *std_search)
{
	ED_SEND_STRING ess;
	GString *str;

	ess.edflag_mask=HAVE_XSEARCH;
	ess.edflag_wanted=HAVE_XSEARCH;

	str=g_string_new("");
	g_string_sprintf(str,"$Search %s %s|",return_path,std_search);
	ess.dont_have_ref=conv_gstring_to_gstring_ref(str);		/* don't free str */

	str=g_string_new("$XSearch ");
	append_MD_to_str(str,file_crc,MD4_DIGEST_LENGTH);
	g_string_sprintfa(str," %lu %s %s|",(unsigned long int)file_length,return_path,std_search);
	ess.have_ref=conv_gstring_to_gstring_ref(str);		/* don't free str */

	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach(user_cnx_entry_list,uc_send_to_all_local_users_a_conditionnal_strref,&ess);
	HL_UNLOCK_READ(user_cnx_entry_lock);

	g_string_ref_free(ess.dont_have_ref);
	g_string_ref_free(ess.have_ref);
}

/**********************************************/
/* function called when a $MD4Set is received */
/***************************************************************/
/* g_crc is the file global CRC (MD_BLOC_SIZE bytes)           */
/* l_crc is the set of partial CRC (nb_seg*MD_BLOC_SIZE bytes) */
/* nickname is the return path nickname                        */
/***************************************************************/
void uc_md4set_received(const guint8 *g_crc, const guint32 file_length, const guint8 *l_crc, const guint32 nb_seg, const char *filename, const char *nickname)
{
	GString *cmd;
	GString_ref *ref;

	/* rebuild a md4set string */
	cmd=g_string_new("$MD4Set ");
	g_string_append(cmd,nickname);
	g_string_append_c(cmd,' ');
	append_MD_to_str(cmd,g_crc,MD_BLOC_SIZE);
	g_string_sprintfa(cmd," %lu ",(unsigned long int)file_length);
	append_MD_to_str(cmd,l_crc,nb_seg*MD_BLOC_SIZE);
	g_string_append_c(cmd,' ');
	g_string_append(cmd,filename);
	g_string_append_c(cmd,'|');
	ref=conv_gstring_to_gstring_ref(cmd);		/* don't free cmd */
	send_gstring_ref_to_named_user(nickname,ref);
	g_string_ref_free(ref);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GED handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***************************/
/* function called on exit */
/***************************/
static int uc_ged_exit(const struct ged_callbacks *ged)
{
	return 0;
}

/********************************/
/* function called periodically */
/********************************/
static void uc_ged_periodic(const struct ged_callbacks *ged)
{
	/* check for inactivity in all cnx every 10 sec if the MAX_CNX_IDLE key exists */
	int max_idle;
	int have_idle_value;

	G_LOCK(conf_file);
	have_idle_value=e_db_int_get(conf_file,"MAX_CNX_IDLE",&max_idle);
	G_UNLOCK(conf_file);

	if(have_idle_value)
	{
		if(max_idle<10)
			max_idle=10;

		HL_LOCK_READ(user_cnx_entry_lock);
		g_hash_table_foreach_remove(user_cnx_entry_list,uc_close_long_time_idle,GINT_TO_POINTER(max_idle));
		HL_UNLOCK_READ(user_cnx_entry_lock);
	}
}

/********************************/
/* function called at each loop */
/********************************/
static void uc_ged_always1(const struct ged_callbacks *ged)
{
	HL_LOCK_READ(user_cnx_entry_lock);
	g_hash_table_foreach_remove(user_cnx_entry_list,process_user_cnx_entry,NULL);
	HL_UNLOCK_READ(user_cnx_entry_lock);
}

static GED_CALLBACKS user_cnx_entry_ged=
					{
						"user cnx ged",
						uc_ged_exit,
						NULL,					/* we have no descriptor to add */
						NULL,					/* and none to scan */
						uc_ged_periodic,
						uc_ged_always1,
						NULL,
						NULL,
					};

static GLOBAL_USER_CALLBACKS user_cnx_entry_glus=
					{
						"user cnx glus",
						uc_send_to_named_user,
						uc_send_to_all_users,
						uc_glus_get_user_info,
						uc_glus_get_users_info,
						uc_glus_get_users_info_by_ip,
						uc_glus_get_nick_lists,
						uc_add_hub_users_stat,
						uc_kick_named_user,
						uc_disconnect_named_user,
						uc_user_is_disconnected,
						uc_user_do_hello,
						uc_user_do_quit,
						uc_user_do_my_info,
						uc_user_do_xsearch,
						uc_md4set_received,
					};

/************************************************************/
/* function initializing the handler of all the connections */
/************************************************************/
GED_CALLBACKS *user_cnx_entry_handler_init(void)
{
   user_cnx_entry_list=g_hash_table_new(g_str_hash,g_str_equal);

	global_user_register(&user_cnx_entry_glus);
	
   return &user_cnx_entry_ged;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------ misc functions ---------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/*******************************************************************/
/* search for the LOCAL_USER_CNX_ENTRY of the given nickname       */
/* output: LOCAL_USER_CNX_ENTRY * of the user or NULL if not found */
/*******************************************************************/
LOCAL_USER_CNX_ENTRY *user_cnx_entry_get_by_nickname(const char *nickname)
{
	if(nickname==NULL)
		return NULL;
	return g_hash_table_lookup(user_cnx_entry_list,nickname);
}

/*************************************/
/* number of users locally connected */
/*************************************/
int user_cnx_entry_get_nb_users(void)
{
	return g_hash_table_size(user_cnx_entry_list);
}

/**************************************/
/* set and clear ext flag of the user */
/**************************************/
void user_cnx_entry_update_ext_flag_by_nickname(const char *nickname, unsigned int flag_to_set, unsigned int flag_to_clear)
{
	LOCAL_USER_CNX_ENTRY *luce;

	if(nickname==NULL)
		return;

	luce=g_hash_table_lookup(user_cnx_entry_list,nickname);
	if(luce)
	{
		luce->ext_flag=((luce->ext_flag|flag_to_set)&~flag_to_clear);
	}
}

/***************************************************************************/
/* rename the given LOCAL_USER_CNX_ENTRY with a uniq invalid user nickname */
/***************************************************************************/
void rename_luce_for_deletion(LOCAL_USER_CNX_ENTRY *luce)
{
	HL_LOCK_WRITE(user_cnx_entry_lock);
	g_hash_table_remove(user_cnx_entry_list,luce->user_nick->str);	/* remove the entry with the old nick */
	g_string_append(luce->user_nick,"$to_delete");
	g_hash_table_insert(user_cnx_entry_list,luce->user_nick->str,luce);	/* and insert the newly named entry */
	HL_UNLOCK_WRITE(user_cnx_entry_lock);
}

