/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * timed_out_string.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: timed_out_string.c,v 2.1 2003/03/07 16:07:54 ericprev Exp $
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#ifdef WIN32
	#include <winsock.h>
#else
	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
#endif
#include "gvar.h"
#include "config.h"
#include "timed_out_string.h"
#include "tos_key.h"
typedef struct
{
	unsigned long int key;			/* it is a key used by the calling functions to scan                   */
											/* only part of the tos_array. For instance, all entries               */
											/* registered by /CSRCH use the key 'CSRC'                             */
	time_t time_out_time;			/* when the given time is reached, this entry is automatically deleted */
	GByteArray *ba[2];				/* 2 byte arrays can be stored and retrieved */
} TOS_ENTRY;

/**********************/
/* array of TOS_ENTRY */
/**********************/
static GArray *tos_array=NULL;

G_LOCK_DEFINE_STATIC(tos_array);

/****************************************************/
/* load current tos entries from the given filename */
/****************************************************/
void load_tos_data(char *filename)
{
	FILE *f;

	f=fopen(filename,"rb");
	if(f==NULL)
	{
		fprintf(stderr,"load_tos_data: No TOS file, TOS ignored: %s\n",strerror(errno));
		return;
	}

	if(tos_array!=NULL)
	{
		fclose(f);
		fprintf(stderr,"load_tos_data: TOS array is not empty, TOS file ignored\n");
		return;
	}

	G_LOCK(tos_array);
	tos_array=g_array_new(FALSE,FALSE,sizeof(TOS_ENTRY));
	do
	{
		TOS_ENTRY nw;
		gint v;
		int k;

		k=fread(&(nw.key),1,sizeof(nw.key),f);

		if(k!=sizeof(nw.key))
		{
			if(k!=0)
				fprintf(stderr,"load_tos_data: TOS file truncated\n");
			break;
		}

		if(fread(&(nw.time_out_time),1,sizeof(nw.time_out_time),f)!=sizeof(nw.time_out_time))
		{
			fprintf(stderr,"load_tos_data: TOS file truncated\n");
			break;
		}

		/* load pointer 1 content */
		if(fread(&v,1,sizeof(v),f)!=sizeof(v))
		{
			fprintf(stderr,"load_tos_data: TOS file truncated\n");
			break;
		}

		if(v==-1)
		{
			nw.ba[0]=NULL;
		}
		else
		{
			nw.ba[0]=g_byte_array_new();
			nw.ba[0]=g_byte_array_set_size(nw.ba[0],v);
			if(fread(nw.ba[0]->data,1,v,f)!=v)
			{
				g_byte_array_free(nw.ba[0],TRUE);
				fprintf(stderr,"load_tos_data: TOS file truncated\n");
				break;
			}
		}

		/* load pointer 2 content */
		if(fread(&v,1,sizeof(v),f)!=sizeof(v))
		{
			g_byte_array_free(nw.ba[0],TRUE);
			fprintf(stderr,"load_tos_data: TOS file truncated\n");
			break;
		}

		if(v==-1)
		{
			nw.ba[1]=NULL;
		}
		else
		{
			nw.ba[1]=g_byte_array_new();
			nw.ba[1]=g_byte_array_set_size(nw.ba[1],v);
			if(fread(nw.ba[1]->data,1,v,f)!=v)
			{
				g_byte_array_free(nw.ba[1],TRUE);
				g_byte_array_free(nw.ba[0],TRUE);
				fprintf(stderr,"load_tos_data: TOS file truncated\n");
				break;
			}
		}

		tos_array=g_array_append_val(tos_array,nw);
	}while(!feof(f));
	G_UNLOCK(tos_array);

	fclose(f);
}

/****************************************************/
/* save current tos entries into the given filename */
/****************************************************/
void save_tos_data(char *filename)
{
	FILE *f;
	int i;

	f=fopen(filename,"wb");
	if(f==NULL)
	{
		fprintf(stderr,"save_tos_data: %s\n",strerror(errno));
		return;
	}

	if(tos_array==NULL)
	{
		fclose(f);
		fprintf(stderr,"save_tos_data: TOS array is empty, nothing to save.\n");
		return;
	}

	G_LOCK(tos_array);
	for(i=0;i<tos_array->len;i++)
	{
		TOS_ENTRY *nw;
		gint v;

		nw=&g_array_index(tos_array,TOS_ENTRY,i);

		if(fwrite(&(nw->key),1,sizeof(nw->key),f)!=sizeof(nw->key))
		{
			fprintf(stderr,"save_tos_data: TOS file truncated\n");
			break;
		}

		if(fwrite(&(nw->time_out_time),1,sizeof(nw->time_out_time),f)!=sizeof(nw->time_out_time))
		{
			fprintf(stderr,"save_tos_data: TOS file truncated\n");
			break;
		}

		/* save pointer 1 content */
		if(nw->ba[0]==NULL)
			v=-1;
		else
			v=nw->ba[0]->len;

		if(fwrite(&v,1,sizeof(v),f)!=sizeof(v))
		{
			fprintf(stderr,"save_tos_data: TOS file truncated\n");
			break;
		}

		if(v>0)
		{
			if(fwrite(nw->ba[0]->data,1,v,f)!=v)
			{
				fprintf(stderr,"save_tos_data: TOS file truncated\n");
				break;
			}
		}

		/* save pointer 2 content */
		if(nw->ba[1]==NULL)
			v=-1;
		else
			v=nw->ba[1]->len;

		if(fwrite(&v,1,sizeof(v),f)!=sizeof(v))
		{
			fprintf(stderr,"save_tos_data: TOS file truncated\n");
			break;
		}

		if(v>0)
		{
			if(fwrite(nw->ba[1]->data,1,v,f)!=v)
			{
				fprintf(stderr,"save_tos_data: TOS file truncated\n");
				break;
			}
		}
	}
	G_UNLOCK(tos_array);

	fclose(f);
}

/************************/
/**   Gui output log   **/
/************************/
#include "toolkit.h"
#ifndef min
#define min(a,b) ((a)>(b))?(b):(a)
#endif

static void gui_send(const unsigned long key, const void *ptr1, int len1, const void *ptr2, int len2, char type)
{
	char	dur = '?';
	
	if (key == TBAN_TOSKEY)
		dur = 't';
	else if (key == PBAN_TOSKEY)
		dur = 'p';
	else if (key == CNX_TOSKEY)
		dur = 'c';

	if ((ptr1!=NULL) && (len1==sizeof(struct in_addr))) /* sould alway be true */
	{
		struct in_addr *inp=(void*)ptr1;

		if ((ptr2==NULL) || (len2 == 0)) /* ban IP */
			printf("_%c%ci:%u.%u.%u.%u\n", dur, type,
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255));
		else /* ban nick */
		{
			char nick[128];

			strncpy_max(nick, ptr2, min(sizeof(nick), (len2+1)));
			printf("_%c%cn:%u.%u.%u.%u %s\n", dur, type,
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255), nick);
		}
			
	}
	return;
}

/****************************************************/
/* add a new timed_out entry to the timed_out array */
/********************************************************************************/
/* input: key= it is a non uniq value used to create group of timed_out entries */
/*        duration= number of seconds before the expiration of the entry.       */
/*        ptr1= pointer on an array of bytes to store.                          */
/*        len1= number of bytes inside ptr1                                     */
/*        ptr2, len2= same as ptr1,len1                                         */
/* ptr1 and/or ptr2 can be NULL.                                                */
/********************************************************************************/
void add_tos_entry(const unsigned long key, const time_t duration, const char *ptr1, const int len1, const char *ptr2, const int len2)
{
	TOS_ENTRY nw;

	nw.key=key;
	nw.time_out_time=time(NULL)+duration;
	if(ptr1==NULL)
	{
		nw.ba[0]=NULL;
	}
	else
	{
		nw.ba[0]=g_byte_array_new();
		nw.ba[0]=g_byte_array_append(nw.ba[0],ptr1,len1);
	}

	if(ptr2==NULL)
	{
		nw.ba[1]=NULL;
	}
	else
	{
		nw.ba[1]=g_byte_array_new();
		nw.ba[1]=g_byte_array_append(nw.ba[1],ptr2,len2);
	}

	G_LOCK(tos_array);
	if(tos_array==NULL)
	{
		tos_array=g_array_new(FALSE,FALSE,sizeof(TOS_ENTRY));
	}

	tos_array=g_array_append_val(tos_array,nw);

	if (gl_gui_mode)
		gui_send(key, ptr1, len1, ptr2, len2, 'b');
	G_UNLOCK(tos_array);
}

/**************************************************************************************/
/* remove the given entry. WARNING: this function doesn't and must not lock tos_array */
/**************************************************************************************/
static inline void delete_tos_entry(int num)
{
	TOS_ENTRY *te;

	te=&(g_array_index(tos_array,TOS_ENTRY,num));
	if (gl_gui_mode)		
		gui_send(te->key, (te->ba[0]==NULL)?NULL:te->ba[0]->data,
						  (te->ba[0]==NULL)?0:te->ba[0]->len,
						  (te->ba[1]==NULL)?NULL:te->ba[1]->data,
						  (te->ba[1]==NULL)?0:te->ba[1]->len,'u');
	if(te->ba[0])
		g_byte_array_free(te->ba[0],TRUE);
	if(te->ba[1])
		g_byte_array_free(te->ba[1],TRUE);

	tos_array=g_array_remove_index_fast(tos_array,num);
}

/******************************************************************/
/* search and remove a tos having the same value as the given one */
/******************************************************************/
/* input: key= key to find                                  */
/*        ptr= array to find                                */
/*        len= ptr array length                             */
/* each entry of the tos is a triplet (key,ptr1,ptr2)       */
/* if side==0, this function destroys the first tos matching*/
/* (key,ptr,x). If side==1, it is for (key,x,ptr)           */
/************************************************************/
/* NOTE: this function MUST be called when the tos_array */
/* is locked.                                            */
/*********************************************************/
static void delete_same_key(const unsigned long key, const char *ptr, const int len, int side)
{
	if(tos_array!=NULL)
	{
		int i;

		for(i=0;i<tos_array->len;i++)
		{
			TOS_ENTRY *te;

			te=&(g_array_index(tos_array,TOS_ENTRY,i));

			/* same key */
			if(te->key!=key)
				continue;

			if((te->ba[side]==NULL)||							/* no array on the side ? */
				(te->ba[side]->len!=len)||						/* array with a != length ? */
				(memcmp(te->ba[side]->data,ptr,len))		/* different array ? */
				)
				continue;
				
			delete_tos_entry(i);
			break;
		}
	}
}

/******************************************************************/
/* search and remove all tos having the same key as the given one */
/******************************************************************/
/* input: key= key to find                                  */
/************************************************************/
void delete_all_same_key(const unsigned long key, GString **dest_base, const char *sender)
{
   G_LOCK(tos_array);
   if(tos_array!=NULL)
   {
      TOS_ENTRY *te=NULL;
      int i;
      int f=0;

      again:
      for(i=0;i<tos_array->len;i++)
      {
         te=&(g_array_index(tos_array,TOS_ENTRY,i));

         /* same key */
         if(te->key==key)
         {
            delete_tos_entry(i);
            f++;
            goto again;
         }
      }

      if((f>0) && (dest_base!=NULL))
      {
         if(key==TBAN_TOSKEY)
            g_string_sprintfa((*dest_base)," %i entries removed from tempbanlist\r\n",f);
         else
            g_string_sprintfa((*dest_base)," %i entries removed from permbanlist\r\n",f);
      }
   }
   G_UNLOCK(tos_array);
}

/*****************************************************************************************/
/* add a new timed_out entry to the timed_out array, 1st value will be uniq in the array */
/*****************************************************************************************/
/* input: key= it is a non uniq value used to create group of timed_out entries */
/*        duration= number of seconds before the expiration of the entry.       */
/*        ptr1= pointer on an array of bytes to store.                          */
/*        len1= number of bytes inside ptr1                                     */
/*        ptr2, len2= same as ptr1,len1                                         */
/* ptr1 and/or ptr2 can be NULL.                                                */
/********************************************************************************/
void add_tos_entry_v1_uniq(const unsigned long key, const time_t duration, const char *ptr1, const int len1, const char *ptr2, const int len2)
{
	TOS_ENTRY nw;

	nw.key=key;
	nw.time_out_time=time(NULL)+duration;
	if(ptr1==NULL)
	{
		nw.ba[0]=NULL;
	}
	else
	{
		nw.ba[0]=g_byte_array_new();
		nw.ba[0]=g_byte_array_append(nw.ba[0],ptr1,len1);
	}

	if(ptr2==NULL)
	{
		nw.ba[1]=NULL;
	}
	else
	{
		nw.ba[1]=g_byte_array_new();
		nw.ba[1]=g_byte_array_append(nw.ba[1],ptr2,len2);
	}

	G_LOCK(tos_array);
	if(tos_array==NULL)
	{
		tos_array=g_array_new(FALSE,FALSE,sizeof(TOS_ENTRY));
	}
	else
	{
		delete_same_key(key,ptr1,len1,0);
	}

	if (gl_gui_mode)
		gui_send(key, ptr1, len1, ptr2, len2, 'b');

	tos_array=g_array_append_val(tos_array,nw);
	G_UNLOCK(tos_array);
}


/************************/
/* get the wanted entry */
/************************************************************/
/* input: key= key to find                                  */
/*        ptr= array to find                                */
/*        len= ptr array length                             */
/*        *out_ptr, *out_len= copy of the returned address. */
/*        content of out_ptr must be freed using free()     */
/* each entry of the tos is a triplet (key,ptr1,ptr2)       */
/* if side==0, this function returns in *out_ptr the first x*/
/* matching (key,ptr,x). If side==1, it is for (key,x,ptr)  */
/************************************************************/
/* output: ==0 not found, ==1, ok                 */
/*   on success, *out_ptr and *out_len are filled */
/**************************************************/
int get_tos_entry(const unsigned long key, const char *ptr, const int len, int side, char **out_ptr, int *out_len)
{
	int ret=0;
	G_LOCK(tos_array);
	if(tos_array!=NULL)
	{
		int i;

		for(i=0;i<tos_array->len;i++)
		{
			TOS_ENTRY *te;

			te=&(g_array_index(tos_array,TOS_ENTRY,i));

			/* same key */
			if(te->key!=key)
				continue;

			if((te->ba[side]==NULL)||							/* no array on the side ? */
				(te->ba[side]->len!=len)||						/* array with a != length ? */
				(memcmp(te->ba[side]->data,ptr,len))		/* different array ? */
				)
				continue;
				
			/* ok, so it is the good entry */
			if(te->ba[side^1]==NULL)
			{
				if(out_len)
					*out_len=0;
				*out_ptr=NULL;
			}
			else
			{
				if(out_len)
					*out_len=te->ba[side^1]->len;
				*out_ptr=malloc(te->ba[side^1]->len);
				if(*out_ptr!=NULL)
				{
					memcpy(*out_ptr,te->ba[side^1]->data,te->ba[side^1]->len);
				}
			}
			ret=1;
			break;
		}
	}
	G_UNLOCK(tos_array);
	return ret;
}

/***************************/
/* delete the wanted entry */
/************************************************************/
/* input: key= key to find                                  */
/*        ptr= array to find                                */
/*        len= ptr array length                             */
/* each entry of the tos is a triplet (key,ptr1,ptr2)       */
/* if side==0, this function returns in *out_ptr the first x*/
/* matching (key,ptr,x). If side==1, it is for (key,x,ptr)  */
/************************************************************/
void delete_this_tos_entry(const unsigned long key, const char *ptr, const int len, int side)
{
	G_LOCK(tos_array);
	delete_same_key(key, ptr, len, side);
	G_UNLOCK(tos_array);
}

/******************************************/
/* scan tos array to find expired entries */
/******************************************/
void timeout_tos(void)
{
	time_t now;

	G_LOCK(tos_array);

	now=time(NULL);
	if(tos_array!=NULL)
	{
		int i;

		for(i=tos_array->len-1;i>=0;i--)
		{
			TOS_ENTRY *te;

			te=&(g_array_index(tos_array,TOS_ENTRY,i));

			if(te->time_out_time<=now)
			{
				delete_tos_entry(i);
			}
		}
	}
	G_UNLOCK(tos_array);
}

/*******************************************************************/
/* call the given function for each tos entry having the given key */
/*******************************************************************/
void scan_all_tos_entry(const unsigned long key, int (*fnc)(const time_t timeout, const char *ptr1, const int len1, const char *ptr2, const int len2, void *xtra),void *xtra)
{
	if(fnc==NULL)
		return;

	G_LOCK(tos_array);
	if(tos_array!=NULL)
	{
		int i;

		for(i=0;i<tos_array->len;i++)
		{
			TOS_ENTRY *te;

			te=&(g_array_index(tos_array,TOS_ENTRY,i));

			if(te->key==key)
			{
				int out;
				out=(*fnc)(te->time_out_time,
							(te->ba[0]==NULL)?NULL:te->ba[0]->data,
							(te->ba[0]==NULL)?0:te->ba[0]->len,
							(te->ba[1]==NULL)?NULL:te->ba[1]->data,
							(te->ba[1]==NULL)?0:te->ba[1]->len,
							xtra);
				if(out!=0)
					break;
			}
		}
	}
	G_UNLOCK(tos_array);
}

#ifdef WIN32
#define BUG_COMPILE 1
#endif
#ifdef __APPLE__
#define BUG_COMPILE 1
#endif

#ifdef BUG_COMPILE
static unsigned long local_key;

static int list(const time_t timeout, const char *ptr1, const int len1, const char *ptr2, const int len2, void *xtra)
{
	gui_send(local_key, ptr1, len1, ptr2, len2, 'b');
	return(0);
}

void	gui_ban_list(const unsigned long key)
{
	local_key = key;
	scan_all_tos_entry(key, list, NULL);
}
#else
void	gui_ban_list(const unsigned long key)
{
	int list(const time_t timeout, const char *ptr1, const int len1, const char *ptr2, const int len2, void *xtra)
	{
		gui_send(key, ptr1, len1, ptr2, len2, 'b');
		return(0);
	}
	scan_all_tos_entry(key, list, NULL);
}
#endif

