/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * network.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: network.c,v 2.1 2003/02/16 10:58:30 eric Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
	#include <getopt.h>
#else
	#ifdef HAVE_SYS_TIME_H
		#include <sys/time.h>
	#endif
 	#include <sys/socket.h>
	#include <unistd.h>
	#include <pwd.h>
	#include <unistd.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <sys/ioctl.h>
	#include <netdb.h>
	#include <net/if.h>
	#ifdef HAVE_UNISTD_H
		#include <unistd.h>
	#endif  /* HAVE_UNISTD_H */
#endif

#ifdef HAVE_ERRNO_H
	#include <errno.h>
#else
	extern int errno;
	#ifndef EINPROGRESS
		#define EINPROGRESS	115
	#endif
	#ifndef EOPNOTSUPP
		#define EOPNOTSUPP	95
	#endif
	#ifndef EINTR
		#define EINTR		4
	#endif
#endif

#include <sys/types.h>
#include <fcntl.h>
#include <string.h>

#ifdef HAVE_NET_ROUTE_H
#include <net/route.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <glib.h>

#include "../config.h"
static struct sockaddr_in INET_ZERO = { AF_INET };

/*************************************************/
/* open a TCP connection with the requested port */
/*************************************************/
/* en entree: port= port demande     */
/* en sortie: numero de socket ou -1 */
/*************************************/
int _x_tcp (int port)
{
	int f;
	unsigned int len;
	struct sockaddr_in me ;
	struct sockaddr_in s_in;
	int zz=1;

	me = s_in = INET_ZERO;

	me.sin_port = htons((unsigned short) port);
	me.sin_family = AF_INET;

	if((f=socket(AF_INET,SOCK_STREAM,0)) == -1) return(-1);

	if(setsockopt(f,SOL_SOCKET,SO_REUSEADDR,(char *)&zz,sizeof(zz)) < 0 ||
		bind(f,(struct sockaddr *) &me,(len = sizeof(me))) < 0)
	{
		perror("setsock or bind");
		close(f); // ok
		return(-1);
	}

	return(f);
}

/*************************************************/
/* open a UDP connection with the requested port */
/*************************************************/
/* en entree: port= port demande     */
/* en sortie: numero de socket ou -1 */
/*************************************/
int _x_udp (int port)
{
	int f, zz;
	unsigned int len;
	struct sockaddr_in me ;
	struct sockaddr_in s_in;

	me = s_in = INET_ZERO;

	me.sin_port = htons((unsigned short) port);
	me.sin_family = AF_INET;

	if((f=socket(AF_INET,SOCK_DGRAM,0)) == -1) return(-1);

	zz=1;

	if(setsockopt(f,SOL_SOCKET,SO_REUSEADDR,(char *)&zz,sizeof(zz)) < 0 ||
		 bind(f,(struct sockaddr *) &me,(len = sizeof(me))) < 0 ||
		 getsockname(f,(struct sockaddr *)&s_in,&len) < 0)
	{
		close(f); // ok
		return(-1);
	}

	return(f);
}	 

/***************************************/
/* conversion nom d'hote -> adresse ip */
/***************************************/
/* en sortie: -1= fail, 0=ok */
/*****************************/
int str_to_inaddr (const char *str, struct in_addr *iap)
{
#ifdef DONT_HAVE_INET_ATON
	*(int *)iap = inet_addr(str);
	if (*(int *)iap == -1)
#else
	if (!inet_aton(str, iap))
#endif
	{
		struct hostent *he;

		if ((he = gethostbyname(str)))
		{
			if ((unsigned)he->h_length > sizeof *iap)
				he->h_length = sizeof *iap;

			memcpy(iap, he->h_addr, (unsigned int)(he->h_length));
		}
		else
		{
			return -1;
		}
	}

	return 0;
}

/*****************************************************/
/* routine fixant le flag non bloquant sur un socket */
/*****************************************************/
void set_non_bloquant_sock(int socket_fd)
{
#ifdef WIN32
	DWORD ioctlEnable=1;
	ioctlsocket (socket_fd, FIONBIO, &ioctlEnable);
	return;
#else
	long retval;

	retval=fcntl(socket_fd,F_GETFL);
	retval|= O_NONBLOCK;

	fcntl(socket_fd,F_SETFL,retval);
#endif	
}


/*************************************************/
/* routine fixant le flag bloquant sur un socket */
/*************************************************/
void set_bloquant_sock(int socket_fd)
{
#ifdef WIN32
	DWORD ioctlEnable=0;
	ioctlsocket (socket_fd, FIONBIO, &ioctlEnable);
	return;
#else
	long retval;

	retval=fcntl(socket_fd,F_GETFL);
	retval&= ~O_NONBLOCK;

	fcntl(socket_fd,F_SETFL,retval);
#endif	
}

/****************************************************/
/* create a socket and connect it to the given host */
/***************************************************************/
/* input: non_block=1 (socket is created with O_NONBLOCK flag) */
/*********************************************************************************************/
/* output: sock_fd or -1,-2 or -3                                                            */
/*         if non_block==1, returned sock_fd must be tested (ready_to_write) before using it */
/*********************************************************************************************/
/* -1=unknown host           */
/* -2=socket creation failed */
/* -3=unable to contact host */
/*****************************/
int create_and_open_sock_on(const char *hostname,unsigned short port, int non_block, char *bind_addr)
{
	struct sockaddr_in addr;
	struct sockaddr_in src;
	int s;
	int code;

	addr.sin_family=AF_INET;
	addr.sin_port=htons(port);
	if(str_to_inaddr(hostname,&(addr.sin_addr)))
		return -1;

	if((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
	{
		perror("socket:");
		return -2;
	}

	src.sin_family = AF_INET;
	src.sin_port=0;
	
	if (bind_addr)
	{
		if (str_to_inaddr(bind_addr,&(src.sin_addr)))
		{
			printf("invalid bind addres\n");
			return -1;
		}
		if (bind(s, (struct sockaddr*)&src,sizeof (src)))
		{
			printf("bind fail\n");
			return -1;
		}
	}
	
	if(non_block)
		set_non_bloquant_sock(s);

	code=connect(s,(struct sockaddr*)&addr,sizeof addr);

	if(non_block==0)
	{
		if(code==-1)
		{
			perror("connect");
			close(s); // ok
			return -3;
		}
	}
	else
	{
		/* socket doesn't block, -1 is allowed if errno==EINPROGRESS */
		if((code==-1)&&(errno!=EINPROGRESS))
		{
			perror("connect");
			close(s); // ok
			return -3;
		}
	}

	return s;
}

/******************************/
/* return the default host IP */
/************************************************************************/
/* the default host IP is the IP of the interface used as default route */
/************************************************************************/
GString *get_default_host_ip(char *bind_addr)
{
	FILE *f;
	char buf[512];

	/* the following script does this: */
	/* 1) extract the interface used by the default route: netstat -rn | grep ^0.0.0.0 | awk '{print $8;} */
	/* 2) obtain the IP of this interface: netstat xxx | fgrep inet | cut -d : -f 2 | awk '{print $1;}' */
	/*
	  const char *cmd=
		"/sbin/ifconfig ppp0 | fgrep inet | cut -d : -f 2 | awk '{print $1;}'";
	*/
	const char *cmd=
		"/sbin/ifconfig `netstat -rn | grep ^0.0.0.0 | awk '{print $8;}'` | fgrep inet | cut -d : -f 2 | awk '{print $1;}'";

	if (bind_addr)
		return g_string_new(bind_addr);

	f=popen(cmd,"r");
	if(f==NULL)
	{
		fprintf(stderr,"Unable to create a shell to obtain default IP.\n");
		return NULL;
	}

	if(fgets(buf,sizeof(buf),f)==NULL)
	{
		pclose(f); // ok
		fprintf(stderr,"No default interface found.\n");
		return NULL;
	}

	pclose(f); // ok

	if(strlen(buf)!=0)
	{
		char *t;

		t=buf;
		while( (*t) && ((*t=='.') || (isdigit(*t))) )
			t++;

		*t='\0';
		return g_string_new(buf);
	}
	else
		return NULL;
}

