/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * multi_private_chat.c: Copyright (C) Uriel Chemouni <uriel@adess.dyndns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: multi_public_chat.c,v 2.19 2003/05/08 08:57:49 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "gvar.h"
#include "multi_public_chat.h"
#include "multi_public_chat_cmd.h"
#include "global_user_if.h"
#include "hl_locks.h"
#include "toolkit.h"

static GHashTable *mpuc_entry_list=NULL;

#ifndef HAVE_NO_MUTEX
HL_MUTEX mpuc_entry_list_mutex= HL_MUTEX_INIT;
#endif

static int first_free_chat_id=0;

static void send_msg_to_mpuc(MPUC_ENTRY *mpuc,const char *base_msg, const char *sender_nickname);
static MPUC_ENTRY *mpuc_entry_get_by_nickname(const char *nickname);
static gboolean is_a_user_of_the_public_chat(MPUC_ENTRY *mpuc, const char *nickname);
static gboolean remove_a_user_of_the_public_chat(MPUC_ENTRY *mpuc, const char *nickname);

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* --------------------- creation/destruction handlers ---------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/****************************/
/* create a new public chat */
/********************************************/
/* output: -1=error, else it is the chat_id */
/********************************************/
static int create_multi_public_chat(const char *nickname, CHAT_TYPE chat_type, const char *password)
{
	MPUC_ENTRY *nw;

	nw=malloc(sizeof(MPUC_ENTRY *));
	if(nw==NULL)
		return -1;

	nw->mpuc_nickname=g_string_new(nickname);
	nw->user_lst=g_ptr_array_new();
	nw->chat_id=first_free_chat_id++;
	nw->chat_type=chat_type;
	if(password==NULL)
		nw->password=NULL;
	else
		nw->password=g_strdup(password);

	HL_LOCK_WRITE(mpuc_entry_list_mutex);
	g_hash_table_insert(mpuc_entry_list,nw->mpuc_nickname->str,nw);
	HL_UNLOCK_WRITE(mpuc_entry_list_mutex);

	/* don't forget to add a new user entry */
	glus_do_hello(nw->mpuc_nickname->str);

	return nw->chat_id;
}

/**********************************/
/* delete an existing public chat */
/****************************************************************************************************/
/* if hash_remove==TRUE, the chat key/value is removed from the hash table                          */
/* else the deleted mpuc pointer remains in the hash table (useful for g_hash_table_foreach_remove) */
/****************************************************************************************************/
static void delete_multi_public_chat(MPUC_ENTRY *mpuc, gboolean hash_remove)
{
	GString *str;
	gchar *tmp;

	if(hash_remove)
		g_hash_table_remove(mpuc_entry_list,mpuc->mpuc_nickname->str);

	/* don't forget to remove the user entry */
	glus_do_quit(mpuc->mpuc_nickname->str);

	/* empty the user list */
	while(mpuc->user_lst->len>0)
	{
		str=g_ptr_array_index(mpuc->user_lst,0);
		tmp=g_strconcat(str->str," has left the chat.|",NULL);
		send_msg_to_mpuc(mpuc,tmp,NULL);
		g_free(tmp);
		g_ptr_array_remove_index_fast(mpuc->user_lst,0);
		g_string_free(str,TRUE);
	}

	g_string_free(mpuc->mpuc_nickname,TRUE);
	g_ptr_array_free(mpuc->user_lst,TRUE);
	if(mpuc->password!=NULL)
		g_free(mpuc->password);

	free(mpuc);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- task handlers ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/******************************************************************************/
/* join a user to the given public chat (no password check is performed here) */
/******************************************************************************/
/* luce can be NULL if the user is not local */
/*********************************************/
static void join_a_user_to_a_public_chat(const char *user_nickname, const char *chat_name, LOCAL_USER_CNX_ENTRY *luce)
{
	MPUC_ENTRY *mpuc;
	GString *incoming_msg;

	HL_LOCK_READ(mpuc_entry_list_mutex);
	mpuc=mpuc_entry_get_by_nickname(chat_name);
	if(mpuc==NULL)
	{
		HL_UNLOCK_READ(mpuc_entry_list_mutex);
		return;	/* MPUC not found */
	}

	if(is_a_user_of_the_public_chat(mpuc,user_nickname)==TRUE)
	{
		HL_UNLOCK_READ(mpuc_entry_list_mutex);
		return;	/* no need to add several times the same user */
	}

	g_ptr_array_add(mpuc->user_lst,g_string_new(user_nickname));
	
	/* notify the incoming to everyone */
	incoming_msg=g_string_new("");
	g_string_sprintf(incoming_msg,"User %s enters in the chat",user_nickname);
	send_msg_to_mpuc(mpuc,incoming_msg->str,NULL);
	g_string_free(incoming_msg,TRUE);

	HL_UNLOCK_READ(mpuc_entry_list_mutex);
}

/******************************************************************************/
/* join a user to the given public chat (no password check is performed here) */
/******************************************************************************/
static void join_a_user_to_a_public_chat_by_mpuc(const char *user_nickname, MPUC_ENTRY *mpuc)
{
	GString *incoming_msg;

	HL_LOCK_READ(mpuc_entry_list_mutex);

	if(is_a_user_of_the_public_chat(mpuc,user_nickname)==TRUE)
	{
		HL_UNLOCK_READ(mpuc_entry_list_mutex);
		return;	/* no need to add several times the same user */
	}

	g_ptr_array_add(mpuc->user_lst,g_string_new(user_nickname));
	
	/* notify the incoming to everyone */
	incoming_msg=g_string_new("");
	g_string_sprintf(incoming_msg,"User %s enters in the chat",user_nickname);
	send_msg_to_mpuc(mpuc,incoming_msg->str,NULL);
	g_string_free(incoming_msg,TRUE);

	HL_UNLOCK_READ(mpuc_entry_list_mutex);
}

/******************************************/
/* remove a user from a multi public chat */
/***********************************************************/
/* output: FALSE= mpuc key/value pair remains in the table */
/*         TRUE= mpuc key/value should be deleted          */
/***********************************************************/
static void remove_leaving_user_out_of_loop(gpointer key_mpuc_nickname, gpointer value_mpuc, gpointer data_unickname)
{
	MPUC_ENTRY *mpuc=value_mpuc;
	const char *nickname=data_unickname;
	gchar *tmp;
	GString *str;

	if(remove_a_user_of_the_public_chat(mpuc,nickname)==FALSE)
		return;

	/* notify its removal to the user */
	str=g_string_new("");
	g_string_sprintf(str,"$To: %s From: %s $You have left the chat.\r\n|", nickname, mpuc->mpuc_nickname->str);
	GLUS_SEND_TO_A_NICK(nickname,str);		/* don't free str */

	/* and also to other user and destroy the chat if required */
	if((mpuc->user_lst->len==0)&&(mpuc->chat_type!=PUBLIC_CHAT))	/* empty PUBLIC_CHAT is allowed */
	{
		delete_multi_public_chat(mpuc,TRUE);
	}
	else
	{
		tmp=g_strconcat(nickname," has left the chat.|",NULL);
		send_msg_to_mpuc(mpuc,tmp,NULL);
		g_free(tmp);
	}
}


/****************************************/
/* handle command sent to a public chat */
/****************************************/
static void public_chat_command(MPUC_ENTRY *mpuc, GString *sender_nickname,  GString *msg_content, GLUS_PARAM *glus_param)
{
	GStringChunk *chunk=NULL;
	GPtrArray *tbl=NULL;

	/* removing the trailing pipe if exists */
	if(msg_content->len>=1)
	{
		if(msg_content->str[msg_content->len-1]=='|')
		{
			g_string_truncate(msg_content,msg_content->len-1);
		}
	}

	split_string_into_array(&tbl,&chunk,msg_content->str," ");

	if(tbl->len>=1)
	{
		/* check all known commands */
		/* +join pubchat [password] */
		if(!strcmp("+join",g_ptr_array_index(tbl,0)))
		{
			/* only user out of the chat can join */
			if(is_a_user_of_the_public_chat(mpuc,sender_nickname->str)==TRUE)
			{
				GString *out;

				out=g_string_new("");
				g_string_sprintf(out,"$To: %s From: %s $<Kernel> You already are in this chat|",
																		sender_nickname->str,mpuc->mpuc_nickname->str);
				GLUS_SEND_TO_A_NICK(sender_nickname->str,out);		/* don't free out */
				goto abrt;
			}
			
			if(mpuc->password!=NULL)
			{
				if(tbl->len<3)
				{
					GString *out;

					out=g_string_new("");
					g_string_sprintf(out,"$To: %s From: %s $<Kernel> This chat is password protected|",
																			sender_nickname->str,mpuc->mpuc_nickname->str);
					GLUS_SEND_TO_A_NICK(sender_nickname->str,out);		/* don't free out */
					goto abrt;
				}

				if(strcmp(mpuc->password,g_ptr_array_index(tbl,2)))
				{
					GString *out;

					out=g_string_new("");
					g_string_sprintf(out,"$To: %s From: %s $<Kernel> Invalid password for this chat|",
																			sender_nickname->str,mpuc->mpuc_nickname->str);
					GLUS_SEND_TO_A_NICK(sender_nickname->str,out);		/* don't free out */
					goto abrt;
				}
			}

			/* password is valid, user is not yet inside */
			join_a_user_to_a_public_chat_by_mpuc(sender_nickname->str,mpuc);
		}
		else if(!strcmp("+leave",g_ptr_array_index(tbl,0)))
		{
			/* only user of the chat can leave */
			if(is_a_user_of_the_public_chat(mpuc,sender_nickname->str)==FALSE)
				goto abrt;

			remove_leaving_user_out_of_loop(mpuc->mpuc_nickname->str,mpuc,sender_nickname->str);
		}
		else if(!strcmp("+list",g_ptr_array_index(tbl,0)))
		{
			GString *out;
			GString *uu;
			int i;

			/* only user of the chat can list */
			if(is_a_user_of_the_public_chat(mpuc,sender_nickname->str)==FALSE)
				goto abrt;

			out=g_string_new("");
			g_string_sprintf(out,"$To: %s From: %s $<Kernel> Users on this chat: ", sender_nickname->str,mpuc->mpuc_nickname->str);
			for(i=0;i<(mpuc->user_lst->len-1);i++)
			{
				uu=g_ptr_array_index(mpuc->user_lst,i);
				g_string_append(out,uu->str);
				g_string_append(out," ,");
			}
			/* don't put a " ," after the last user */
			i=mpuc->user_lst->len-1;
			uu=g_ptr_array_index(mpuc->user_lst,i);
			g_string_append(out,uu->str);
			g_string_append(out,"\r\n|");

			GLUS_SEND_TO_A_NICK(sender_nickname->str,out);		/* don't free out */
		}
		else
		{
			GString *out;

			out=g_string_new("");
			g_string_sprintf(out,"$To: %s From: %s $<Kernel> Unknown command: '%s'|",
																		sender_nickname->str,mpuc->mpuc_nickname->str,msg_content->str);
			GLUS_SEND_TO_A_NICK(sender_nickname->str,out);		/* don't free out */
		}
	}

	abrt:
	/* free allocated data */
	if(tbl!=NULL)
		g_ptr_array_free(tbl,TRUE);
	if(chunk!=NULL)
		g_string_chunk_free(chunk);
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- I/O handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/******************************************************/
/* send a message to all users of a multi public chat */
/******************************************************/
/****************************************************************************************/
/* create "$To: destnick From: chatnick $base_msg" and send it to all users of the chat */
/****************************************************************************************/
static void send_msg_to_mpuc(MPUC_ENTRY *mpuc,const char *base_msg, const char *sender_nickname)
{
	int i;
	GString *msg;

	if(sender_nickname==NULL)
	{
		for(i=0;i<mpuc->user_lst->len;i++)
		{
			char *nickname;

			nickname=((GString*)g_ptr_array_index(mpuc->user_lst,i))->str;
	
			msg=g_string_new(NULL);
			g_string_sprintf(msg,"$To: %s From: %s $%s|",nickname,mpuc->mpuc_nickname->str,base_msg);
			GLUS_SEND_TO_A_NICK(nickname,msg);		/* don't free msg */
		}
	}
	else
	{
		for(i=0;i<mpuc->user_lst->len;i++)
		{
			char *nickname;

			nickname=((GString*)g_ptr_array_index(mpuc->user_lst,i))->str;
	
			if(strcmp(sender_nickname,nickname))
			{
				msg=g_string_new(NULL);
				g_string_sprintf(msg,"$To: %s From: %s $%s|",nickname,mpuc->mpuc_nickname->str,base_msg);
				GLUS_SEND_TO_A_NICK(nickname,msg);		/* don't free msg */
			}
		}
	}
}

/********************************************/
/* find a multi public chat by its nickname */
/********************************************/
static MPUC_ENTRY *mpuc_entry_get_by_nickname(const char *nickname)
{
	return g_hash_table_lookup(mpuc_entry_list,nickname);
}

/*****************************************************************/
/* check if the given nickname is in the given multi public chat */
/*****************************************************************/
static gboolean is_a_user_of_the_public_chat(MPUC_ENTRY *mpuc, const char *nickname)
{
	int i;

	for(i=0;i<mpuc->user_lst->len;i++)
	{
		if(!strcmp(nickname,((GString*)g_ptr_array_index(mpuc->user_lst,i))->str))
			return TRUE;
	}

	return FALSE;
}

/************************************************************/
/* remove the given nickname of the given multi public chat */
/************************************************************/
/* output: TRUE= user found, FALSE= user not found */
/***************************************************/
static gboolean remove_a_user_of_the_public_chat(MPUC_ENTRY *mpuc, const char *nickname)
{
	int i;

	for(i=0;i<mpuc->user_lst->len;i++)
	{
		GString *str;
		str=((GString*)g_ptr_array_index(mpuc->user_lst,i));
		if(!strcmp(nickname,str->str))
		{
			g_ptr_array_remove_index_fast(mpuc->user_lst,i);
			g_string_free(str,TRUE);
			return TRUE;
		}
	}

	return FALSE;
}

/**********************************************************************/
/* extract the sender nickname and the message content from a message */
/**********************************************************************/
/* incoming msg format "$To: chatname From: nickname $<nickname> msg" */
/**********************************************************************/
/* output: 0=ok, !=0 = error */
/*****************************/
static int extract_nickname_and_msg_content_from_msg(GString *str, GString **nick, GString **msg_content)
{
	char *t;

	if(strncmp(str->str,"$To: ",5))
		return 1;		/* not a private message */

	t=strchr(str->str+1 /* skip the first $ */, '$');
	if(t==NULL)
		return 1;		/* no 2nd '$' in the string */

	*msg_content=g_string_new(t+1);		/* extract the message content */

	*nick=g_string_new(str->str);
	g_string_truncate(*nick,t-str->str);	/* discard the '$' and anything following it */
	t=strstr((*nick)->str,"From: ");
	if(t==NULL)
	{
		g_string_free(*msg_content,TRUE);
		*msg_content=NULL;
		g_string_free(*nick,TRUE);
		*nick=NULL;
		return 1;		/* no From: field */
	}

	g_string_erase(*nick,0,(t+6)-(*nick)->str);		/* remove everything before the nick (including the "From: " */
	t=strchr((*nick)->str,' ');
	if(t!=NULL)
		g_string_truncate(*nick,t-(*nick)->str);	/* discard space following the nick */

	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GLUS handlers ------------------------------ */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************************************************************/
/* send a message to a user having the given name and required user_flag and ext_flag */
/* zero *_msks force GLUS to ignore these flags                                       */
/* output: 0=ok else user not found                                                   */
/**************************************************************************************/
/* NOTE: we receive here the normal message but also '+' command sent by user */
/******************************************************************************/
/* incoming msg format "$To: chatname From: nickname $<nickname> msg" */
/**********************************************************************/
static int mpuc_send_to_named_user(GLUS_PARAM *glus_param)
{
	MPUC_ENTRY *mpuc;
	GString *sender_nickname=NULL;
	GString *msg_content=NULL;

	HL_LOCK_READ(mpuc_entry_list_mutex);
	mpuc=mpuc_entry_get_by_nickname(glus_param->nickname);
	if(mpuc==NULL)
	{
		HL_UNLOCK_READ(mpuc_entry_list_mutex);
		return 1;	/* MPUC not found */
	}

	if(extract_nickname_and_msg_content_from_msg(glus_param->msg->data,&sender_nickname,&msg_content))
	{
		if(sender_nickname!=NULL)
			g_string_free(sender_nickname,TRUE);
		if(msg_content!=NULL)
			g_string_free(msg_content,TRUE);

		HL_UNLOCK_READ(mpuc_entry_list_mutex);
		return 0;		/* MPUC found but invalid content */
	}

	if(msg_content->str[0]=='+')
	{
		public_chat_command(mpuc,sender_nickname, msg_content, glus_param);
	}
	else
	{
		/* only user of the chat can talk */
		if(is_a_user_of_the_public_chat(mpuc,sender_nickname->str)==FALSE)
		{
			g_string_free(sender_nickname,TRUE);
			g_string_free(msg_content,TRUE);
			HL_UNLOCK_READ(mpuc_entry_list_mutex);
			return 0;		/* MPUC found but user is not part of it */
		}

		send_msg_to_mpuc(mpuc,msg_content->str,sender_nickname->str);
	}
	HL_UNLOCK_READ(mpuc_entry_list_mutex);

	g_string_free(sender_nickname,TRUE);
	g_string_free(msg_content,TRUE);
	return 0;
}

/***************************************************************************************/
/* send a message to all users having required user_flag and ext_flag                  */
/* zero *_msks force GLUS to ignore these flags                                        */
/* if nickname is not NULL, the given user is ignored and will not receive the message */
/***************************************************************************************/
static void mpuc_send_to_all_users(GLUS_PARAM *glus_param)
{
	/* it is not possible to broadcast on multi public chat, thus there is nothing to do */
}

/*******************************************************/
/* create a GLUS_USER_INFO structure from a MPUC_ENTRY */
/*******************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static GLUS_USER_INFO *mpuc_get_user_info_from_mpuc_entry(MPUC_ENTRY *mpuc)
{
	GLUS_USER_INFO *gui=NULL;

	gui=malloc(sizeof(GLUS_USER_INFO));
	if(gui!=NULL)
	{
		gui->cnx_start_time=0;		/* no meaning */
		switch(mpuc->chat_type)
		{
			case PUBLIC_CHAT: gui->user_cnx_type="GChat";	/* Global chat */
									break;
			case PRIVATE_CHAT:gui->user_cnx_type="UChat";	/* User chat */
									break;
		}
		gui->user_nick=g_string_new(mpuc->mpuc_nickname->str);
		gui->shared_size=0;		/* no meaning */
		gui->user_description=g_string_new("");
		gui->user_mail=g_string_new("");
		gui->user_flag=1;			/* default flag: 1 */
		gui->ext_flag=0;			/* default flag: 0 */
		gui->privilege=0;			/* no privilege */
		gui->client_version=g_string_new("");
		gui->user_ip.s_addr=0;		/* no meaning */
	}
	else
		fprintf(stderr,"mpuc_get_user_info: out of memory.\n");

	return gui;
}

/*********************************************/
/* retrieve user information from a nickname */
/*********************************************/
static GLUS_USER_INFO *mpuc_get_user_info(const char *nickname)
{
	MPUC_ENTRY *mpuc;
	GLUS_USER_INFO *gui=NULL;
	
	HL_LOCK_READ(mpuc_entry_list_mutex);
	mpuc=g_hash_table_lookup(mpuc_entry_list,nickname);
	if(mpuc!=NULL)
		gui=mpuc_get_user_info_from_mpuc_entry(mpuc);

	HL_UNLOCK_READ(mpuc_entry_list_mutex);
	return gui;
}

/* -------------------------------------------------------------------------- */
/*************************************************/
/* add the glus user info of a user to the array */
/*****************************************************************************/
/* entry: key= GString of user_nick (the value is inside the next parameter) */
/*        value= LOCAL_USER_CNX_ENTRY * to process                           */
/*        user_data= GPtrArray of GLUS_USER_INFO                             */
/*****************************************************************************/
/* Note: the array has LOCK_READ set */
/*************************************/
static void add_public_chat_info_to_list(gpointer key, gpointer value, gpointer user_data)
{
   GPtrArray *array_of_gui_ptr=user_data;
   MPUC_ENTRY *mpuc=value;

   g_ptr_array_add(array_of_gui_ptr,mpuc_get_user_info_from_mpuc_entry(mpuc));
}

/******************************************************/
/* get all users. (add GLUS_USER_INFO * to the array) */
/******************************************************/
static void mpuc_get_users_info(GPtrArray *array_of_gui_ptr)
{
	HL_LOCK_READ(mpuc_entry_list_mutex);
	g_hash_table_foreach(mpuc_entry_list,add_public_chat_info_to_list,array_of_gui_ptr);
	HL_UNLOCK_READ(mpuc_entry_list_mutex);
	return;		/* multi public chat has no IP, thus there is nothing to do */
}

/* -------------------------------------------------------------------------- */
/**************************************************************************/
/* get all users having the given IP. (add GLUS_USER_INFO * to the array) */
/**************************************************************************/
static void mpuc_get_users_info_by_ip(GPtrArray *array_of_gui_ptr, struct in_addr ip)
{
	return;		/* multi public chat has no IP, thus there is nothing to do */
}

/*************************************************/
/* add a public chat 'nickname' in the user lust */
/*************************************************/
static void add_public_chat_to_user_nick_list(gpointer key_mpuc_nickname, gpointer value_mpuc, gpointer data_list)
{
	MPUC_ENTRY *mpuc=value_mpuc;
	GString *user_list=data_list;

	g_string_append(user_list,mpuc->mpuc_nickname->str);
	g_string_append(user_list,"$$");
}

/************************************************************************/
/* add the nicks to the provided strings (each nick is '$$' terminated) */
/************************************************************************/
/* list[0]= user nick list */
/* list[1]= op nick list   */
/***************************/
static void mpuc_get_nick_lists(GString **lists)
{
	HL_LOCK_READ(mpuc_entry_list_mutex);
	g_hash_table_foreach(mpuc_entry_list,add_public_chat_to_user_nick_list,(void*)lists[0]);
	HL_UNLOCK_READ(mpuc_entry_list_mutex);
}

/****************************************************************/
/* add the shared size and number of users to current variables */
/****************************************************************/
static void mpuc_add_hub_users_stat(guint64 *shared_size, unsigned int *nb_users)
{
#warning code to write ?
}

/******************************************/
/* kick the user having the given name    */
/* the user is kicked by kicking_nickname */
/* output: 0=ok else user not found       */
/******************************************/
static int mpuc_kick_named_user(const char *kicking_nickname, const char *nickname)
{
	return -1;	/* multi public chat has no kickable users */
}

/*************************************************/
/* disconnect the user having the given nickname */
/* output: 0=ok else user not found              */
/*************************************************/
static int mpuc_disconnect_named_user(const char *nickname)
{
	return -1;	/* multi public chat has no disconnectable users */
}

/******************************************/
/* remove a user from a multi public chat */
/***********************************************************/
/* output: FALSE= mpuc key/value pair remains in the table */
/*         TRUE= mpuc key/value should be deleted          */
/***********************************************************/
static gboolean remove_leaving_user(gpointer key_mpuc_nickname, gpointer value_mpuc, gpointer data_unickname)
{
	MPUC_ENTRY *mpuc=value_mpuc;
	const char *nickname=data_unickname;
	gchar *tmp;
	GString *str;

	if(remove_a_user_of_the_public_chat(mpuc,nickname)==FALSE)
		return FALSE;		/* keep the chat */

	/* notify its removal to the user */
	str=g_string_new("");
	g_string_sprintf(str,"$To: %s From: %s $You have left the chat.\r\n|", nickname, mpuc->mpuc_nickname->str);
	GLUS_SEND_TO_A_NICK(nickname,str);		/* don't free str */

	/* and also to other user and destroy the chat if required */
	if((mpuc->user_lst->len==0)&&(mpuc->chat_type!=PUBLIC_CHAT))	/* empty PUBLIC_CHAT is allowed */
	{
		delete_multi_public_chat(mpuc,FALSE);
		return TRUE;		/* destroy the chat */
	}
	else
	{
		tmp=g_strconcat(nickname," has left the chat.|",NULL);
		send_msg_to_mpuc(mpuc,tmp,NULL);
		g_free(tmp);
		return FALSE;		/* keep the chat */
	}
}

/**************************************************************/
/* function called after a user was disconnected from the hub */
/**************************************************************/
static void mpuc_user_is_disconnected(const char *nickname)
{
	HL_LOCK_WRITE(mpuc_entry_list_mutex);
	g_hash_table_foreach_remove(mpuc_entry_list,remove_leaving_user,(void*)nickname);
	HL_UNLOCK_WRITE(mpuc_entry_list_mutex);
}

/**************************************************************************** */
/**************************************************************************** */
/****************************** GLUS functions ****************************** */
/**************************************************************************** */
/**************************************************************************** */
static GLOBAL_USER_CALLBACKS mpuc=
								{
									"Multi public chat GLUS",
									mpuc_send_to_named_user,
									mpuc_send_to_all_users,
									mpuc_get_user_info,
									mpuc_get_users_info,
									mpuc_get_users_info_by_ip,
									mpuc_get_nick_lists,
									mpuc_add_hub_users_stat,
									mpuc_kick_named_user,
									mpuc_disconnect_named_user,
									mpuc_user_is_disconnected,
									NULL,	// mpuc_user_do_hello
									NULL,	// mpuc_user_do_quit
									NULL,	// mpuc_user_do_myinfo
									NULL, // mpuc_user_do_xsearch
									NULL, // mpuc_md4set_received
								};

/***************************************************/
/* register multipublicchat user virtual functions */
/***************************************************/
void multi_public_chat_init(void)
{
	mpuc_entry_list=g_hash_table_new(g_str_hash,g_str_equal);

	global_user_register(&mpuc);
}


/**************************************************************************** */
/**************************************************************************** */
/***************************** hub_cmd functions **************************** */
/**************************************************************************** */
/**************************************************************************** */

/*****************************/
/* use to join a public chat */
/*****************************/
int cmd_join_public_chat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base,
               const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	int i;
	GString *outstr;

	/************************************************************************/
	/* cmd = '+join'                                                        */
	/* ptr = luce of the user sending the message                           */
	/* dest_base = "<Kernel> " (if the command was sent from a public chat) */
	/*          or "$To: sender From: dest $" (if the command was sent from */
	/*                                         a private chat)              */
	/* str= initial string "+join ...."                                     */
	/* xtra_param= NULL                                                     */
	/* splitted_str = array of strings. It is 'str' splitted along ' '      */
	/************************************************************************/
	i=splitted_str->len;
	if((i<2)||(i>3))
	{
		g_string_append((*dest_base),"<Kernel> bad syntax. Syntax: +join chatname [chatpassword]|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	outstr=g_string_new("");
	g_string_sprintf(outstr,"$To: %s From: %s $%s|",(char*)(g_ptr_array_index(splitted_str,1)),luce->user_nick->str,str->str);
	GLUS_SEND_TO_A_NICK(g_ptr_array_index(splitted_str,1),outstr);	/* don't free outstr */

	/* 3 cases can appear here: */
	/* - the chat is local => we go to mpuc_send_to_named_user */
	/* - the chat is remote => we go to the hub relay which will call mpuc_send_to_named_user on the remote side */
	/* - the chat does not exist => the message is lost */
	return 0;
}

/***************************************/
/* list all users of a chat of the hub */
/***************************************/
int cmd_list_public_chat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base,
               const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	GString *outstr;

	/************************************************************************/
	/* cmd = '+list'                                                        */
	/* ptr = luce of the user sending the message                           */
	/* dest_base = "<Kernel> " (if the command was sent from a public chat) */
	/*          or "$To: sender From: dest $" (if the command was sent from */
	/*                                         a private chat)              */
	/* str= initial string "+list ...."                                     */
	/* xtra_param= NULL                                                     */
	/* splitted_str = array of strings. It is 'str' splitted along ' '      */
	/************************************************************************/
	if((splitted_str->len)!=2)
	{
		g_string_append((*dest_base),"<Kernel> bad syntax. Syntax: +list chatname|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	outstr=g_string_new("");
	g_string_sprintf(outstr,"$To: %s From: %s $%s|",(char*)(g_ptr_array_index(splitted_str,1)),luce->user_nick->str,str->str);
	GLUS_SEND_TO_A_NICK(g_ptr_array_index(splitted_str,1),outstr);	/* don't free outstr */

	/* 3 cases can appear here: */
	/* - the chat is local => we go to mpuc_send_to_named_user */
	/* - the chat is remote => we go to the hub relay which will call mpuc_send_to_named_user on the remote side */
	/* - the chat does not exist => the message is lost */
	return 0;
}

/***********************/
/* leave a public chat */
/***********************/
int cmd_leave_public_chat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base,
               const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	GString *outstr;

	/************************************************************************/
	/* cmd = '+leave'                                                       */
	/* ptr = luce of the user sending the message                           */
	/* dest_base = "<Kernel> " (if the command was sent from a public chat) */
	/*          or "$To: sender From: dest $" (if the command was sent from */
	/*                                         a private chat)              */
	/* str= initial string "+leave ...."                                    */
	/* xtra_param= NULL                                                     */
	/* splitted_str = array of strings. It is 'str' splitted along ' '      */
	/************************************************************************/
	if((splitted_str->len)!=2)
	{
		g_string_append((*dest_base),"<Kernel> bad syntax. Syntax: +leave chatname|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	outstr=g_string_new("");
	g_string_sprintf(outstr,"$To: %s From: %s $%s|",(char*)(g_ptr_array_index(splitted_str,1)),luce->user_nick->str,str->str);
	GLUS_SEND_TO_A_NICK(g_ptr_array_index(splitted_str,1),outstr);	/* don't free outstr */

	/* 3 cases can appear here: */
	/* - the chat is local => we go to mpuc_send_to_named_user */
	/* - the chat is remote => we go to the hub relay which will call mpuc_send_to_named_user on the remote side */
	/* - the chat does not exist => the message is lost */
	return 0;
}

/****************************/
/* create a new public chat */
/****************************/
int cmd_create_public_chat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base,
               const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	int i;
	int ret;
	int multi_allow;
	int chat_id;

	/************************************************************************/
	/* cmd = '+new_chat'                                                    */
	/* ptr = luce of the user sending the message                           */
	/* dest_base = "<Kernel> " (if the command was sent from a public chat) */
	/*          or "$To: sender From: dest $" (if the command was sent from */
	/*                                         a private chat)              */
	/* str= initial string "+new_chat ...."                                 */
	/* xtra_param= NULL                                                     */
	/* splitted_str = array of strings. It is 'str' splitted along ' '      */
	/************************************************************************/
	G_LOCK(conf_file);
	ret=e_db_int_get(conf_file,"MULTICHAT",&multi_allow);
	G_UNLOCK(conf_file);

	if(ret==0)
	{
		g_string_sprintfa((*dest_base),"MULTICHAT key (int) does not exist, multichat creation is disabled\r\n|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	switch(multi_allow)
	{
		case 0:	/* disabled for everyone */
					g_string_sprintfa((*dest_base),"multichat creation is disabled\r\n|");
					send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
					return 0;
		
		case 1:	/* op and master only */
					if (!(luce->privilege & OP_PRIV))
					{
						g_string_sprintfa((*dest_base),"multichat creation is only available to Op users\r\n|");
						send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
						return 0;
					}
					break;
	
		case 2:	/* allowed for everyone */
					break;
	}
	
	i=splitted_str->len;
	if((i<2)||(i>3))
	{
		g_string_append((*dest_base),"bad syntax. Syntax: +new_chat chatname [chatpassword]|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	/* chat creation is only on the local hub, there is no relay */
	chat_id=create_multi_public_chat(g_ptr_array_index(splitted_str,1), 
												PRIVATE_CHAT,
												(i==3)?g_ptr_array_index(splitted_str,2):NULL);	/* add password if exists */
	if(chat_id==-1)
	{
		g_string_append((*dest_base),"Not enough memory to create a new public chat|");
		send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
		return 0;
	}

	/* don't forget to auto-join a user to the newly created chat */
	join_a_user_to_a_public_chat(luce->user_nick->str,g_ptr_array_index(splitted_str,1),luce);

	g_string_sprintfa((*dest_base),"chat %s created|", (char*)g_ptr_array_index(splitted_str,1));
	send_const_str_to_luce(luce,(*dest_base)->str);	/* we cannot use *dest_base because its reference cannot be stolen */
	return 0;
}

