/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_cnx_handshake.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hub_cnx_handshake.c,v 2.12 2003/04/13 07:50:18 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <glib.h>

#include "hub_cnx_handshake.h"
#include "ged.h"
#include "bin_xf_io.h"
#include "gvar.h"
#include "main.h"
#include "key.h"
#include "tos_key.h"
#include "hub_cnx_lst.h"
#include "hub_passwd.h"
#include "md5.h"
#include "network.h"

static GArray *hub_in_handshake=NULL;

static guint8 local_hub_id[HUB_ID_LEN];
static char *local_hub_name=NULL;

/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */
/* ********************* hub connection handshake handler ******************* */
/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */

/****************************************************/
/* register a new socket into the handshake handler */
/****************************************************/
/* if remote_addr ==NULL, the function computes it */
/***************************************************/
void hub_handshake_register_new_fd(int sock_fd, struct in_addr *remote_addr)
{
	HUB_HSHAKE nw;
	int i;
	GByteArray *gba;

	/* reset the socket to its original mode to avoid data loss during transfer */
	set_bloquant_sock(sock_fd);

	nw.bxfio=create_bin_xfio(sock_fd,remote_addr);
	memset(nw.remote_hub_id,0,HUB_ID_LEN);

	/* generate the varykey */
	for(i=0;i<MD_BLOC_SIZE;i++)
  		nw.varykey[i]=rand();

	memset(nw.used_clus_id,0,MD_BLOC_SIZE);
	memset(nw.wanted_md_result,0,MD_BLOC_SIZE);
	nw.log_step=STEP_ID_SENT;

	nw.f_hub_name=NULL;
	nw.f_hc_type=0;
	nw.f_hop_cost=0;

	g_array_append_val(hub_in_handshake,nw);

	/* create initial stage bloc */
	gba=chunk_convert_param_to_mem(CC_HUB_LOG_S0,2,
	                                 SC_PARAM_PTR(SC_HUB_ID, HUB_ID_LEN, local_hub_id),
	                                 SC_PARAM_PTR(SC_HUB_VARIx, MD_BLOC_SIZE, nw.varykey)
	                              );

	bin_xfio_append_new_gba_to_glist_outgoing(nw.bxfio,gba,FALSE);	/* don't duplicate gba */
}

/*************************************/
/* destroy an entry under monitoring */
/**********************************************/
/* if keep_xfio == FALSE, the xfio is deleted */
/* else, the xfio is left untouched           */
/**********************************************/
static void hub_handshake_destroy_hub_hshake(int entry_idx, gboolean keep_xfio)
{
	HUB_HSHAKE *ch;

	ch=&(g_array_index(hub_in_handshake,HUB_HSHAKE,entry_idx));

	if(keep_xfio==FALSE)
		delete_bin_xfio(ch->bxfio);

	if(ch->f_hub_name!=NULL)
		free(ch->f_hub_name);
	g_array_remove_index_fast(hub_in_handshake,entry_idx);
}

/*******************************************************************/
/* convert a registered HUB_HSHAKE into a registered HUB_CNX_ENTRY */
/*******************************************************************/
static void hub_handshake_convert_hub_hshake_to_hub_cnx_entry(HUB_HSHAKE *ch, int entry_idx)
{
	int flag;
	HUB_CNX_ENTRY *hce;
	GByteArray *gba;

	/* set the keep-alive flag to detect broken connection */
	flag=1;
   setsockopt(ch->bxfio->sock_fd, SOL_SOCKET, SO_KEEPALIVE, &flag, sizeof(flag));

	hce=hc_create_entry(ch->bxfio, ch->f_hub_name, ch->f_hc_type, ch->f_hop_cost,ch->remote_hub_id, ch->used_clus_id);

	hub_handshake_destroy_hub_hshake(entry_idx, TRUE);	/* don't delete BIN_XF_IO */

	if(hce==NULL)		/* entry creation failed, abort */
		return;

	/* retrieve the remote user list */
	gba=chunk_convert_param_to_mem(CC_USER_LST_REQ,0);
	bin_xfio_append_new_gba_to_glist_outgoing(hce->hub_bxfio,gba,FALSE);   /* don't duplicate gba */
}


/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------------ misc functions ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static void disp_hub_hshake_info(HUB_HSHAKE *ch)
{
	unsigned int hip;

	hip=ntohl(ch->bxfio->user_ip.s_addr);
	printf("hubloginfailure %hhu.%hhu.%hhu.%hhu\n",
			 (unsigned char)(hip>>24)&0xff,
			 (unsigned char)(hip>>16)&0xff,
			 (unsigned char)(hip>>8) &0xff,
			 (unsigned char) hip	  &0xff);
	return;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------- step handler ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/***********************************************************/
/* the CC_HUB_LOG_S0 was sent, we wait for a CC_HUB_LOG_S0 */
/***********************************************************/
/* next steps: on_success, STEP_CHALLENGE_SENT */
/*             else CNX_CLOSED is set          */
/***********************************************/
static void hsf_step_id_sent(HUB_HSHAKE *ch)
{
	CHUNK_CORE *cc;
	SUBCHUNK_CORE *sc;
	guint8 temp_md[MD_BLOC_SIZE];
	guint8 md_val[HUB_ID_LEN+MD_BLOC_SIZE+MD_BLOC_SIZE];	/* len: hub_id + varykey + clus_id */
	GByteArray *gba;
	
	cc=bin_xfio_take_first_chunk_of_glist_incoming(ch->bxfio);
	if(cc==NULL)
	{
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	if(cc->chunk_type!=CC_HUB_LOG_S0)
	{
		fprintf(stderr,"Want CC_HUB_CHALLENGE, have: %d\n",cc->chunk_type);
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	/* retrieve remote_hub_id */
	sc=chunk_get_subcore(cc,SC_HUB_ID);
	if((sc==NULL)||(sc->chunk_size!=HUB_ID_LEN))
	{
		fprintf(stderr,"CC_HUB_CHALLENGE has no SC_HUB_ID or with an invalid length\n");
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;
		return;
	}
	memcpy(ch->remote_hub_id,sc->chunk_content,HUB_ID_LEN);

	/* get clust_id from passwd file and remote_hub_id */
	if(hub_passwd_get_clus_id_from_hub_id(ch->remote_hub_id,ch->used_clus_id)==FALSE)
	{
		fprintf(stderr,"CC_HUB_CHALLENGE: no known CLUS_ID for this HUB_ID\n");
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;
		return;
	}

	/* retrieve varykey */
	sc=chunk_get_subcore(cc,SC_HUB_VARIx);
	if((sc==NULL)||(sc->chunk_size!=MD_BLOC_SIZE))
	{
		fprintf(stderr,"CC_HUB_CHALLENGE has no SC_HUB_VARIx or with an invalid length\n");
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;
		return;
	}

	memcpy(&md_val[0],ch->remote_hub_id,HUB_ID_LEN);
	memcpy(&md_val[HUB_ID_LEN],sc->chunk_content,MD_BLOC_SIZE);
	memcpy(&md_val[HUB_ID_LEN+MD_BLOC_SIZE],ch->used_clus_id,MD_BLOC_SIZE);
	/* compute md5(remote_hub_id.sc->chunk_content[SC_HUB_VARIx].clust_id) into temp_md */
	do_md5(md_val,HUB_ID_LEN+MD_BLOC_SIZE+MD_BLOC_SIZE, temp_md);
	
	/* send the challenge */
	gba=chunk_convert_param_to_mem(CC_HUB_CHALLENGE,1,
	                                 SC_PARAM_PTR(SC_HUB_CHALLENGE, MD_BLOC_SIZE, temp_md)
	                              );
	bin_xfio_append_new_gba_to_glist_outgoing(ch->bxfio,gba,FALSE);	/* don't duplicate gba */

	memcpy(&md_val[0],local_hub_id,HUB_ID_LEN);
	memcpy(&md_val[HUB_ID_LEN],ch->varykey,MD_BLOC_SIZE);
	memcpy(&md_val[HUB_ID_LEN+MD_BLOC_SIZE],ch->used_clus_id,MD_BLOC_SIZE);
	/* compute wanted_md_result= md5(local_hub_id.ch->varykey,clus_id) */
	do_md5(md_val,HUB_ID_LEN+MD_BLOC_SIZE+MD_BLOC_SIZE, ch->wanted_md_result);

	chunk_free(cc);
	ch->log_step=STEP_CHALLENGE_SENT;
}
		
/*****************************************************************/
/* the CC_HUB_CHALLENGE was sent, we wait for a CC_HUB_CHALLENGE */
/*****************************************************************/
/* next steps: on_success, STEP_LOG_CHECK */
/*             else CNX_CLOSED is set     */
/******************************************/
static void hsf_step_challenge_sent(HUB_HSHAKE *ch)
{
	CHUNK_CORE *cc;
	SUBCHUNK_CORE *sc;
	GByteArray *gba;

	cc=bin_xfio_take_first_chunk_of_glist_incoming(ch->bxfio);
	if(cc==NULL)
	{
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	if(cc->chunk_type!=CC_HUB_CHALLENGE)
	{
		fprintf(stderr,"Want CC_HUB_CHALLENGE, have: %d\n",cc->chunk_type);
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	sc=chunk_get_subcore(cc,SC_HUB_CHALLENGE);
	if((sc==NULL)||(sc->chunk_size!=MD_BLOC_SIZE))
	{
		fprintf(stderr,"CC_HUB_CHALLENGE has no SC_HUB_CHALLENGE or with an invalid length\n");
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;
		return;
	}

	if(memcmp(ch->wanted_md_result,sc->chunk_content,MD_BLOC_SIZE))
	{
		gint8 ret_val=ERR_INVALID_CHALLENGE_REPLY;
		fprintf(stderr,"CC_HUB_CHALLENGE's SC_HUB_CHALLENGE doesn't match with wanted one\n");
		disp_hub_hshake_info(ch);
		chunk_free(cc);

		gba=chunk_convert_param_to_mem(CC_HUB_LOGIN,1,
	                                 SC_PARAM_GINT8(SC_NUM_ERROR, ret_val)
	                              );
		bin_xfio_append_new_gba_to_glist_outgoing(ch->bxfio,gba,FALSE);	/* don't duplicate gba */
		ch->bxfio->closed=CNX_WILL_DIE;
		return;
	}

	/* the challenge is valid, send a CC_HUB_LOGIN */
	{
		guint16 hop_cost=htons(1);
		guint8 hc_type=HC_TYPE_3;		
#warning connection is default to HC_TYPE_3
		gba=chunk_convert_param_to_mem(CC_HUB_LOGIN,3,
												SC_PARAM_PTR(SC_HUB_NAME,strlen(local_hub_name),local_hub_name),
                                 	SC_PARAM_GUINT8(SC_HUB_CNX_TYPE, hc_type),
                                 	SC_PARAM_GUINT16(SC_HUB_HOP_COST, hop_cost)
												);
	}
	bin_xfio_append_new_gba_to_glist_outgoing(ch->bxfio,gba,FALSE);	/* don't duplicate gba */

	chunk_free(cc);
	ch->log_step=STEP_LOG_CHECK;
}

/*********************************************************/
/* the CC_HUB_LOGIN was sent, we wait for a CC_HUB_LOGIN */
/*********************************************************/
/* next steps: on_success, STEP_LOG_DONE */
/*             else CNX_CLOSED is set    */
/*****************************************/
static void hsf_step_log_check(HUB_HSHAKE *ch)
{
	CHUNK_CORE *cc;
	SUBCHUNK_CORE *sc;

	cc=bin_xfio_take_first_chunk_of_glist_incoming(ch->bxfio);
	if(cc==NULL)
	{
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	if(cc->chunk_type!=CC_HUB_LOGIN)
	{
		fprintf(stderr,"Want CC_HUB_LOGIN, have: %d\n",cc->chunk_type);
		abrt:
		disp_hub_hshake_info(ch);
		chunk_free(cc);
		ch->bxfio->closed=CNX_CLOSED;	/* immediat termination */
		return;
	}

	/* check if the reply contains an error code */
	sc=chunk_get_subcore(cc,SC_NUM_ERROR);
	if(sc!=NULL)
	{
		fprintf(stderr,"CC_HUB_LOGIN returns the error %s\n",sc_num_err_to_str(subchunk_get_gint(sc)));
		goto abrt;
	}

	/* get remote hub name */
	sc=chunk_get_subcore(cc,SC_HUB_NAME);
	if(sc==NULL)
	{
		fprintf(stderr,"CC_HUB_LOGIN contains no SC_HUB_NAME\n");
		goto abrt;
	}

	ch->f_hub_name=malloc(sc->chunk_size+1);
	if(ch->f_hub_name==NULL)
	{
		fprintf(stderr,"CC_HUB_LOGIN: out of memory\n");
		goto abrt;
	}
	memcpy(ch->f_hub_name,sc->chunk_content,sc->chunk_size);
	ch->f_hub_name[sc->chunk_size]='\0';

	/* get hub cnx type */
	sc=chunk_get_subcore(cc,SC_HUB_CNX_TYPE);
	if(sc==NULL)
	{
		fprintf(stderr,"CC_HUB_LOGIN contains no SC_HUB_CNX_TYPE\n");
		goto abrt;
	}

	ch->f_hc_type=subchunk_get_guint(sc);
#warning perhaps we should check if the 2 connections types are identical

	/* get hub hop cost */
	sc=chunk_get_subcore(cc,SC_HUB_HOP_COST);
	if(sc==NULL)
	{
		fprintf(stderr,"CC_HUB_LOGIN contains no SC_HUB_HOP_COST\n");
		goto abrt;
	}

	ch->f_hop_cost=subchunk_get_guint(sc);
	
	chunk_free(cc);
	ch->log_step=STEP_LOG_DONE;
}


/* -------------------------------------------------------------------------- */
/*************************************/
/* all functions of the step handler */
/************************************************************************************/
/* on problem, the xfio status must be changed to CNX_CLOSED (immediat termination) */
/* or to CNX_WILL_DIE (delay termination until output buffer is flushed or timeout) */
/************************************************************************************/

#define STEP_FLAGS_NONE 0
#define STEP_FLAGS_LINK 1

typedef struct 
{
	void (*fnc)(HUB_HSHAKE *);
	int step_flags;
} HSTEP_FUNCTIONS;

static HSTEP_FUNCTIONS hstep_function[]=
				{
					{hsf_step_id_sent, STEP_FLAGS_NONE},          /* STEP_ID_SENT function */
					{hsf_step_challenge_sent, STEP_FLAGS_NONE},   /* STEP_CHALLENGE_SENT function */
					{hsf_step_log_check, STEP_FLAGS_NONE},        /* STEP_LOG_CHECK function */
					{NULL, STEP_FLAGS_NONE},                      /* final step (STEP_LOG_DONE) */
				};

/*******************************/
/* perform the following steps */
/*******************************/
static void do_steps(HUB_HSHAKE *ch)
{
	int flags;
	int cur_step;

	do
	{
		cur_step=ch->log_step;

		flags=hstep_function[cur_step].step_flags;
		(hstep_function[ch->log_step].fnc)(ch);
	}while((ch->bxfio->closed==CNX_OPENED)&&
	       (cur_step!=ch->log_step)&&			/* prevent eternal loop on a stage without I/O */
	       (flags==STEP_FLAGS_LINK));
}
	
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GED handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**************************/
/* function called at end */
/**************************/
static int ged_hub_hshake_exit(const GED_CALLBACKS *ged)
{
	return 0;
}

/********************************/
/* function called periodically */
/********************************/
static void ged_hub_hshake_periodic(const struct ged_callbacks *ged)
{
	int i;
	HUB_HSHAKE *ch;

	/* expire connection opened for a too long time */
	/* reverse loop to handle entry destruction without tips */
	for(i=hub_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(hub_in_handshake,HUB_HSHAKE,i));

		if((gl_cur_time - ch->bxfio->start_time)> MAX_HUB_HANDSHAKE_TIME)
		{	/* no need to check for closed connection, they are managed in the always1 function */
			hub_handshake_destroy_hub_hshake(i,FALSE);
		}
	}
}

/********************************/
/* function called at each loop */
/********************************/
static void ged_hub_hshake_always1(const struct ged_callbacks *ged)
{
	int i;
	HUB_HSHAKE *ch;

	/* expire connection opened for a too long time */
	/* reverse loop to handle entry destruction without tips */
	for(i=hub_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(hub_in_handshake,HUB_HSHAKE,i));
#ifdef DEBUG
		printf("ch: %p  step: %d\n",ch, (int)(ch->log_step));
#endif
		if(BIN_XFIO_IS_CLOSED(ch->bxfio))
		{
			hub_handshake_destroy_hub_hshake(i,FALSE);
			continue;
		}

		if(BIN_XFIO_INPUT_AVAILABLE(ch->bxfio))
		{
			do_steps(ch);

			if(BIN_XFIO_IS_CLOSED(ch->bxfio))
			{	/* maybe the connection is dead */
				hub_handshake_destroy_hub_hshake(i,FALSE);
				continue;
			}
		}
	}

	/* convert successful handshake into full local user entry */
	for(i=hub_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(hub_in_handshake,HUB_HSHAKE,i));
		if(ch->log_step==STEP_LOG_DONE)
		{
			hub_handshake_convert_hub_hshake_to_hub_cnx_entry(ch,i);
		}
	}
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------- initialization of the handler --------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static GED_CALLBACKS hub_hshake_ged=
				{
					"hub connection handshake GED", /* ged_name For debug */
					ged_hub_hshake_exit, /* destructor */
               NULL,
               NULL,
               ged_hub_hshake_periodic,
               ged_hub_hshake_always1,
               NULL,
					NULL
				};

/***********************************************/
/* function initializing the handshake handler */
/***********************************************/
GED_CALLBACKS *hub_handshake_init(void)
{
	char *hubname;

	/* retrieve the HUBNAME (if not defined, it is redefined to "noname") */
	G_LOCK(conf_file);
	hubname=e_db_str_get(conf_file,"HUBNAME");
	if(hubname==NULL)
	{
		e_db_str_set(conf_file,"HUBNAME","noname");
		hubname=strdup("noname");
	}
	G_UNLOCK(conf_file);
	if(!strcmp(hubname,"noname"))
	{
		fprintf(stderr,"WARNING: you must use a different HUBNAME value. Use -dbset to modify it\n"
		               "         and then restart the hub.\n");
	}

	/* load local_hub_id value */
	do_md5(hubname,strlen(hubname),local_hub_id);

	local_hub_name=hubname;	/* don't free hubname */

	/* set the HUB_ID in the database to the computed value */
	{
		char hub_id[MD_BLOC_SIZE*2+1];
		int i;

		for(i=0;i<MD_BLOC_SIZE;i++)
		{
			sprintf(hub_id+i*2,"%02X",((unsigned int)(local_hub_id[i]))&255);
		}
		hub_id[MD_BLOC_SIZE*2]='\0';

		G_LOCK(conf_file);
		e_db_str_set(conf_file,"HUB_ID",hub_id);
		G_UNLOCK(conf_file);
	}

	hub_in_handshake=g_array_new(FALSE,FALSE,sizeof(HUB_HSHAKE));


	return &hub_hshake_ged;
}

