/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hub_cmd.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hub_cmd.c,v 2.30 2003/06/03 17:23:39 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
	#include <getopt.h>
#else
	#ifdef HAVE_SYS_TIME_H
		#include <sys/time.h>
	#endif
 	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#ifdef HAVE_UNISTD_H
		#include <unistd.h>
	#endif  /* HAVE_UNISTD_H */
#endif

#ifdef HAVE_ERRNO_H
	#include <errno.h>
#else
	extern int errno;
#endif

#include <glib.h>
#include "hub_cmd.h"
#include "gvar.h"
#include "macro.h"
#include "passwd.h"
#include "toolkit.h"
#include "e_db.h"
#include "timed_out_string.h"
#include "tos_key.h"
#include "main.h"
#include "multi_public_chat_cmd.h"
#include "emb_perl.h"
#include "seen_list.h"
#include "afk_list.h"
#include "msg_list.h"
#include "user_cnx_lst.h"
#include "global_user_if.h"
#include "hub_passwd.h"
#include "md5.h"
#include "hub_cnx_lst.h"
#include "user_cnx_lst.h"

#ifndef INADDR_ANY
#define INADDR_ANY      0x00000000
#endif
#ifndef INADDR_NONE
#define INADDR_NONE     0xFFFFFFFF
#endif

#ifdef __CYGWIN__
#define GLIB_INT_64_BUG
#endif

#ifdef WIN32
#define GLIB_INT_64_BUG
#endif

#ifdef GLIB_INT_64_BUG
char	*llunsigned_to_str(guint64 a);
#endif


static int do_reload(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	load_dyn_conf();
	g_string_sprintfa((*dest_base),"Reload done|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return (0);
}

#ifdef DEBUG

static int do_dump(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{

	if(splitted_str->len<2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		int i;
		const char *user;
		CNX_ENTRY *ptr2;

		for(i=1; i<splitted_str->len; i++)
		{
			user=g_ptr_array_index(splitted_str,i);
			if(user!=NULL)
			{
				GString *tmp;

				ptr2 = find_cnx_by_nickname(user); 
				tmp = g_string_new((*dest_base)->str);
				
				if (ptr2 == NULL)
					g_string_sprintfa(tmp, "nick : %s not found in strucure|", user);
				else
				{
#ifdef GLIB_INT_64_BUG
					g_string_sprintfa(tmp, "user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, share=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
#else
					g_string_sprintfa(tmp, "user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, share=%Lu, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
#endif					
									ptr2->user_nick->str,
									ptr2->sock_fd,
									ptr2->user_cnx_type,
									ptr2->is_op,
									inet_ntoa(ptr2->ip),
#ifdef GLIB_INT_64_BUG
									llunsigned_to_str(ptr2->shared_size),
#else
									ptr2->shared_size,
#endif
									ptr2->incoming_commands,
									ptr2->outgoing_commands,
									ptr2->server_id,
									ptr2,
									ptr2->next);
				}
				send_const_str_to_luce(luce,tmp->str);
				g_string_free(tmp, TRUE);
			}
		}
	}
	return (0);
}

static int do_dump_queue(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	CNX_ENTRY	 *ptr2;
	GString	*tmp;

	for (ptr2=cnx_lst->wait; ptr2; ptr2 = ptr2->next)
	{
		tmp = g_string_new((*dest_base)->str);
		if (ptr2->handshake_done==1)
		{
			g_string_sprintfa(tmp,"user_nick=%s, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",
						ptr2->user_nick->str,
						ptr2->sock_fd,
						ptr2->user_cnx_type,
						ptr2->is_op,
						inet_ntoa(ptr2->ip),
						ptr2->incoming_commands,
						ptr2->outgoing_commands,
						ptr2->server_id,
						ptr2,
						ptr2->next);
		}
		else
		{
			g_string_sprintfa(tmp,"%duser_nick=?????, sock_fd=%d, user_cnx_type=%s, is_op=%i, IP=%s, incomming=%p, outcomming=%p, server_id=%d, local=%p, next=%p|",ptr2->connected,
						ptr2->sock_fd,
						ptr2->user_cnx_type,
						ptr2->is_op,
						inet_ntoa(ptr2->ip),
						ptr2->incoming_commands,
						ptr2->outgoing_commands,
						ptr2->server_id,
						ptr2,
						ptr2->next);
		}
		send_const_str_to_luce(luce,tmp->str);
		g_string_free(tmp, TRUE);
	}
	return(0);
}

#endif

static int do_set(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s. Usage : set var value\r\n|",cmd);
	else if (!strcmp(g_ptr_array_index(splitted_str,1), "MODERATE"))
	{
		if (gl_gui_mode == 0)
			printf("(%s)", (char *)g_ptr_array_index(splitted_str,2));
		sscanf(g_ptr_array_index(splitted_str,2), "%d", &gl_moderate_mode);
		if (gl_moderate_mode)
			g_string_sprintfa((*dest_base)," moderate mode enabled\r\n|");
		else
			g_string_sprintfa((*dest_base)," moderate mode disabled\r\n|");
	}
	else
	{
		g_string_sprintfa((*dest_base),"arg 1 in MODERATE\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return(0);
}

/****************************/
/* get ip from a named user */
/****************************/
static int do_get_ip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s user_name\r\n|",cmd);
	}
	else
	{
		const char *user;

		user=g_ptr_array_index(splitted_str,1);
		if(user!=NULL)
		{
			GLUS_USER_INFO *gui;

			gui=glus_get_user_info(user);
			if (gui)
			{
				g_string_sprintfa((*dest_base),"ip:%s %s\r\n|", gui->user_nick->str, inet_ntoa(gui->user_ip));
				glus_free_user_info(gui);
			}
			else
			{
				g_string_sprintfa((*dest_base),"user %s not found.\r\n|", user);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************/
/* get type from a username */
/****************************/
static int do_is_reg(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s user_name\r\n|",cmd);
	}
	else
	{
		const char *user;

		user=g_ptr_array_index(splitted_str,1);
		if(user!=NULL)
		{
			int ret;
			char type = '?';
			
			ret = get_account_type(user, &type);
			if (ret)
			{
				g_string_sprintfa((*dest_base),"type:%s %c\r\n|", user, type);
			}
			else
			{
				g_string_sprintfa((*dest_base),"type:%s ?\r\n|", user);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}


/****************************************/
/* give the username having the same IP */
/**********************************-*****/
static int do_rev_ip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s user_name\r\n|",cmd);
	}
	else
	{
		const char *str_ip;
		str_ip=g_ptr_array_index(splitted_str,1);

		if(str_ip!=NULL)
		{
			int i;
			struct in_addr ip;

			if (inet_aton(str_ip,&ip) == 0)	/* address invalid ? */
				g_string_sprintfa((*dest_base),"unable to parse %s\r\n|", str_ip);
			else
			{
				GPtrArray *gui_array;

				gui_array=glus_get_users_info_by_ip(ip);
				for(i=0;i<gui_array->len;i++)
				{
					GLUS_USER_INFO *gui;

					gui=g_ptr_array_index(gui_array,i);
					g_string_sprintfa((*dest_base),"rev:%s %s\r\n",inet_ntoa(gui->user_ip), gui->user_nick->str);
				}

				glus_free_user_info_ptrarray(gui_array);
				g_string_append((*dest_base),"end of list\r\n|");
			}
		}
		else
		{
			g_string_sprintfa((*dest_base),"internal error in %s\r\n|", __FILE__);
		}
	} 
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************/
/* set a new hub password entry */
/********************************/
static int do_hpset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 5)		/* at least 4 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id cluster_id hub_addr flags\r\n|",cmd);
	}
	else
	{
		HUB_PASS_ROW nw_hpr;
		gchar **fields;

		if(strlen(g_ptr_array_index(splitted_str,1))!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
			goto abrt;
		}

		if(strlen(g_ptr_array_index(splitted_str,2))!=(2*MD_BLOC_SIZE))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  cluster_id is a %d characters string|",cmd,MD_BLOC_SIZE*2);
			goto abrt;
		}

		nw_hpr.entry_flags=HPR_EF_BUSY;
		id_ascii_to_bin(g_ptr_array_index(splitted_str,1),nw_hpr.hub_id);
		id_ascii_to_bin(g_ptr_array_index(splitted_str,2),nw_hpr.cluster_id);
		strncpy_max(nw_hpr.hub_address, g_ptr_array_index(splitted_str,3), sizeof(nw_hpr.hub_address));
		memset(nw_hpr.reserved,0,sizeof(nw_hpr.reserved));
		nw_hpr.special_flags=0;

		fields=g_strsplit(g_ptr_array_index(splitted_str,4),".",0);
		if(fields)
		{
			char **flag;
			char *option;

			flag=fields;
			while((option=*flag++)!=NULL)
			{
				if(!strcmp(option,"None"))
					continue;
				if(!strcmp(option,"Autostart"))
				{
					nw_hpr.special_flags|=HPR_SF_AUTO_START;
					continue;
				}
				if(!strcmp(option,"Rejected"))
				{
					nw_hpr.special_flags|=HPR_SF_REJECTED;
					continue;
				}

				g_string_sprintfa((*dest_base),"Unkown special flag '%s' (ignored)\r\n",option);
			}
			g_strfreev(fields);
		}
		
		ucmd_hp_set(&nw_hpr);
		g_string_sprintfa((*dest_base),"%s done\r\n|",cmd);
	}
	abrt:
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**************************************************************************************************/
/* convert a HUB_PASS_ROW structure into a human readable form and append it to the given gstring */
/**************************************************************************************************/
static void append_decoded_hpr(GString *dest_base, HUB_PASS_ROW *hpr)
{
	if(hpr->entry_flags==HPR_EF_EMPTY)	/* the entry is empty */
		return;

	g_string_append(dest_base,"HubID: ");
	append_MD_to_str(dest_base,hpr->hub_id,HUB_ID_LEN);
	g_string_append(dest_base,"\r\n   ClusterID: ");
	append_MD_to_str(dest_base,hpr->cluster_id,HUB_ID_LEN);
	g_string_append(dest_base,"\r\n   HubAddr: ");
	g_string_sprintfa(dest_base,"%s\r\n   Flags:",hpr->hub_address);
	
	if(hpr->special_flags==0)
		g_string_append(dest_base," None\r\n");
	else
	{
		if(hpr->special_flags&HPR_SF_AUTO_START)
			g_string_append(dest_base," AutoStart");
		if(hpr->special_flags&HPR_SF_REJECTED)
			g_string_append(dest_base," Rejected(invalid_cluster_id)");
		g_string_append(dest_base,"\r\n");
	}
}

/****************************/
/* get a hub password entry */
/****************************/
static int do_hpget(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			HUB_PASS_ROW *hpr;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			hpr=ucmd_hp_get(hub_id);
			if(hpr==NULL)
			{
				g_string_sprintfa((*dest_base),"No known hub with the ID %s\r\n|",str_id);
			}
			else
			{
				append_decoded_hpr((*dest_base),hpr);
				g_string_append_c((*dest_base),'|');

				free(hpr);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************/
/* delete a hub password entry */
/*******************************/
static int do_hpdel(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			gboolean erased;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			erased=ucmd_hp_del(hub_id);
			if(erased==FALSE)
			{
				g_string_sprintfa((*dest_base),"Fail to delete the hub with the ID %s\r\n|",str_id);
			}
			else
			{
				g_string_sprintfa((*dest_base),"hub with the ID %s was deleted\r\n|",str_id);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************************/
/* schedule a hub entry for connection */
/***************************************/
static int do_hpcnx(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		char *str_id=g_ptr_array_index(splitted_str,1);

		if(strlen(str_id)!=(2*HUB_ID_LEN))
		{
			g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n  hub_id is a %d characters string|",cmd,HUB_ID_LEN*2);
		}
		else
		{
			gboolean erased;
			guint8 hub_id[HUB_ID_LEN];

			/* convert the ascii ID to its binary version */
			id_ascii_to_bin(str_id,hub_id);

			erased=ucmd_hp_reset_last_cnx_attempt(hub_id);
			if(erased==FALSE)
			{
				g_string_sprintfa((*dest_base),"The hub with the ID %s was not found\r\n|",str_id);
			}
			else
			{
				g_string_sprintfa((*dest_base),"hub with the ID %s was scheduled for connection attempt in the next 60 seconds\r\n|",str_id);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************/
/* get all hub password entries */
/********************************/
static int do_hpgetall(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 1)		/* at least the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s\r\n|",cmd);
	}
	else
	{
		HUB_PASS_ROW *hpr;
		int nb_rows;
		int i;

		hpr=ucmd_hp_get_rows(&nb_rows);
		if(nb_rows==0)
		{
			g_string_sprintfa((*dest_base),"No known hub ID\r\n|");
		}
		else
		{
			for(i=0;i<nb_rows;i++)
			{
				append_decoded_hpr((*dest_base),&(hpr[i]));
			}
			g_string_append_c((*dest_base),'|');
			free(hpr);
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**************************************/
/* get all running cluster connection */
/**************************************/
static int do_hpstat(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 1)		/* at least the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s\r\n|",cmd);
	}
	else
	{
		g_string_sprintfa((*dest_base),"Cluster connections:\r\n");
		
		hub_cnx_append_cluster_stats(*dest_base);

		g_string_sprintfa((*dest_base),"End of list.\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************************/
/* compute the MD5 hash of the given string */
/********************************************/
static int do_md5cmd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"usage: %s hub_id\r\n|",cmd);
	}
	else
	{
		guint8 md[MD_BLOC_SIZE];
		char *str=g_ptr_array_index(splitted_str,1);

		do_md5(str,strlen(str),md);

		g_string_sprintfa((*dest_base),"MD5hash of '%s' is ",str);
		append_MD_to_str((*dest_base),md,MD_BLOC_SIZE);

		g_string_sprintfa((*dest_base),"\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/************************************/
/* display the list of all commands */
/************************************/
static int do_help(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	g_string_sprintfa((*dest_base),"** Operator command list:\r\n"
				 "\r\n"
				 "-help                         display this help\r\n"
				 "+help                         normal user help\r\n"
			 	 "\r\n"
				 "-kick {nick OR ip} reason     kick the given user out of the hub\r\n"
				 "-skick nick reason            silent kick (Without printing anything on the chat)\r\n"
				 "-redir nick address reason    redirect the given user to this address\r\n"
				 "-bcast msg                    send the given message to everyone\r\n"
				 "-users			the amount of users since hubstart\r\n"
				 "-version nick			the client-version used by the given user\r\n"
				 "\r\n"
				 "-getip nick                   Get IP of a user.\r\n"
				 "-revip ip                     Reveal's all the users having the given IP.\r\n"
				 "-showtype nick                show the type of a user account.\r\n"
				 "\r\n"
				 "-tbanip ip [ip...]            add temporary ban on the given ip(s)\r\n"
				 "-tbankick nick reason         kick and tempban a user\r\n"
				 "-tunbanip ip [ip...]          remove temporary ban of the given ip(s)\r\n"
				 "-tunbannick nick              remove temporary ban of the given nickname\r\n"
				 "-tbanlist                     print the list of temporary banned ip(s)\r\n"
				 "-pbanip ip [ip...]            add permanent ban on the given ip(s)\r\n"
				 "-pbankick nick reason         kick and permban a user\r\n"
				 "-punbanip ip [ip...]          remove permanent ban of the given ip(s)\r\n"
				 "-punbannick nick              remove permanent ban of the given nickname\r\n"
				 "-pbanlist                     print the list of permanently banned ip(s)\r\n"
				 "-cleartban						  clearing tempbanlist\r\n"
				 "-clearpban						  clearing permbanlist\r\n"
				 "-users                        number of users seen since the hub start\r\n"
				 "\r\n"
				 "-perlclear                    clear all perl autoloaded handlers\r\n"
				 "-perlreset                    restart perl interpreter\r\n"
				 "\r\n"
				 "\r\n"
				 "*** Master command list:\r\n"
				 "\r\n"
				 "-ulist		display all existing accounts\r\n"
				 "-olist		display all OP existing accounts\r\n"
				 "-mlist		display all MASTER existing accounts\r\n"
				 "\r\n"
				 "-add nick passwd type         create a new registered user\r\n"
				 "-del nick                     delete a new registered user\r\n"
				 "-rename oldnick newnick       rename a registered user\r\n"
				 "-passwd nick newpasswd        change the password of a registered user\r\n"
				 "-type nick newtype            change the type of a registered user\r\n"
				 "\r\n"
				 "-dblist                           display database\r\n"
				 "-dbshow key                   display one database key\r\n"
				 "-dbadd key type value         create a new key in the database\r\n"
				 "-dbset key value              change the value of a key in the database\r\n"
				 "-dbdel key                    delete a key of the database\r\n"
				 "-reload                       update some variables in the hub\r\n"
				 "\r\n"
				 "-set VARIABLE VALUE           custom function (new)\r\n"
				 "-startprg name                start the given external program (-e required)\r\n"
				 "                              during hub start and name must be a valid name\r\n"
				 "                              in this directory\r\n"
				 "\r\n"
				 "-redirall address             redirect all users to this address\r\n"
				 "-rediralllocal address        redirect all local users to this address\r\n"
#ifdef DEBUG
				 "-dump user [users...]         dump data from the internal structure (use with care)\r\n"
				 "-dumpqueue                    dump login queue list (use with care)\r\n"
#endif
				 "\r\n"
				 "\r\n"
				 "*** cluster command list:\r\n"
				 "-hpset hub_id cluster_id hub_address flags\r\n"
				 "                              set an entry in the hub password file for the given hub.\r\n"
				 "                              hub_address can be either a FQDN or an IP. hub_id is the HUB_ID\r\n"
				 "                              key of the remote hub. cluster_id is MD5 of the cluster name.\r\n"
				 "                              flags value contains at least one of the following keywords.\r\n"
				 "                              keywords are separated by a dot (.) and contains no space.\r\n"
				 "                              Keywords: None, Autostart, Rejected.\r\n"
				 "-hpdel hub_id                 remove an entry in the hub password file for the given hub.\r\n"
				 "-hpcnx hub_id                 for a connection attempt for the given hub.\r\n"
				 "                              Note: the attempt will occur in the next 60 seconds.\r\n"
				 "-hpget hub_id                 retrieve the hub password file entry having the given hub_id.\r\n"
				 "-hpgetall                     retrieve all the hub password file entries.\r\n"
				 "-hpstat                       list the established cluster connections.\r\n"
				 "-md5 string                   compute the MD5 of the given string. Use this command to compute\r\n"
				 "                              the cluster ID from the cluster name for example\r\n"
				 "\r\n"
				 "\r\n"
				 "Note: If a parameter contains spaces, put the parameter between quotes\r\n"
				 "Note2: A quoted parameter cannot contains quote itself\r\n"
				 "\r\n|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************************************************************************/
/* display the list of all existing account with their account type and their password */
/***************************************************************************************/
static int do_ulist(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	GString *out;
	GString *(*fnc)(void)=(void*)xtra_param;

	out=(fnc)();
	g_string_sprintfa((*dest_base),"%s|",out->str);
	g_string_free(out,TRUE);
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************************************/
/* add new registered user to the registered user list */
/*******************************************************/
static int do_add(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=4)		/* 3 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *t;

		/* check login */
		t=g_ptr_array_index(splitted_str,1);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(account_exist(t,NULL,NULL))
		{
			g_string_sprintfa((*dest_base),"%s: an account with this nickname still exists.\r\n|",cmd);
			goto leave;
		}

		/* check password */
		t=g_ptr_array_index(splitted_str,2);
		if((strlen(t)+1)>PASS_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Password too long (max: %d).\r\n|",cmd,PASS_PASS_LEN-1);
			goto leave;
		}

		/* check type */
		t=g_ptr_array_index(splitted_str,3);
		if(strlen(t)!=1)
		{
			g_string_sprintfa((*dest_base),"%s: Account type is only 1 character (" STR_KNOW_USER_TYPE ").\r\n|",cmd);
			goto leave;
		}

		if(strchr(KNOW_USER_TYPE, t[0]) == NULL)
		{
			g_string_sprintfa((*dest_base),"%s: Account type can only be " STR_KNOW_USER_TYPE ".\r\n|",cmd);
			goto leave;
		}

		if(add_user_to_ulist(g_ptr_array_index(splitted_str,1),g_ptr_array_index(splitted_str,2),*t)!=0)
		{
			g_string_sprintfa((*dest_base),"%s: Error while creating nickname %s.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s: nickname %s created.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**********************************************************/
/* delete a registered user from the registered user list */
/**********************************************************/
static int do_del(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameter + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *t;

		/* check login */
		t=g_ptr_array_index(splitted_str,1);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(!account_exist(t,NULL,NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname exists.\r\n|",cmd);
			goto leave;
		}

		if(del_user_to_ulist(g_ptr_array_index(splitted_str,1))!=0)
		{
			g_string_sprintfa((*dest_base),"%s: Error while deleting nickname %s.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s: nickname %s deleted.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/******************************************************************/
/* rename an existing registered user of the registered user list */
/******************************************************************/
static int do_rename(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)		/* 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *t;

		/* check old login */
		t=g_ptr_array_index(splitted_str,1);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Old Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(!account_exist(t,NULL,NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname exists.\r\n|",cmd);
			goto leave;
		}

		/* check new login */
		t=g_ptr_array_index(splitted_str,2);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: New Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(account_exist(t,NULL, NULL))
		{
			g_string_sprintfa((*dest_base),"%s: An account having the new name still exists.\r\n|",cmd);
			goto leave;
		}

		if(rename_user_to_ulist(g_ptr_array_index(splitted_str,1),g_ptr_array_index(splitted_str,2))!=0)
		{
			g_string_sprintfa((*dest_base),"%s: Error while updating nickname %s.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s: nickname %s updated.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/******************************************************/
/* change the password of an existing registered user */
/******************************************************/
static int do_passwd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)		/* 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *t;

		/* check old login */
		t=g_ptr_array_index(splitted_str,1);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(!account_exist(t,NULL,NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname exists.\r\n|",cmd);
			goto leave;
		}

		/* check password */
		t=g_ptr_array_index(splitted_str,2);
		if((strlen(t)+1)>PASS_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Password too long (max: %d).\r\n|",cmd,PASS_PASS_LEN-1);
			goto leave;
		}

		if(chg_user_passwd_to_ulist(g_ptr_array_index(splitted_str,1),g_ptr_array_index(splitted_str,2))!=0)
		{
			g_string_sprintfa((*dest_base),"%s: Error while updating password of nickname %s.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s: password of nickname %s updated.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************************/
/* change an existing registered user type */
/*******************************************/
static int do_type(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)		/* 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *t;

		/* check old login */
		t=g_ptr_array_index(splitted_str,1);
		if((strlen(t)+1)>NICK_PASS_LEN)
		{
			g_string_sprintfa((*dest_base),"%s: Nickname too long (max: %d).\r\n|",cmd,NICK_PASS_LEN-1);
			goto leave;
		}

		/* account already exists ? */
		if(!account_exist(t,NULL,NULL))
		{
			g_string_sprintfa((*dest_base),"%s: No account with this nickname exists.\r\n|",cmd);
			goto leave;
		}

		/* check type */
		t=g_ptr_array_index(splitted_str,2);
		if(strlen(t)!=1)
		{
			g_string_sprintfa((*dest_base),"%s: Account type is only 1 character (" STR_KNOW_USER_TYPE ").\r\n|",cmd);
			goto leave;
		}

		if (strchr(KNOW_USER_TYPE, t[0]) == NULL)
		{
			g_string_sprintfa((*dest_base),"%s: Account type can only be " STR_KNOW_USER_TYPE ".\r\n|",cmd);
			goto leave;
		}

		if(chg_user_type_to_ulist(g_ptr_array_index(splitted_str,1),*t)!=0)
		{
			g_string_sprintfa((*dest_base),"%s: Error while updating type of nickname %s.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s: type of nickname %s updated.\r\n|",cmd,(char*)g_ptr_array_index(splitted_str,1));
		}
	}
	leave:

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*************************/
/* dump current database */
/*************************/
static int do_db(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=1)		/* 0 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GStringChunk *gsc=NULL;
		GPtrArray *gpa_name=NULL;
		GPtrArray *gpa_type=NULL;
		GPtrArray *gpa_val=NULL;

		if(!get_all_db_keys(&gpa_name,&gpa_type,&gpa_val,&gsc))
		{
			int i;
			GString *out;

			out=g_string_new("");

			/* the database content is sent in multiple small messages */
			/* to avoid buffer overflow in DC */
			for(i=0;i<gpa_name->len;i++)
			{
				out=g_string_assign(out,(*dest_base)->str);
				g_string_sprintfa(out,"'%s'\t'%s'\t'%s'|",
											(char*)g_ptr_array_index(gpa_type,i),
											(char*)g_ptr_array_index(gpa_name,i),
											(char*)g_ptr_array_index(gpa_val,i));
				send_const_str_to_luce(luce,out->str);
			}
			g_string_free(out,TRUE);
		}
		
		if(gsc!=NULL)
			g_string_chunk_free(gsc);
		if(gpa_name!=NULL)
			g_ptr_array_free(gpa_name,TRUE);
		if(gpa_type!=NULL)
			g_ptr_array_free(gpa_type,TRUE);
		if(gpa_val!=NULL)
			g_ptr_array_free(gpa_val,TRUE);

		g_string_sprintfa((*dest_base),"End of database dump|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/**********************/
/* dump the key value */
/**********************/
static int do_dbshow(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *tp;
		char *kn;

		G_LOCK(conf_file);
		kn=g_ptr_array_index(splitted_str,1);

		tp=e_db_type_get(conf_file,kn);
		if(tp==NULL)
		{
			g_string_sprintfa((*dest_base),"no key named '%s'\r\n|",kn);
		}
		else
		{
			if(!strcmp(tp,"int"))
			{
				/* value is an int */
				int vl;

				if(!e_db_int_get(conf_file,kn,&vl))
					vl=0;

				g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%d'|",
											tp,kn,vl);
			}
			else if(!strcmp(tp,"float"))
			{
				/* value is a float */
				float vl;

				if(!e_db_float_get(conf_file,kn,&vl))
					vl=0;

				g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%f'|",
											tp,kn,vl);
			}
			else
			{
				/* value is a string */
				char *vl;

				vl=e_db_str_get(conf_file,kn);
				if(vl==NULL)
				{
					g_string_sprintfa((*dest_base),"'%s'\t'%s'\t''|", tp,kn);
				}
				else
				{
					g_string_sprintfa((*dest_base),"'%s'\t'%s'\t'%s'|",
											tp,kn,vl);
					free(vl);
				}
			}
			free(tp);
		}
		G_UNLOCK(conf_file);
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*********************/
/* add the key value */
/*********************/
static int do_dbadd(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=4)		/* 3 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *tp;
		char *kn;

		G_LOCK(conf_file);

		kn=g_ptr_array_index(splitted_str,1);

		tp=e_db_type_get(conf_file,kn);
		if(tp!=NULL)
		{
			g_string_sprintfa((*dest_base),"a key with this name still exists. Error\r\n|");
			free(tp);
		}
		else
		{
			/* check type before adding */
			tp=g_ptr_array_index(splitted_str,2);
			if((strcmp(tp,"int")) && (strcmp(tp,"str")) && (strcmp(tp,"float")))
			{
				g_string_sprintfa((*dest_base),"Unknown type. Only 'int', 'str' and 'float' are allowed. Error\r\n|");
			}
			else
			{
				if(!strcmp(tp,"int"))
				{
					e_db_int_set(conf_file,kn,atoi(g_ptr_array_index(splitted_str,3)));
				}
				else if(!strcmp(tp,"str"))
				{
					e_db_str_set(conf_file,kn,g_ptr_array_index(splitted_str,3));
				}
				else
				{
					e_db_float_set(conf_file,kn,atof(g_ptr_array_index(splitted_str,3)));
				}
				g_string_sprintfa((*dest_base),"Key %s added.\r\n|",kn);
			}
		}
		G_UNLOCK(conf_file);
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/************************/
/* update the key value */
/************************/
static int do_dbset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<3)		/* at least 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *tp;
		char *kn;

		G_LOCK(conf_file);

		kn=g_ptr_array_index(splitted_str,1);

		tp=e_db_type_get(conf_file,kn);
		if(tp==NULL)
		{
			g_string_sprintfa((*dest_base),"No key with this name still exists. Error\r\n|");
		}
		else
		{
			if(!strcmp(tp,"int"))
			{
				e_db_int_set(conf_file,kn,atoi(g_ptr_array_index(splitted_str,2)));
			}
			else if(!strcmp(tp,"str"))
			{
				e_db_str_set(conf_file,kn,g_ptr_array_index(splitted_str,2));
			}
			else
			{
				e_db_float_set(conf_file,kn,atof(g_ptr_array_index(splitted_str,2)));
			}
			g_string_sprintfa((*dest_base),"Key %s updated.\r\n|",kn);
			free(tp);
		}
		G_UNLOCK(conf_file);
		load_dyn_conf();
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/********************************************/
/* test if the given key is a protected key */
/********************************************/
/* out: 0=not protected, 1=yes */
static int is_a_protected_key(char *kn)
{
	static const char *tbl_prot[]={
												"HUBNAME",
												"HUBADDR",
												"HUBDESC",
												"REG_HUB",
												"REG_SERV",
												"MAXUSER",
												"REDIR_FULL",
												"REDIR_ADDR",
												"ONLY_REG",
												NULL
											};
	int i;
	
	i=0;
	while(tbl_prot[i]!=NULL)
	{
		if(!strcmp(kn,tbl_prot[i]))
			return 1;
		i++;
	}

	return 0;
}

/*********************/
/* del the key value */
/*********************/
static int do_dbdel(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char *kn;

		kn=g_ptr_array_index(splitted_str,1);
		if(!is_a_protected_key(kn))
		{
			e_db_data_del(conf_file,kn);
			g_string_sprintfa((*dest_base),"Key %s deleted.\r\n|",kn);
		}
		else
		{
			g_string_sprintfa((*dest_base),"Key %s is a protected key, you cannot delete it.\r\n|",kn);
		}
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*****************************/
/* start an external program */
/*****************************/
static int do_startprg(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	char *progname;
	gchar *p;
	struct stat st;

	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		goto abrt;
	}

	if(ext_prog_dir==NULL)
	{
		g_string_sprintfa((*dest_base),"Hub has been started without external program directory.\r\n|");
		goto abrt;
	}

	progname=g_ptr_array_index(splitted_str,1);

	if(strchr(progname,'/')!=NULL)
	{
		g_string_sprintfa((*dest_base),"You cannot start an external program containing / in its name.\r\n|");
		goto abrt;
	}

	p=g_strconcat(ext_prog_dir,"/",progname,NULL);
	if(stat(p,&st)==-1)
	{
		g_string_sprintfa((*dest_base),"Error on %s: %s\r\n|",progname,strerror(errno));
	}
	else
	{
		/*
		start_new_ext_prog(progname)
		*/
		if (glus_user_is_connected(progname)==TRUE)
			g_string_sprintfa((*dest_base),"%s is already take by an other user\r\n|",progname);
		else
		{
			add_prog_to_start(progname);
			g_string_sprintfa((*dest_base),"%s started\r\n|",progname);
		}
	}
		
	g_free(p);

	abrt:
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***********************************************/
/* now lets disconnect given user from do_kick */
/***********************************************/
void proceed_user_kick(char *nick, LOCAL_USER_CNX_ENTRY *luce, GString *preason, GString **dest_base, int ban_type)
{
	if (preason!=NULL && strcmp("...", preason->str))
	{
		int bot_kick;
		GString *out;
		char *kickmsg = NULL;
		out=g_string_new("");

		if(ban_type==0)
				kickmsg="is disconnecting";
		if(ban_type==1)
				kickmsg="is kicking";
		if(ban_type==2)
				kickmsg="is kicking & banning";

		g_string_sprintf(out,"<%s> %s %s %s because: %s|",
										luce->user_nick->str,
										luce->user_nick->str,
										kickmsg,
										nick,
										preason->str);
										
		G_LOCK(conf_file);
		if (!e_db_int_get(conf_file,"DISPLAY_BOT_KICK",&bot_kick))
			bot_kick=1;
		G_UNLOCK(conf_file);

		if(bot_kick==0 && (luce->privilege & BOT_PRIV))
		{
			GLUS_SEND_TO_A_NICK(nick,out);	/* don't free out */
		}
		else
			GLUS_SEND_TO_EVERYONE(out);      /* don't free out */

	}
	/* kick the user */
	if (ban_type==0)
		glus_disconnect_named_user(nick);
	else
		glus_kick_named_user(luce->user_nick->str, nick);
}

/******************************************/
/* kick the given nick                    */
/* check if nick exists as ip or nickname */
/* and decide what to do                  */
/******************************************/
static int do_flood(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	/* 2 parameters (-flood nick) + reason */
	if (splitted_str->len < 3)
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
		return 0;
	}
	else
	{
		const char *nick;
		GLUS_USER_INFO *gui_nick;
		GString *preason;
		char *reason;
		int flood_count;

		G_LOCK(conf_file);
		if (!e_db_int_get(conf_file,"FLOOD_COUNT",&flood_count))
         flood_count=1;
      G_UNLOCK(conf_file); 

		reason=g_ptr_array_index(splitted_str,2);
		preason=g_string_new(reason);
		if (splitted_str->len > 3)
		{
			int i;
			for (i=3;i<splitted_str->len;i++)
			{
				if(g_ptr_array_index(splitted_str,i)!=NULL)
					g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
			}
		}
		nick=g_ptr_array_index(splitted_str,1);

		gui_nick = glus_get_user_info((char *)nick);
		
		if(gui_nick)
		{
			int i;
			int ban_type = 0;
			GString *mymsg;
			const unsigned int user_flag=1;

			mymsg=g_string_new("");

			g_string_sprintf(mymsg,"$ $DSL%c$$0$",(char)user_flag);
			for(i=0;i<flood_count;i++)
			{
				GString *floodmsg;
				floodmsg=g_string_new("");
				g_string_sprintf(floodmsg,"$Hello flood%i|$MyINFO $ALL flood%i %s|$To: %s From: flood%i $<flood%i> %s|",i,i,mymsg->str,gui_nick->user_nick->str,i,i,preason->str);
				glus_send_to_named_user(gui_nick->user_nick->str,0,0,0,0,floodmsg);
			}
			g_string_free(mymsg,TRUE);

			if(gui_nick->ext_flag & USER_TBAN)
				ban_type = 1;
			if(gui_nick->ext_flag & USER_PBAN)
				ban_type = 2;
			proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, ban_type);
			glus_free_user_info(gui_nick);
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
			send_const_str_to_luce(luce,(*dest_base)->str);
		}
		g_string_free(preason,TRUE);
	}
	printf("flood finished\n");
	return 0;
}

static int do_kick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	/* 2 parameters (-kick nick) + reason */
	if ((splitted_str->len < 2) || 
		(splitted_str->len < 3  && !(luce->privilege & MASTER_PRIV)))
	{ /* only a hub-master should be allowed to kick without a reason !!! */
			g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
			send_const_str_to_luce(luce,(*dest_base)->str);
			return 0;
	}
	else
	{
		const char *nick;
		char *reason=NULL;
		struct sockaddr_in addr_in;
		GLUS_USER_INFO *gui_nick;
		GString *preason=NULL;
		
		if (splitted_str->len > 2)
		{
			/* preparing kicking reason */
			reason=g_ptr_array_index(splitted_str,2);		
			preason=g_string_new(reason);
			if (splitted_str->len > 3)
			{
				int i;
				for (i=3;i<splitted_str->len;i++)
				{
					if(g_ptr_array_index(splitted_str,i)!=NULL)
						g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
				}
			}
		}

		nick=g_ptr_array_index(splitted_str,1);

		/* check if we've got a nick using this name */
		gui_nick = glus_get_user_info((char *)nick);

		/* lets see if we can use the given nick as an IP */
		if (inet_aton(nick,&(addr_in.sin_addr)) != 0)
		{	/* take the string as an IP */
			
			GPtrArray *gui_array;
			/********************************************************************/
			/* got an IP, but maybe where's more than one user with the same ip */
			/* so collect them into an array first and kick them all            */
			/* the nick should be unique                                        */
			/********************************************************************/

			gui_array=glus_get_users_info_by_ip(addr_in.sin_addr);

			/* are we able to use both nick and ip ? what should I do now ? */
			/* first check if we find both nick & ip */
			if(gui_nick && gui_array->len>0)
			{
				int i;
				int use_both=FALSE;

				for(i=0;i<gui_array->len;i++)
				{
					GLUS_USER_INFO *gui_ip;
					gui_ip=g_ptr_array_index(gui_array,i);

					/* the user is using his IP as nick as well ? */
					if(!strcmp(gui_ip->user_nick->str,gui_nick->user_nick->str))
						use_both=TRUE;
				}

				if(use_both==TRUE)
				{	/* found user using ip as nick as well -> kick it !!! */
					GLUS_USER_INFO *gui_ip;

					for(i=0;i<gui_array->len;i++)
					{
						if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
						{
							int ban_type = 0;
							if(gui_ip->ext_flag & USER_TBAN)
								ban_type = 1;
							if(gui_ip->ext_flag & USER_PBAN)
								ban_type = 2;
							proceed_user_kick(gui_ip->user_nick->str, luce, preason, dest_base, ban_type);
						}
					}
				}
				else
				{	/* different users found -> reply data to decide */
					GLUS_USER_INFO *gui_ip;

					g_string_sprintfa((*dest_base),"Attention found more than one user to kick:\r\n");
					g_string_sprintfa((*dest_base),"IP: %s	Nick: %s\r\n",inet_ntoa(gui_nick->user_ip), gui_nick->user_nick->str);
					for(i=0;i<gui_array->len;i++)
					{
						if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
							g_string_sprintfa((*dest_base),"IP: %s	Nick: %s\r\n",inet_ntoa(gui_ip->user_ip), gui_ip->user_nick->str);
					}
					g_string_sprintfa((*dest_base),"Please try the -kick command with the Parameter(ip/nick) you've not used|");
					send_const_str_to_luce(luce,(*dest_base)->str);
				}
			}
			else if(gui_nick && gui_array->len==0)
			{	/* found nick only -> kick it */
				int ban_type = 0;
				if(gui_nick->ext_flag & USER_TBAN)
					ban_type = 1;
				if(gui_nick->ext_flag & USER_PBAN)
					ban_type = 2;
				proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, ban_type);
			}
			else if(gui_array->len>0)
			{
				int i;
				GLUS_USER_INFO *gui_ip;

				for (i=0;i<gui_array->len;i++)
				{
					if((gui_ip=g_ptr_array_index(gui_array,i))!=NULL)
					{
						int ban_type = 0;
						if(gui_ip->ext_flag & USER_TBAN)
							ban_type = 1;
						if(gui_ip->ext_flag & USER_PBAN)
							ban_type = 2;
						proceed_user_kick(gui_ip->user_nick->str, luce, preason, dest_base, ban_type);
					}
				}
			}
			else
			{
				g_string_sprintfa((*dest_base),"%s not found|",nick);
				send_const_str_to_luce(luce,(*dest_base)->str);
			}

			if(gui_nick)
				glus_free_user_info(gui_nick);
			glus_free_user_info_ptrarray(gui_array);
		}
		else if (gui_nick)
		{
			int ban_type = 0;
			if(gui_nick->ext_flag & USER_TBAN)
				ban_type = 1;
			if(gui_nick->ext_flag & USER_PBAN)
				ban_type = 2;
			proceed_user_kick(gui_nick->user_nick->str, luce, preason, dest_base, ban_type);
			glus_free_user_info(gui_nick);
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
			send_const_str_to_luce(luce,(*dest_base)->str);
		}
		if(preason!=NULL)
			g_string_free(preason,TRUE);
	}
	/* no need for success-message - we have one in Main-chat */
	return 0;
}

static int do_ban_kick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<3)		/* at least 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		char	*nick;
		char	*reason;
		int	temp_ban;
		GLUS_USER_INFO	*gui;

		nick=g_ptr_array_index(splitted_str,1);
		reason=g_ptr_array_index(splitted_str,2);

		/* update the kicking user message */
		gui = glus_get_user_info(nick);

		if (gui)
		{
			if (gui->ext_flag & SHILD_MODE)
			{
				g_string_sprintfa((*dest_base),"%s is a VIP|",nick);
				send_const_str_to_luce(luce,(*dest_base)->str);
			}
			else
			{
				unsigned long key=(unsigned long)xtra_param;

				GString *preason;

				G_LOCK(conf_file);
				if(key==TBAN_TOSKEY)
				{
					if(!e_db_int_get(conf_file,"KICKTEMPBAN2",&temp_ban))
						temp_ban=1;
				}
				else
				{
					if(!e_db_int_get(conf_file,"KICKPERMBAN",&temp_ban))
						temp_ban=7;
				}
				G_UNLOCK(conf_file);

				preason=g_string_new(reason);
				if(splitted_str->len>3)
				{
					int i;
					for(i=3;i<splitted_str->len;i++)
					{
						if(g_ptr_array_index(splitted_str,i)!=NULL)
							g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
					}
				}
				/* g_string_sprintfa((*dest_base),"%s kicked (%s)|",nick, inet_ntoa(gui->user_ip)); */
				/* this should never happen - only rightclick-kick sends this message */
				/* if (strcmp("...", reason)) */
				{
					GString *msg;
					msg=g_string_new("");

					/* broadcast the kicking message */
					g_string_sprintf(msg,"<%s> %s is kicking %s because: %s|",
										luce->user_nick->str,
										luce->user_nick->str,
										nick,
										preason->str);

					GLUS_SEND_TO_EVERYONE(msg);      /* don't free msg */
				}
				
				add_tos_entry_v1_uniq(key,temp_ban*24*3600,(void*)(&(gui->user_ip)),sizeof(struct in_addr),nick,strlen(nick));

				/* why adding user in ban-list twice ? -> disconnect only */
				glus_disconnect_named_user(nick);

				/* { */
					/* GString *msg; */
					/* msg=g_string_new(""); */
					/* g_string_sprintf(msg,"<Hub-Security> The user %s was banned & kicked by %s. IP: %s|", */
											/* nick,luce->user_nick->str,inet_ntoa(gui->user_ip)); */
					/* GLUS_SEND_TO_EVERYONE(msg); */      /* don't free msg */
				/* } */

				g_string_free(preason,TRUE);
			}
			glus_free_user_info(gui);
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
			send_const_str_to_luce(luce,(*dest_base)->str);
		}
	}
	/* no success-message needed in cause of having one it in the main-chat */
	return 0;
}


static int do_close(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
   if(splitted_str->len<3)    /* min 2 parameters + the command */
   {
      g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
      send_const_str_to_luce(luce,(*dest_base)->str);
   }
   else
   {
      char    *reason;
      char    *nick;
		GLUS_USER_INFO *gui;

      nick=g_ptr_array_index(splitted_str,1);
      reason=g_ptr_array_index(splitted_str,2);

      /* update the kicking user message */
		gui = glus_get_user_info(nick);
      if (gui)
      {
         int i;
         GString *str;
         GString *preason;

         preason=g_string_new(reason);
         if(splitted_str->len>3)
         {
            for(i=3;i<splitted_str->len;i++)
            {
               if(g_ptr_array_index(splitted_str,i)!=NULL)
                  g_string_sprintfa(preason," %s",(char*)g_ptr_array_index(splitted_str,i));
            }
         }
         str=g_string_new("");
         g_string_sprintf(str,"<%s> %s has disconnected %s because: %s|",
                                    luce->user_nick->str,
                                    luce->user_nick->str,
                                    nick,
                                    preason->str);

			GLUS_SEND_TO_EVERYONE(str);      /* don't free str */

			glus_disconnect_named_user(nick);

			glus_free_user_info(gui);
      }
      else
      {
         g_string_sprintfa((*dest_base),"%s is not in the hub!|",nick);
         send_const_str_to_luce(luce,(*dest_base)->str);
      }
   }
	/* no success-message needed in cause of having one it in the main-chat */
   return 0;
}

static int do_silence_kick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=3)		/* 2 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		char	*nick;
		char	*reason;
		int temp_ban;
		GLUS_USER_INFO *gui;

		nick=g_ptr_array_index(splitted_str,1);
		reason=g_ptr_array_index(splitted_str,2);

		/* update the kicking user message */
		gui = glus_get_user_info(nick);

		if (gui)
		{
			if (gui->ext_flag & SHILD_MODE)
			{
				g_string_sprintfa((*dest_base),"%s is a VIP|",nick);
				glus_free_user_info(gui);
			}
			else
			{
				g_string_sprintfa((*dest_base),"%s kicked (%s)|",nick, inet_ntoa(gui->user_ip));
				G_LOCK(conf_file);
				if(!e_db_int_get(conf_file,"KICKTEMPBAN",&temp_ban))
						temp_ban=0;
				G_UNLOCK(conf_file);

				if(temp_ban!=0)
					add_tos_entry_v1_uniq(TBAN_TOSKEY,temp_ban*60,(void*)(&(gui->user_ip)),sizeof(struct in_addr),NULL,0);
				glus_disconnect_named_user(nick);
				glus_free_user_info(gui);
			}
		}
		else
		{
			g_string_sprintfa((*dest_base),"%s not found|",nick);
#if 0
			glus_free_user_info(gui); /* don't free gui, it is NULL */
#endif
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

static int do_tutel(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if ((splitted_str->len != 2) && ((splitted_str->len != 3)))		 /* 2 parameters + the command or more */
	{
		g_string_sprintfa((*dest_base),"usage:%s user_name Tutor\r\n|",cmd);
	}
	else
	{
#if 1
#warning code disabled (useful ?)
#else
		CNX_ENTRY *ent;
		
		ent = find_cnx_by_nickname(g_ptr_array_index(splitted_str,1));
		if (ent != NULL)
		{
			if (ent->tutor)
				free(ent->tutor);
			if (splitted_str->len == 3)
				ent->tutor = strdup(g_ptr_array_index(splitted_str,2));
			g_string_sprintfa((*dest_base),"Tutel done.|");
		}
		else
#endif
		{
			g_string_sprintfa((*dest_base),"no such username.|");
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

static int do_silence(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len < 3)		/* 2 parameters + the command or more*/
	{
		g_string_sprintfa((*dest_base),"usage:%s TYPE(in pPcCsSdDvVlLrRaAoO) username [username ...]\r\n"
		"P->PM\r\n"
		"C->Chat\r\n"
		"S->Search\r\n"
		"D->Download\r\n"
		"V->Vip(Shild)\r\n"
		"L->LOCK_PM_TUTOR\r\n"
		"R->REV_SILENT_CNX\r\n"
		"A->OTIST CC (N/A)\r\n"
		"O->OTIST PM\r\n"	
		"|",cmd);
	}
	else
	{
		char		*type;
		int			i;
		unsigned int	flag = 0;
		unsigned int	unflag = 0;

		type = g_ptr_array_index(splitted_str,1);

		if (strchr(type, 'P'))
			flag |= SILENT_PM;
		if (strchr(type, 'p'))
			unflag |= SILENT_PM;
			
		if (strchr(type, 'C'))
			flag |= SILENT_CC;
		if (strchr(type, 'c'))
			unflag |= SILENT_CC;
			
		if (strchr(type, 'S'))
			flag |= SILENT_SR;
		if (strchr(type, 's'))
			unflag |= SILENT_SR;
			
		if (strchr(type, 'D'))
			flag |= SILENT_CNX;
		if (strchr(type, 'd'))
			unflag |= SILENT_CNX;
			
		if (strchr(type, 'V'))
			flag |= SHILD_MODE;
		if (strchr(type, 'v'))
			unflag |= SHILD_MODE;
			
		if (strchr(type, 'L')) 
			flag |= LOCK_PM_TUTOR;
		if (strchr(type, 'l'))
			unflag |= LOCK_PM_TUTOR;
			
		if (strchr(type, 'R'))
			flag |= REV_SILENT_CNX;
		if (strchr(type, 'r'))
			unflag |= REV_SILENT_CNX;
			
		if (strchr(type, 'A'))
			flag |= OTIST_CC;
		if (strchr(type, 'a'))
			unflag |= OTIST_CC;

		if (strchr(type, 'O'))
			flag |= OTIST_PM;
		if (strchr(type, 'o'))
			unflag |= OTIST_PM;


		for (i = 2; i < splitted_str->len; i++)
		{
			user_cnx_entry_update_ext_flag_by_nickname(g_ptr_array_index(splitted_str,i),flag,unflag);
		}
		g_string_sprintfa((*dest_base)," done|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************/
/* redirect the given nick */
/***************************/
static int do_redir(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=4)		/* 3 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GString *rd_msg;
		char *rd;
		char *nm;
		char *msg;
		rd_msg=g_string_new("");

		nm=g_ptr_array_index(splitted_str,1);
		if (gl_gui_mode == 0)
			printf("redirect by: %s\n",nm);
		rd=g_ptr_array_index(splitted_str,2);
		if (gl_gui_mode == 0)
			printf("redirect to: %s\n",rd);
		msg=g_ptr_array_index(splitted_str,3);
		if (gl_gui_mode == 0)
			printf("redirect message: %s\n",msg);

		/* first the message to display */
		if((strncmp(msg,"You are being re-directed to",28))!=0)
			g_string_sprintf(rd_msg,"$To: %s From: %s $<%s> You are being re-directed to %s because: %s|",nm,luce->user_nick->str,luce->user_nick->str,rd,msg);
		else
			g_string_sprintf(rd_msg,"$To: %s From: %s $<%s> %s|",nm,luce->user_nick->str,luce->user_nick->str,msg);

		/* and then the redirection command */
		g_string_sprintfa(rd_msg,"$ForceMove %s|",rd);

		GLUS_SEND_TO_A_NICK(nm,rd_msg);		/* don't free rd_msg */

		/* g_string_sprintfa((*dest_base),"%s redirected\r\n|",nm); */
	}

	/* send_const_str_to_luce(luce,(*dest_base)->str); */
	return 0;
}

/**********************/
/* redirect all users */
/**********************/
static int do_redirall(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2) {	/* 1 parameter + the command */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GString *rd_msg;
		char *rd;
		rd_msg=g_string_new("");

		rd=g_ptr_array_index(splitted_str,1);

		/* first the message to display */
		g_string_sprintf(rd_msg,"<Hub-Security> You are being redirected to %s.|",rd);

		/* and then the redirection command */
		g_string_sprintfa(rd_msg,"$ForceMove %s|",rd);

#warning full cluster or local user only ?
		GLUS_SEND_TO_EVERYONE(rd_msg);		/* don't free rd_msg */
		
		g_string_sprintfa((*dest_base),"All users has been redirected\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

#if 0
/****************************/
/* redirect all local users */
/****************************/
static int do_local_redirall(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len != 2) {	/* 1 parameter + the command */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		GString *rd_msg;
		char *rd;
		rd_msg=g_string_new("");

		rd=g_ptr_array_index(splitted_str,1);

		/* first the message to display */
		g_string_sprintf(rd_msg,"<Hub-Security> You are being redirected to %s.|",rd);

		/* and then the redirection command */
		g_string_sprintf(rd_msg,"$ForceMove %s|",rd);

#warning full cluster or local user only ?
		GLUS_SEND_TO_EVERYONE(rd_msg);		/* don't free rd_msg */
		
		g_string_sprintfa((*dest_base),"All users has been redirected\r\n|");
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}
#endif

/****************************/
/* send a broadcast message */
/****************************/
static int do_bcast(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		GString *out;
		out=g_string_new("");
		g_string_sprintf(out,"$To: you From: Hub-Mass-Message $<Hub-Mass-Message> %s|", (char*)g_ptr_array_index(splitted_str,1));
		GLUS_SEND_TO_EVERYONE(out);		/* don't free out */
		/* g_string_sprintfa((*dest_base),"Broadcast done\r\n|"); */
	}
	/* send_const_str_to_luce(luce,(*dest_base)->str); */
	return 0;
}

static int do_version(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)		/* 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	else
	{
		GLUS_USER_INFO *gui;
		char *nick;
		nick=g_ptr_array_index(splitted_str,1);

		gui=glus_get_user_info(nick);

		if(gui!=NULL)
		{
			g_string_sprintfa((*dest_base),"the client-version of '%s' is: %s\r\n|",
			                 nick,gui->client_version->str);

			send_const_str_to_luce(luce,(*dest_base)->str);

			glus_free_user_info(gui);
		}
	}
	return 0;
}

/****************************************************/
/* add a temporary/permanent ban on the given ip(s) */
/****************************************************/
static int do_banip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<2 || splitted_str->len>3)		/* at least 1 parameter/max 2 parameters + the command */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	else if(splitted_str->len==3 && atol(g_ptr_array_index(splitted_str,2))==0)
		g_string_sprintfa((*dest_base),"invalid time for %s\r\n|",cmd);
	else
	{
		unsigned long key=(unsigned long)xtra_param;
		time_t duration;
		unsigned long temp_ban=0;
		const char *ip;
		struct sockaddr_in addr_in;
		GLUS_USER_INFO *gui;
		unsigned int dummy;

		if(splitted_str->len==2)
		{
			G_LOCK(conf_file);
			if(key==TBAN_TOSKEY)
			{
				if(!e_db_int_get(conf_file,"KICKTEMPBAN",(void*)&temp_ban))
					temp_ban=1;
				/* KICKTEMPBAN specified in minutes */
				duration=temp_ban*60;
			}
			else
			{
				if(!e_db_int_get(conf_file,"KICKPERMBAN",(void*)&temp_ban))
					temp_ban=7;
				/* KICKPERMBAN specified in days */
				duration=temp_ban*24*3600;
			}
			G_UNLOCK(conf_file);
		}
		else
		{
			temp_ban=atol(g_ptr_array_index(splitted_str,2));
			if(key==TBAN_TOSKEY)
				duration=temp_ban*60;
			else
				duration=temp_ban*24*3600;
		}

		ip=g_ptr_array_index(splitted_str,1);
		if((sscanf(ip,"%u.%u.%u.%u",&dummy,&dummy,&dummy,&dummy)==4) &&		/* be sure to have 4 digits because inet_aton can success with less */
			(inet_aton(ip,&(addr_in.sin_addr)) != 0))	/* take the string as an IP*/
 		{
			int i;
			unsigned int flag = 0;
			
			GPtrArray *gui_array;

			/* lets see if this IP is used by an user in this hub to set the ban-flag */

			if(key==TBAN_TOSKEY)
				flag |= USER_TBAN;
			if(key==PBAN_TOSKEY)
				flag |= USER_PBAN;
			gui_array=glus_get_users_info_by_ip(addr_in.sin_addr);
			for(i=0;i<gui_array->len;i++)
			{
				GLUS_USER_INFO *gui_ip;
				gui_ip=g_ptr_array_index(gui_array,i);
				user_cnx_entry_update_ext_flag_by_nickname(gui_ip->user_nick->str,flag,0);
			}
         glus_free_user_info_ptrarray(gui_array);

			if(duration>0)
				add_tos_entry_v1_uniq(key,duration,(void*)&(addr_in.sin_addr),sizeof(struct in_addr),NULL,0);
	
			/* fix me add brcastt ban addr to the cluster */
			/* send_banip_entry_to_hub(struct sockaddr_in , duration); */
				
			g_string_sprintfa((*dest_base),"ban IP %s|",ip);
		}
		else
		{
			gui=glus_get_user_info(ip);			/* take the string as an nickname*/
			if(gui == NULL)
			{
				g_string_sprintfa((*dest_base),"'%s' is not a valid IP or not a current username.|",ip);
			}
			else
			{
				if(duration>0)
				{
					unsigned int flag = 0;

					if(key==TBAN_TOSKEY)
						flag |= USER_TBAN;
					if(key==PBAN_TOSKEY)
						flag |= USER_PBAN;
					add_tos_entry_v1_uniq(key, duration, (void*)&(gui->user_ip), sizeof(struct in_addr), gui->user_nick->str, strlen(gui->user_nick->str));
					user_cnx_entry_update_ext_flag_by_nickname(ip,flag,0);
				}
				g_string_sprintfa((*dest_base),"ban IP %s|",inet_ntoa(gui->user_ip));
				glus_free_user_info(gui);
			}
		}
	}
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************************************/
/* add a temporary/permanent ban on the given nick(s) */
/****************************************************/
static int do_bannick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	unsigned long key=(unsigned long)xtra_param;
	time_t duration;
	int temp_ban;

	G_LOCK(conf_file);
	if(key==TBAN_TOSKEY)
	{
		if(!e_db_int_get(conf_file,"KICKTEMPBAN2",&temp_ban))
			temp_ban=1;
	}
	else
	{
		if(!e_db_int_get(conf_file,"KICKPERMBAN",&temp_ban))
			temp_ban=7;
	}
	G_UNLOCK(conf_file);
	duration=temp_ban*24*3600;

	if(splitted_str->len<2)    /* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		char *nick=NULL;
		for(i=1;i<splitted_str->len;i++)
		{
			nick=g_ptr_array_index(splitted_str,i);
			add_tos_entry_v1_uniq(key, duration, NULL,0,nick, strlen(nick));
         /* send_ban_entry_to_hub((char *)g_ptr_array_index(splitted_str,1), duration); */
         /* is this valid for nick-ban ??? */
			g_string_sprintfa((*dest_base),"ban Nick %s\r\n",nick);
			if(nick)
				nick=NULL;
		}
		g_string_sprintfa((*dest_base),"Ban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************************************/
/* delete a temporary/permanent ban on the given ip(s) */
/*******************************************************/
static int do_unbanip(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	unsigned long key=(unsigned long)xtra_param;

	if(splitted_str->len<2)		/* at least 1 parameters + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		const char *ip;

		for(i=1;i<splitted_str->len;i++)
		{
			ip=g_ptr_array_index(splitted_str,i);
			if(ip!=NULL)
			{
				struct in_addr inp;

				if(inet_aton(ip,&inp)==0)
				{
					g_string_sprintfa((*dest_base),"Fail to unban IP '%s'. Invalid address\r\n",ip);
				}
				else
				{
					delete_this_tos_entry(key,(void*)&inp,sizeof(inp),0);
					g_string_sprintfa((*dest_base),"unban IP %s\r\n",ip);
				}
			}
		}

		g_string_sprintfa((*dest_base),"Unban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/*******************************************************/
/* delete a temporary/permanent ban on the given ip(s) */
/*******************************************************/
static int do_unbannick(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<2)	 /* 1 parameter + the command */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
		int i;
		unsigned long key=(unsigned long)xtra_param;
		const char *nick;

		for (i=1;i<splitted_str->len;i++)
		{
			nick=g_ptr_array_index(splitted_str,i);
			if(nick!=NULL)
			{
				delete_this_tos_entry(key,(void*)nick,strlen(nick),1);
				g_string_sprintfa((*dest_base),"unban Nick %s\r\n",nick);
			}
		}
		g_string_sprintfa((*dest_base),"Unban done\r\n|");
	}

	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

static int append_ban_entry(const time_t timeout, const char *ptr1, const int len1, const char *ptr2, const int len2, void *xtra)
{
	/* warning: ptr1 is always a pointer on a valid struct in_addr (and len1 is its length) */
	/*          ptr2 is either a nickname (and len2 is its length) but it can also be NULL */
	GString **dest_base=xtra;

	if((ptr1!=NULL)&&(len1==sizeof(struct in_addr)))
	{
		struct in_addr *inp=(void*)ptr1;
		char timestamp[512];
#ifdef WIN32
		strftime(timestamp,sizeof(timestamp),"%c",localtime(timeout));
#else
		struct tm tm;
		localtime_r(&timeout,&tm);
		strftime(timestamp,sizeof(timestamp),"%c",&tm);
#endif


		if((ptr2==NULL)||(len2==0))
		{
			/* inet_ntoa is not thread safe, it works with a static buffer, we will do the conversion manually */
			g_string_sprintfa((*dest_base),"IP: %u.%u.%u.%u - ban until %s\r\n",
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255),
									timestamp
									);
		}
		else
		{
			char nick[256];

			strncpy_max(nick,ptr2,min(sizeof(nick),(len2+1)));
			/* inet_ntoa is not thread safe, it works with a static buffer, we will do the conversion manually */
			g_string_sprintfa((*dest_base),"IP: %u.%u.%u.%u	Nick: %s - ban until %s\r\n",
									(unsigned)((ntohl(inp->s_addr)>>24)&255),
									(unsigned)((ntohl(inp->s_addr)>>16)&255),
									(unsigned)((ntohl(inp->s_addr)>>8)&255),
									(unsigned)((ntohl(inp->s_addr))&255),
									nick,
									timestamp
									);
		}
	}
	else if(ptr2!=NULL)
	{
		struct tm tm;
		char nick[256];
		char timestamp[512];
		localtime_r(&timeout,&tm);
		strftime(timestamp,sizeof(timestamp),"%c",&tm);

		strncpy_max(nick,ptr2,min(sizeof(nick),(len2+1)));
		g_string_sprintfa((*dest_base),"Nick: %s - ban until %s\r\n",
									nick,
									timestamp
									);
   }
	return 0;	/* continue */
}

/********************************************/
/* print the temporary/permanently ban list */
/********************************************/
static int do_banlist(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	unsigned long key=(unsigned long)xtra_param;

	if(key==TBAN_TOSKEY)
		g_string_sprintfa((*dest_base),"Tban list:\r\n");
	else
		g_string_sprintfa((*dest_base),"Pban list:\r\n");

	scan_all_tos_entry(key,append_ban_entry,dest_base);

	g_string_sprintfa((*dest_base),"|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/***************************/
/* clearing wanted banlist */
/***************************/
static int do_clearban(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
   if(splitted_str->len!=1)      /* the command only */
   {
      g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
   }
   else
   {
      unsigned long key=(unsigned long)xtra_param;
      delete_all_same_key(key, dest_base, luce->user_nick->str);
      if (key==TBAN_TOSKEY)
         g_string_sprintfa((*dest_base),"Clearing Tempban-list done\r\n|");
      else
         g_string_sprintfa((*dest_base),"Clearing Permban-list done\r\n|");
   }

   send_const_str_to_luce(luce,(*dest_base)->str);
   return 0;
}

/******************************/
/* reset the perl interpreter */
/******************************/
static int do_perlreset(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=1)		/* the command only */
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
#ifdef WITH_PERL
		restart_perl();
#endif
		g_string_sprintfa((*dest_base),"Perl interpreter restarted\r\n|");
		send_const_str_to_luce(luce,(*dest_base)->str);
	}
	return 0;
}
/**************************************/ 
/* clear all autoloaded perl handlers */ 
/**************************************/ 
static int do_clear_autoloaded_perl_handlers(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)                  
{                                
	if (splitted_str->len!=1)
	{                /* the command only */
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
	}
	else
	{
#ifdef WITH_PERL
		clear_autoloaded_perl_handlers();
#endif
		g_string_sprintfa((*dest_base),"All autoloaded perl handlers cleared\r\n|");
		send_const_str_to_luce(luce,(*dest_base)->str);
	}                        
	return 0;                
}                                

	
typedef struct
{
	 const char *cmd;
	 size_t cmd_len;
	 int (*fnc)(const char *cmd, LOCAL_USER_CNX_ENTRY *ptr, GString **dest_base, 
					const GString *str, char *xtra_param, const GPtrArray *splitted_str);
			/* function to call. (ERROR: the following sentence is FALSE => This function MUST free str when it is not useful anymore) */
			/* the function must return >=0 if the connection must remain alive (no error (0) or */
			/* recoverable error (>0)) and a value <0 if there is a fatal error) */
			/* *dest_base is a the beginning of the reply to send. To create the full reply, */
			/* add the message you want to send and ends the string with a |. Use send_const_str_to(ptr,(*dest_base)->str) */
			/* to send the reply */
	 char *xtra_param;
	int op_level;		/* it is a bit mask. If (1<<is_op) is set, the user can use this command */
} KNOWN_CMD;

static KNOWN_CMD hub_cmd[]={
		{"-add ",         sizeof("-add ")-1,            do_add,        NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-bcast ",       sizeof("-bcast ")-1,          do_bcast,      NULL,OP_PRIV},					/* ok v0.4.0 */
		{"-clearpban",    sizeof("-clearpban")-1,       do_clearban,   (void*)PBAN_TOSKEY,MASTER_PRIV},
		{"-cleartban",    sizeof("-cleartban")-1,       do_clearban,   (void*)TBAN_TOSKEY,MASTER_PRIV},
      {"-close ",       sizeof("-close ")-1,          do_close,      NULL,OP_PRIV},

		/* database management */
		{"-dbadd ",       sizeof("-dbadd ")-1,          do_dbadd,      NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-dbdel ",       sizeof("-dbdel ")-1,          do_dbdel,      NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-dblist",       sizeof("-dblist")-1,          do_db,         NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-dbset ",       sizeof("-dbset ")-1,          do_dbset,      NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-dbshow ",      sizeof("-dbshow ")-1,         do_dbshow,     NULL,MASTER_PRIV},			/* ok v0.4.0 */
		{"-del ",         sizeof("-del ")-1,            do_del,        NULL,MASTER_PRIV},			/* ok v0.4.0 */
#ifdef DEBUG
		{"-dump ",        sizeof("-dump ")-1,           do_dump,       NULL,MASTER_PRIV},			/* disabled */
		{"-dumpqueue",    sizeof("-dumpqueue")-1,       do_dump_queue, NULL,MASTER_PRIV},			/* disabled */
#endif
		{"-flood ",			sizeof("-flood ")-1,				do_flood,		NULL, OP_PRIV},
		{"-getip ",       sizeof("-getip ")-1,          do_get_ip,     NULL, OP_PRIV},				/* ok v0.4.0 */
		{"-help",         sizeof("-help")-1,            do_help,       NULL, OP_PRIV},				/* ok v0.4.0 */
		{"-hpcnx ",       sizeof("-hpcnx ")-1,          do_hpcnx,      NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-hpdel ",       sizeof("-hpdel ")-1,          do_hpdel,      NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-hpget ",       sizeof("-hpget ")-1,          do_hpget,      NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-hpgetall",     sizeof("-hpgetall")-1,        do_hpgetall,   NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-hpset ",       sizeof("-hpset ")-1,          do_hpset,      NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-hpstat",       sizeof("-hpstat")-1,          do_hpstat,     NULL, MASTER_PRIV},			/* ok v0.4.0 */
		{"-kick ",        sizeof("-kick ")-1,           do_kick,       NULL, OP_PRIV},				/* ok v0.4.0 */
		{"-md5 ",         sizeof("-md5 ")-1,            do_md5cmd,     NULL, OP_PRIV|MASTER_PRIV},	/* ok v0.4.0 */
		{"-mlist",        sizeof("-mlist")-1,           do_ulist,      (void*)master_user_list, MASTER_PRIV},	/* ok v0.4.0 */
		{"-olist",        sizeof("-olist")-1,           do_ulist,      (void*)op_user_list, MASTER_PRIV},	/* ok v0.4.0 */
		{"-passwd ",      sizeof("-passwd ")-1,         do_passwd,     NULL, MASTER_PRIV},						/* ok v0.4.0 */
		{"-pbanip ",      sizeof("-pbanip ")-1,         do_banip,      (void*)PBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-pbankick ",		sizeof("-pbankick ")-1,			do_ban_kick,	(void*)PBAN_TOSKEY,OP_PRIV},
		{"-pbanlist",     sizeof("-pbanlist")-1,        do_banlist,    (void*)PBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-pbannick ",    sizeof("-pbannick ")-1,       do_bannick,    (void*)PBAN_TOSKEY,OP_PRIV},
		{"-perlclear",    sizeof("-perlclear")-1,       do_clear_autoloaded_perl_handlers, NULL, OP_PRIV},	/* ok v0.4.0 */
		{"-perlreset",    sizeof("-perlreset")-1,       do_perlreset,  NULL,OP_PRIV},								/* ok v0.4.0 */
		{"-punbanip ",    sizeof("-punbanip ")-1,       do_unbanip,    (void*)PBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-punbannick ",  sizeof("-punbannick ")-1,     do_unbannick,  (void*)PBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-redir ",       sizeof("-redir ")-1,          do_redir,      NULL, OP_PRIV},							/* ok v0.4.0 */
		{"-redirall ",    sizeof("-redirall ")-1,       do_redirall,   NULL, MASTER_PRIV},						/* ok v0.4.0 */
#if 0
		{"-rediralllocal ",sizeof("-rediralllocal ")-1, do_local_redirall,NULL, MASTER_PRIV},					/* disabled */
#endif
		{"-reload",       sizeof("-reload")-1,          do_reload,     NULL, MASTER_PRIV},						/* ok v0.4.0 */
		{"-rename ",      sizeof("-rename ")-1,         do_rename,     NULL, MASTER_PRIV},						/* ok v0.4.0 */
		{"-revip ",       sizeof("-revip ")-1,          do_rev_ip,     NULL, OP_PRIV},							/* ok v0.4.0 */
		{"-set ",         sizeof("-set ")-1,            do_set,        NULL, MASTER_PRIV},						/* ok v0.4.0 */
		{"-showtype ",    sizeof("-showtype ")-1,       do_is_reg,     NULL, OP_PRIV},							/* ok v0.4.0 */
		{"-silence ",     sizeof("-silence ")-1,        do_silence,    NULL, BOT_PRIV | MASTER_PRIV},
		{"-skick ",       sizeof("-skick ")-1,          do_silence_kick,NULL, OP_PRIV},
		{"-startprg ",    sizeof("-startprg ")-1,       do_startprg,   NULL, MASTER_PRIV},
		{"-tbanip ",      sizeof("-tbanip ")-1,         do_banip,      (void*)TBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-tbankick ",		sizeof("-tbankick ")-1,			do_ban_kick,	(void*)TBAN_TOSKEY,OP_PRIV},
		{"-tbanlist",     sizeof("-tbanlist")-1,        do_banlist,    (void*)TBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-tbannick ",    sizeof("-tbannick ")-1,       do_bannick,    (void*)TBAN_TOSKEY,OP_PRIV},
		{"-tunbanip ",    sizeof("-tunbanip ")-1,       do_unbanip,    (void*)TBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
		{"-tunbannick ",	sizeof("-tunbannick ")-1,		do_unbannick,	(void*)TBAN_TOSKEY,OP_PRIV},			/* ok v0.4.0 */
#if 0
		{"-tutel ",       sizeof("-tutel ")-1,          do_tutel,      NULL,   BOT_PRIV | MASTER_PRIV},		/* disabled */
#endif
		{"-type ",        sizeof("-type ")-1,           do_type,       NULL, MASTER_PRIV},						/* ok v0.4.0 */
		{"-ulist",        sizeof("-ulist")-1,           do_ulist,      (void*)full_user_list, MASTER_PRIV},/* ok v0.4.0 */
		{"-users",			sizeof("-users")-1,				do_users,		NULL,OP_PRIV},								/* ok v0.4.0 */
		{"-version ",		sizeof("-version ")-1,			do_version,		NULL,OP_PRIV},								/* ok v0.4.0 */
   	{NULL,0,NULL,NULL},
};

/*********************************************************************************/
/* run the given command. Incoming string is splitted, access rights are checked */
/*********************************************************************************/
static int run_hub_cmd(KNOWN_CMD *hcmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest, const GString *inp)
{
	if ((hcmd->op_level == 0) || ((luce->privilege & hcmd->op_level) != 0))
	{
		GStringChunk *chunk=NULL;
		GPtrArray *tbl=NULL;
		int ret=0;

		split_string_into_array(&tbl,&chunk,inp->str," ");
		/* now, we can process the command */
		if((chunk!=NULL)&&(tbl!=NULL))
		{
			ret=(hcmd->fnc)(hcmd->cmd,luce,dest,inp,hcmd->xtra_param,tbl);
		}

		/* free allocated data */
		if(tbl!=NULL)
			g_ptr_array_free(tbl,TRUE);
		if(chunk!=NULL)
			g_string_chunk_free(chunk);
		return ret;
	}
	else
	{
		/* not enough rights */
		g_string_sprintfa((*dest),"You are not allowed to use this command.|");
		send_const_str_to_luce(luce,(*dest)->str);
		return 0;
	}
}

/*************************************************************************/
/* the given string has been sent to "Hub-Security", it may be a command */
/* This function processes the string and performs the wanted action	 */
/*************************************************************************/
/* output: =0: ok								 */
/*		   <0: fatal error (=> abort connection) */
/*		   >0: recoverable error (continue)		 */
/*************************************************/
/* the last char of string is a '|' */
int process_hub_command(LOCAL_USER_CNX_ENTRY *luce,char *string)
{
	int ret=0;
	GString *inp=NULL;
	GString *dest=NULL;
	char *t;
	int fnd=0;

	if (gl_gui_mode == 0)
		printf("%s\n",string);

	t=strchr(string,' ');
	if (t == NULL)
		goto eoc;
	SKIP_SPACE(t);

	if(*t=='\0')
	{
		ret=1;
		goto eoc;		/* non fatal error: nothing entered */
	}

	inp=g_string_new(t);	/* remove trash before the command */
	if(inp->len>1)
		inp=g_string_truncate(inp,inp->len-1);		/* remove trailing | */

	/* build the beginning of the query reply */
	dest=g_string_new("");
	g_string_sprintf(dest,"$To: %s From: Hub-Security $",luce->user_nick->str);
	{
		int min = 0, max = ARRAY_SIZE (hub_cmd) - 2;

		do
		{
			int i = (min + max) / 2;
			int cmp = strncmp(inp->str, hub_cmd[i].cmd, hub_cmd[i].cmd_len);
			if (cmp == 0)
			{
				fnd=1;
				ret=run_hub_cmd(&hub_cmd[i],luce,&dest,inp);
				break;
			}
			else if (cmp < 0)
				max = i - 1;
			else
				min = i + 1;
		}
		while (min <= max);
	}

	if(!fnd)
	{
		g_string_sprintfa(dest,"Unknown command: %s|",inp->str);
		send_const_str_to_luce(luce,dest->str);
	}
	eoc:
	if(dest!=NULL)
		g_string_free(dest,TRUE);
	if (inp!=NULL)
		g_string_free(inp,TRUE);
	return 0;
}

/*****************************************/
/* display the list of all user commands */
/*****************************************/
static int do_user_help(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	g_string_sprintfa((*dest_base),	"** User command list:\r\n"
											 "\r\n"
											 "+help\r\n"
											 "               display this help\r\n"
											 "+me text\r\n"
											 "               irc like /me\r\n"
											 "+new_chat chatname [password]\r\n"
											 "               create a new multi-users chat named \"chatname\" with an optional\r\n"
											 "               password. The multi-users chat (mchat in short) works like the\r\n"
											 "               normal private chat except you can be more than 2 users in this chat.\r\n"
											 "               Note: it is not possible to create two mchats with the same name.\r\n"
											 "+list chatname\r\n"
											 "               list the users in the chat named \"chatname\".\r\n"
											 "+join chatname [password]\r\n"
											 "               connect to a multi-users chat named \"chatname\".\r\n"
											 "+leave chatname\r\n"
											 "               leave to a multi-users chat named \"chatname\".\r\n"
											 "+seen nick\r\n"
											 "               showing when a user entered or left the hub.\r\n"
											 "+afk message\r\n"
											 "               add an 'away-from-keyboard'-message\r\n"
											 "+back\r\n"
											 "               delete the afk message\r\n"
											 "+msg dest_nick message\r\n"
											 "               post a message to a user not in the hub\r\n"
											 "+clearmsg\r\n"
											 "               delete messages created with +msg\r\n"
			

											 "\r\n|");
	send_const_str_to_luce(luce,(*dest_base)->str);
	return 0;
}

/****************************************/
/*     a +me function like in irc       */
/****************************************/
static int do_user_me(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param, const GPtrArray *splitted_str)
{
	GString *out2;
	const char	*rcv;

	if (dest_base == NULL)
		return 0;
	
	if (luce->ext_flag & SILENT_CC)
	{
		return 0;
	}
	rcv = str->str + (sizeof("me ") - 1);
	if (*rcv == '\0')
		return 0;

	out2=g_string_new("");
	if ((*dest_base)->str[0] == '<')
	{
		g_string_sprintfa(out2,"<*%s%s|", luce->user_nick->str, rcv);
		
		GLUS_SEND_TO_EVERYONE(out2);			/* don't free out2 */
	}
	else
	{
		char	*ptr_nick;
		char	dest_nick[64];
		int	i;

		ptr_nick = (*dest_base)->str + strlen("$To: ");
		ptr_nick = strchr(ptr_nick, ' '); /* This strchr can not retur NULL cause the string is create by the hub */
		ptr_nick += strlen(" From: ");
		for (i = 0; (i < 63) && ptr_nick[i] && ptr_nick[i] != ' '; i++)
			dest_nick[i] = ptr_nick[i];
		dest_nick[i] = '\0';
		g_string_sprintfa(out2,"$To: %s From: %s $<*%s*%s|", dest_nick, luce->user_nick->str, luce->user_nick->str, rcv);
		GLUS_SEND_TO_A_NICK(dest_nick,out2);		/* don't free out2 */
	}
	return 0;
}

static int do_seen(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len!=2)
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
		return 0;
	}
	find_seen_by_nickname(luce, dest_base, splitted_str->pdata[1]);
	return 0;
}

static int do_afk(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<2)
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
		return 0;
	}
	else
	{
		GString *message;
		int i;
		message=g_string_new("");
		for(i=1;splitted_str->len>i;i++)
		{
			g_string_sprintfa(message," %s",(char*)g_ptr_array_index(splitted_str,i));
		}
		g_string_sprintfa(message,"|");
		add_afk_msg(luce,dest_base,message);
		g_string_free(message,TRUE);
	}
	return 0;
}

static int do_back(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	del_afk_entry(luce,dest_base);
	return 0;
}

static int do_msg(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	if(splitted_str->len<3)
	{
		g_string_sprintfa((*dest_base),"invalid number of parameters for %s\r\n|",cmd);
		send_const_str_to_luce(luce,(*dest_base)->str);
		return 0;
	}
	else
	{
		GString *message;
		GString *dest_base_cpy;
		int i;

		dest_base_cpy=g_string_new((*dest_base)->str);
		message=g_string_new("");
		for(i=2;splitted_str->len>i;i++)
		{
			g_string_sprintfa(message," %s",(char*)g_ptr_array_index(splitted_str,i));
		}
		g_string_sprintfa(message,"|");
		add_msg_msg(luce,dest_base,(char*)g_ptr_array_index(splitted_str,1),message);
		g_string_free(message,TRUE);
		g_string_free(dest_base_cpy,TRUE);
	}
	return 0;
}

static int do_clearmsg(const char *cmd, LOCAL_USER_CNX_ENTRY *luce, GString **dest_base, const GString *str, char *xtra_param,const GPtrArray *splitted_str)
{
	del_msg_entry(luce,*dest_base);
	return 0;
}


#define EVERYONE_MSK 0
static KNOWN_CMD user_hub_cmd[]={
	{"+afk ",             sizeof("+afk ")-1,              do_afk,                 NULL,EVERYONE_MSK},		/* to rewrite */
	{"+back",             sizeof("+back")-1,              do_back,                NULL,EVERYONE_MSK},		/* to rewrite */
	{"+clearmsg",         sizeof("+clearmsg")-1,          do_clearmsg,            NULL,EVERYONE_MSK},		/* to rewrite */
	{"+help",             sizeof("+help")-1,              do_user_help,           NULL,EVERYONE_MSK},		/* ok v0.4.0 */
	{"+join ",            sizeof("+join ")-1,             cmd_join_public_chat,   NULL,EVERYONE_MSK},		/* ok v0.4.0 */
	{"+leave ",           sizeof("+leave ")-1,            cmd_leave_public_chat,  NULL,EVERYONE_MSK},		/* ok v0.4.0 */
	{"+list",             sizeof("+list")-1,              cmd_list_public_chat,   NULL,EVERYONE_MSK},		/* ok v0.4.0 */
	{"+me ",              sizeof("+me ")-1,               do_user_me,             NULL,EVERYONE_MSK},		/* to rewrite */
	{"+msg ",             sizeof("+msg ")-1,              do_msg,                 NULL,EVERYONE_MSK},		/* to rewrite */
	{"+new_chat ",        sizeof("+new_chat ")-1,         cmd_create_public_chat, NULL,EVERYONE_MSK},		/* ok v0.4.0 */
	{"+seen ",            sizeof("+seen ")-1,             do_seen,                NULL,EVERYONE_MSK},		/* to rewrite */
	{NULL,0,NULL,NULL},
};

/*************************************************************************/
/* the given string has been sent to any user but always starts with a + */
/* This function processes the string and performs the wanted action     */
/*************************************************************************/
/* output: =0: ok                                */
/*         <0: fatal error (=> abort connection) */
/*         >0: recoverable error (continue)      */
/*************************************************/
int process_user_hub_command(LOCAL_USER_CNX_ENTRY *luce,char *string, char *remote_name)
{
	int ret=0;
	GString *inp;
	GString *dest=NULL;
	char *t;
#if 0
	int fnd=0;
#endif

	inp=g_string_new(string);

	t=inp->str;
	SKIP_SPACE(t);

	if(*t=='\0')
	{
		ret=1;
		goto eoc;		/* non fatal error: nothing entered */
	}

	/* remote trash before the command */
	inp=g_string_erase(inp,0,t-inp->str);
	if(inp->len>1)
		inp=g_string_truncate(inp,inp->len-1);		/* remove trailing | */
	{
		int min = 0, max = ARRAY_SIZE (user_hub_cmd) - 2;

		do
		{
			int i = (min + max) / 2;
			int cmp = strncmp(inp->str,user_hub_cmd[i].cmd,user_hub_cmd[i].cmd_len);
			
			if (cmp == 0)
			{
#if 0
				fnd=1;
#endif
				/* build the beginning of the query reply */
				dest=g_string_new("");
				if((remote_name==NULL)||(strlen(remote_name)==0))
					g_string_sprintf(dest,"<Kernel> ");
				else
					g_string_sprintf(dest,"$To: %s From: %s $",luce->user_nick->str, remote_name);
				ret=run_hub_cmd(&user_hub_cmd[i],luce,&dest,inp);
				break;
			}
			else if (cmp < 0)
				max = i - 1;
			else
				min = i + 1;
		}
		while (min <= max);
	}

#if 0
	/* if the command is not found, we don't return any error message because */
	/* the command can be handled by a script */
	if(!fnd)
	{
		g_string_sprintfa(dest,"Unknown command: %s|",inp->str);
		send_str_to(ptr,dest->str);
	}
#endif
	eoc:
	if(dest!=NULL)
		g_string_free(dest,TRUE);
	g_string_free(inp,TRUE);

	return 0;
}

typedef struct
{
	const char *cmd;
	void (*fnc)(const char *evt_name, const char *evt_emitter, int nb_args, va_list ap, char *xtra_param);
	/* NOTE: all parameters are read-only because you may have more than 1 event handler for each event */
	char *xtra_param;
} KNOWN_EVENT;

static KNOWN_EVENT hub_event_hdl[]={
	{"quit",add_seen_entry,NULL},
	{"quit",del_afk_on_quit,NULL},
	{NULL,NULL,NULL},
};

/**************************************************************************************************/
/* in the same way as it exists for PERL scripts, there is an event handler for embedded commands */
/**************************************************************************************************/
/* evt_name and evt_emitter are the same as PERL scripts one */
/* ap is a va_list of nb_args (char *)                       */
/*************************************************************/
void hub_event_command(const char *evt_name, const char *evt_emitter, int nb_args, va_list ap)
{
	int i;
	i=0;
	while(hub_event_hdl[i].cmd!=NULL)
	{
		if(!strcmp(evt_name,hub_event_hdl[i].cmd))
		{
			(hub_event_hdl[i].fnc)(evt_name,evt_emitter,nb_args,ap,hub_event_hdl[i].xtra_param);
		}
		i++;
	}
}


