/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * hl_locks.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: hl_locks.h,v 2.0 2003/01/30 12:50:45 eric Exp $
*/

#ifndef __HL_LOCKS_H__
#define __HL_LOCKS_H__

#ifndef HAVE_NO_PTHREAD

#include <pthread.h>
/* this set of macros introduces high level mutex allowing multiple */
/* readers or 1 write to own the locks                              */
/* it requires 3 values:                                            */
/* 1 pthread_mutex_t named xxx_mutex;                               */
/* 1 pthread_cond_t  named xxx_cond;                                */
/* 1 int             named xxx_nb_readers                           */
typedef struct
{
	pthread_mutex_t xxx_mutex;
	pthread_cond_t  xxx_cond;
	int				 xxx_nb_readers;
} HL_MUTEX;

/* macro to initialize non "volatile" HL_MUTEX (the "global" one) */
#define HL_MUTEX_INIT {PTHREAD_MUTEX_INITIALIZER, PTHREAD_COND_INITIALIZER, 0};

/* macro to initialize "volatile" HL_MUTEX */
#define HL_MUTEX_INIT_VOLATILE(x)											\
			{																			\
				pthread_mutex_init(&(x.xxx_mutex),NULL);\
				pthread_cond_init(&(x.xxx_cond),NULL);						\
				x.xxx_nb_readers=0;												\
			}

#define HL_MUTEX_EXIT_VOLATILE(x)											\
			{																			\
				pthread_mutex_destroy(&(x.xxx_mutex));						\
				pthread_cond_destroy(&(x.xxx_cond));						\
			}
				

/******************/
/* easiest macros */
/******************/
/* to lock and unlock for reading */
#define HL_LOCK_READ(v)			LOCK_MUTEX_FOR_READ(&(v))
#define HL_UNLOCK_READ(v)		UNLOCK_MUTEX_FOR_READ(&(v))
/* to lock and unlock for writing */
#define HL_LOCK_WRITE(v)		LOCK_MUTEX_FOR_WRITE(&(v))
#define HL_UNLOCK_WRITE(v)		UNLOCK_MUTEX_FOR_WRITE(&(v))

#ifndef DEBUG
/* lock for read just increases the number of readers */
#define LOCK_MUTEX_FOR_READ(v)								\
			{															\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				((v)->xxx_nb_readers)++;						\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
			}

/* unlock for read just decreases the number of readers */
/* if the number of readers becomes null, it tries to   */
/* wakeup potential writers                             */
#define UNLOCK_MUTEX_FOR_READ(v)								\
			{															\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				((v)->xxx_nb_readers)--;						\
				if((v)->xxx_nb_readers==0)						\
					pthread_cond_signal(&((v)->xxx_cond));	\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
			}

/* lock for write try to lock until no reader here */
#define LOCK_MUTEX_FOR_WRITE(v)								\
			{															\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				while( ((v)->xxx_nb_readers)!=0 )			\
				{	/* there is some readers */				\
					pthread_cond_wait(&((v)->xxx_cond),		\
											&((v)->xxx_mutex));	\
				}														\
			}

/* unlock for write releases the mutex and wakeup potential writers */
#define UNLOCK_MUTEX_FOR_WRITE(v)							\
			{															\
				pthread_cond_signal(&((v)->xxx_cond));		\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
			}
#else

/* lock for read just increases the number of readers */
#define LOCK_MUTEX_FOR_READ(v)								\
			{															\
				fprintf(stderr,"lock for read (%08lX):" #v ":",(long)v);\
				fflush(stderr);									\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				((v)->xxx_nb_readers)++;						\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
				fprintf(stderr," done (%08lX)\n",(long)v);					\
			}

/* unlock for read just decreases the number of readers */
/* if the number of readers becomes null, it tries to   */
/* wakeup potential writers                             */
#define UNLOCK_MUTEX_FOR_READ(v)								\
			{															\
				fprintf(stderr,"unlock for read (%08lX):" #v ":",(long)v);\
				fflush(stderr);									\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				((v)->xxx_nb_readers)--;						\
				if((v)->xxx_nb_readers==0)						\
					pthread_cond_signal(&((v)->xxx_cond));	\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
				fprintf(stderr," done (%08lX)(%d)\n",(long)v,(v)->xxx_nb_readers);\
			}

/* lock for write try to lock until no reader here */
#define LOCK_MUTEX_FOR_WRITE(v)								\
			{															\
				fprintf(stderr,"lock for write (%08lX):" #v ":",(long)v);\
				fflush(stderr);									\
				pthread_mutex_lock(&((v)->xxx_mutex));		\
				while( ((v)->xxx_nb_readers)!=0 )			\
				{	/* there is some readers */				\
					pthread_cond_wait(&((v)->xxx_cond),		\
											&((v)->xxx_mutex));	\
				}														\
				fprintf(stderr," done (%08lX)\n",(long)v);					\
			}

/* unlock for write releases the mutex and wakeup potential writers */
#define UNLOCK_MUTEX_FOR_WRITE(v)							\
			{															\
				fprintf(stderr,"unlock for write (%08lX):" #v ":",(long)v);\
				fflush(stderr);									\
				pthread_cond_signal(&((v)->xxx_cond));		\
				pthread_mutex_unlock(&((v)->xxx_mutex));	\
				fprintf(stderr," done (%08lX)\n",(long)v);					\
			}
#endif

#else
#define HL_LOCK_READ(v)
#define HL_UNLOCK_READ(v)

#define HL_LOCK_WRITE(v)
#define HL_UNLOCK_WRITE(v)

#endif /*HAVE_NO_PTHREAD*/

#endif
