/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * do_hub_reg.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 * patialy rewrite by Uriel Chemouni <uriel@xor.dyndns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: do_hub_reg.c,v 2.3 2003/02/16 11:04:46 eric Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#ifdef WIN32
	#include <windows.h>
	#include <winsock2.h>
	#include <getopt.h>
#else
	#include <sys/socket.h>
	#ifdef HAVE_UNISTD_H
		#include <unistd.h>
	#endif  /* HAVE_UNISTD_H */
#endif

#include <errno.h>
#include <string.h>
#ifndef HAVE_NO_PTHREAD
	#include <pthread.h>
#endif

#ifdef HAVE_SIGNAL_H
	#include <signal.h>
	#define HAVE_SIGNAL 1
#else
	#undef HAVE_SIGNAL
#endif

#include <glib.h>

#include "do_hub_reg.h"
#include "main.h"
#include "gvar.h"
#include "key.h"
#include "network.h"
#include "global_user_if.h"

#if defined(__CYGWIN__) || defined(WIN32)
#define GLIB_INT_64_BUG
#endif

#ifdef GLIB_INT_64_BUG
char	*llunsigned_to_str(guint64 a);
#endif


/*************************/
/* output: 0=ok, 1=error */
/*************************/
static int send_this(int sck, GString *out)
{
	int ln;

	ln=send(sck, out->str,out->len,MSG_NOSIGNAL);
	if(ln==-1)
	{
		return 1;
	}
	if(ln==0)
	{
		return 1;
	}
	return 0;
}

static void perform_registration(char *main_server_addr, char *this_serv)
{
	int sck=-1;
	char buf[512];
	int ln;
	char *t;
	struct sockaddr_in lcl;
	int len_lcl=sizeof(lcl);
	unsigned int lport;
	unsigned char magic;

	GString *in,*out;
	
#ifdef DEBUG
	printf("hub registration in progress\n");
#endif

	sck=create_and_open_sock_on(main_server_addr,2501,0,gl_bind_addr);
	if(sck<0)
	{
		perror("create - registration failed");
		return;
	}

#ifdef DEBUG
	printf("hub registration (1)\n");
#endif

	if(getsockname(sck,(void*)&lcl,&len_lcl)!=0)
	{
		perror("getsockname - registration failed");
		close(sck);
		return;
	}

	lport=ntohs(lcl.sin_port);

	magic=lport+(lport>>8);		/* add the 2 bytes of the local port and create a 1 byte magic value */
	
	ln=recv(sck, buf,sizeof(buf)-1,MSG_NOSIGNAL);

#ifdef DEBUG
	printf("hub registration (2)\n");
#endif

	if(ln==-1)
	{
		perror("recv - registration failed");
		close(sck);
		return;
	}
	if(ln==0)
	{
		printf("recv no data - registration failed");
		close(sck);
		return;
	}

	buf[ln]='\0';

	in=g_string_new(buf);

	if(strncmp(buf,"$Lock ",6))
	{
		printf("no $Lock: %s",in->str);
		g_string_free(in,TRUE);
		close(sck);
		return;
	}

	in=g_string_erase(in,0,6);

	t=strstr(in->str," Pk=");
	if(t==NULL)
	{
		printf("no Pk= %s",in->str);
		g_string_free(in,TRUE);
		close(sck);
		return;
	}

	in=g_string_truncate(in,t-in->str);

#ifdef DEBUG
	printf("hub registration (3)\n");
#endif

	out=compute_hub_reg_access_key(in,magic);

	out=g_string_prepend(out,"$Key ");
	out=g_string_append_c(out,'|');
	out=g_string_append(out,this_serv);
	send_this(sck,out);

	g_string_free(in,TRUE);
	g_string_free(out,TRUE);

#ifdef DEBUG
	printf("hub registration (4)\n");
#endif

	printf("hub registration done\n");

	/* the registration thread *MUST* be closed each time */
	/* if it don't nmdc server won't register the hub */
	if(sck!=-1)
	{
		shutdown(sck,2);
		close(sck);
		sck=-1;
	}
	return;
}

G_LOCK_DEFINE_STATIC(hub_reg_in_progress);

#include "color_debug.h"

#ifndef HAVE_NO_PTHREAD
/***************************/
/* hub registration thread */
/***************************/
static void *do_hub_reg(void *ptr)
{
	char *hub_name;
	char *hubaddr;
	char *hubdesc;
	char *reg_serv;
	guint64 shared_size;
	unsigned int nb_users;

	set_user_id();	/* change user ID if required */

	/* printf("starting hub registration thread\n"); */
	if(G_TRYLOCK(hub_reg_in_progress)==TRUE)
	{
		G_LOCK(conf_file);
		hub_name=e_db_str_get(conf_file,"HUBNAME");
		hubaddr=e_db_str_get(conf_file,"HUBADDR");
		hubdesc=e_db_str_get(conf_file,"HUBDESC");
		reg_serv=e_db_str_get(conf_file,"REG_SERV");
		G_UNLOCK(conf_file);
	
		if(reg_serv!=NULL)
		{
			GString *fhd;
			glus_hub_users_stat(&shared_size,&nb_users);

			fhd=g_string_new("");
			g_string_sprintfa(fhd,"%s|%s",
								hub_name!=NULL?hub_name:"",
								hubaddr!=NULL?hubaddr:"");
			if (reg_hub_port == 0)
			{
				if(hub_port!=411)
					g_string_sprintfa(fhd,":%hu",hub_port);
			}
			else
			{
				if(reg_hub_port!=411)
					g_string_sprintfa(fhd,":%hu",reg_hub_port);
			}
#ifdef GLIB_INT_64_BUG
			g_string_sprintfa(fhd,"|%s|%u|%s|",
								hubdesc!=NULL?hubdesc:"",
								nb_users,
								llunsigned_to_str(shared_size));
#else
			g_string_sprintfa(fhd,"|%s|%u|%Lu|",
								hubdesc!=NULL?hubdesc:"",
								nb_users,
								shared_size);
#endif
			perform_registration(reg_serv,fhd->str);
			printf("registrer : "COLOGREEN1"%s"COLONONE"\n", fhd->str);
			g_string_free(fhd,TRUE);
		}
	
		if(hub_name!=NULL)
			free(hub_name);
		if(hubaddr!=NULL)
			free(hubaddr);
		if(hubdesc!=NULL)
			free(hubdesc);
		if(reg_serv!=NULL)
			free(reg_serv);
		G_UNLOCK(hub_reg_in_progress);
	}
	else
	{
		printf("earlier hub registration still in progress.\n");
	}
	pthread_exit(NULL);
	return(NULL);
}
#endif
/***********************************************/
/* check hub registration on DC hublist server */
/***********************************************/
void check_hub_registration(void)
{
#ifndef HAVE_NO_PTHREAD
	static time_t last_hub_registration=0;
	time_t cur_time;
	int do_hub_registration=0;

	G_LOCK(conf_file);
	e_db_int_get(conf_file,"REG_HUB",&do_hub_registration);
	G_UNLOCK(conf_file);

	if(do_hub_registration==0)
		return;

	cur_time=time(NULL);
	if((cur_time-last_hub_registration)>DELAY_BETWEEN_REGISTRATION)
	{
		static pthread_t thread_id;			/* these 2 variables must exist as long as the thread exists */
		static pthread_attr_t thread_attr;

		pthread_attr_init (&thread_attr);
		pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
		if(pthread_create(&thread_id,&thread_attr,(void*)do_hub_reg,NULL)!=0)
		{
			last_hub_registration+=10;		 /* fail -> retry 10 seconds later */
		}
		else
			last_hub_registration=cur_time;
	}
#endif
}

/*
This part had been remove from dchub 0.2.5 to be remplace by the old registration function
this part of the code will be user in dechub 0.3.0
*/

#if 0

static char *gen_reg_str()
{
	char *hubname;
	char *hubaddr;
	char *hubdesc;
	guint64 shared_size;
	unsigned int nb_users;
	static GString *fhd = NULL;
	int	str_gen = 0;

	/* load conffile */

	if (fhd == NULL)
		fhd=g_string_new(""); /* init output buffer */

	G_LOCK(conf_file);
	hubname=e_db_str_get(conf_file,"HUBNAME");
	hubaddr=e_db_str_get(conf_file,"HUBADDR");
	hubdesc=e_db_str_get(conf_file,"HUBDESC");
	G_UNLOCK(conf_file);
	
	/* if have reg addr */
	/* get hub info */
	HL_LOCK_READ(cnx_struct_lock);
	get_hub_user_stat(&shared_size,&nb_users);
	HL_UNLOCK_READ(cnx_struct_lock);

#ifdef GLIB_INT_64_BUG
	g_string_sprintfa(fhd,"%s|%s:%hu|%s|%u|%s|",
				hubname!=NULL?hubname:"",
				hubaddr!=NULL?hubaddr:"",
					(reg_hub_port==0)?((hub_port!=411)?hub_port:411):((reg_hub_port!=411)?reg_hub_port:411),
				hubdesc!=NULL?hubdesc:"",
				nb_users,
				llunsigned_to_str(shared_size));
#else
	g_string_sprintfa(fhd,"%s|%s:%hu|%s|%u|%Lu|",
				hubname!=NULL?hubname:"",
				hubaddr!=NULL?hubaddr:"",
					(reg_hub_port==0)?((hub_port!=411)?hub_port:411):((reg_hub_port!=411)?reg_hub_port:411),
				hubdesc!=NULL?hubdesc:"",
				nb_users,
				shared_size);
	str_gen = 1;
#endif
	if(hubname!=NULL)
		free(hubname);
	if(hubaddr!=NULL)
		free(hubaddr);
	if(hubdesc!=NULL)
		free(hubdesc);
	if (str_gen)
	{
		printf("trying to reg:%s\n", fhd->str);
		return(fhd->str);
	}
	return (NULL);
}

void try_new_registration(const char *main_server_addr)
{
#ifndef HAVE_NO_PTHREAD
	gl_reg_sock = create_and_open_sock_on(main_server_addr,2501,0,gl_bind_addr);
#else
	fprintf(stderr, "start bind red serv (if dchub get stuck here install Linux.)\n");
	gl_reg_sock = create_and_open_sock_on(main_server_addr,2501,1,gl_bind_addr);
	fprintf(stderr, "end bind reg serv\n");
#endif
	if (gl_reg_sock < 0)
	{
		perror("create - registration failed");
		gl_reg_sock = -1;
		return;
	}
	return;
}

void	check_hub_registration(void)
{
	char buf[512];
	int ln;
	char *t;
	struct sockaddr_in lcl;
	int len_lcl=sizeof(lcl);
	unsigned int lport;
	unsigned char magic;
	GString *in,*out;

	printf("Check hub_reg\n");	
	if(getsockname(gl_reg_sock,(void*)&lcl,&len_lcl)!=0)
	{
		perror("getsockname - registration failed");
		close(gl_reg_sock);
		gl_reg_sock=-1;
		return;
	}
	lport=ntohs(lcl.sin_port);
	magic=lport+(lport>>8); /* add the 2 bytes of the local port and create a 1 byte magic value */
	ln=recv(gl_reg_sock, buf,sizeof(buf)-1,MSG_NOSIGNAL);
	if(ln==-1)
	{
		perror("recv - registration failed");
		goto end_reg;
	}
	if(ln==0)
	{
		fprintf(stderr, "recv no data - registration failed\n");
		goto end_reg;
	}
	buf[ln]='\0';

	in=g_string_new(buf);
	if(strncmp(buf,"$Lock ",6))
	{
		printf("no $Lock: %s\n",in->str);
		g_string_free(in,TRUE);
		goto end_reg;
	}
	in=g_string_erase(in,0,6);
	t=strstr(in->str," Pk=");
	if(t==NULL)
	{
		printf("no Pk= %s\n",in->str);
		g_string_free(in,TRUE);
		goto end_reg;
	}
	in=g_string_truncate(in,t-in->str);
	out=compute_hub_reg_access_key(in,magic);
	out=g_string_prepend(out,"$Key ");
	out=g_string_append_c(out,'|');
	out=g_string_append(out,gen_reg_str());
	send_this(gl_reg_sock,out);
	g_string_free(in,TRUE);
	g_string_free(out,TRUE);
	printf("hub registration done\n");
 end_reg:
	shutdown(gl_reg_sock,2);
	close(gl_reg_sock);
	gl_reg_sock=-1;
	return;
}
#ifndef HAVE_NO_PTHREAD

static void	*self_reg_loop(void *ptr)
{
	fprintf(stderr, "Registration thread started\n");
	set_user_id();	/* change user ID if required */

	while (1)
	{
		int		do_hub_registration = 0;
		char	*reg_serv = NULL;

		G_LOCK(conf_file);
		e_db_int_get(conf_file,"REG_HUB",&do_hub_registration);
		G_UNLOCK(conf_file);

		if (do_hub_registration == 0)
			goto skip;

		G_LOCK(conf_file);
		reg_serv=e_db_str_get(conf_file,"REG_SERV");
		G_UNLOCK(conf_file);

		if (reg_serv)
		{
			try_new_registration(reg_serv); /* start a registration */
			free(reg_serv);
		}
		if(gl_reg_sock!=-1)
			check_hub_registration();
		skip:
		sleep(DELAY_BETWEEN_REGISTRATION);
	}
	return (NULL);
}
#endif

int	strat_reg_thread(void)
{
#ifndef HAVE_NO_PTHREAD
	static pthread_t thread_id;
	static pthread_attr_t thread_attr;

	pthread_attr_init (&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
	if(pthread_create(&thread_id,&thread_attr,(void*)self_reg_loop,NULL)!=0)
	{
		fprintf(stderr, "cannot create registration thread! The hub won't be able to register on the registration server\n");
		return (1);
	}
#endif
	return (0);
}

#endif
