/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * chunk.h: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: chunk.h,v 2.12 2003/03/04 17:40:27 ericprev Exp $
*/

#ifndef __CHUNK_H__
#define __CHUNK_H__

/*
Chunk format:

 1 guint16: chunk_type
 1 guint16: nb_subchunk (=N)
 1 guint32: chunk_length  	(the size does not includes these header, only the size users by sub-chunks)
 1 guint32: CRC            (CRC computed on the above bytes) 
 N sub-chunks

Sub-Chunk format:
 1 guint16: subchunk_type
 1 guint16: subchunk_length (=M) (the size is the size of the data of the sub-chunk without this header)
 M bytes.
 
All Values are in network byte order (==big endian).
 */
#define XS_GUINT32(gba,offset)		(GET_UAA_GUINT32(&(gba->data[offset])))
#define XS_GUINT16(gba,offset)		(GET_UAA_GUINT16(&(gba->data[offset])))


#define HCHUNK_SIZE (2*sizeof(guint16)+sizeof(guint32)+sizeof(guint32))
#define HCHUNKF_CTYPE(gba) (XS_GUINT16(gba,0))
#define HCHUNKF_NB_CHUNK(gba) (XS_GUINT16(gba,sizeof(guint16)))
#define HCHUNKF_CHUNK_LEN(gba) (XS_GUINT32(gba,2*sizeof(guint16)))

#define SHCHUNK_SIZE (2*sizeof(guint16))
#define SHCHUNKF_CTYPE(gba,offset) (XS_GUINT16(gba,offset+0))
#define SHCHUNKF_SCHUNK_LEN(gba,offset) (XS_GUINT16(gba,offset+sizeof(guint16)))
#define SHCHUNKF_DATA_ADR(gba,offset) (&(gba->data[offset+SHCHUNK_SIZE]))


typedef struct
{
	guint16 chunk_type;	/* host format */
	guint16 chunk_size;	/* host format */
	unsigned char *chunk_content;
} SUBCHUNK_CORE;

typedef struct
{
	guint16 chunk_type;	/* host format */
	guint16 nb_subchunk;
	SUBCHUNK_CORE *subchunk;
} CHUNK_CORE;

/************************************************************/
/* convert a chunk into its planar structure (to save/send) */
/************************************************************/
GByteArray *chunk_convert_chunk_to_mem(CHUNK_CORE *cc);

/******************************************************/
/* convert parameters into a planar structure         */
/* use SC_PARAM_* to provide parameters without error */
/******************************************************/
GByteArray *chunk_convert_param_to_mem(guint16 chunk_type, guint16 nb_subchunk, ...);
#define SC_PARAM_GINT8(name,var)    (int)name,(int)sizeof(gint8),(void*)&(var)
#define SC_PARAM_GINT16(name,var)   (int)name,(int)sizeof(gint16),(void*)&(var)
#define SC_PARAM_GINT32(name,var)   (int)name,(int)sizeof(gint32),(void*)&(var)
#define SC_PARAM_GUINT8(name,var)   (int)name,(int)sizeof(guint8),(void*)&(var)
#define SC_PARAM_GUINT16(name,var)  (int)name,(int)sizeof(guint16),(void*)&(var)
#define SC_PARAM_GUINT32(name,var)  (int)name,(int)sizeof(guint32),(void*)&(var)
#define SC_PARAM_GUINT64(name,var)  (int)name,(int)sizeof(guint64),(void*)&(var)
#define SC_PARAM_PTR(name,size,addr)   (int)name,(int)size,(void*)addr


/*******************************************/
/* convert a planar structure into a chunk */
/************************************************************************************/
/* output: the decoded chunk_core and data used by the decoder are removed from gba */
/*         else NULL and too_small is set to TRUE if there is not enough data       */
/*         available.                                                               */
/*         if *too_small==FALSE and the output is NULL, is_invalid is set to TRUE   */
/*         if the buffer has no meaning                                             */
/************************************************************************************/
CHUNK_CORE *chunk_convert_mem_to_chunk(GByteArray *gba, gboolean *too_small, gboolean *is_invalid);

/*******************************/
/* free memory used by a chunk */
/*******************************/
void chunk_free(CHUNK_CORE *cc);

/* -------------- misc functions -------------- */
/********************************************/
/* create a new chunk core without subchunk */
/********************************************/
CHUNK_CORE *chunk_core_new(guint16 chunk_type);

/********************************/
/* append a subchunk to a chunk */
/*************************************************************************/
/* note: The chunk content is duplicated and a copy is used in the chunk */
/*************************************************************************/
void chunk_core_append_subchunk(CHUNK_CORE *cc,guint16 chunk_type, guint16 chunk_size, const void *chunk_content);

/**********************************************************************/
/* search inside a CHUNK_CORE for a SUBCHUNK_CORE having the given ID */
/**********************************************************************/
/* output: NULL if not found, else a pointer on the SUBCHUNK_CORE */
/******************************************************************/
SUBCHUNK_CORE *chunk_get_subcore(CHUNK_CORE *cc, guint16 chunk_type);

/*******************************************************************/
/* extract gint or guint from a subchunk core. SC size must be the */
/* sizeof g*int8, g*int16 or g*int32                               */
/*******************************************************************/
gint32 subchunk_get_gint(SUBCHUNK_CORE *sc);
guint32 subchunk_get_guint(SUBCHUNK_CORE *sc);
guint64 subchunk_get_guint64(SUBCHUNK_CORE *sc);

/************************************************************************************************/
/* extract a string from a subchunk and ASSIGN it to the given gstring (the gstring must exist) */
/************************************************************************************************/
/* output: TRUE if the previous value is different of the new one, else FALSE */
/******************************************************************************/
gboolean subchunk_assign_gstring(GString *str,SUBCHUNK_CORE *sc); 

/****************************************/
/* convert chunk error code into string */
/****************************************/
const char *sc_num_err_to_str(gint err_code);

/* -------------- misc macro -------------- */
/* retrieve a value and set it without default one  */
/* updated_ is set to TRUE if the value has changed */

#define CHUNK_GET_AND_SET_UVAL_ND(dest_,chunk_,sc_code_,updated_) \
			{                                                       \
				SUBCHUNK_CORE *sc__;                                 \
				if((sc__=chunk_get_subcore(chunk_,sc_code_))!=NULL)  \
				{                                                    \
				   guint32 old_val_;                                 \
					old_val_=(dest_);                                 \
					(dest_)=subchunk_get_guint(sc__);                 \
               if(old_val_!=(dest_))                             \
						updated_=TRUE;                                 \
				}                                                    \
			}

#define CHUNK_GET_AND_SET_SVAL_ND(dest_,chunk_,sc_code_,updated_) \
			{                                                       \
				SUBCHUNK_CORE *sc__;                                 \
				if((sc__=chunk_get_subcore(chunk_,sc_code_))!=NULL)  \
				{                                                    \
				   gint32 old_val_;                                  \
					old_val_=(dest_);                                 \
					(dest_)=subchunk_get_gint(sc__);                  \
               if(old_val_!=(dest_))                             \
						updated_=TRUE;                                 \
				}                                                    \
			}

#define CHUNK_GET_AND_SET_ULVAL_ND(dest_,chunk_,sc_code_,updated_) \
			{                                                       \
				SUBCHUNK_CORE *sc__;                                 \
				if((sc__=chunk_get_subcore(chunk_,sc_code_))!=NULL)  \
				{                                                    \
				   guint64 old_val_;                                 \
					old_val_=(dest_);                                 \
					(dest_)=subchunk_get_guint64(sc__);               \
               if(old_val_!=(dest_))                             \
						updated_=TRUE;                                 \
				}                                                    \
			}

#define CHUNK_GET_AND_SET_NNSTR_in_gstring_ND(dest_,chunk_,sc_code_,updated_)        \
			{                                                       \
				SUBCHUNK_CORE *sc__;                                 \
				if((sc__=chunk_get_subcore(chunk_,sc_code_))!=NULL)  \
				{                                                    \
					updated_|=subchunk_assign_gstring((dest_),sc__);  \
				}                                                    \
			}

#endif
