/* 
 * Copyright (C) 2005 Laird Breyer
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * Author:   Laird Breyer <laird@lbreyer.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>

#if defined HAVE_UNISTD_H
#include <unistd.h> 
#endif

#include <locale.h>

#if defined HAVE_LANGINFO_H
#include <langinfo.h>
#if !defined CODESET
/* on OpenBSD, CODESET doesn't seem to be defined - 
   we use 3, which should be US-ASCII, but it's not ideal... */
#define CODESET 3
#endif
#endif

#include "util.h" 
#include "dbacl.h" 
#include "hypex.h"


/* global variables */

extern hash_bit_count_t default_max_hash_bits;
extern hash_count_t default_max_tokens;

extern hash_bit_count_t default_max_grow_hash_bits;
extern hash_count_t default_max_grow_tokens;

hash_bit_count_t decimation;

extern options_t u_options;
extern options_t m_options;
extern char *extn;

extern token_order_t ngram_order; /* defaults to 1 */

/* for option processing */
extern char *optarg;
extern int optind, opterr, optopt;

extern char *progname;
extern char *inputfile;
extern long inputline;

int exit_code = 0; /* default */

extern long system_pagesize;

extern void *in_iobuf;
extern void *out_iobuf;


/***********************************************************
 * MISCELLANEOUS FUNCTIONS                                 *
 ***********************************************************/

static void usage(char **argv) {
  fprintf(stderr, 
	  "\n");
  fprintf(stderr, 
	  "hypex CATDUMP1 CATDUMP2...\n");
  fprintf(stderr, 
	  "\n");
  fprintf(stderr, 
	  "      computes the Chernoff information and related quantities for\n");
  fprintf(stderr, 
	  "      a binary hypothesis test based on the category dump files CATDUMP1, CATDUMP2.\n");
  fprintf(stderr, 
	  "\n");
  fprintf(stderr, 
	  "hypex -V\n");
  fprintf(stderr, 
	  "\n");
  fprintf(stderr, 
	  "      prints program version.\n");
}



/***********************************************************
 * FILE MANAGEMENT FUNCTIONS                               *
 ***********************************************************/
bool_t load_category_dump(FILE *input, hypex_t *hypex) {
  int extra_lines = 0;

  if( hypex->stage > 2 ) {
    errormsg(E_WARNING, "too many categories.\n");
    return 0;
  }

  set_iobuf_mode(input);

  inputline = 0;

  while( fill_textbuf(input, &extra_lines) ) {
    inputline++;
    
  }


  return (++hypex->stage);
}


/***********************************************************
 * MAIN FUNCTIONS                                          *
 ***********************************************************/
int set_option(int op, char *optarg) {
  int c = 0;
  switch(op) {
  case 'V':
    fprintf(stdout, "hypex version %s\n", VERSION);
    fprintf(stdout, COPYBLURB, "hypex");
    exit(1);
    break;
  default:
    c--;
    break;
  }
  return c;
}

void sanitize_options() {

  /* consistency checks */

  /* decide if we need some options */

}


int main(int argc, char **argv) {

  FILE *input;
  signed char op;
  hypex_t hypex;

  progname = "hypex";
  inputfile = "";
  inputline = 0;

  memset(&hypex, 0, sizeof(hypex_t));

  /* set up internationalization */
  if( !setlocale(LC_ALL, "") ) {
    errormsg(E_WARNING,
	    "could not set locale, internationalization disabled\n");
  } else {
    if( u_options & (1<<U_OPTION_VERBOSE) ) {
      errormsg(E_WARNING,
	      "international locales not supported\n");
    }
  }

#if defined(HAVE_GETPAGESIZE)
  system_pagesize = getpagesize();
#endif
  if( system_pagesize == -1 ) { system_pagesize = BUFSIZ; }

  /* parse the options */
  while( (op = getopt(argc, argv, 
		      "V")) > -1 ) {
    set_option(op, optarg);
  }

  /* end option processing */
  sanitize_options();

  init_buffers();

  /* now process each file on the command line,
     or if none provided read stdin */
  while( (optind > -1) && *(argv + optind) ) {
    /* if it's a filename, process it */
    input = fopen(argv[optind], "rb");
    if( input ) {
      inputfile = argv[optind];

      if( u_options & (1<<U_OPTION_VERBOSE) ) {
	fprintf(stdout, "processing file %s\n", argv[optind]);
      }

      if( !load_category_dump(input, &hypex) ) {
	usage(argv);
	optind = -2;
      }

      fclose(input);

    } else { /* unrecognized file name */

      errormsg(E_ERROR, "couldn't open %s\n", argv[optind]);
      usage(argv);
      exit(0);

    }
    optind++;
  }
  
  cleanup_buffers();

  return exit_code;
}


