/* This file is part of the db4o object database http://www.db4o.com

Copyright (C) 2004 - 2009  Versant Corporation http://www.versant.com

db4o is free software; you can redistribute it and/or modify it under
the terms of version 3 of the GNU General Public License as published
by the Free Software Foundation.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program.  If not, see http://www.gnu.org/licenses/. */
using System.IO;
using Db4oUnit;
using Db4oUnit.Extensions.Fixtures;
using Db4objects.Db4o;
using Db4objects.Db4o.Config;
using Db4objects.Db4o.Defragment;
using Db4objects.Db4o.Foundation.IO;
using Db4objects.Db4o.IO;
using Db4objects.Db4o.Tests.Common.Defragment;

namespace Db4objects.Db4o.Tests.Common.Defragment
{
	public class DefragEncryptedFileTestCase : ITestLifeCycle, IOptOutNoFileSystemData
	{
		private static readonly string Original = Path.GetTempFileName();

		private static readonly string Defraged = Original + ".bk";

		/// <exception cref="System.Exception"></exception>
		public virtual void SetUp()
		{
			Cleanup();
		}

		/// <exception cref="System.Exception"></exception>
		public virtual void TearDown()
		{
			Cleanup();
		}

		private void Cleanup()
		{
			File4.Delete(Original);
			File4.Delete(Defraged);
		}

		public static void Main(string[] args)
		{
			new ConsoleTestRunner(typeof(DefragEncryptedFileTestCase)).Run();
		}

		/// <exception cref="System.Exception"></exception>
		public virtual void TestCOR775()
		{
			Prepare();
			VerifyDB();
			DefragmentConfig config = new DefragmentConfig(Original, Defraged);
			config.ForceBackupDelete(true);
			//config.storedClassFilter(new AvailableClassFilter());
			config.Db4oConfig(GetConfiguration());
			Db4objects.Db4o.Defragment.Defragment.Defrag(config);
			VerifyDB();
		}

		private void Prepare()
		{
			Sharpen.IO.File file = new Sharpen.IO.File(Original);
			if (file.Exists())
			{
				file.Delete();
			}
			IObjectContainer testDB = OpenDB();
			DefragEncryptedFileTestCase.Item item = new DefragEncryptedFileTestCase.Item("richard"
				, 100);
			testDB.Store(item);
			testDB.Close();
		}

		private void VerifyDB()
		{
			IObjectContainer testDB = OpenDB();
			IObjectSet result = testDB.QueryByExample(typeof(DefragEncryptedFileTestCase.Item
				));
			if (result.HasNext())
			{
				DefragEncryptedFileTestCase.Item retrievedItem = (DefragEncryptedFileTestCase.Item
					)result.Next();
				Assert.AreEqual("richard", retrievedItem.name);
				Assert.AreEqual(100, retrievedItem.value);
			}
			else
			{
				Assert.Fail("Cannot retrieve the expected object.");
			}
			testDB.Close();
		}

		private IObjectContainer OpenDB()
		{
			IConfiguration db4oConfig = GetConfiguration();
			IObjectContainer testDB = Db4oFactory.OpenFile(db4oConfig, Original);
			return testDB;
		}

		private IConfiguration GetConfiguration()
		{
			IConfiguration config = Db4oFactory.NewConfiguration();
			config.ActivationDepth(int.MaxValue);
			config.CallConstructors(true);
			IoAdapter ioAdapter = new DefragEncryptedFileTestCase.MockIOAdapter(new RandomAccessFileAdapter
				(), "db4o");
			config.Io(ioAdapter);
			return config;
		}

		public class Item
		{
			public string name;

			public int value;

			public Item(string name, int value)
			{
				this.name = name;
				this.value = value;
			}
		}

		public class MockIOAdapter : IoAdapter
		{
			private IoAdapter ioAdapter;

			private string password;

			private long pos;

			public MockIOAdapter(IoAdapter ioAdapter, string password)
			{
				this.ioAdapter = ioAdapter;
				this.password = password;
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override void Close()
			{
				ioAdapter.Close();
			}

			public override void Delete(string path)
			{
				ioAdapter.Delete(path);
			}

			public override bool Exists(string path)
			{
				return ioAdapter.Exists(path);
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override long GetLength()
			{
				return ioAdapter.GetLength();
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override IoAdapter Open(string path, bool lockFile, long initialLength, bool
				 readOnly)
			{
				return new DefragEncryptedFileTestCase.MockIOAdapter(ioAdapter.Open(path, lockFile
					, initialLength, readOnly), password);
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override int Read(byte[] bytes, int length)
			{
				ioAdapter.Read(bytes);
				for (int i = 0; i < length; i++)
				{
					bytes[i] = (byte)(bytes[i] - password.GetHashCode());
				}
				ioAdapter.Seek(pos + length);
				return length;
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override void Seek(long pos)
			{
				this.pos = pos;
				ioAdapter.Seek(pos);
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override void Sync()
			{
				ioAdapter.Sync();
			}

			/// <exception cref="Db4objects.Db4o.Ext.Db4oIOException"></exception>
			public override void Write(byte[] buffer, int length)
			{
				for (int i = 0; i < length; i++)
				{
					buffer[i] = (byte)(buffer[i] + password.GetHashCode());
				}
				ioAdapter.Write(buffer, length);
				Seek(pos + length);
			}
		}
	}
}
