/* Example program demonstrating the multi-thread capabilities/interface
   of Dazuko.
   Written by John Ogness <jogness@antivir.de>

   Copyright (C) 2003, H+BEDV Datentechnik GmbH
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

   3. Neither the name of Dazuko nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE.
*/

/* This program is a simple application demonstrating how to interface
   with Dazuko usint the multi-thread interface (_TS). The program
   instructs Dazuko to detect open, close, and execute accesses within
   the /home and /usr/home directories (or any subdirectory thereof).
   The program than prints out the accesses and instructs Dazuko to
   allow access.

   An application communicates with Dazuko through a device. After
   installing Dazuko (as kernel module or after building in the
   kernel itself), the device node must be created. The major device
   number is needed. This can be found by looking at /proc/devices.

   example:
   > grep dazuko /proc/devices
   < 254 dazuko
   > mknod -m 600 /dev/dazuko c 254 0

   After Dazuko is loaded and the device node is created, this example
   program SHOULD run (as is) without any problems.

   Note: For security reasons, Dazuko will only interact with applications
         running as root. Therefore, this example program must also be run
         as root.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/types.h>
#include "dazukoio.h"

#define NUMBER_OF_THREADS 5

int	RUNNING = 1;		/* flag for main loop */
int	id[NUMBER_OF_THREADS]; 	/* id's for threads */

void sigterm(int sig)
{
	/* exit the loop on TERM or INT signal */

	printf("detected signal: %d\n", sig);

	RUNNING = 0;
	signal(sig, sigterm);
}

void print_access(struct access_t *acc_ptr, int thread_id)
{
	/* print access data */

	switch (acc_ptr->event)
	{
		case ON_OPEN:
			printf("OPEN ");
			break;
		case ON_CLOSE:
			printf("CLOSE ");
			break;
		case ON_CLOSE_MODIFIED:
			printf("CLOSE (modified) ");
			break;
		case ON_EXEC:
			printf("EXEC ");
			break;
		default:
			printf("???? ");
			break;
	}
	printf("flags:%d ", acc_ptr->o_flags);
	printf("mode:%d ", acc_ptr->o_mode);
	printf("uid:%d ", acc_ptr->uid);
	printf("pid:%d ", acc_ptr->pid);
	printf("file:%s ", acc_ptr->filename);
	printf(" (thread %d)", thread_id);
	printf("\n");

	fflush(stdout);
}

void* main_loop(void *param)
{
	struct dazuko_t d;
	struct access_t	acc;
	int thread_id = 0;

	if (param == NULL)
	{
		printf("error: invalid thread parameter\n");
		return NULL;
	}

	thread_id = *((int *)param);

	/* initialize dazuko structure */
	if (dazukoInitialize_TS(&d) != 0)
	{
		printf("error: failed to initialize dazuko structure (thread %d)\n", thread_id);
		return NULL;
	}

	/* register with dazuko */
	if (dazukoRegister_TS(&d, "DAZUKO_MT_EXAMPLE") != 0)
	{
		printf("error: failed to register with Dazuko (thread %d)\n", thread_id);
		return NULL;
	}

	printf("registered with Dazuko successfully (thread %d)\n", thread_id);

	/* set access mask */
	if (dazukoSetAccessMask_TS(&d, ON_OPEN | ON_CLOSE | ON_CLOSE_MODIFIED | ON_EXEC) != 0)
	{
		printf("error: failed to set access mask (thread %d)\n", thread_id);
		dazukoUnregister_TS(&d);
		return NULL;
	}

	printf("set access mask successfully (thread %d)\n", thread_id);

	/* set scan path */
	if (dazukoAddIncludePath_TS(&d, "/home/") != 0)
	{
		printf("error: failed to add /home/ include path (thread %d)\n", thread_id);
		dazukoUnregister_TS(&d);
		return NULL;
	}

	/* set scan path */
	if (dazukoAddIncludePath_TS(&d, "/usr/home/") != 0)
	{
		printf("error: failed to add /usr/home/ include path (thread %d)\n", thread_id);
		dazukoUnregister_TS(&d);
		return NULL;
	}

	printf("set scan path successfully (thread %d)\n", thread_id);

	while (RUNNING)
	{
		/* get an access */
		if (dazukoGetAccess_TS(&d, &acc) == 0)
		{
			print_access(&acc, thread_id);

			/* always allow access */
			acc.deny = 0;

			/* return access (IMPORTANT, the kernel is waiting for us!) */
			if (dazukoReturnAccess_TS(&d, &acc) != 0)
			{
				printf("error: failed to return access (thread %d)\n", thread_id);
				RUNNING = 0;
			}
		}
		else
		{
			printf("warning: failed to get an access (thread %d)\n", thread_id);
			RUNNING = 0;
		}
	}

	/* unregister with Dazuko */
	if (dazukoUnregister_TS(&d) != 0)
	{
		printf("error: failed to unregister with Dazuko (thread %d)\n", thread_id);
		return NULL;
	}

	printf("unregistered with Dazuko successfully (thread %d)\n", thread_id);
 
	return NULL;
}

int main(void)
{
	pthread_t tid[NUMBER_OF_THREADS];
	int i;

	/* make sure we are root */
	if (getuid() != 0)
	{
		printf("only root can run this program\n");
		return -1;
	}

	/* detect TERM signals */
	signal(SIGTERM, sigterm);

	/* detect INT signals */
	signal(SIGINT, sigterm);

	for (i=0 ; i<NUMBER_OF_THREADS ; i++)
	{
		/* set the id for the thread */
		id[i] = i;

		/* create the thread */
		pthread_create(&(tid[i]), NULL, main_loop, &(id[i]));
	}

	for (i=0 ; i<NUMBER_OF_THREADS ; i++)
	{
		/* wait for the thread to finish */
		pthread_join(tid[i], NULL);
	}

	return 0;
}
