/* 
** Send requests to the WebDAV server.
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include "config.h"

#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include <sys/stat.h>

#include <ne_alloc.h>
#include <ne_auth.h>
#include <ne_basic.h>
#include <ne_dates.h>
#include <ne_locks.h>
#include <ne_props.h>
#include <ne_request.h>
#include <ne_session.h>
#include <ne_socket.h>
#include <ne_string.h>
#include <ne_uri.h>
#include <ne_utils.h>

#include "dav_debug.h"
#include "defaults.h"
#include "mount_davfs.h"
#include "webdav.h"


/* Data Types */
/*============*/

/* Data structures used as userdata in neon callback functions. */

typedef struct {
    char *path;                 /* The *not* url-encoded path. */
    dav_props *results;         /* Start of the linked list of dav_props. */
} propfind_context;

typedef struct {
    ne_lock_store *locks;       /* The lock store of this session. */
    char *owner;                /* Lock owner name used by this session. */
    int ret;                    /* Will be set by lock_result(). */
} lock_context;

typedef struct {
    int error;                  /* An error occured while reading/writing. */
    const char *file;           /* cache_file to store the data in. */
    int fd;                     /* file descriptor of the open cache file. */
} get_context;


/* Private constants */
/*===================*/

/* Properties to be retrieved from the server. This constants
   are used by dav_get_collection(). */
enum {
    NAME = 0,
    TYPE,
    LENGTH,
    CREATION,
    MODIFIED,
    EXECUTE,
    END
};

static const ne_propname prop_names[] = {
    [NAME] = {"DAV:", "displayname"},
    [TYPE] = {"DAV:", "resourcetype"},
    [LENGTH] = {"DAV:", "getcontentlength"},
    [CREATION] ={"DAV:", "creationdate"},
    [MODIFIED] = {"DAV:", "getlastmodified"},
    [EXECUTE] = {"http://apache.org/dav/props/", "executable"},
    [END] = {NULL, NULL}
};


/* Private global variables */
/*==========================*/

/* The neon session. It also holds the name of the server (including scheme
   and host port if different from default), the path of the root collection,
   the lock store and the credentials for server and proxy. */
static ne_session *session;

/* Whether a terminal is available to communicate with the user.
 * Should be reset with set_no_terminal() when forking into daemon mode. */
static int have_terminal = 1;


/* Private function prototypes and inline functions */
/*==================================================*/

static inline char *dir_path(const char *path) {

    char *base = (char *) ne_get_session_private(session, "path");
    char *tmp_path = ne_concat(base, path, "/", NULL);
    char *spath = ne_path_escape(tmp_path);
    NE_FREE(tmp_path);
    return spath;
}

static inline char *file_path(const char *path) {

    char *base = (char *) ne_get_session_private(session, "path");
    char *tmp_path = ne_concat(base, path, NULL);
    char *spath = ne_path_escape(tmp_path);
    NE_FREE(tmp_path);
    return spath;
}

static int get_error(int ret, const char *method);

static int get_ne_error(const char *method);

/* Call-back functions for neon. */

static int auth(void *userdata, const char *realm, int attempt, char *user,
                char *pwd);

#if NE_VERSION_MINOR == 24
static void block_writer(void *userdata, const char *block, size_t length);
#else
static int block_writer(void *userdata, const char *block, size_t length);
#endif

static void lock_result(void *userdata, const struct ne_lock *lock, 
			                  const char *uri, const ne_status *status);

static void prop_result(void *userdata, const char *href,
                         const ne_prop_result_set *set);

static int ssl_verify(void *userdata, int failures,
                      const ne_ssl_certificate *cert);


/* Public functions */
/*==================*/

void dav_init_webdav(const dav_args *args) {

    if (ne_sock_init() != 0)
        error(EXIT_FAILURE, 0, "Socket library initalization failed");

    ne_uri *uri = (ne_uri *) ne_malloc(sizeof(ne_uri));
    if (ne_uri_parse(args->url, uri) != 0 || uri->host == NULL)
        error(EXIT_FAILURE, 0, "Invalid URL.");
    if (uri->scheme == NULL)
        uri->scheme = ne_strdup("http");
    if (uri->port == 0)
        uri->port = ne_uri_defaultport(uri->scheme);

    session = ne_session_create(uri->scheme, uri->host, uri->port);

    char *path = ne_path_unescape(uri->path);
    while (strlen(path) > 0 && *(path + strlen(path) -1) == '/')
        *(path + strlen(path) -1) = '\0';
    char *unescaped_path = ne_strdup(path);
    NE_FREE(path);
    ne_set_session_private(session, "path", unescaped_path);

    char *server;
    if (uri->port != ne_uri_defaultport(uri->scheme)) {
        if (asprintf(&server, "%s://%s:%i", uri->scheme, uri->host, uri->port)
                < 0)
            abort();
    } else {
        server = ne_concat(uri->scheme, "://", uri->host, NULL);
    }
    ne_uri_free(uri);
    NE_FREE(uri);
    ne_set_session_private(session, "server", server);

#if NE_VERSION_MINOR == 24
    ne_set_expect100(session, 1);
#endif

    ne_set_read_timeout(session, args->read_timeout);

    char *useragent = ne_concat(PACKAGE_TARNAME, "/", PACKAGE_VERSION, NULL);
    ne_set_useragent(session, useragent);
    NE_FREE(useragent);
    
    if (args->locks) {
        ne_lock_store *lock_store = ne_lockstore_create();
        ne_lockstore_register(lock_store, session);
        ne_set_session_private(session, "locks", lock_store);
    }

    if (args->username != NULL)
        ne_set_session_private(session, "user", ne_strdup(args->username));
    if (args->password != NULL)
        ne_set_session_private(session, "pwd", ne_strdup(args->password));
    ne_set_server_auth(session, auth, "server");
    
    if (args->useproxy && args->p_host != NULL) {
        ne_session_proxy(session, args->p_host, args->p_port);
        if (args->p_user != NULL)
            ne_set_session_private(session, "p_user", ne_strdup(args->p_user));
        if (args->p_passwd != NULL)
            ne_set_session_private(session, "p_pwd", ne_strdup(args->p_passwd));
        ne_set_proxy_auth(session, auth, "proxy");
    }
    
    if (strcmp(ne_get_scheme(session), "https") == 0) {
#if NE_VERSION_MINOR == 24
        if (!ne_supports_ssl())
#else
        if (!ne_has_support(NE_FEATURE_SSL))
#endif
            error(EXIT_FAILURE, 0, "Neon library does not support TLS/SSL.");
        ne_ssl_set_verify(session, ssl_verify, NULL);
        ne_ssl_trust_default_ca(session);
    }
}


void dav_close_webdav(void) {

    if(session == NULL)
        return;

    ne_lock_store *locks = (ne_lock_store *) ne_get_session_private(session,
                                                                    "locks");
    if (locks != NULL) {
        struct ne_lock *lock = ne_lockstore_first(locks);
        while (lock != NULL) {
            DBG1("  UNLOCK %s", lock->uri.path);
            int ret = ne_unlock(session, lock);
            ret = get_error(ret, "UNLOCK");
            lock = ne_lockstore_next(locks);
        }
        ne_lockstore_destroy(locks);
    }

    char *tofree = (char *) ne_get_session_private(session, "server");
    DBG0("Closing connection to");
    DBG1("  %s", tofree);
    NE_FREE(tofree);
    tofree = (char *) ne_get_session_private(session, "path");
    DBG1("  %s", tofree);
    NE_FREE(tofree);
    tofree = (char *) ne_get_session_private(session, "user");
    NE_FREE(tofree);
    tofree = (char *) ne_get_session_private(session, "pwd");
    NE_FREE(tofree);
    tofree = (char *) ne_get_session_private(session, "p_user");
    NE_FREE(tofree);
    tofree = (char *) ne_get_session_private(session, "p_pwd");
    NE_FREE(tofree);
    ne_session_destroy(session);
    ne_sock_exit();
}


int dav_delete(const char *path) {

    char *spath = file_path(path);
    DBG1("  DELETE %s", spath);
    int ret = ne_delete(session, spath);
    ret = get_error(ret, "DELETE");
    NE_FREE(spath);
    return ret;
}


int dav_delete_collection(const char *path) {

    char *spath = dir_path(path);
    DBG1("  DELETE %s", spath);
    int ret = ne_delete(session, spath);
    ret = get_error(ret, "DELETE");
    NE_FREE(spath);
    return ret;
}


int dav_get_collection(dav_props **props, const char *path) {

    propfind_context ctx;
    ctx.results = NULL;
    char *base = (char *) ne_get_session_private(session, "path");
    ctx.path = ne_concat(base, path, "/", NULL);
    char *spath = ne_path_escape(ctx.path);

    ne_propfind_handler *pfh = ne_propfind_create(session, spath, NE_DEPTH_ONE);
    DBG1("  PROPFIND %s", spath);
    int ret = ne_propfind_named(pfh, prop_names, prop_result, &ctx);
    ret = get_error(ret, "PROPFIND");
    if (ret != 0) {
        while(ctx.results != NULL) {
            NE_FREE(ctx.results->name);
            dav_props *tofree = ctx.results;
            ctx.results = ctx.results->next;
            NE_FREE(tofree);
        }    
    }
    ne_propfind_destroy(pfh);

    NE_FREE(spath);
    NE_FREE(ctx.path);
    *props = ctx.results;
    return ret;
}


#if NE_VERSION_MINOR == 24

int dav_get_file(const char *cache_path, const char *path, char **etag,
                 time_t *mtime, off_t *size) {

    char *spath = file_path(path);
    ne_request *req = ne_request_create(session, "GET", spath);

    char *mod_time = NULL;
    if (*etag != NULL)
        ne_add_request_header(req, "If-Non-Match", *etag);
    if (*mtime != 0) {
        mod_time = ne_rfc1123_date(*mtime);
        ne_add_request_header(req, "If-Modified-Since", mod_time);
    }

    char *r_mod_time = NULL;
    ne_add_response_header_handler(req, "Last-Modified", ne_duplicate_header,
                                   &r_mod_time);
    char *r_etag = NULL;
    ne_add_response_header_handler(req, "ETag", ne_duplicate_header, &r_etag);
    off_t total = 0;
    ne_add_response_header_handler(req, "Content-Length",
                                   ne_handle_numeric_header, &total);

    get_context ctx;
    ctx.error = 0;
    ctx.file = cache_path;
    ctx.fd = 0;
    ne_add_response_body_reader(req, ne_accept_2xx, block_writer, &ctx);

    DBG1("  GET %s", spath);
    int ret = ne_request_dispatch(req);
    ret = get_error(ret, "GET");

    if (ctx.error)
        ret = EIO;

    const ne_status *status = ne_get_status(req);
    if (ret == 0 && status->code != 304) {
        if (r_mod_time != NULL)
            *mtime = ne_httpdate_parse(r_mod_time);
        NE_FREE(*etag);
        if (r_etag != NULL)
            *etag = ne_strdup(r_etag);
        *size = total;
    }

    ne_request_destroy(req);
    if (ctx.fd > 0)
        close(ctx.fd);
    NE_FREE(r_etag);
    NE_FREE(r_mod_time);
    NE_FREE(mod_time);
    NE_FREE(spath);
    return ret;
}

#else     /* NE_VERSION_MINOR */

int dav_get_file(const char *cache_path, const char *path, char **etag,
                 time_t *mtime, off_t *size) {

    char *spath = file_path(path);
    ne_request *req = ne_request_create(session, "GET", spath);

    char *mod_time = NULL;
    if (*etag != NULL)
        ne_add_request_header(req, "If-Non-Match", *etag);
    if (*mtime != 0) {
        mod_time = ne_rfc1123_date(*mtime);
        ne_add_request_header(req, "If-Modified-Since", mod_time);
    }

    get_context ctx;
    ctx.file = cache_path;
    ctx.fd = 0;
    ne_add_response_body_reader(req, ne_accept_2xx, block_writer, &ctx);

    DBG1("  GET %s", spath);
    int ret = ne_request_dispatch(req);
    ret = get_error(ret, "GET");

    const ne_status *status = ne_get_status(req);
    if (ret == 0 && status->code != 304) {
        const char *value = ne_get_response_header(req, "Last-Modified");
        if (value != NULL)
            *mtime = ne_httpdate_parse(value);
        value = ne_get_response_header(req, "ETag");
        if (value != NULL)
            *etag = ne_strdup(value);
        value = ne_get_response_header(req, "Content-Length");
        if (value != NULL)
            *size = strtol(value, NULL, 10);
    }

    ne_request_destroy(req);
    if (ctx.fd > 0)
        close(ctx.fd);
    NE_FREE(spath);
    return ret;
}

#endif    /* NE_VERSION_MINOR */


int dav_head(const char *path) {

    char *spath = file_path(path);
    ne_request *req = ne_request_create(session, "HEAD", spath);

    DBG1("  HEAD %s", spath);
    int ret = ne_request_dispatch(req);
    ret = get_error(ret, "HEAD");
    ne_request_destroy(req);

    return ret;
}


int dav_lock(const char *path, int *locked) {

    ne_lock_store *locks = (ne_lock_store *) ne_get_session_private(session,
                                                                    "locks");
    if (locks == NULL) {
        *locked = 0;
        return 0;
    }

    ne_uri uri;   
    ne_fill_server_uri(session, &uri);
    uri.path = file_path(path);
    uri.authinfo = NULL;

    struct ne_lock *lock = ne_lockstore_findbyuri(locks, &uri);
    if (lock != NULL) {
        NE_FREE(uri.scheme);
        NE_FREE(uri.host);
        NE_FREE(uri.path);
        *locked = 1;
        return 0;
    }

    lock = ne_lock_create();
    lock->uri = uri;
    lock->owner = ne_concat((char *) ne_get_session_private(session, "user"),
                            "/", PACKAGE_STRING, NULL);
    DBG1("  LOCK %s", uri.path);
    int ret = ne_lock(session, lock);
    ret = get_error(ret, "LOCK");

    if (ret == 0) {
        ne_lockstore_add(locks, lock);
        *locked = 1;
        return 0;
    } else {
        if (ret == EACCES && dav_lock_discover(uri.path) == 0)
            *locked = (ne_lockstore_findbyuri(locks, &uri) == NULL);
        ne_lock_destroy(lock);
        return (*locked == 1) ? 0 : ret;
    }
}


int dav_lock_discover(const char *path) {

    lock_context ctx;
    ctx.locks = (ne_lock_store *) ne_get_session_private(session, "locks");
    if (ctx.locks == NULL)
        return 0;
    ctx.owner = ne_concat((char *) ne_get_session_private(session, "user"),
                          "/", PACKAGE_STRING, NULL);
    ctx.ret = 0;
    char *spath = file_path(path);

    DBG1("  LOCKDISCOVER %s", spath);
    int ret = ne_lock_discover(session, spath, lock_result, &ctx);
    ret = get_error(ret, "LOCKDISCOVER");
    NE_FREE(spath);
    NE_FREE(ctx.owner);

    return (ctx.ret == 0) ? ret : ctx.ret;
}


int dav_make_collection(const char *path) {

    char *spath = dir_path(path);
    DBG1("  MKCOL %s", spath);
    int ret = ne_mkcol(session, spath);
    ret = get_error(ret, "MKCOL");
    NE_FREE(spath);
    return ret;
}


int dav_move(const char *src, const char *dst) {

    char *src_path = file_path(src);
    char *dst_path = file_path(dst);
    DBG1("  MOVE %s", src_path);
    DBG1("       %s", dst_path);
    int ret = ne_move(session, 1, src_path, dst_path); 
    ret = get_error(ret, "MOVE");
    NE_FREE(src_path);
    NE_FREE(dst_path);
    return ret;
}


int dav_move_collection(const char *src, const char *dst) {

    char *src_path = dir_path(src);
    char *dst_path = dir_path(dst);
    DBG1("  MOVE %s", src_path);
    DBG1("       %s", dst_path);
    int ret = ne_move(session, 1, src_path, dst_path); 
    ret = get_error(ret, "MOVE");
    NE_FREE(src_path);
    NE_FREE(dst_path);
    return ret;
}


#if NE_VERSION_MINOR == 24

int dav_put(const char *cache_path, const char *path, char **etag,
            time_t *mtime) {

    char *spath = file_path(path);
    char *r_mod_time = NULL;
    char *r_etag = NULL;
    char *date = NULL;
    ne_request *req;
    int ret = 0;

    if (ne_get_session_private(session, "locks") == NULL
            && (*mtime != 0 || *etag != NULL)) {

        req = ne_request_create(session, "HEAD", spath);
        ne_add_response_header_handler(req, "Last-Modified",
                                       ne_duplicate_header, &r_mod_time);
        ne_add_response_header_handler(req, "ETag", ne_duplicate_header,
                                       &r_etag);
        DBG1("  HEAD %s", spath);
        ret = ne_request_dispatch(req);
        ret = get_error(ret, "HEAD");
        ne_request_destroy(req);
        if (*etag != NULL && r_etag != NULL && strcmp(*etag, r_etag) != 0)
            ret = EINVAL;
        if (*mtime != 0 && r_mod_time != NULL
                && *mtime < ne_httpdate_parse(r_mod_time))
            ret = EINVAL;
        NE_FREE(r_mod_time);
        NE_FREE(r_etag);
        if (ret != 0) {
            NE_FREE(spath);
            return ret;
        }
    }

    int fd = open(cache_path, O_RDONLY);
    if (fd <= 0) {
        NE_FREE(spath);
        return EIO;
    }

    req = ne_request_create(session, "PUT", spath);
    ne_lock_using_resource(req, spath, 0);
    if (ne_set_request_body_fd(req, fd) != 0) {
        ne_request_destroy(req);
        close(fd);
        NE_FREE(spath);
        return EIO;
    }
    ne_add_response_header_handler(req, "Date", ne_duplicate_header, &date);

    DBG1("  PUT %s", spath);
    ret = ne_request_dispatch(req);
    ret = get_error(ret, "PUT");
    ne_request_destroy(req);
    close(fd);

    if (ret != 0) {
        NE_FREE(date);
        NE_FREE(spath);
        return ret;
    }
        
    req = ne_request_create(session, "HEAD", spath);
    ne_add_response_header_handler(req, "Last-Modified", ne_duplicate_header,
                                   &r_mod_time);
    ne_add_response_header_handler(req, "ETag", ne_duplicate_header, &r_etag);

    DBG1("  HEAD %s", spath);
    ret = ne_request_dispatch(req);
    ret = get_error(ret, "HEAD");
    ne_request_destroy(req);

    if (r_mod_time != NULL) {
        *mtime = ne_httpdate_parse(r_mod_time);
    } else if (date != NULL) {
        *mtime = ne_httpdate_parse(date);
    } else {
        *mtime = time(NULL);
    }
    NE_FREE(*etag);
    if (r_etag != NULL)
        *etag = ne_strdup(r_etag);

    NE_FREE(date);
    NE_FREE(r_etag);
    NE_FREE(r_mod_time);
    NE_FREE(spath);
    return 0;
}

#else     /* NE_VERSION_MINOR */

int dav_put(const char *cache_path, const char *path, char **etag,
            time_t *mtime) {

    char *spath = file_path(path);
    const char *value = NULL;
    ne_request *req;
    int ret = 0;

    if (ne_get_session_private(session, "locks") == NULL
            && (*mtime != 0 || *etag != NULL)) {

        req = ne_request_create(session, "HEAD", spath);
        DBG1("  HEAD %s", spath);
        ret = ne_request_dispatch(req);
        ret = get_error(ret, "PUT");
        if (ret == 0) {
            if (*etag != NULL) {
                value = ne_get_response_header(req, "ETag");
                if (strcmp(*etag, value) != 0)
                    ret = EINVAL;
            }
            if (*mtime != 0) {
                value = ne_get_response_header(req, "Last-Modified");
                if (*mtime < ne_httpdate_parse(value))
                    ret = EINVAL;
            }
        }
        ne_request_destroy(req);
        if (ret != 0) {
            NE_FREE(spath);
            return ret;
        }
    }

    int fd = open(cache_path, O_RDONLY);
    if (fd <= 0) {
        NE_FREE(spath);
        return EIO;
    }
    struct stat st;
    if (fstat(fd, &st) != 0) {
        close(fd);
        NE_FREE(spath);
        return EIO;
    }

    req = ne_request_create(session, "PUT", spath);
    ne_set_request_expect100(req, 1);
    ne_lock_using_resource(req, spath, 0);
    ne_set_request_body_fd(req, fd, 0, st.st_size);

    DBG1("  PUT %s", spath);
    ret = ne_request_dispatch(req);
    ret = get_error(ret, "PUT");
    if (ret == 0) {
        value = ne_get_response_header(req, "Date");
        if (value != NULL)
            *mtime = ne_httpdate_parse(value);
    }
    ne_request_destroy(req);
    close(fd);

    if (ret != 0) {
        NE_FREE(spath);
        return ret;
    }
        
    req = ne_request_create(session, "HEAD", spath);
    DBG1("  HEAD %s", spath);
    ret = ne_request_dispatch(req);
    ret = get_error(ret, "HEAD");
    if (ret == 0) {
        value = ne_get_response_header(req, "Last-Modified");
        if (value != NULL)
            *mtime = ne_httpdate_parse(value);
        NE_FREE(*etag);
        value = ne_get_response_header(req, "ETag");
        if (value != NULL)
            *etag = ne_strdup(value);
    }
    ne_request_destroy(req);

    NE_FREE(spath);
    return 0;
}

#endif    /* NE_VERSION_MINOR */


int dav_set_execute(const char *path, int set) {

    ne_proppatch_operation op[2];
    op[0].name = &prop_names[EXECUTE];
    op[0].type = ne_propset;
    if (set) {
        op[0].value = ne_strdup("T");
    } else {
        op[0].value = ne_strdup("F");
    }
    op[1].name = NULL;

    char *spath = file_path(path);
    DBG1("  PROPPATCH %s", spath);
    int ret = ne_proppatch(session, spath, &op[0]);
    ret = get_error(ret, "PROPPATCH");
    NE_FREE(spath);

    return ret;
}


void dav_set_no_terminal(void) {
    have_terminal = 0;
}


int dav_unlock(const char *path, int *locked) {

    ne_lock_store *locks = (ne_lock_store *) ne_get_session_private(session,
                                                                    "locks");
    if (locks == NULL) {
        *locked = 0;
        return 0;
    }

    ne_uri uri;   
    ne_fill_server_uri(session, &uri);
    uri.path = file_path(path);
    uri.authinfo = NULL;

    struct ne_lock *lock = ne_lockstore_findbyuri(locks, &uri);
    NE_FREE(uri.scheme);
    NE_FREE(uri.host);
    NE_FREE(uri.path);
    if (lock == NULL) {
        *locked = 0;
        return 0;
    }

    DBG1("  UNLOCK %s", lock->uri.path);
    int ret = ne_unlock(session, lock);
    ret = get_error(ret, "UNLOCK");
    if (ret == 0) {
        ne_lockstore_remove(locks, lock);
        ne_lock_destroy(lock);
        *locked = 0;
    }

    return ret;
}



/* Private functions */
/*===================*/

/* Returns a file error code according to NE_ERROR ret from the last WebDAV
   method call. If ret has value NE_ERROR the error code from the session is
   fetched and translated.
   ret    : the error code returned from NEON.
   method : name of the WebDAV method, used for debug messages.
   return value : a file error code according to errno.h. */
static int get_error(int ret, const char *method) {

    int err;
    switch (ret) {
    case NE_OK:
    case NE_ERROR:
        err = get_ne_error(method);
        break;
    case NE_LOOKUP:
        DBG1("  %s neon: NE_LOOKUP failure", method);
        err = EAGAIN;
        break;
    case NE_AUTH:
        DBG1("  %s neon: NE_AUTH failure", method);
        err = EPERM;
        break;
    case NE_PROXYAUTH:
        DBG1("  %s: NE_PROXYAUTH failure", method);
        err = EPERM;
        break;
    case NE_CONNECT:
        DBG1("  %s: NE_CONNECT failure", method);
        err = EAGAIN;
        break;
    case NE_TIMEOUT:
        DBG1("  %s: NE_TIMEOUT failure", method);
        err = EAGAIN;
        break;
    case NE_FAILED:
        DBG1("  %s neon: NE_FAILED failure", method);
        err = EINVAL;
        break;
    case NE_RETRY:
        DBG1("  %s: NE_RETRY failure", method);
        err = EAGAIN;
        break;
    case NE_REDIRECT:
        DBG1("  %s: NE_REDIRECT failure", method);
        err = ENOENT;
        break;
    default:
        DBG1("  %s: Unknown error", method);
        err = EIO;
        break;
    }

    return err;
}

/* Get the error from the session and translates it into a file error code.
   method : name of the WebDAV method, used for debug messages.
   return value : a file error code according to errno.h. */
static int get_ne_error(const char *method) {

    const char *text = ne_get_error(session);
    DBG2("  %s: %s", method, text);

    char *tail;
    int err = strtol(text, &tail, 10);
    if (tail == text)
        return EIO;

    switch (err) {
        case 200:
        case 201:
        case 204:
        case 207:
        case 304:
            return 0;
        case 401:
        case 403:
        case 405:
        case 407:
            return EPERM;
        case 301:
        case 303:
        case 404:
        case 410:
            return ENOENT;
        case 408:
        case 500:
        case 503:
        case 504:
            return EAGAIN;
        case 423:
            return EACCES;
        case 406:
        case 409:
        case 412:
        case 414:
        case 415:
        case 416:
        case 424:
            return EINVAL;
        case 507:
            return ENOSPC;
        default:
            return EIO;
    }
}


/* Call-back functions for neon. */

/* Call-back-function called by neon to get creditentials.
   userdata must be a string with value "server" or "proxy", to decide what
   the creditentials are needed for. The creditentials must be stored
   in the session as private data with names user, pwd, p-user or p_pwd.
   If attempt is 0, the creditentials are copied to user and pwd.
   If attempt > 0, this is logged as an error.
   userdata : what the credentials are needed for (server or proxy).
   realm    : ignored, as no terminal is available to tell the user.
   attempt  : number of attempt to get credentials. If not 0 an error occured.
   user     : username will be returned in this parameter.
   pwd      : password will be returned in this parameter.
   return value : value if attempt. neon will not call this function again if
                  it is greater than 0. */
static int auth(void *userdata, const char *realm, int attempt, char *user,
                char *pwd) {

    DBG1("  Neon wants creditentials for %s.", realm);
    if (attempt != 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "Authentication failure:");
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "  %s", realm);
        return attempt;
    }

    char *s;
    if (strcmp((char *) userdata, "server") == 0) {
        s = (char *) ne_get_session_private(session, "user");
        if (s != NULL)
            strncpy(user, s, NE_ABUFSIZ - 1);
        s = (char *) ne_get_session_private(session, "pwd");
        if (s != NULL)
            strncpy(pwd, s, NE_ABUFSIZ - 1);
    } else if (strcmp((char *) userdata, "proxy") == 0) {
        s = (char *) ne_get_session_private(session, "p_user");
        if (s != NULL)
            strncpy(user, s, NE_ABUFSIZ - 1);
        s = (char *) ne_get_session_private(session, "p_pwd");
        if (s != NULL)
            strncpy(pwd, s, NE_ABUFSIZ - 1);
    }
    return 0;
}


/* Call back function called by neon to store data retrieved with GET from the
   server in the cach_file.
   userdata : get_context structure, containing the name of the cache_file, the
              file descriptor, and a boolean variable indicating an error.
   block    : buffer containing the data.
   length   : length of data in the buffer. */
#if NE_VERSION_MINOR == 24

static void block_writer(void *userdata, const char *block, size_t length) {

    get_context *ctx = userdata;
    if (ctx->fd == 0)
        ctx->fd = open(ctx->file, O_WRONLY | O_CREAT | O_TRUNC);
    if (ctx->fd <= 0)
        ctx->error = EIO;

    while (!ctx->error && length > 0) {
        ssize_t ret = write(ctx->fd, block, length);
        if (ret < 0) {
            ctx->error = 1;
            syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
                   "Error writing to cache file.");
        } else {
            length -= ret;
            block += ret;
        }
    }
}

#else     /* NE_VERSION_MINOR */

static int block_writer(void *userdata, const char *block, size_t length) {

    get_context *ctx = userdata;
    if (ctx->fd == 0)
        ctx->fd = open(ctx->file, O_WRONLY | O_CREAT | O_TRUNC);
    if (ctx->fd <= 0) {
        ne_set_error(session, "0 Could not open cache file.");
        return EIO;
    }

    while (length > 0) {
        ssize_t ret = write(ctx->fd, block, length);
        if (ret < 0) {
            ne_set_error(session, "0 Error writing to cache file.");
            syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
                   "Error writing to cache file.");
            return EIO;
        } else {
            length -= ret;
            block += ret;
        }
    }
    return 0;
}

#endif    /* NE_VERSION_MINOR */


/* Callback function called by ne_lock_discover(). The lock_context structure
   userdata contains the name of lock owner used for this session. If the
   owner of lock is the same it is copied into the lock store, else the return
   value in userdata will be set to EACCESS.
   userdata : a lock_context structure containing the lock store, owner name
              and an integer that will be set to the return value.
   lock     : a lock found by ne_lock_discover() on the server.
   uri      : not used.
   status   : not used. */ 
static void lock_result(void *userdata, const struct ne_lock *lock, 
			                  const char *uri, const ne_status *status) {

    lock_context *ctx = userdata;
    if (lock == NULL)
        return;

    if (strcmp(lock->owner, ctx->owner) == 0) {
        ne_lockstore_add(ctx->locks, ne_lock_copy(lock));
    } else if (lock->scope == ne_lockscope_exclusive) {
        ctx->ret = EACCES;
    }
}


/* Callback function call by ne_propfind_named().
   userdata must be a pointer to a propfind_context structure.
   The properties are evaluated from href and set and filled into a dav_props
   structure which is added to the linked list in userdata. The properties to
   be retrieved are taken from the global constant prop_names[].
   Names stored in the dav_props structure are without leading and trailing
   slashes. They are relative to the directory stored as href in userdata.
   The relative name of this directory is the empty string.
   userdata : a propfind_context structure containing the path of the
              collection and the anchor of the linked list of properties.
   href     : value of the href propertiy returned by the server. It may be
              the complete URL of the collection or the path only.
   set      : contains the properties returned from the server.*/
static void prop_result(void *userdata, const char *href,
                        const ne_prop_result_set *set) {

    propfind_context *ctx = (propfind_context *) userdata;
    if (ctx == NULL)
        return;
 
    char *url = ne_path_unescape(href);
    char *server = (char *) ne_get_session_private(session, "server");
    char *path = url;
    if (strstr(url, server) == url)
        path = url + strlen(server);
    if (strstr(path, ctx->path) != path)
        return;

    dav_props *result = ne_calloc(sizeof(dav_props));
    const char *data;

    if (strcmp(path, ctx->path) == 0) {
        result->name = ne_strdup("");
    } else {
        data = ne_propset_value(set, &prop_names[NAME]);
        if (data != NULL) {
            result->name = ne_path_unescape(data);
        } else {
            result->name = ne_path_unescape(path + strlen(ctx->path));
        }
        while (strlen(result->name) > 0
                && *(result->name + strlen(result->name) -1) == '/')
            *(result->name + strlen(result->name) -1) = '\0';
    }
    NE_FREE(url);
    
    data = ne_propset_value(set, &prop_names[TYPE]);
    if (data != NULL && strstr(data, "collection") != NULL)
            result->is_dir = 1;

    data = ne_propset_value(set, &prop_names[LENGTH]);
    if (data != NULL)
         result->size = atoi(data);
    
    data = ne_propset_value(set, &prop_names[CREATION]);
    if (data != NULL)
        result->ctime = ne_httpdate_parse(data);

    data = ne_propset_value(set, &prop_names[MODIFIED]);
    if (data != NULL)
        result->mtime = ne_httpdate_parse(data);

    data = ne_propset_value(set, &prop_names[EXECUTE]);
    if (data == NULL) {
        result->is_exec = -1;
    } else if (*data == 'T') {
        result->is_exec = 1;
    }

    result->next = ctx->results;
    ctx->results = result;
}


/* Callback function called by neon when it cannot verify the server
   certificate cert. If a terminal is available the user is informed
   about the kind of error, the issuer, the subject and the fingerprint
   of the certificate and may decide to accept the certificate.
   If the user does not accept, or no terminal is available, the
   certificate is not accepted. In any case the error is logged.
   userdata : not used.
   failures : a constant indicating the kind of error.
   cert     : the server certificate that could not be verified by neon.
   return value : 0 accept the certificate for this session.
                  -1 don't accept the certificate. */
static int ssl_verify(void *userdata, int failures,
                      const ne_ssl_certificate *cert) {

    char *issuer = ne_ssl_readable_dname(ne_ssl_cert_issuer(cert));
    char *subject = ne_ssl_readable_dname(ne_ssl_cert_subject(cert));
    char *digest = ne_calloc(NE_SSL_DIGESTLEN);
    if (ne_ssl_cert_digest(cert, digest) != 0)
        NE_FREE(digest);

    char *error_msg;
    if (failures & NE_SSL_NOTYETVALID) {
        error_msg = "The server certificate is not yet valid.";
    } else if (failures & NE_SSL_EXPIRED) {
        error_msg = "The server certificate has expired.";
    } else if (failures & NE_SSL_IDMISMATCH) {
        error_msg = "The server certificate does not match the server name.";
    } else {
        error_msg = "The server certificate is not trusted.";
    }

    int ret = -1;
    if (have_terminal) {
        printf("%s\n", error_msg);
        printf("  presented for: %s\n", ne_ssl_cert_identity(cert));
        printf("  Issuer:  %s\n", issuer);
        printf("  Subject: %s\n", subject);
        printf("  Fingerprint: %s\n", digest);
        printf("\nYou only should accept this certificate, if you can\n");
        printf("verify the fingerprint! The server might be faked\n");
        printf("or there might be a man-in-the-middle-attack.\n");
        printf("Accept certificate for this session? [y,N] ");
        char *s = NULL;
        size_t n = 0;
        ssize_t len = 0;
        len = getline(&s, &n, stdin);
        if (len < 0)
            abort();
        if (*s == 'y' || *s == 'Y')
            ret = 0;
        NE_FREE(s);
    }

    syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), error_msg);
    syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "  issuer:  %s", issuer);
    syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "  subject: %s", subject);
    if (ret == 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "  Accepted by user.");
    }

    NE_FREE(issuer);
    NE_FREE(subject);
    NE_FREE(digest);
    return ret;
}
