/* 
** DAV directory chache
 */

/*
 *  This file is part of davfs2.
 *
 *  davfs2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  davfs2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with davfs2; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include "config.h"

#include <dirent.h>
#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <grp.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>
#include <utime.h>

#include <sys/stat.h>

#include <ne_alloc.h>
#include <ne_dates.h>
#include <ne_string.h>
#include <ne_xml.h>

#include "dav_debug.h"
#include "defaults.h"
#include "mount_davfs.h"
#include "webdav.h"
#include "cache.h"


/* Data Types */
/*============*/


/* Public global variables */
/*=========================*/

/* Alignment boundary of dav_node in byte.
 * Used to compute a hash value and file numbers from node pointers. */
int alignment;


/* Private constants */
/*===================*/

/* Constants describing different types of XML elements and attributes
   in cache index file. */
enum {
    ROOT = 1,
    DDIR,
    REG,
    NAME,
    CACHE_PATH,
    SIZE,
    MODE,
    UID,
    GID,
    ATIME,
    MTIME,
    CTIME,
    SMTIME,
    ETAG,
    END
};

static const char* const type[] = {
    [ROOT] = "root",
    [DDIR] = "dir",
    [REG] = "reg",
    [NAME] = "name",
    [CACHE_PATH] = "cache_path",
    [SIZE] = "size",
    [MODE] = "mode",
    [UID] = "uid",
    [GID] = "gid",
    [ATIME] = "atime",
    [MTIME] = "mtime",
    [CTIME] = "ctime",
    [SMTIME] = "smtime",
    [ETAG] = "etag",
    [END] = NULL
};


/* Private global variables */
/*==========================*/

/* Root node of the directory cache. */
static dav_node *root;

/* Directory for local buckups. */
static dav_node *backup;

/* List of invalid nodes. */
static dav_node *invalid;

/* A hash table to store the nodes. The hash is computed from the pointer
   to the node, which is also the node number. */
static dav_node **table;

/* Size of the hash table. */
static size_t table_size;

/* Defaults for file ownership and mode. */
static uid_t default_uid;
static gid_t default_gid;
static mode_t default_file_mode;
static mode_t default_dir_mode;
static mode_t file_umask;
static mode_t dir_umask;

/* Directory for cached files and directories. */
static char *cache_dir;

/* Actual size of the cached files. */
static size_t cache_size;

/* Maximum cache size. If open files require more space, this will
   be ignored. */
static size_t max_cache_size;

/* Pointer to the zap_file-function. Will be set by
   dav_register_kernel_interface(), which must be called by kernel-interface.
   Default is a dummy function, that does nothing. */
dav_zap_file_fn zap_file;

/* How long results of PROPFIND for directories is considered valid. */
static time_t expire;

/* Retry time in case of an I/O-error. */
static time_t retry;

/* Minimum retry time. */
static time_t min_retry;

/* Maximum retry time. */
static time_t max_retry;


/* Private function prototypes */
/*=============================*/

/* Node maintenance. */
static void add_node(dav_node *parent, dav_props *props);
static void attr_from_cache_file(dav_node *node);
static void backup_node(dav_node *orig);
static int check_cache_path(dav_node *node);
static void clean_tree(dav_node *node);
static int close_invalid(dav_node *node, int flags);
static void delete_cache_file(dav_node *node);
static void delete_node(dav_node *node);
static void delete_tree(dav_node *node);
static dav_node *new_node(dav_node *parent, mode_t mode);
static void remove_node(dav_node *node);
static void remove_from_table(dav_node *node);
static void remove_from_tree(dav_node *node);
static void set_cache_file_times(dav_node *node);
static int update_cache_file(dav_node *node, int locked);
static int update_directory(dav_node *dir);
static void update_node(dav_node *node, dav_props *props);

/* Get information about node. */
static int exists(const dav_node *node);
static dav_node *get_child(const dav_node *parent, const char *name);
static int has_permission(const dav_node *node, uid_t uid, int how);
static int is_busy(const dav_node *node);
static inline int is_cached(const dav_node *node) {
    return (S_ISREG(node->mode) && node->cache_path != NULL);
}
static inline int is_dir(const dav_node *node) {
    return S_ISDIR(node->mode);
}
static inline int is_dirty(const dav_node *node) {
    return (S_ISREG(node->mode) && !node->created &&
            node->mtime > node->smtime);
}
static inline int is_reg(const dav_node *node) {
    return S_ISREG(node->mode);
}
static char *strcat_path(const dav_node *node, const char *name);

/* Permanent cache maintenance. */
static void check_cache_dir(const dav_args *args);
static void delete_cache(void);
static void parse_index(void);
static int write_node(dav_node *node, FILE *file, const char *indent);
static int xml_cdata_decimal(void *userdata, int state, const char *cdata,
                             size_t len);
static int xml_cdata_mode(void *userdata, int state, const char *cdata,
                          size_t len);
static int xml_cdata_string(void *userdata, int state, const char *cdata,
                            size_t len);
static int xml_end_dir(void *userdata, int state, const char *nspace,
                       const char *name);
static int xml_end_reg(void *userdata, int state, const char *nspace,
                       const char *name);
static int xml_start_decimal(void *userdata, int parent, const char *nspace,
                             const char *name, const char **atts);
static int xml_start_dir(void *userdata, int parent, const char *nspace,
                         const char *name, const char **atts);
static int xml_start_mode(void *userdata, int parent, const char *nspace,
                          const char *name, const char **atts);
static int xml_start_reg(void *userdata, int parent, const char *nspace,
                         const char *name, const char **atts);
static int xml_start_root(void *userdata, int parent, const char *nspace,
                          const char *name, const char **atts);
static int xml_start_string(void *userdata, int parent, const char *nspace,
                            const char *name, const char **atts);

/* Auxiliary. */
static int test_alignment();
static void zap_file_dummy(dav_node *node);


/* Public functions */
/*==================*/

/* Initializing and closing the cache. */

void dav_init_cache(const dav_args *args) {

    alignment = test_alignment();
    DBG1("  Alignment of dav_node: %i", alignment);

    if (zap_file == NULL)
        zap_file = &zap_file_dummy;

    default_uid = args->uid;
    default_gid = args->gid;

    default_file_mode = args->file_mode | S_IFREG;
    default_dir_mode = args->dir_mode | S_IFDIR;
    file_umask = args->file_umask;
    dir_umask = args->dir_umask;

    table_size = args->table_size;
    table = ne_calloc(sizeof(*table) * table_size);

    expire = args->expire;
    min_retry = args->retry;
    max_retry = args->max_retry;
    
    DBG0("  Checking cache directory.");
    max_cache_size = args->cache_size;
    check_cache_dir(args);

    root = new_node(NULL, default_dir_mode);
    root->name = ne_strdup("");

    DBG0("  Reading stored cache data.");
    parse_index();

    backup = get_child(root, DAV_BACKUP_DIR);
    if (backup == NULL) {
        backup = new_node(root, S_IFDIR | S_IRWXU);
        backup->name = ne_strdup(DAV_BACKUP_DIR);
    }

    int ret = update_directory(root);
    if (ret == EPERM)
        error(EXIT_FAILURE, 0, "Authentication with server or proxy failed.\n"
              "  Look up the log files for details.");
    if (ret != 0)
        error(0, 0, "Connection failed, mounting anyway."
              "  File system will only be usable when connection comes up.");
}


void dav_close_cache(void) {

    zap_file = &zap_file_dummy;

    clean_tree(root);

    char *new_index = ne_concat(cache_dir, "/", DAV_INDEX, ".new", NULL);
    DBG1("  Creating index %s.", new_index);
    FILE *new_file = fopen(new_index, "w");
    if (new_file != NULL) {

        int ret = fprintf(new_file,
                          "<?xml version=\"1.0\" encoding=\"iso-8859-1\" ?>\n");
        if (ret >= 0)
            ret = fprintf(new_file, "<d:%s xmlns:d=\"%s\">\n", type[ROOT],
                          DAV_XML_NS);
        if (ret >= 0)
            ret =fprintf(new_file, "  <d:%s>%o</d:%s>\n", type[MODE],
                         root->mode, type[MODE]);
        while (ret >= 0 && root->childs != NULL)
            ret = write_node(root->childs, new_file, "  ");
        if (ret >= 0)
            ret = fprintf(new_file, "</d:%s>\n", type[ROOT]);

        fclose(new_file);
        remove_from_table(root);
        delete_node(root);

        if (ret >= 0) {
            DBG0("  Replacing old index");
            char *old_index = ne_concat(cache_dir, "/", DAV_INDEX, NULL);
            if (rename(new_index, old_index) != 0)
                syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR), "Could not replace\n"
                       "  %s\n  with %s.", old_index, new_index);
            NE_FREE(old_index);
        } else {
            syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
                   "Error writing to new index file\n  %s.", new_index);
        }
    } else {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Could not create new index file for\n  %s.", cache_dir);
    }
    NE_FREE(new_index);
}


void dav_idle(void) {

    int i;
    for (i = 0; i < table_size; i++) {
        dav_node *node = table[i];
        while (node != NULL) {

            if (is_reg(node) && node->o_write == 0
                    && (is_dirty(node) || node->created)
                    && node->parent != backup) {

                int set_exec = node->created
                               && (node->mode & (S_IXUSR | S_IXGRP | S_IXOTH));
                char *path = strcat_path(node, NULL);
                int ret = dav_put(node->cache_path, path, &node->etag,
                                  &node->smtime);
                if (ret == 0) {
                    if (set_exec)
                        dav_set_execute(path, 1);
                    if (node->locked)
                        dav_unlock(path, &node->locked);
                    node->created = 0;
                    node->mtime = node->smtime;
                    set_cache_file_times(node);
                }
                if (ret == EACCES || ret == EINVAL || ret == ENOENT
                        || ret == EPERM || ret == ENOSPC) {
                    backup_node(node);
                    if (node->locked)
                        dav_unlock(path, &node->locked);
                    delete_cache_file(node);
                }
                NE_FREE(path);
            }

            node = node->table_next;
        }
    }
}


const char *dav_register_kernel_interface(dav_zap_file_fn fn) {

    zap_file = fn;
    return cache_dir;
}


/* Upcalls from the kernel, via the interface module. */

int dav_access(dav_node *node, uid_t uid, int how) {

    if (!exists(node))
        return ENOENT;
    DBG_PATH(node, NULL);
    if (!has_permission(node, uid, how))
        return EACCES;
    return 0;
}


int dav_close(dav_node *node, uid_t uid, int flags) {

    if (!exists(node))
        return close_invalid(node, flags);
    DBG_PATH(node, NULL);

    if (is_dir(node)) {
        node->atime = time(NULL);
        return 0;
    }

    DBG2("  o_read %i, o_write %i", node->o_read, node->o_write);
    attr_from_cache_file(node);
    if ((O_ACCMODE & flags) == O_RDONLY) {
        node->o_read = (node->o_read > 0) ? (node->o_read - 1) : 0;
    } else {
        node->o_write = (node->o_write > 0) ? (node->o_write - 1) : 0;
        if (node->o_write == 0 && node->locked && !is_dirty(node)
                && !node->created) {
            char *path = strcat_path(node, NULL);
            dav_unlock(path, &node->locked);
            NE_FREE(path);
        }
    }

    return 0;
}


/* TODO: Replace dav_head. File may be created onthe server. Check for
   non-existent or empty instead. */
int dav_create(dav_node **nodep, dav_node *parent, const char *name,
               uid_t uid, int excl, mode_t mode) {

    if (!exists(parent))
        return ENOENT;
    DBG_PATH(parent, name);
    if (!is_dir(parent))
        return ENOTDIR;
    if (parent == backup)
        return EINVAL;
    if (!has_permission(parent, uid, X_OK | W_OK))
        return EACCES;

    if (get_child(parent, name) != NULL)
        return EEXIST;

    struct passwd *pw = getpwuid(uid);
    if (pw == NULL)
        return EINVAL;
    *nodep = new_node(parent, mode | S_IFREG);
    (*nodep)->name = ne_strdup(name);
    (*nodep)->uid = uid;
    (*nodep)->gid = pw->pw_gid;
    (*nodep)->created = 1;
    int ret = check_cache_path(*nodep);

    if (ret == 0) {
        char *path = strcat_path(parent, name);
        if (dav_head(path) == 0) {
            delete_cache_file(parent);
            ret = EEXIST;
        }
        ret = dav_lock(path, &(*nodep)->locked);
        NE_FREE(path);
    }

    if (ret != 0) {
        remove_from_tree(*nodep);
        remove_from_table(*nodep);
        delete_node(*nodep);
    }

    return ret;
}


int dav_getattr(dav_node *node, uid_t uid) {

    if (!exists(node))
        return ENOENT;
    DBG_PATH(node, NULL);
    if (node->parent != NULL && !has_permission(node->parent, uid, X_OK | R_OK))
        return EACCES;

    if (is_dir(node)) {
        update_directory(node);
    } else {
        attr_from_cache_file(node);
    }

    return 0;
}


int dav_lookup(dav_node **nodep, dav_node *parent, const char *name,
               uid_t uid) {

    if (!exists(parent))
        return ENOENT;
    DBG_PATH(parent, name);
    if (!is_dir(parent))
        return ENOTDIR;
    if (!has_permission(parent, uid, X_OK | R_OK))
        return EACCES;

    update_directory(parent);
    *nodep = get_child(parent, name);

    return (*nodep == NULL) ? ENOENT : 0;
}


int dav_mkdir(dav_node **nodep, dav_node *parent, const char *name, uid_t uid,
              mode_t mode) {

    if (!exists(parent))
        return ENOENT;
    DBG_PATH(parent, name);
    if (!is_dir(parent))
        return ENOTDIR;
    if (parent == backup)
        return EINVAL;
    if (!has_permission(parent, uid, X_OK | W_OK))
        return EACCES;

    if (get_child(parent, name) != NULL)
        return EEXIST;

    struct passwd *pw = getpwuid(uid);
    if (pw == NULL)
        return EINVAL;

    char *path = strcat_path(parent, name);
    int ret = dav_make_collection(path);
    NE_FREE(path);

    if (ret == 0) {
        *nodep = new_node(parent, mode | S_IFDIR);
        (*nodep)->name = ne_strdup(name);
        (*nodep)->uid = uid;
        (*nodep)->gid = pw->pw_gid;
    }

    return ret;
}


int dav_open(dav_node *node, uid_t uid, int flags) {

    if (!exists(node))
        return ENOENT;
    DBG_PATH(node, NULL);

    int how;
    if ((O_ACCMODE & flags) == O_WRONLY) {
        how = W_OK;
    } else if ((O_ACCMODE & flags) == O_RDONLY) {
        how = R_OK;
    } else {
        how = R_OK | W_OK;
    }
    if (!has_permission(node, uid, how))
        return EACCES;

    if (is_dir(node)) {
        if (how & W_OK)
            return EPERM;
        update_directory(node);
        node->atime = time(NULL);
        return 0;
    }

    DBG2("  o_read %i, o_write %i", node->o_read, node->o_write);
    int ret = 0;
    if ((O_ACCMODE & flags) == O_RDONLY) {

        ret = update_cache_file(node, node->locked);
        if (ret == 0)
            ++node->o_read;

    } else {

        char *path = strcat_path(node, NULL);
        int locked = node->locked;
        if (!node->locked && node->parent != backup)
            ret = dav_lock(path, &node->locked);
        NE_FREE(path);
        if (ret == 0 && (flags & O_TRUNC || node->created)) {
            ret = check_cache_path(node);
            if (ret == 0) {
                node->atime = time(NULL);
                node->mtime = node->atime;
                set_cache_file_times(node);
                ++node->o_write;
            }
        } else if (ret == 0) {
            ret = update_cache_file(node, locked);
            if (ret == 0)
                ++node->o_write;
        }
    }

    return ret;
}


int dav_remove(dav_node *parent, const char *name, uid_t uid) {

    if (!exists(parent))
        return ENOENT;
    DBG_PATH(parent, name);
    if (!is_dir(parent))
        return ENOTDIR;
    if (!has_permission(parent, uid, X_OK | W_OK))
        return EACCES;

    dav_node *node = get_child(parent, name);
    if (node == NULL)
        return ENOENT;
    if (is_dir(node))
        return EISDIR;

    if (node->parent != backup) {
        char *path = strcat_path(node, NULL);
        int ret = 0;
        if (node->locked)
            ret = dav_unlock(path, &node->locked);
        if (ret == 0) {
            ret = dav_delete(path);
            if (ret == ENOENT)
                ret = 0;
            if (ret == EACCES) {
                ret = dav_lock_discover(path);
                if (ret == 0)
                    ret = dav_delete(path);
            }
        }
        NE_FREE(path);
        if (ret != 0)
            return EIO;
    }

    remove_from_tree(node);
    remove_from_table(node);
    zap_file(node);
    delete_node(node);
    return 0;
}


int dav_rename(dav_node *src_parent, const char *src_name,
               dav_node *dst_parent, const char *dst_name, uid_t uid) {

    if (!exists(src_parent) || !exists(dst_parent))
        return ENOENT;
    DBG_PATH(src_parent, src_name);
    DBG_PATH(dst_parent, dst_name);
    if (!is_dir(src_parent) || !is_dir(dst_parent))
        return ENOTDIR;
    if (dst_parent == backup)
        return EINVAL;
    if (!has_permission(src_parent, uid, X_OK | W_OK)
            || !has_permission(dst_parent, uid, X_OK | W_OK))
        return EACCES;

    dav_node *src = get_child(src_parent, src_name);
    dav_node *dst = get_child(dst_parent, dst_name);
    if (src == NULL)
        return ENOENT;
    if (src == backup || (dst != NULL && dst == backup))
        return EINVAL;

    if (is_dir(src)) {
        if (dst != NULL && !is_dir(dst))
            return ENOTDIR;
    } else {
        if (dst != NULL && is_dir(dst))
            return EISDIR;
    }
    if (dst != NULL && is_busy(dst))
        return EBUSY;

    char *dst_path = strcat_path(dst_parent, dst_name);
    if (dst == NULL && dav_head(dst_path) != ENOENT) {
        NE_FREE(dst_path);
        return EEXIST;
    }
    char *src_path = strcat_path(src, NULL);

    int ret = 0;
    attr_from_cache_file(src);
    if (is_dir(src)) {

        ret = dav_move_collection(src_path, dst_path);

    } else if (src->created || is_dirty(src) || src->parent == backup) {

        if (dst != NULL) {
            ret = dav_delete(dst_path);
            if (ret == ENOENT)
                ret = 0;
        }
        if (ret == 0) {
            if (src->locked) {
                dav_unlock(src_path, &src->locked);
            }
            if (src->parent != backup)
                dav_delete(src_path);
            dav_lock(dst_path, &src->locked);
            NE_FREE(src->etag);
            src->smtime = 0;
            src->created = 1;
        }

    } else {

        ret = dav_move(src_path, dst_path);
        if (ret == 0) {
            if (src->mode & (S_IXUSR | S_IXGRP | S_IXOTH))
                dav_set_execute(dst_path, 1);
            if (src->locked) {
                dav_unlock(src_path, &src->locked);
                dav_lock(dst_path, &src->locked);
            }
        }
    }

    NE_FREE(src_path);
    NE_FREE(dst_path);
    if (ret != 0)
        return EIO;

    if (dst != NULL)
        remove_node(dst);

    if (src_parent != dst_parent) {
        remove_from_tree(src);
        src->parent = dst_parent;
        src->next = dst_parent->childs;
        dst_parent->childs = src;
    }
    NE_FREE(src->name);
    src->name = ne_strdup(dst_name);
    src->ctime = time(NULL);
    src->atime = src->ctime;

    if (is_dir(src)) {
        --src_parent->nref;
        ++dst_parent->nref;
    }

    delete_cache_file(src_parent);
    delete_cache_file(dst_parent);
    src_parent->utime = src->ctime;
    dst_parent->utime = src->ctime;

    return 0;
}


int dav_rmdir(dav_node *parent, const char *name, uid_t uid) {

    if (!exists(parent))
        return ENOENT;
    DBG_PATH(parent, name);
    if (!is_dir(parent))
        return ENOTDIR;
    if (!has_permission(parent, uid, X_OK | W_OK))
        return EACCES;

    dav_node *node = get_child(parent, name);
    if (node == NULL)
        return ENOENT;
    if (node == backup)
        return EINVAL;
    if (!is_dir(node))
        return ENOTDIR;
    if (node->childs != 0)
        return ENOTEMPTY;

    char *path = strcat_path(node, NULL);
    int ret = dav_delete_collection(path);
    NE_FREE(path);
    
    if (ret == 0) {
        remove_node(node);
        return 0;
    } else {
        return EIO;
    }
}


int dav_root(dav_node **nodep, uid_t uid) {

    if (uid != 0)
        return EPERM;
    *nodep = root;
    return 0;
}


int dav_setattr(dav_node *node, uid_t uid, int sm, mode_t mode,
                int so, uid_t owner, int sg, gid_t gid, int sat, time_t atime,
                int smt, time_t mtime) {

    if (!exists(node))
        return ENOENT;
    DBG_PATH(node, NULL);
    if (node->parent != NULL && !has_permission(node->parent, uid, X_OK))
        return EACCES;

    if (so) {
        DBG1("  set owner to %i", owner);
        if (uid != 0 && (uid != owner || uid != node->uid))
            return EPERM;
        if (getpwuid(owner) == NULL)
            return EINVAL;
    }
    if (sg) {
        DBG1("  set group to %i", gid);
        if (uid != node->uid && uid != 0)
            return EPERM;
        if (uid != 0) {
            struct passwd *pw = getpwuid(uid);
            if (pw->pw_gid != gid) {
                struct group *gr = getgrgid(gid);
                char **member = gr->gr_mem;
                while (*member != NULL && strcmp(*member, pw->pw_name) != 0)
                    ++member;
                if (*member == NULL)
                    return EPERM;
            }
        }
        if (getgrgid(gid) == NULL)
            return EINVAL;
    }
    if (sm) {
        DBG1("  set mode to %o", mode);
        if (uid != node->uid && uid != 0)
            return EPERM;
        if (S_ISDIR(node->mode) && (mode & dir_umask))
            return EINVAL;
        if (S_ISREG(node->mode) && (mode & file_umask))
            return EINVAL;
    }

    if (so) {
        node->uid = owner;
    }
    if (sg) {
        node->gid = gid;
    }
    if (sm) {
        if (node != backup && !node->created
                && (node->mode & (S_IXUSR | S_IXGRP | S_IXOTH))
                && !(mode & (S_IXUSR | S_IXGRP | S_IXOTH))) {
            char *path = strcat_path(node, NULL);
            dav_set_execute(path, 0);
            NE_FREE(path);
        }
        if (node != backup && !node->created
                && !(node->mode & (S_IXUSR | S_IXGRP | S_IXOTH))
                && (mode & (S_IXUSR | S_IXGRP | S_IXOTH))) {
            char *path = strcat_path(node, NULL);
            dav_set_execute(path, 1);
            NE_FREE(path);
        }
        node->mode = (node->mode & ~DAV_A_MASK) | mode;
    }
    if (sat) {
        DBG1("  set atime to %s", ne_rfc1123_date(atime));
        node->atime = atime;
    }
    if (smt) {
        DBG1("  set mtime to %s", ne_rfc1123_date(mtime));
        node->mtime = mtime;
    }

    node->ctime = time(NULL);
    return 0;
}


int dav_sync(dav_node *node) {

    if (!exists(node)) {
        dav_node *n = invalid;
        while (n != NULL && n != node)
            n = n->next;
        if (n == NULL)
            return ENOENT;
    }

    if (node->fd_write == 0)
        return EINVAL;

    return fsync(node->fd_write);
}


/* Private functions */
/*===================*/

/* Node maintenance. */

/* Creates a new node taking properties from props and adds it to the
   child list of parent and to the hash table.
   If the name in props equals DAV_BACKUP_DIR no node will be created.
   parent : the parent directory node for the new node.
   props  : properties retrieved from the server. Will be freed. */
static void add_node(dav_node *parent, dav_props *props) {

    if ((parent == root && strcmp(props->name, DAV_BACKUP_DIR) == 0)) {
        NE_FREE(props->name);
        NE_FREE(props);
        return;
    }

    dav_node *node;

    if (props->is_dir) {
        node = new_node(parent, default_dir_mode);
    } else {
        node = new_node(parent, default_file_mode);
        node->size = props->size;
        if (props->is_exec == 1) {
            node->mode |= (node->mode & S_IRUSR) ? S_IXUSR : 0;
            node->mode |= (node->mode & S_IRGRP) ? S_IXGRP : 0;
            node->mode |= (node->mode & S_IROTH) ? S_IXOTH : 0;
        } else {
            node->mode &= ~(S_IXUSR | S_IXGRP | S_IXOTH);
        }
        node->mode &= ~file_umask;
    }

    node->name = props->name;
    node->mtime = props->mtime;
    node->smtime = props->mtime;
    node->ctime = (props->ctime > 0) ? props->ctime : props->mtime;

    NE_FREE(props);
}


/* Updates size, mtime and atime of the node to the values of the cached file.
   The private global variable cache_size is updated too.
   For regular file nodes only.
   node : the node to be updated. */
static void attr_from_cache_file(dav_node *node) {

    if (!is_cached(node))
        return;

    struct stat st;
    if (stat(node->cache_path, &st) != 0)
        return;

    cache_size = cache_size - node->size + st.st_size;
    node->size = st.st_size;
    node->atime = (st.st_atime > node->atime) ? st.st_atime : node->atime;
    node->mtime = (st.st_mtime > node->mtime) ? st.st_mtime : node->mtime;
}


/* Creates a new file in DAV_BACKUP_DIR. The name will be the name of the
   cache file of orig and attributes will be taken from orig. The cache
   fiel will be moved from orig to the new node.
   orig : the node to be backed up. */
static void backup_node(dav_node *orig) {

    dav_node *node = new_node(backup, orig->mode);
    node->name = ne_strdup(orig->cache_path + strlen(cache_dir) +1);
    node->cache_path = orig->cache_path;
    orig->cache_path = NULL;
    node->size = orig->size;
    node->uid = orig->uid;
    node->gid = orig->gid;
    node->mtime = orig->mtime;
}


/* If no cache file exists, it creates a new empty cache file. The name
   is concatenated of the node name and a random postfix.
   node : node that needs a cache file.
   return value : 0 on success, EIO if no cache file could be created. */
static int check_cache_path(dav_node *node) {

    if (node->cache_path != NULL && access(node->cache_path, F_OK) == 0)
        return 0;

    NE_FREE(node->cache_path);
    node->smtime = 0;
    NE_FREE(node->etag);
    node->cache_path = ne_concat(cache_dir, "/", node->name, "-XXXXXX", NULL);
    int fd = mkstemp(node->cache_path);
    if (fd <= 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Error %i creating: %s", errno, node->cache_path);
        NE_FREE(node->cache_path);
        return EIO;
    }
    close(fd);
    return 0;
}


/* Scans the directory tree starting from node and
   - saves dirty files to the server
   - removes any file nodes without cached file
   - removes all dir nodes that have not at least one file node below.
   Short: removes everthing that is not necessary to correctly reference
   the cached files.
   DAV_BACKUP_DIR will never be removed.
   node : the node to be cleaned. */
static void clean_tree(dav_node *node) {

    if (node == backup)
        return;

    if (is_dir(node)) {

        dav_node *child = node->childs;
        while (child != NULL) {
            dav_node *next = child->next;
            if (child != backup)
                clean_tree(child);
            child = next;
        }
        if (node->childs == NULL && node != root) {
            remove_from_tree(node);
            remove_from_table(node);
            delete_node(node);
        }

    } else if (!is_cached(node)) {

        if (node->locked) {
            char *path = strcat_path(node, NULL);
            dav_unlock(path, &node->locked);
            NE_FREE(path);
        }
        remove_from_tree(node);
        remove_from_table(node);
        delete_node(node);

    } else if (is_dirty(node) || node->created) {

        char *path = strcat_path(node, NULL);
        dav_put(node->cache_path, path, &node->etag, &node->smtime);
        if (node->locked)
            dav_unlock(path, &node->locked);
        NE_FREE(path);

    }
}


/* Closes a node in the list of invalid nodes. When there are no more opens
   for read or write, the node is removed from the list and deleted.
   node  : the node to close.
   flags : file access mode.
   return value : ENOENT if the node does not exist, 0 otherwise. */
static int close_invalid(dav_node *node, int flags) {

    dav_node **np = &invalid;
    while (*np != NULL && (*np)->next != node)
        np = &(*np)->next;
    if (*np == NULL)
        return ENOENT;

    DBG2("  o_read %i, o_write %i", node->o_read, node->o_write);
    if ((O_ACCMODE & flags) == O_RDONLY) {
        node->o_read = (node->o_read > 0) ? (node->o_read - 1) : 0;
    } else {
        node->o_write = (node->o_write > 0) ? (node->o_write - 1) : 0;
    }

    if (node->o_write == 0 && node->o_read == 0) {
        *np = node->next;
        zap_file(node);
        delete_node(node);
    }
    return 0;
}


/* Deletes the cache file of node. Also freeing etag and adjusting
   the global variable cache_size. */
static void delete_cache_file(dav_node *node) {

    if (is_reg(node)) { 
        NE_FREE(node->etag);
        if (node->cache_path != NULL)
            cache_size -= node->size;
    }
    if (node->fd_read > 0) {
        close(node->fd_read);
        node->fd_read = 0;
    }
    if (node->fd_write > 0) {
        close(node->fd_write);
        node->fd_write = 0;
    }
    if (node->cache_path != NULL) {
        remove(node->cache_path);
        NE_FREE(node->cache_path);
    }
}


/* Frees any resources held by node and finally frees node. */
static void delete_node(dav_node *node) {

    DBG1("  Deleting node %p.", node);
    NE_FREE(node->name);
    delete_cache_file(node);
    NE_FREE(node);
}


/* Deletes the tree starting at node. The tree is freed uncontionally, no
   checks for lost update problem and the like are done. */
static void delete_tree(dav_node *node) {

    while (node->childs != NULL)
        delete_tree(node->childs);
    if (node != root) {
        remove_from_tree(node);
        remove_from_table(node);
        NE_FREE(node->name);
        if (node->cache_path != NULL)
            remove(node->cache_path);
        NE_FREE(node->cache_path);
        NE_FREE(node->etag);
        NE_FREE(node);
    }
}


/* Creates a new node. mode must have the I_ISDIR or I_ISREG bit set.
   node->mode is set to mode, but checked against the umask. All other
   members are set to reasonable defaults. The new node will be inserted
   into the child list of parent and the hash table.
   parent : the parent of the new node.
   mode   : the mode of the new node.
   return value : a pointer to the new node. */
static dav_node *new_node(dav_node *parent, mode_t mode) {

    dav_node *node = (dav_node *) ne_malloc(sizeof(dav_node));

    node->name = NULL;
    node->cache_path = NULL;
    node->size = 0;
    node->uid = default_uid;
    node->gid = default_gid;
    node->atime = time(NULL);
    node->mtime = node->atime;
    node->ctime = node->atime;
    if (S_ISDIR(mode)) {
        node->mode = mode & ~dir_umask;
        node->nref = 2;
        node->utime = node->atime;
    } else {
        node->mode = mode & ~file_umask;
        node->locked = 0;
        node->smtime = 0;
    }
    node->o_read = 0;
    node->o_write = 0;
    node->fd_read = 0;
    node->fd_write = 0;
    node->etag = NULL;
    node->created = 0;

    node->parent = parent;
    node->childs = NULL;
    if (parent != NULL) {
        if (S_ISDIR(mode))
            ++parent->nref;
        delete_cache_file(parent);
        node->next = parent->childs;
        parent->childs = node;
    } else {
        node->next = NULL;
    }

    int i = ((uint) node / alignment) % table_size;
    node->table_next = table[i];
    table[i] = node;

    DBG2("  New node: %p->%p", node->parent, node);
    return node;
}


/* Frees locks, removes the node from the tree and from the hash table,
   and deletes it.
   Depending on the kind of node and its state additional action will be taken:
   - For directories the complete tree below is removed too.
   - If a regular file is dirty or open for writing, a backup DAV_BACKUP_DIR
     will be created, that holds the cached local copy of the file.
   - If a file is open, it will be moved to the list of invalid
     nodes instead of beeing deleted.
   node : the node to remove. */
static void remove_node(dav_node *node) {

    remove_from_tree(node);
    remove_from_table(node);

    if (is_dir(node)) {

        while (node->childs != NULL)
            remove_node(node->childs);
        zap_file(node);
        delete_node(node);

    } else {

        if (node->locked) {
            char *path = strcat_path(node, NULL);
            dav_unlock(path, &node->locked);
            NE_FREE(path);
        }

        attr_from_cache_file(node);
        if (is_dirty(node) || node->o_write > 0)
            backup_node(node);

        if (node->o_read > 0 || node->o_write > 0) {
            DBG1("  Invalidating node %p.", node);
            node->next = invalid;
            invalid = node;
        } else {
            zap_file(node);
            delete_node(node);
        }

    }
}


/* Removes a node from the hash table. */
static void remove_from_table(dav_node *node) {

    int i = ((uint) node / alignment) % table_size;
    dav_node **np = &table[i];
    while (*np != NULL && *np != node)
        np = &(*np)->table_next;
    if (*np != NULL)
        *np = (*np)->table_next;
}


/* Removes a node from the directory tree. The root node can not be removed. */
static void remove_from_tree(dav_node *node) {

    if (node->parent == NULL)
        return;

    dav_node **np = &node->parent->childs;
    while (*np != NULL && *np != node)
        np = &(*np)->next;
    if (*np != NULL) {
        *np = node->next;
        delete_cache_file(node->parent);
        if (is_dir(node))
            --node->parent->nref;
    }
}


/* Sets atime and mtime of the cache file to the values of node. */
static void set_cache_file_times(dav_node *node) {

    struct utimbuf t;
    t.actime = node->atime;
    t.modtime = node->mtime;
    utime(node->cache_path, &t);
}


/* Updates the cached file from the server if necessary and possible, or
   retrieves one from the server if no cache file exists.
   It is not necessary or possible if
   - node is in DAV_BACKUP_DIR
   - node has a cache file and holds a lock on the server
   - node is open for writing
   - node is dirty
   - it has been updated within the last second.
   If the node is dirty but not open for write, it will be stored back on
   the server.
   node : the cache file of this node has to be updated.*/
static int update_cache_file(dav_node *node, int locked) {

DBG0("update 1");
    int atime = node->atime;
    attr_from_cache_file(node);

    if (node->parent == backup
            || node->created
            || node->o_write > 0
            || (is_cached(node) && locked)
            || (is_cached(node) && time(NULL) < (atime +1)))
        return 0;

    char *path = strcat_path(node, NULL);

    int ret = 0;
    if (is_dirty(node)) {
DBG0("update dirty");
        ret = dav_put(node->cache_path, path, &node->etag, &node->smtime);
        NE_FREE(path);
        if (ret == 0) {
            node->mtime = node->smtime;
            set_cache_file_times(node);
        }
        return 0;
    }

    if (check_cache_path(node) != 0)
        return EIO;

DBG0("update 2");
    if (is_cached(node))
        cache_size -= node->size;
    ret = dav_get_file(node->cache_path, path, &node->etag, &node->smtime,
                       &node->size);
DBG1("update size %li", node->size);
    NE_FREE(path);
    if (ret != 0) {
        return EIO;
    }
    cache_size += node->size;

    node->atime = time(NULL);
    node->mtime = node->smtime;
    set_cache_file_times(node);
    
DBG0("update 3");
    return 0;
}


/* If dir->utime is reached it will get information from the server and update
   dir and its child nodes accordingly. Else it will do nothing. If dir is
   DAV_BACKUP_DIR it will also do nothing.
   To update dir and its child nodes it gets a property list from the server.
   Attributes of the nodes are updated, new nodes created and nodes that have
   been removed on the server will be removed.
   If files are changed locally and changed or removed on the server, a backup
   file in DAV_BACKUP_DIR will be created. DAV_BACKUP_DIR as well as files
   created locally and not yet stored on the server will not be removed.
   The utime of dir will be updated too.
   dir : the direcotry node to update. */
static int update_directory(dav_node *dir) {

    if (dir == backup)
        return 0;
    if (dir->utime > time(NULL))
        return 0;

    char *path = strcat_path(dir, NULL);
    dav_props *props = NULL;
    int ret = dav_get_collection(&props, path);
    NE_FREE(path);
    if (ret != 0) {
        dir->utime = time(NULL) + retry;
        retry = 2 * retry;
        retry = (retry > max_retry) ? max_retry : retry;
        return ret;
    } else {
        dir->utime = time(NULL) + expire;
        retry = min_retry;
    }

    dav_node *child = dir->childs;
    while (child != NULL) {
        dav_node *next = child->next;
        dav_props *p = props;
        dav_props **pp = &props;
        while (p != NULL && strcmp(p->name, child->name) != 0) {
            pp = &p->next;
            p = p->next;
        }
        if (p != NULL) {
            *pp = p->next;
            update_node(child, p);
        } else if (!child->created && child != backup) {
            remove_node(child);
        }
        child = next;
    }

    while (props != NULL) {
        dav_props *next = props->next;
        if (strlen(props->name) > 0) {
            add_node(dir, props);
        } else {
            NE_FREE(props->name);
            if (props->mtime > dir->atime)
                dir->atime = props->mtime;
            if (props->mtime > dir->mtime) {
                delete_cache_file(dir);
                dir->mtime = props->mtime;
            }
            if (props->ctime > dir->ctime)
                dir->ctime = props->ctime;
            NE_FREE(props);
        }
        props = next;
    }

    DBG2("  Directory updated: %p->%p", dir->parent, dir);
    DBG_PATH(dir, NULL);
    return 0;
}


/* Updates the properties of node according to props and frees props.
   If props is incompatibel with node or indicates a lost update problem,
   a new node is created from props and the old node is deleted, creating
   a local back up if necessary.
   node  : the node to be updated. It must not be the root node and have a
           valid parent.
   props : the properties retrieved from the server. They will be freed. */
static void update_node(dav_node *node, dav_props *props) {

    if (node == backup) {
        NE_FREE(props->name);
        NE_FREE(props);
        return;
    }

    if ((is_dir(node) && !props->is_dir)
            || (!is_dir(node) && props->is_dir)) {
        add_node(node->parent, props);
        remove_node(node);
        return;
    }

    if (is_reg(node))
        attr_from_cache_file(node);

    if (node->created) {
        if (!node->locked || props->size > 0) {
            add_node(node->parent, props);
            remove_node(node);
        } else {
            NE_FREE(props->name);
            NE_FREE(props);
        }
        return;
    }

    if (is_dirty(node) || node->o_write > 0) {
        if (props->mtime > node->smtime) {
            add_node(node->parent, props);
            remove_node(node);
        } else {
            NE_FREE(props->name);
            NE_FREE(props);
        }
        return;
    }

    if (props->mtime > node->atime)
        node->atime = props->mtime;
    if (props->mtime > node->mtime) {
        node->mtime = props->mtime;
        if (is_dir(node)) {
            node->utime = time(NULL);
        } else {
            node->smtime = props->mtime;
        }
        delete_cache_file(node);
    }
    if (props->ctime > node->ctime)
        node->ctime = props->ctime;

    if (is_reg(node)) {
        if (props->is_exec == 1) {
            node->mode |= (node->mode & S_IWUSR) ? S_IXUSR : 0;
            node->mode |= (node->mode & S_IWGRP) ? S_IXGRP : 0;
            node->mode |= (node->mode & S_IWOTH) ? S_IXOTH : 0;
            node->mode &= ~file_umask;
        } else if (props->is_exec == 0) {
            node->mode &= ~(S_IXUSR | S_IXGRP | S_IXOTH);
        }
        if (props->size != 0)
            node->size = props->size;
    }

    NE_FREE(props->name);
    NE_FREE(props);

    DBG2("  Node updated: %p->%p", node->parent, node);
}


/* Get information about node. */

/* Checks whether node exists. The parent directory is updated if
   necessary. */
static int exists(const dav_node *node) {

    int i = ((uint) node / alignment) % table_size;
    dav_node *n = table[i];
    while (n != NULL && n != node)
        n = n->table_next;

    if (n != NULL && n->parent != NULL) {
        update_directory(n->parent);
        n = table[i];
        while (n != NULL && n != node)
            n = n->table_next;
    }

    return (n != NULL);
}


/* Searches parents list of childs for a child with name name and returns
   a pointer to this child or NULL of none found.
   parent : a directory node.
   name   : the name of the child to search.
   return value : a pointer to the child or NULL. */
static dav_node *get_child(const dav_node *parent, const char *name) {

    dav_node *node = parent->childs;
    while (node != NULL && strcmp(name, node->name) != 0)
        node = node->next;
    return node;
}


/* Checks whether user uid has access to node according to how.
   In any case the user must have execute permission for the parent of node
   and all of its parents up to the root node.
   int how : How to acces the node. May be any combination of R_OK, W_OK, X_OK
             and F_OK.
   return value: 1 access is allowed.
                 0 access is denied. */
static int has_permission(const dav_node *node, uid_t uid, int how) {

    if (uid == 0)
        return 1;

    if (node->parent != NULL && !has_permission(node->parent, uid, X_OK))
        return 0;

    mode_t a_mode = (how & R_OK) ? (S_IRUSR | S_IRGRP | S_IROTH) : 0;
    a_mode |= (how & W_OK) ? (S_IWUSR | S_IWGRP | S_IWOTH) : 0;
    a_mode |= (how & X_OK) ? (S_IXUSR | S_IXGRP | S_IXOTH) : 0;

    if (!(~node->mode & S_IRWXO & a_mode))
        return 1;

    if (node->uid == uid && !(~node->mode & S_IRWXU & a_mode))
        return 1;

    struct passwd *pw = getpwuid(uid);
    if (pw == NULL)
        return 0;
    if (pw->pw_gid != node->gid) {
        struct group *grp = getgrgid(node->gid);
        if (grp == NULL)
            return 0;
        char **members = grp->gr_mem;
        while (*members != NULL && strcmp(*members, pw->pw_name) == 0)
            members++;
        if (*members == NULL)
            return 0;
    }
    if (!(~node->mode & S_IRWXG & a_mode))
        return 1;

    return 0;
}


/* A node is considered busy if it is open for writing or, in case of a
   directory, if in the tree below the node there is any file open for write.
   return value : 1 if busy, 0 if not. */
static int is_busy(const dav_node *node) {

    dav_node *child = node->childs;
    while (child != NULL) {
        if (is_busy(child))
            return 1;
    }

    if (is_dir(node))
        return 0;
    return (node->o_write > 0);
}


/* Returns the path of node, starting with the root node. If name is not NULL,
   it is appended as the last component of the path.
   return value : the path. It always starts with "/", but never has a
                  trailing slash. The path string is newly allocated and must
                  be freed by the calling function. */
static char *strcat_path(const dav_node *node, const char *name) {

    char *path;
    if (name == NULL) {
        path = ne_strdup(node->name);
    } else {
        path = ne_concat(node->name, "/", name, NULL);
    }

    dav_node *n = node->parent;
    while (n != NULL) {
        char *new_path = ne_concat(n->name, "/", path, NULL); 
        NE_FREE(path);
        path = new_path;
        n = n->parent;
    }

    return path;
}


/* Permanent cache maintenance. */

/* Checks whether there is an cache directory for the server in
   args->url. If not it will create one.
   In case of an error it will print an error message and terminate
   the program.
   args : structure containing the url of the server. */
static void check_cache_dir(const dav_args *args) {

    char *index = ne_concat(args->cache_dir, "/", DAV_INDEX, NULL);
    FILE *idx = fopen(index, "a+");
    if (idx == NULL) {
        error(EXIT_FAILURE, 0, "Could not open index file\n  %s.", index);
    }

    char *line = NULL;
    size_t n = 0;
    int len = getline(&line, &n, idx);
    while (len > 0 && strstr(line, args->url) != line)
        len = getline(&line, &n, idx);
    if (len > 0) {
        char *nl = strchr(line, '\n');
        if (nl != 0)
            *nl = '\0';
        if (strlen(line) > (strlen(args->url) + 1))
            cache_dir = ne_strdup(line + strlen(args->url)+ 1);
    }

    if (cache_dir != NULL && access(cache_dir, F_OK) != 0) {
        if (mkdir(cache_dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)
                != 0)
            error(EXIT_FAILURE, 0, "Could not create cache directory.");
    }

    if (cache_dir == NULL) {
        cache_dir = ne_concat(args->cache_dir, "/", "XXXXXX", NULL);
        if (mkdtemp(cache_dir) == NULL)
            error(EXIT_FAILURE, 0, "Could not create cache directory.");
        DBG1("    Writing entry for\n   %s.", args->url);
        fputs(args->url, idx);
        fputs(" ", idx);
        fputs(cache_dir, idx);
        if (fputs("\n", idx) == EOF)
            error(EXIT_FAILURE, 0, "Error writing to index file\n  %s.", index);
    }

    fclose(idx);
}


/* Deletes all files in the cache directory. Will be called if there are
   it proves to be useless because of errors. */
static void delete_cache(void) {

    DIR *dir = opendir(cache_dir);
    if (dir != NULL) {
        struct dirent *de = readdir(dir);
        while (de != NULL) {
            if (strcmp(de->d_name, ".") != 0 && strcmp(de->d_name, "..") != 0 )
                remove(de->d_name);
            de = readdir(dir);
        }
    }
    closedir(dir);
    cache_size = 0;
}


/* Reads the index file of the cache and creates nodes from the cached
   data. Will be called when the cache module is initialized.
   If an error occurs all nodes created up to this will be deleted and
   also all files in the cache. */
#if NE_VERSION_MINOR == 24

static void parse_index(void) {

    char *index = ne_concat(cache_dir, "/", DAV_INDEX, NULL);
    FILE *idx = fopen(index, "r");
    if (idx == NULL)
        return;

    ne_xml_parser *parser = ne_xml_create();
    ne_xml_push_handler(parser, xml_start_root, NULL, NULL, &root);
    ne_xml_push_handler(parser, xml_start_dir, NULL, xml_end_dir, &root);
    ne_xml_push_handler(parser, xml_start_reg, NULL, xml_end_reg, &root);
    ne_xml_push_handler(parser, xml_start_mode, xml_cdata_mode, NULL, &root);
    ne_xml_push_handler(parser, xml_start_decimal, xml_cdata_decimal, NULL,
                        &root);
    ne_xml_push_handler(parser, xml_start_string, xml_cdata_string, NULL,
                        &root);

    char *buf = ne_malloc(DAV_XML_BUF_SIZE);
    size_t len = fread(buf, 1, DAV_XML_BUF_SIZE, idx);
    while (len > 0 && ne_xml_valid(parser)) {
        ne_xml_parse(parser, buf, len);
        len = fread(buf, 1, DAV_XML_BUF_SIZE, idx);
    }
    NE_FREE(buf);

    int ret = (ne_xml_valid(parser)) ? 0 : -1;
    ne_xml_destroy(parser);
    if (ret != 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Error parsing index file:\n  %s", ne_xml_get_error(parser));
        delete_tree(root);
        delete_cache();
    }
    fclose(idx);
    NE_FREE(index);
}

#else     /* NE_VERSION_MINOR */

static void parse_index(void) {

    char *index = ne_concat(cache_dir, "/", DAV_INDEX, NULL);
    FILE *idx = fopen(index, "r");
    if (idx == NULL)
        return;

    ne_xml_parser *parser = ne_xml_create();
    ne_xml_push_handler(parser, xml_start_root, NULL, NULL, &root);
    ne_xml_push_handler(parser, xml_start_dir, NULL, xml_end_dir, &root);
    ne_xml_push_handler(parser, xml_start_reg, NULL, xml_end_reg, &root);
    ne_xml_push_handler(parser, xml_start_mode, xml_cdata_mode, NULL, &root);
    ne_xml_push_handler(parser, xml_start_decimal, xml_cdata_decimal, NULL,
                        &root);
    ne_xml_push_handler(parser, xml_start_string, xml_cdata_string, NULL,
                        &root);

    char *buf = ne_malloc(DAV_XML_BUF_SIZE);
    size_t len = fread(buf, 1, DAV_XML_BUF_SIZE, idx);
    int ret = ne_xml_parse(parser, buf, len);
    while (len > 0 && ret == 0) {
        len = fread(buf, 1, DAV_XML_BUF_SIZE, idx);
        ret = ne_xml_parse(parser, buf, len);
    }
    NE_FREE(buf);

    ne_xml_destroy(parser);
    if (ret != 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "Error parsing index file:\n  %s", ne_xml_get_error(parser));
        delete_tree(root);
        delete_cache();
    }
    fclose(idx);
    NE_FREE(index);
}

#endif    /* NE_VERSION_MINOR */


/* Creates an entry for node in the index file file, removes the node from
   the tree and the hash table and deletes the node. The entry will be
   indented by indent to get proper alignment of nested entries.
   node   : the node.
   file   : the index file for this cache directory.
   indent : a string of spaces to indent the entry.
   return value : 0 on success, -1 if an error occurs. */
static int write_node(dav_node *node, FILE *file, const char *indent) {

    if (is_dir(node)) {
        DBG1("  Writing directory %p.", node);
        if (fprintf(file, "%s<d:%s>\n", indent, type[DDIR]) < 0)
            return -1;
    } else {
        DBG1("  Writing file %p.", node);
        if (fprintf(file, "%s<d:%s>\n", indent, type[REG]) < 0)
            return -1;
    }
    char *ind = ne_concat(indent, "  ", NULL);

    if (fprintf(file, "%s<d:%s>%s</d:%s>\n", ind, type[NAME], node->name,
                type[NAME]) < 0)
        return -1;

    if (is_reg(node) && node->cache_path != NULL) {
        if (fprintf(file, "%s<d:%s>%s</d:%s>\n", ind, type[CACHE_PATH],
                    node->cache_path, type[CACHE_PATH]) < 0)
            return -1;
        if (fprintf(file, "%s<d:%s>%li</d:%s>\n", ind, type[SIZE], node->size,
                    type[SIZE]) < 0)
            return -1;
        if (fprintf(file, "%s<d:%s>%li</d:%s>\n", ind, type[SMTIME],
                    node->smtime, type[SMTIME]) < 0)
            return -1;
        if (node->etag != NULL) {
            if (fprintf(file, "%s<d:%s>%s</d:%s>\n", ind, type[ETAG],
                       node->etag, type[ETAG]) < 0)
            return -1;
        }
        NE_FREE(node->cache_path);
    }

    if (fprintf(file, "%s<d:%s>%o</d:%s>\n", ind, type[MODE], node->mode,
                type[MODE]) < 0)
        return -1;
    if (fprintf(file, "%s<d:%s>%i</d:%s>\n", ind, type[UID], node->uid,
                type[UID]) < 0)
        return -1;
    if (fprintf(file, "%s<d:%s>%i</d:%s>\n", ind, type[GID], node->gid,
                type[GID]) < 0)
        return -1;
    if (fprintf(file, "%s<d:%s>%li</d:%s>\n", ind, type[ATIME], node->atime,
                type[ATIME]) < 0)
        return -1;
    if (fprintf(file, "%s<d:%s>%li</d:%s>\n", ind, type[MTIME], node->mtime,
                type[MTIME]) < 0)
        return -1;
    if (fprintf(file, "%s<d:%s>%li</d:%s>\n", ind, type[CTIME], node->ctime,
                type[CTIME]) < 0)
        return -1;

    if (is_dir(node)) {
        while (node->childs != NULL) {
            if (write_node(node->childs, file, ind) < 0)
                return -1;
        }
        if (fprintf(file, "%s</d:%s>\n", indent, type[DDIR]) < 0)
            return -1;
    } else {
        if (fprintf(file, "%s</d:%s>\n", indent, type[REG]) < 0)
            return -1;
    }

    NE_FREE(ind);
    remove_from_tree(node);
    remove_from_table(node);
    delete_node(node);

    return 0;
}


/* Call back function to retrieve a decimal value from cdata and add it to
   the appropriate member of a node.
   userdata : a pointer to the node.
   state    : a constant indicating the member of node.
   cdata    : string representation of a decimal number.
   len      : length of cdata.
   return value : 0 on success, -1 if an error occurs. */
static int xml_cdata_decimal(void *userdata, int state, const char *cdata,
                             size_t len) {

    char *s = ne_strndup(cdata, len);
    char *tail;

    long int n = strtol(s, &tail, 10);
    if (*tail != '\0') {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML parse %s error: %s", type[state], s);
        NE_FREE(s);
        return -1;
    }
    NE_FREE(s);

    switch (state) {
    case SIZE:
        (*((dav_node **) userdata))->size = n;
        break;
    case UID:
        (*((dav_node **) userdata))->uid = n;
        break;
    case GID:
        (*((dav_node **) userdata))->gid = n;
        break;
    case ATIME:
        (*((dav_node **) userdata))->atime = n;
        break;
    case MTIME:
        (*((dav_node **) userdata))->mtime = n;
        break;
    case CTIME:
        (*((dav_node **) userdata))->ctime = n;
        break;
    case SMTIME:
        (*((dav_node **) userdata))->smtime = n;
        break;
    default:
        return -1;
    }

    return 0;
}


/* Call back function to parse cdata for the value of the mode member of a node.
   The mode is represented in cdata as an octal number.
   userdata : a pointer to the node.
   state    : a constant indicating the member of node.
   cdata    : string representation of an octal number.
   len      : length of cdata.
   return value : 0 on success, -1 if an error occurs. */
static int xml_cdata_mode(void *userdata, int state, const char *cdata,
                          size_t len) {

    char *s = ne_strndup(cdata, len);
    char *tail;
    (*((dav_node **) userdata))->mode = strtol(s, &tail, 8);
    if (*tail != '\0') {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML parse %s error: %s", type[state], s);
        NE_FREE(s);
        return -1;
    }
    NE_FREE(s);

    return 0;
}


/* Call back function to copy the string in cdata into the appropriate member
   of a node. The string is newly allocated.
   userdata : a pointer to the node.
   state    : a constant indicating the member of node.
   cdata    : the string.
   len      : length of cdata.
   return value : 0 on success, -1 if an error occurs. */
static int xml_cdata_string(void *userdata, int state, const char *cdata,
                            size_t len) {

    switch (state) {
    case NAME:
        (*((dav_node **) userdata))->name = ne_strndup(cdata, len);
        break;
    case CACHE_PATH:
        (*((dav_node **) userdata))->cache_path = ne_strndup(cdata, len);
        break;
    case ETAG:
        (*((dav_node **) userdata))->etag = ne_strndup(cdata, len);
        break;
    default:
        return -1;
    }

    return 0;
}


/* Call back function, called when the closing tag of a directory entry is
   found. It will test whether the directory node has at least a name,
   otherwise the directory node will be removed and deleted. It will
   replace the pointer in userdata by a pointer to the parenter of the
   node, stepping up one level in the tree of nodes.
   userdata : a pointer to the directory node.
   state    : not used.
   nspace   : not used.
   name     : not used.
   return value : allways 0. */
static int xml_end_dir(void *userdata, int state, const char *nspace,
                       const char *name) {

    dav_node *dir = *((dav_node **) userdata);
    *((dav_node **) userdata) = dir->parent;

    if (dir->name == NULL) {
        remove_from_tree(dir);
        remove_from_table(dir);
        delete_node(dir);
    }

    return 0;
}


/* Call back function, called when the closing tag of an entry for a regular
   file node is found. The node must have a name and a valid cache file,
   otherwise the node will be removed and deleted. 
   If valid the private global variable cache_size will be updated.
   userdata : a pointer to the node.
   state    : not used.
   nspace   : not used.
   name     : not used.
   return value : allways 0. */
static int xml_end_reg(void *userdata, int state, const char *nspace,
                       const char *name) {

    dav_node *reg = *((dav_node **) userdata);
    *((dav_node **) userdata) = reg->parent;

    if (is_cached(reg))
        cache_size += reg->size;

    if (reg->name == NULL || reg->cache_path == NULL
            || access(reg->cache_path, F_OK) != 0) {
        remove_from_tree(reg);
        remove_from_table(reg);
        delete_node(reg);
        return 0;
    }

    return 0;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property, whether it is responsible for this
   kind and whether this property is legal as child of parent. It returns a
   constant indicating the result of this test.
   userdata : pointer to the node, not used.
   parent   : constant indicating the kind of the parent property.
   nspace   : not used.
   name     : the name of the property.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, parent must not contain this property.
                  otherwise: constant indicating the type of property. */
static int xml_start_decimal(void *userdata, int parent, const char *nspace,
                             const char *name, const char **atts) {

    int ret;
    if (strcmp(name, type[SIZE]) == 0) {
        ret = SIZE;
    } else if (strcmp(name, type[UID]) == 0) {
        ret = UID;
    } else if (strcmp(name, type[GID]) == 0) {
        ret = GID;
    } else if (strcmp(name, type[ATIME]) == 0) {
        ret = ATIME;
    } else if (strcmp(name, type[MTIME]) == 0) {
        ret = MTIME;
    } else if (strcmp(name, type[CTIME]) == 0) {
        ret = CTIME;
    } else if (strcmp(name, type[SMTIME]) == 0) {
        ret = SMTIME;
    } else {
        return 0;
    }

    if (parent != DDIR && parent != REG) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: decimal type inside %s", type[parent]);
        return -1;
    }

    return ret;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property, whether it is responsible for this
   kind and whether this property is legal as child of parent. It returns a
   constant indicating the result of this test.
   This function is responsible for properties of type DDIR (directory node).
   If it is a DDIR property, a new directory node will be created. Its parent
   will be userdata. userdata will be replaced by a pointer to the new
   directory node, thus stepping down one level in the tree.
   userdata : pointer to the current directory node.
   parent   : constant indicating the kind of the parent property, must be a
              directory.
   nspace   : not used.
   name     : the name of the property, must be 'dir'.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, parent must not contain this property.
                  otherwise: constant DDIR. */
static int xml_start_dir(void *userdata, int parent, const char *nspace,
                         const char *name, const char **atts) {

    if (strcmp(name, type[DDIR]) != 0)
        return 0;

    if (parent != DDIR && parent != ROOT) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: dir type inside %s", type[parent]);
        return -1;
    }

    dav_node *dir = new_node(*((dav_node **) userdata), default_dir_mode);
    *((dav_node **) userdata) = dir;

    return DDIR;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property, whether it is responsible for this
   kind and whether this property is legal as child of parent. It returns a
   constant indicating the result of this test.
   This function is responsible for propertiies of type mode.
   userdata : not used.
   parent   : constant indicating the kind of the parent property, must be a
              directory or a regular file.
   nspace   : not used.
   name     : the name of the property, must be 'mode'.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, parent must not contain this property.
                  otherwise: constant MODE. */
static int xml_start_mode(void *userdata, int parent, const char *nspace,
                          const char *name, const char **atts) {

    if (strcmp(name, type[MODE]) != 0)
        return 0;

    if (parent != DDIR && parent != REG && parent != ROOT) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: mode type inside %s", type[parent]);
        return -1;
    }

    return MODE;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property, whether it is responsible for this
   kind and whether this property is legal as child of parent. It returns a
   constant indicating the result of this test.
   This function is responsible for properties of type REG (regular file).
   If it is a REG property, a new file node will be created. Its parent
   will be userdata. userdata will be replaced by a pointer to the new
   file node, thus stepping down one level in the tree.
   userdata : pointer to the current directory node.
   parent   : constant indicating the kind of the parent property, must be a
              regular file.
   nspace   : not used.
   name     : the name of the property, must be 'reg'.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, parent must not contain this property.
                  otherwise: constant REG. */
static int xml_start_reg(void *userdata, int parent, const char *nspace,
                         const char *name, const char **atts) {

    if (strcmp(name, type[REG]) != 0)
        return 0;

    if (parent != DDIR && parent != ROOT) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: reg type inside %s", type[parent]);
        return -1;
    }

    dav_node *reg = new_node(*((dav_node **) userdata), default_file_mode);
    *((dav_node **) userdata) = reg;

    return REG;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property. It returns a constant indicating the
   result of this test.
   This function is responsible for the root property. It is the top level
   property and must not have a parent.
   userdata : not used.
   parent   : constant indicating the kind of the parent property, must be 0.
   nspace   : not used.
   name     : the name of the property, must be 'root'.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, root property with parent not equal 0.
                  otherwise: constant ROOT. */
static int xml_start_root(void *userdata, int parent, const char *nspace,
                          const char *name, const char **atts) {

    if (strcmp(name, type[ROOT]) != 0)
        return 0;

    if (parent != 0) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: root type inside %s", type[parent]);
        return -1;
    }

    return ROOT;
}


/* Call back function called when the start tag of an XML property is found.
   Tests name for the kind of property, whether it is responsible for this
   kind and whether this property is legal as child of parent. It returns a
   constant indicating the result of this test.
   This function is responsible for properties that contain a string.
   userdata : not used.
   parent   : constant indicating the kind of the parent property, must be a
              directory or a regular file.
   nspace   : not used.
   name     : the name of the property.
   atts     : not used.
   return value : 0 not responsible for this kind of property.
                  -1 XML error, parent must not contain this property.
                  otherwise: constant indicating the type of property. */
static int xml_start_string(void *userdata, int parent, const char *nspace,
                            const char *name, const char **atts) {

    if (parent != DDIR && parent != REG) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: string type inside %s", type[parent]);
        return -1;
    }

    int ret;
    if (strcmp(name, type[NAME]) == 0) {
        ret = NAME;
    } else if (strcmp(name, type[CACHE_PATH]) == 0) {
        ret = CACHE_PATH;
    } else if (strcmp(name, type[ETAG]) == 0) {
        ret = ETAG;
    } else {
        return 0;
    }

    if (parent != DDIR && parent != REG) {
        syslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),
               "XML error: string type inside %s", type[parent]);
        return -1;
    }

    return ret;
}


/* Auxiliary. */

/* Tries to evaluate the alignment of structure dav_node. It allocates
   dav_node structures and random length strings alternatively and inspects the
   address.
   return value : the alignment (e.g. alignment = 4 means addresses
                  are always multiples of 4 */
static int test_alignment() {

    srand(time(0));
    int align = 64;
    int trials = 100;
    char *s[trials];
    dav_node *n[trials];

    int j = 0;
    while (align > 0 && j < trials) {
        s[j] = (char *) ne_malloc((rand() / (RAND_MAX / 1024)) % (4 *align));
        n[j] = (dav_node *) ne_malloc(sizeof(dav_node));
        while (align > 0 && ((uint) n[j] % align) > 0)
            align /= 2;
        ++j;
    }

    for (j = 0; j < trials; j++) {
        NE_FREE(n[j]);
        NE_FREE(s[j]);
    }
    return align;
}

 /* Does nothing. Used as long as the kernel interface module did not register
    a proper function. */
static void zap_file_dummy(dav_node *node) {
    return;
}
