/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#include "datasearchpopup.h"

#include "project.h"
#include "dateedit.h"
#include "timeedit.h"
#include "datafield.h"
#include "datatable.h"
#include "formlayout.h"
#include "datatableeditorfactory.h"

#include <qregexp.h>

#include <kdebug.h>

DataSearchPopup::DataSearchPopup( const DataTableSearch::ComponentList &components,
                                  DataTableSearch::SearchLevel level,
                                  QWidget *parent, const char *name )
            : KDialogBase( parent, name, true, name, Ok ),
            m_originals( components ),
            m_updates( DataTableSearch::ComponentList() ),
            m_level( level )
{
    m_formBox = new QFrame( this );
    setMainWidget( m_formBox );

    m_formLayout = new FormLayout( mainWidget(), 5, 5 );

    switch ( m_level )
    {
    case DataTableSearch::Simple:
        break;
    case DataTableSearch::Advanced:
        addAdvancedPrompts();
        break;
    case DataTableSearch::Custom:
        addCustomPrompts();
        break;
    default:
        break;
    }
}

DataSearchPopup::~DataSearchPopup()
{}

void DataSearchPopup::addAdvancedPrompts()
{
    int i = 0;
    DataTableSearch::ComponentList::ConstIterator it = m_originals.begin();
    for ( ; it != m_originals.end(); ++it )
    {
        if ( !( *it ).isPrompt() )
            continue;

        DataEditorBase *editor =
            new DataEditorBase( ( *it ).dataField()->label(),
                                ( *it ).dataField(),
                                ( *it ).dataTable()->dataTableEditorFactory(),
                                mainWidget() );

        m_editors[ QString::number( ++i ) ] = editor;
        if ( m_formLayout->isEmpty() )
            editor->setFocus();
        m_formLayout->add( editor );
        editor->show();
    }
}

void DataSearchPopup::addCustomPrompts()
{
    DataTableSearch::ComponentList::ConstIterator it = m_originals.begin();
    for ( ; it != m_originals.end(); ++it )
    {
        addPromptsFromCustomQuery( ( *it ).dataTable(), ( *it ).query() );
    }
}

static QStringList listFromRx( const QString str, QRegExp rx, int cap )
{
    QStringList list;
    int pos = 0;
    while ( pos >= 0 )
    {
        pos = rx.search( str, pos );
        if ( pos >= 0 )
        {
            list.append( rx.cap( cap ) );
            pos += rx.matchedLength();
        }
    }
    return list;
}

static DataField *fetchField( DataTable *dataTable, const QStringList &fullName )
{
    DataField *field = 0L;
    DataTableList dataTables = dataTable->project()->dataTablesInDataTableTree( dataTable );
    DataTableList::Iterator it = dataTables.begin();
    for ( ; it != dataTables.end(); ++it )
    {
        if ( ( *it )->alias() == fullName[0] )
        {
            field = ( *it )->dataField( fullName[1] );
            break;
        }
    }
    return field;
}

void DataSearchPopup::addPromptsFromCustomQuery( DataTable *dataTable, const QString &query )
{
    QRegExp fieldRx( "(\\[f:\\w+\\.\\w+(?::\\d)?\\])" );
    QStringList fields = listFromRx( query, fieldRx, 1 );
    QStringList::iterator it = fields.begin();
    for ( ; it != fields.end(); ++it )
    {
        QRegExp nameRx( "\\[f:(\\w+\\.\\w+)(?::\\d)?\\]" );
        QStringList name = listFromRx( ( *it ), nameRx, 1 );
        QStringList fullName = QStringList::split( '.', name.join("") );

        DataField *field = 0L;
        if ( dataTable->alias() == fullName[0] )
            field = dataTable->dataField( fullName[1] );
        else
            field = fetchField( dataTable, fullName );

        if (field)
        {
            if ( m_editors.contains( ( *it ) ) )
                continue;

            DataEditorBase *editor =
                new DataEditorBase( field->label(),
                                    field,
                                    dataTable->dataTableEditorFactory(),
                                    mainWidget() );

            m_editors[ ( *it ) ] = editor;
            if ( m_formLayout->isEmpty() )
                editor->setFocus();
            m_formLayout->add( editor );
            editor->show();
        }
    }
}

DataSearchPopup::Result DataSearchPopup::exec()
{
    if ( m_formLayout->isEmpty() )
        return Result();

    Result r;
    r.result = DialogCode( KDialogBase::exec() );
    r.components = m_updates;
    return r;
}

void DataSearchPopup::accept()
{
    switch ( m_level )
    {
    case DataTableSearch::Simple:
        break;
    case DataTableSearch::Advanced:
        resolveAdvancedPrompts();
        break;
    case DataTableSearch::Custom:
        resolveCustomPrompts();
        break;
    default:
        break;
    }

    KDialogBase::accept();
}

void DataSearchPopup::resolveAdvancedPrompts()
{
    DataTableSearch::ComponentList::ConstIterator it = m_originals.begin();
    int i = 0;
    for ( ; it != m_originals.end(); ++it )
    {
        if ( ( *it ).isPrompt() )
        {
            QString value = valueFromEditor( m_editors[ QString::number( ++i ) ] );
            m_updates.append( DataTableSearch::Component( value,
                                                          ( *it ).dataTable(),
                                                          ( *it ).dataField(), true,
                                                          ( *it ).matchMode() ) );
        }
        else
            m_updates.append( ( *it ) );
    }
}

void DataSearchPopup::resolveCustomPrompts()
{
    DataTableSearch::ComponentList::ConstIterator it = m_originals.begin();
    for ( ; it != m_originals.end(); ++it )
    {
        resolvePromptsFromCustomQuery( ( *it ), ( *it ).query() );
    }
}

void DataSearchPopup::resolvePromptsFromCustomQuery( const DataTableSearch::Component &component,
                                                     const QString &query )
{
    QString resolvedQuery = query;
    QRegExp fieldRx( "(\\[f:\\w+\\.\\w+(?::\\d)?\\])" );
    QStringList fields = listFromRx( query, fieldRx, 1 );
    QStringList::iterator it = fields.begin();
    for ( ; it != fields.end(); ++it )
    {
        DataEditorBase *base = m_editors[ ( *it ) ];
        if ( !base )
            continue;

        QString value = valueFromEditor( m_editors[ ( *it ) ] );
        resolvedQuery = resolvedQuery.replace( ( *it ), "'" + value + "'" );
    }

    m_updates.append( DataTableSearch::Component( resolvedQuery,
                                                  component.dataTable(),
                                                  component.dataField(),
                                                  component.isPrompt(),
                                                  component.matchMode() ) );
}

QString DataSearchPopup::valueFromEditor( DataEditorBase *base )
{
    QWidget *editor = base->getEditor();

    if ( DataLineEdit *line = ::qt_cast<DataLineEdit*>( editor ) )
        return line->text();

    else if ( DateEdit *date = ::qt_cast<DateEdit*>( editor ) )
        return date->date().toString("yyyy-M-d");

    else if ( TimeEdit *time = ::qt_cast<TimeEdit*>( editor ) )
        return time->getTime().toString("hh:mm:ss");

    else if ( DateTimeEdit* datetime = ::qt_cast<DateTimeEdit*>( editor ) )
        return datetime->dateTime().toString("yyyy-M-d hh:mm:ss");

    else if ( RelationCombo * relation = ::qt_cast<RelationCombo*>( editor ) )
        return relation->lineEdit()->text();

    else if ( DataComboBox * combo = ::qt_cast<DataComboBox*>( editor ) )
        return combo->lineEdit()->text();

    return QString::null;
}

#include "datasearchpopup.moc"
