/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "DarkroomView.h"

#include <png.h>

#include <QGraphicsPixmapItem>
#include <QWheelEvent>
#include <QScrollBar>

#include <KDebug>
#include <klocale.h>
#include <KFileDialog>
#include <KMessageBox>

#include <threadweaver/ThreadWeaver.h>

#include "Darkroom.h"
#include "ExportCodec.h"
#include "RawImageInfo.h"
#include "JobExport.h"
#include "JobPreview.h"
#include "PostProcessor.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"
#include "HistogramWidget.h"

struct DarkroomView::Private {
  Darkroom* darkRoom;
  RawImageInfoSP currentRawFile;
  QGraphicsPixmapItem* pixmapItem;
  QGraphicsScene* scene;
  float scale;
  int m_i_last_x;
  int m_i_last_y;
  JobPreview* jobPreview;
};

DarkroomView::DarkroomView(Darkroom* _parent) : QGraphicsView(_parent), d(new Private)
{
  d->darkRoom = _parent;
  d->currentRawFile = 0;
  // Init the scene
  d->scene = new QGraphicsScene;
  setScene( d->scene );
  d->pixmapItem = new QGraphicsPixmapItem;
  d->scene->addItem( d->pixmapItem );
  
  setCursor (Qt::OpenHandCursor);
  setMouseTracking (true);
  d->jobPreview = 0;
}

DarkroomView::~DarkroomView()
{
  delete d;
}

void DarkroomView::setRawImageInfo( RawImageInfoSP v)
{
  d->currentRawFile = v;
  
  if( d->currentRawFile->previewInfo() )
  {
    setImage( d->currentRawFile->previewInfo() );
  } else {
    updatePreview();
  }
}

void DarkroomView::fileSaveAs()
{
  if( not d->currentRawFile ) return ;
  QFileInfo fi = d->currentRawFile->fileInfo();
  
  QString extension = ExportCodec::codec( d->currentRawFile->processingOptions().asString("CodecId", "PNG") )->extension();
  
  QString file = KFileDialog::getSaveFileName( KUrl(fi.baseName() + "." + extension ), "*." + extension );
  if( not file.isEmpty() and not file.isNull() )
  {
    if( QFile( file).exists())
    {
      if( KMessageBox::warningContinueCancel( this, i18n("File %1 already exists, overwrite?", file), i18n("Overwrite?")) != KMessageBox::Continue )
      {
          return;
      }
    }
    d->currentRawFile->setProcessingOptions( d->darkRoom->processingOptions() );
    JobExport::weaver()->enqueue( new JobExport( d->currentRawFile, file ) );
  }
}

void DarkroomView::zoomIn()
{
  scale( 1.5, 1.5);
}

void DarkroomView::zoomOut()
{
  scale( 1.0 / 1.5, 1.0 / 1.5);
}

void DarkroomView::actualSize()
{
  setMatrix( QMatrix() );
}

void DarkroomView::fitToPage()
{
  fitInView( d->pixmapItem, Qt::KeepAspectRatio );
}

void DarkroomView::updatePreview()
{
  kDebug() << "updatePreview " << d->currentRawFile;
  if( not d->currentRawFile) return;
  if(JobPreview::weaver()->dequeue( d->jobPreview ))
  {
    delete d->jobPreview;
    d->jobPreview = 0;
  }
  d->currentRawFile->setProcessingOptions( d->darkRoom->processingOptions() ); // TODO do this inside Darkroom before calling updatePreview when a parameter is updated
  JobPreview* job = new JobPreview(d->currentRawFile );
  connect( job, SIGNAL(imageFinished(const PreviewInfo*)), this, SLOT(setImage( const PreviewInfo* ) ) );
  connect( job, SIGNAL(imageFinished(const PreviewInfo*)), d->currentRawFile.data(), SLOT(setPreviewInfo( const PreviewInfo* ) ) );
  connect( job, SIGNAL(done(ThreadWeaver::Job *)), this, SLOT(jobDone(ThreadWeaver::Job *)));
  connect( job, SIGNAL(emitIntermediateData( uchar *, uint , uint  )), d->darkRoom, SLOT(setIntermediateData(uchar*, uint, uint)) );
  JobPreview::weaver()->enqueue( job );
  d->jobPreview = job;
}

void DarkroomView::jobDone(ThreadWeaver::Job * job)
{
  if( job == d->jobPreview )
  {
    d->jobPreview = 0;
  }
  delete job;
}

void DarkroomView::setImage( const PreviewInfo* _previewInfo )
{
  d->pixmapItem->setPixmap( QPixmap::fromImage( _previewInfo->asQImage() ) );
  d->darkRoom->histogramWidget()->updateData( (uchar*) _previewInfo->data().data(), _previewInfo->width(), _previewInfo->height(), true, false );
}

void DarkroomView::wheelEvent ( QWheelEvent * event )
{
  if( event->modifiers() == Qt::ControlModifier )
  {
    if( event->delta() > 0 )
    {
      zoomIn();
    } else {
      zoomOut();
    }
  } else {
    QGraphicsView::wheelEvent( event );
  }
}

// ------------------------------------------------------------------------
void DarkroomView::mousePressEvent (QMouseEvent *e)
{
  if (e -> button () == Qt::LeftButton)
  {
    d -> m_i_last_x = e -> x ();
    d -> m_i_last_y = e -> y ();
    setCursor (Qt::ClosedHandCursor);
  }
}

// ------------------------------------------------------------------------
void DarkroomView::mouseReleaseEvent (QMouseEvent *e)
{
  if (e -> button () == Qt::LeftButton)
    setCursor (Qt::OpenHandCursor);
}

// ------------------------------------------------------------------------
void DarkroomView::mouseMoveEvent (QMouseEvent *e)
{
  if (e -> buttons () & Qt::LeftButton)
  {
    int dx = e -> x () - d -> m_i_last_x;
    int dy = e -> y () - d -> m_i_last_y;
    verticalScrollBar () -> setValue (
        verticalScrollBar () -> value () - dy);
    horizontalScrollBar () -> setValue (
        horizontalScrollBar () -> value () - dx);
    d -> m_i_last_x = e -> x ();
    d -> m_i_last_y = e -> y ();
  }
}

#include "DarkroomView.moc"
