/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "RawImageInfo.h"

#include <QFileInfo>
#include <QMutex>
#include <QPainter>
#include <QPixmap>

#include <KDebug>
#include <KIcon>

#include <threadweaver/ThreadWeaver.h>

#include "JobThumbnail.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"

struct RawImageInfo::Private {
  Private() : lock(QMutex::Recursive) {}
  QFileInfo fileInfo;
  QImage imagePreview;
  bool previewUptodate;
  QPixmap pixmapPreview;
  QPixmap icon;
  bool iconUptodate;
  QMutex lock;
  Status status;
  PreviewInfo* previewInfo;
  ProcessingOptions processingOptions;
  void updatePreview();
  void updateIcon();
};

void RawImageInfo::Private::updatePreview()
{
  if( previewUptodate ) return;
  pixmapPreview = QPixmap::fromImage( imagePreview );
  previewUptodate = true;
}

void RawImageInfo::Private::updateIcon()
{
  if( iconUptodate ) return;
  icon = pixmapPreview.copy();
  QString statusIconName;
  switch( status )
  {
    case IDLING:
      break;
    case LOADING:
      statusIconName = "document-open";
      break;
    case DECODING:
    case POSTPROCESSING:
    case PREVIEWING:
      statusIconName = "system-run";
      break;
    case SAVING:
      statusIconName = "document-save";
      break;
    case DONE:
      statusIconName = "dialog-ok-apply";
      break;
    case ERROR:
      statusIconName = "edit-delete";
      break;
  }
  if( not statusIconName.isEmpty() )
  {
    KIcon statusIcon( statusIconName );
    QPainter painter( &icon );
    painter.drawPixmap( 10, 10, statusIcon.pixmap(32,32) );
  }
  iconUptodate = true;
}

RawImageInfo::RawImageInfo(const QFileInfo& fileInfo ) : d(new Private)
{
  d->fileInfo = fileInfo;
  KIcon icon("image-x-dcraw");
  d->pixmapPreview = icon.pixmap( 100 );
  d->previewUptodate = true;
  d->iconUptodate = false;
  d->previewInfo = 0;
  JobThumbnail::weaver()->enqueue( new JobThumbnail( RawImageInfoSP(this)) );
  setStatus( LOADING );
}

RawImageInfo::~RawImageInfo()
{
  delete d->previewInfo;
  delete d;
}

const QFileInfo& RawImageInfo::fileInfo() const
{
  return d->fileInfo;
}

const QPixmap& RawImageInfo::thumbnail() const
{
  QMutexLocker locker(&d->lock);
  d->updatePreview();
  return d->pixmapPreview;
}

const QPixmap& RawImageInfo::icon() const
{
  QMutexLocker locker(&d->lock);
  d->updatePreview();
  d->updateIcon();
  return d->icon;
}

void RawImageInfo::setThumbnail(const QImage& _image)
{
  QMutexLocker locker(&d->lock);
  d->imagePreview = _image;
  d->previewUptodate = false;
  d->iconUptodate = false;
  setStatus( IDLING );
  emit( previewChanged( ) );
}

void RawImageInfo::setStatus( Status _newStatus )
{
  QMutexLocker locker(&d->lock);
  d->status = _newStatus;
  d->iconUptodate = false;
  emit( iconChanged() );
}

void RawImageInfo::setPreviewInfo( const PreviewInfo* previewInfo)
{
  kDebug() << "============== setPreview info for " << this;
  d->previewInfo = const_cast<PreviewInfo*>(previewInfo);
  if( d->previewInfo )
  {
    d->previewInfo->appendToCache();
  }
}

const PreviewInfo* RawImageInfo::previewInfo() const
{
  return d->previewInfo;
}

const ProcessingOptions& RawImageInfo::processingOptions() const
{
  return d->processingOptions;
}

void RawImageInfo::setProcessingOptions( const ProcessingOptions& _processingOptions )
{
  d->processingOptions = _processingOptions;
}

#include "RawImageInfo.moc"
