/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobPreview.h"

#include <QFileInfo>
#include <qendian.h>

#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/kdcraw.h>

#include "RawImageInfo.h"
#include "PostProcessor.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"

using namespace KDcrawIface;

struct JobPreview::Private {
  Private(const ProcessingOptions& _processingOptions) : processor(_processingOptions), processingOptions( _processingOptions )
  {
  }
  RawImageInfoSP rawImageInfo;
  RawDecodingSettings settings;
  PostProcessor processor;
  ProcessingOptions processingOptions;
  static ThreadWeaver::Weaver* weaver;
  int priority;
};

ThreadWeaver::Weaver* JobPreview::Private::weaver = 0;

JobPreview::JobPreview( RawImageInfoSP _rawImageInfo, int _priority ) : d(new Private(_rawImageInfo->processingOptions()))
{
  Q_ASSERT( _rawImageInfo );
  d->rawImageInfo = _rawImageInfo;
  d->settings = _rawImageInfo->processingOptions().rawDecodingSettings();
  d->processor.setConvertToSRGB( true );
  d->priority = _priority;
}

JobPreview::~JobPreview()
{
  delete d;
}

ThreadWeaver::Weaver* JobPreview::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
    Private::weaver->setMaximumNumberOfThreads( 1 );
  }
  return Private::weaver;
}

int JobPreview::priority() const
{
  return d->priority;
}

void JobPreview::run()
{
  QByteArray imageData;
  int width, height, rgbmax;
  KDcraw dcraw;
  d->rawImageInfo->setStatus( RawImageInfo::PREVIEWING );
  if(dcraw.decodeHalfRAWImage( d->rawImageInfo->fileInfo().absoluteFilePath().toUtf8(), d->settings, imageData, width, height, rgbmax))
  {
    bool sixteenBitsImage = d->processingOptions.asBool("SixteenBits");
    for( int y = 0; y < height; ++y)
    {
      for(int x = 0; x < width; ++x )
      {
          quint16* ptr = (quint16*)(imageData.data() + (y * width + x )* 3 *sizeof(quint16));
          for( int i = 0; i < 3; ++i)
          {
            ptr[i] = qFromBigEndian(ptr[i]);
          }
      }
    }
    emit( emitIntermediateData( (uchar*)imageData.data(), width, height ) );
    d->processor.apply16( imageData );
    PreviewInfo* pi = new PreviewInfo( d->rawImageInfo, width, height, imageData, true );
    d->rawImageInfo->setStatus( RawImageInfo::IDLING );
    emit( imageFinished( pi ) );
  }
  // TODO signal failure
}

#include "JobPreview.moc"
