/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobExport.h"

#include <QFileInfo>

#include <KDebug>
#include <KMessageBox>
#include <KLocale>

#include <threadweaver/ThreadWeaver.h>

#include <libkdcraw/kdcraw.h>
#include <libkdcraw/rawfiles.h>

#include "ExportCodec.h"
#include "PostProcessor.h"
#include "ProcessingOptions.h"
#include "RawImageInfo.h"

using namespace KDcrawIface;

struct JobExport::Private {
  Private(const ProcessingOptions& _processingOptions) : processingOptions( _processingOptions )
  {
  }
  RawImageInfoSP rawImageInfo;
  QString dstFileName;
  RawDecodingSettings settings;
  ProcessingOptions processingOptions;
  static ThreadWeaver::Weaver* weaver;
};

ThreadWeaver::Weaver* JobExport::Private::weaver = 0;

JobExport::JobExport( RawImageInfoSP _rawImageInfo, const QString& _dstFileName) : d(new Private(_rawImageInfo->processingOptions()))
{
  Q_ASSERT( _rawImageInfo );
  d->rawImageInfo = _rawImageInfo;
  d->dstFileName = _dstFileName;
  d->settings = _rawImageInfo->processingOptions().rawDecodingSettings();
}

JobExport::~JobExport()
{
  delete d;
}

ThreadWeaver::Weaver* JobExport::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
  }
  return Private::weaver;
}

void JobExport::run()
{
  d->rawImageInfo->setStatus( RawImageInfo::DECODING );
  QByteArray imageData;
  int width, height, rgbmax;
  KDcrawIface::KDcraw dcraw;  
  if( dcraw.decodeRAWImage( d->rawImageInfo->fileInfo().absoluteFilePath(), d->settings, imageData, width, height, rgbmax) )
  {
    d->rawImageInfo->setStatus( RawImageInfo::SAVING );
    
    ExportCodec* codec = ExportCodec::codec( d->processingOptions.asString( "CodecId", "PNG" ) );
    if(codec->writeFile( d->rawImageInfo, d->dstFileName, imageData, width, height, d->processingOptions ) )
    {
      d->rawImageInfo->setStatus( RawImageInfo::DONE );
    } else {
      d->rawImageInfo->setStatus( RawImageInfo::ERROR );
    }
  } else {
    d->rawImageInfo->setStatus( RawImageInfo::ERROR );
  }
}
