{-# OPTIONS -fglasgow-exts -fffi #-}
module System.Posix where

import Foreign
import Foreign.Ptr
import Foreign.C.Error
import Foreign.C.Types
import Foreign.C.String

import Monad

import IO

foreign import ccall "_stat" c_stat :: CString -> Ptr () -> IO Int


type FileOffset = CInt
 
data FileStatus = FileStatus {
    accessTime :: EpochTime,
    modificationTime :: EpochTime,
    fileSize :: FileOffset,
    isRegularFile :: Bool,
    isDirectory :: Bool,
    isSymbolicLink :: Bool
  }

s_ifdir, s_ifreg :: CShort
s_ifdir = 0x4000 :: CShort
s_ifreg = 0x8000 :: CShort

getFileStatus     :: FilePath -> IO FileStatus
getFileStatus path = do
    p <- mallocBytes 36
    r <- path `withCString` (`c_stat` p)
    when (r /= 0) $ throwErrno path
    md <- peek_short p 6
    sz <- peek_int p 20
    at <- peek_int p 24
    mt <- peek_int p 28
    free p
    let isReg = md .&. s_ifreg == s_ifreg
    let isDir = md .&. s_ifdir == s_ifdir
    return (FileStatus (EpochTime (fromInteger at)) (EpochTime (fromInteger mt)) sz isReg isDir False)
    where peek_int p o = do i <- peek (castPtr (plusPtr p o)) :: IO CInt
                            return (fromIntegral i)
          peek_short p o = do peek (castPtr (plusPtr p o)) :: IO CShort

getSymbolicLinkStatus :: FilePath -> IO FileStatus
getSymbolicLinkStatus = getFileStatus

foreign import ccall "sys/utime.h _utime" c_utime :: CString -> Ptr a -> IO CInt

setFileTimes :: FilePath -> EpochTime -> EpochTime -> IO ()
setFileTimes path (EpochTime atime) (EpochTime mtime) = path `withCString` \s -> do
    p <- mallocBytes 8
    poke (castPtr p :: Ptr Int32) (fromIntegral atime)
    poke (castPtr (plusPtr p 4) :: Ptr Int32) (fromIntegral mtime)
    c_utime s p
    free p
    return ()



foreign import ccall "time" c_ctime :: CInt -> IO CInt

epochTime :: IO EpochTime
epochTime = do
    t <- c_ctime 0
    return (EpochTime (fromIntegral t))

newtype EpochTime = EpochTime Int32
 deriving (Eq, Ord, Num, Real, Integral, Enum, Show)

foreign import ccall "winbase.h SleepEx" c_SleepEx :: CULong -> CUInt -> IO CInt

sleep :: Integer -> IO CInt
sleep n = c_SleepEx (1000 * fromIntegral n) 1

handleToFd :: Handle -> IO Int
handleToFd _ = fail "handleToFd not supported!"
