%  Copyright (C) 2005 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\documentclass{article}

\usepackage{verbatim}
\newenvironment{code}{\verbatim}{\endverbatim}

\begin{document}

\title{The DarcsIO library}
\author{David Roundy}

\maketitle

\begin{abstract}
This paper hasn't been written.
\end{abstract}

\begin{comment}
\begin{code}
module DarcsIO ( ReadableDirectory(..), WriteableDirectory(..),
               ) where

import System.IO.Error ( isDoesNotExistError )
import Control.Monad.Error
import System.Directory ( getDirectoryContents, createDirectory,
                          removeDirectory, removeFile,
                          renameFile, renameDirectory,
                          doesDirectoryExist, doesFileExist,
                        )

import FastPackedString ( PackedString, readFilePS, linesPS,
                          unlinesPS, nilPS, packString, unpackPS,
                        )
import DarcsUtils ( withCurrentDirectory )
import Printer ( Doc, renderPS )
import FileName ( FileName, fn2fp, fp2fn )
import Lock ( writeBinFile, readBinFile, writeAtomicFilePS )
import Workaround ( isExecutable, setExecutable )
\end{code}
\end{comment}

\begin{comment}
\begin{code}

class MonadPlus m => ReadableDirectory m where
    mDoesDirectoryExist :: FileName -> m Bool
    mDoesFileExist :: FileName -> m Bool
    mIsFileExecutable :: FileName -> m Bool
    mInCurrentDirectory :: FileName -> m a -> m a
    mGetDirectoryContents :: m [FileName]
    mReadBinFile :: FileName -> m String
    mReadBinFile f = liftM unpackPS $ mReadFilePS f
    mReadFilePS :: FileName -> m PackedString
    mReadFilePSs :: FileName -> m [PackedString]
    mReadFilePSs f = linesPS `liftM` mReadFilePS f

class ReadableDirectory m => WriteableDirectory m where
    mWithCurrentDirectory :: FileName -> m a -> m a
    mSetFileExecutable :: FileName -> Bool -> m ()
    mWriteBinFile :: FileName -> String -> m ()
    mWriteBinFile fn s = mWriteFilePS fn $ packString s
    mWriteFilePS :: FileName -> PackedString -> m ()
    mWriteFilePSs :: FileName -> [PackedString] -> m ()
    mWriteFilePSs f ss = mWriteFilePS f (unlinesPS ss)
    mCreateDirectory :: FileName -> m ()
    mRemoveDirectory :: FileName -> m ()
    mWriteDoc :: FileName -> Doc -> m ()
    mWriteDoc f d = mWriteFilePS f (renderPS d)
    mCreateFile :: FileName -> m ()
    mCreateFile f = mWriteFilePS f nilPS
    mRemoveFile :: FileName -> m ()
    mRename :: FileName -> FileName -> m ()
    mModifyFilePS :: FileName -> (PackedString -> m PackedString) -> m ()
    mModifyFilePS f j = do ps <- mReadFilePS f
                           ps' <- j ps
                           mWriteFilePS f ps'
    mModifyFilePSs :: FileName -> ([PackedString] -> m [PackedString]) -> m ()
    mModifyFilePSs f j = do ps <- mReadFilePSs f
                            ps' <- j ps
                            mWriteFilePSs f ps'

instance ReadableDirectory IO where
    mDoesDirectoryExist = doesDirectoryExist . fn2fp
    mDoesFileExist = doesFileExist . fn2fp
    mIsFileExecutable = isExecutable . fn2fp
    mInCurrentDirectory = withCurrentDirectory . fn2fp
    mGetDirectoryContents = map fp2fn `liftM` getDirectoryContents "."
    mReadBinFile = readBinFile . fn2fp
    mReadFilePS = readFilePS . fn2fp

instance WriteableDirectory IO where
    mWithCurrentDirectory = mInCurrentDirectory
    mSetFileExecutable = setExecutable . fn2fp
    mWriteBinFile = writeBinFile . fn2fp
    mWriteFilePS = writeAtomicFilePS . fn2fp
    mCreateDirectory = createDirectory . fn2fp
    mRemoveFile = removeFile . fn2fp
    mRemoveDirectory = removeDirectory . fn2fp
    mRename a b = (renameDirectory x y `mplus` renameFile x y)
                  -- We need to catch does not exist errors, since older
                  -- versions of darcs allowed users to rename nonexistent
                  -- files.  :(
                  `catch` (\e -> if isDoesNotExistError e
                                 then return ()
                                 else ioError e)
        where x = fn2fp a
              y = fn2fp b
\end{code}
\end{comment}

\end{document}
