%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module PatchApply ( applyToPop, patchChanges, empty_markedup_file,
                    markup_file, force_replace_slurpy,
                    apply_to_filepath, apply_to_filepaths, apply_to_slurpy,
                    LineMark(..), MarkedUpFile )
       where

import Prelude hiding ( pi )
import AntiMemo ( AntiMemo )
import FastPackedString ( PackedString, packString, splitPS, concatPS,
                          breakPS, nullPS, nilPS )
import FileName ( fn2ps, fn2fp, fp2fn )
import PatchInfo ( PatchInfo )
import PopulationData ( Population(..), PopTree(..), Info(..), DirMark(..) )
import Data.Maybe ( catMaybes )
import PatchCore ( Patch(..), DirPatchType(..), FilePatchType(..), is_merger )
import PatchCommute ( merger_equivalent, applyBinary, try_tok_internal,
                      movedirfilename )
import Control.Monad ( liftM )
import SlurpDirectory ( FileContents, Slurpy,
                        slurp_addfile, slurp_removefile, slurp_modfile,
                        slurp_adddir, slurp_removedir,
                        slurp_move, slurp_runfunc )
import RegChars ( regChars )
import RepoPrefs ( change_prefval )
import AntiMemo ( readAntiMemo )
#include "impossible.h"
\end{code}

\section{Introduction}

A patch describes a change to the tree.  It could be either a primitive
patch (such as a file add/remove, a directory rename, or a hunk replacement
within a file), or a composite patch describing many such changes.  Every
patch type must satisfy the conditions described in this appendix.  The
theory of patches is independent of the data which the patches manipulate,
which is what makes it both powerful and useful, as it provides a framework
upon which one can build a revision control system in a sane manner.

Although in a sense, the defining property of any patch is that it can be
applied to a certain tree, and thus make a certain change, this change does
not wholly define the patch.  A patch is defined by a
\emph{representation}, together with a set of rules for how it behaves
(which it has in common with its patch type).  The \emph{representation} of
a patch defines what change that particular patch makes, and must be
defined in the context of a specific tree.  The theory of patches is a
theory of the many ways one can change the representation of a patch to
place it in the context of a different tree.  The patch itself is not
changed, since it describes a single change, which must be the same
regardless of its representation\footnote{For those comfortable with
quantum mechanics, think of a patch as a quantum mechanical operator, and
the representation as the basis set.  The analogy breaks down pretty
quickly, however, since an operator could be described in any complete
basis set, while a patch modifying the file {\tt foo} can only be described
in the rather small set of contexts which have a file {\tt foo} to be
modified.}.

So how does one define a tree, or the context of a patch? The simplest way
to define a tree is as the result of a series of patches applied to the
empty tree\footnote{This is very similar to the second-quantized picture,
in which any state is seen as the result of a number of creation operators
acting on the vacuum, and provides a similar set of simplifications---in
particular, the exclusion principle is very elegantly enforced by the
properties of the anti-hermitian fermion creation operators.}.  Thus, the
context of a patch consists of the set of patches that precede it.

\section{Applying patches}

\begin{code}
apply_to_filepaths :: Patch -> [FilePath] -> [FilePath]
apply_to_filepaths pa fs = concatMap (atof pa) fs
    where atof (Move f f') fi = [fn2fp $ movedirfilename f f' (fp2fn fi)]
          atof (FP f RmFile) fi | fn2fp f == fi = []
          atof (DP f RmDir) fi | fn2fp f == fi = []
          atof (NamedP _ _ p) fi = atof p fi
          atof (ComP []) fi = [fi]
          atof (ComP (p:ps)) fi = case atof p fi of
                                  [fi'] -> atof (ComP ps) fi'
                                  _ -> []
          atof (Split ps) fi = atof (ComP ps) fi
          atof p fi | is_merger p = atof (merger_equivalent p) fi
          atof _ fi = [fi]

apply_to_filepath :: Patch -> FilePath -> FilePath
apply_to_filepath (Move f f') fi = fn2fp $ movedirfilename f f' (fp2fn fi)
apply_to_filepath (FP f RmFile) fi | fn2fp f == fi = ""
apply_to_filepath (DP f RmDir) fi | fn2fp f == fi = ""
apply_to_filepath (NamedP _ _ p) fi = apply_to_filepath p fi
apply_to_filepath (ComP []) fi = fi
apply_to_filepath (ComP (p:ps)) fi =
    apply_to_filepath (ComP ps) $ apply_to_filepath p fi
apply_to_filepath (Split ps) fi = apply_to_filepath (ComP ps) fi
apply_to_filepath p fi | is_merger p =
    apply_to_filepath (merger_equivalent p) fi
apply_to_filepath _ fi = fi

apply_to_slurpy :: Patch -> Slurpy -> Maybe Slurpy

apply_to_slurpy (NamedP _ _ p) s = apply_to_slurpy p s
apply_to_slurpy p s | is_merger p =
    apply_to_slurpy (merger_equivalent p) s
apply_to_slurpy (Merger _ _ _ _ _ _) _ = impossible
apply_to_slurpy (ComP []) s = Just s
apply_to_slurpy (ComP (p:ps)) s =
  apply_to_slurpy p s >>= apply_to_slurpy (ComP ps)
apply_to_slurpy (Split []) s = Just s
apply_to_slurpy (Split (p:ps)) s =
  apply_to_slurpy p s >>= apply_to_slurpy (Split ps)

apply_to_slurpy (FP f RmFile) s = slurp_removefile f s
apply_to_slurpy (FP f AddFile) s = slurp_addfile f s
apply_to_slurpy (FP f (Hunk line old new)) s =
    slurp_modfile f (applyHunkLines line old new) s
apply_to_slurpy (FP f (TokReplace tcs old new)) s =
    slurp_modfile f (applyTokReplace tcs old new) s
apply_to_slurpy (FP f (Binary o n)) s =
    slurp_modfile f (applyBinary o n) s

apply_to_slurpy (DP d AddDir) s = slurp_adddir d s
apply_to_slurpy (DP d RmDir) s = slurp_removedir d s

apply_to_slurpy (Move f f') s = slurp_move f f' s
apply_to_slurpy (ChangePref p f t) s =
    slurp_runfunc (change_prefval p f t) s

force_replace_slurpy :: Patch -> Slurpy -> Maybe Slurpy
force_replace_slurpy (FP f (TokReplace tcs old new)) s =
    slurp_modfile f (forceTokReplace tcs old new) s
force_replace_slurpy _ _ = bug "Can only force_replace_slurpy on a replace."
\end{code}

\subsection{Hunk patches}

Hunks are an example of a complex filepatch.  A hunk is a set of lines of a
text file to be replaced by a different set of lines.  Either of these sets
may be empty, which would mean a deletion or insertion of lines.
\begin{code}
applyHunkLines :: Int -> AntiMemo [PackedString] -> AntiMemo [PackedString]
               -> FileContents -> Maybe FileContents
applyHunkLines _ o n fc
    | null (readAntiMemo o) && null (readAntiMemo n) = Just fc
applyHunkLines l _ _ _ | l < 0 = bug "Patch.applyHunkLines: After -ve lines?"
applyHunkLines l ooo nnn (ccc,_) = if readAntiMemo ahl == Nothing
                                   then Nothing
                                   else Just $ (fromJust `fmap` ahl, Nothing)
    where splitAtN 0 xs = Just ([], xs)
          splitAtN i (x:xs) = case splitAtN (i-1) xs of
                                  Just (ys, zs) -> Just (x:ys, zs)
                                  Nothing -> Nothing
          splitAtN _ [] = Nothing
          dropPrefix [] ys = Just ys
          dropPrefix (x:xs) (y:ys) | x == y = dropPrefix xs ys
          dropPrefix _ _ = Nothing
          actualo = readAntiMemo ooo
          lengtho = length actualo
          isok = case splitAtN (l - 1) (readAntiMemo ccc) of
                 Just (_,post) -> take lengtho post == actualo
                 Nothing -> False
          ahl = if not isok
                then return Nothing
                else do c <- ccc
                        n <- nnn
                        case splitAtN (l - 1) c of
                            Just (pre, post) ->
                                return $ Just (pre++n++drop lengtho post)
                            Nothing -> impossible
\end{code}

\subsection{Token replace patches}\label{token_replace}

Although most filepatches will be hunks, darcs is clever enough to support
other types of changes as well.  A ``token replace'' patch replaces all
instances of a given token with some other version.  A token, here, is
defined by a regular expression, which must be of the simple [a--z\ldots]\ type,
indicating which characters are allowed in a token, with all other
characters acting as delimiters.  For example, a C identifier would be a
token with the flag \verb![A-Za-z_0-9]!.

\begin{code}
forceTokReplace :: String -> String -> String
                -> FileContents -> Maybe FileContents
forceTokReplace t os ns (c,_) = Just (map forceReplace `fmap` c, Nothing)
    where o = packString os
          n = packString ns
          tokchar = regChars t
          toks_and_intratoks ps | nullPS ps = []
          toks_and_intratoks ps =
              let (before,s') = breakPS tokchar ps
                  (tok, after) = breakPS (not . tokchar) s'
                  in before : tok : toks_and_intratoks after
          forceReplace ps = concatPS $ map o_t_n $ toks_and_intratoks ps
          o_t_n s | s == o = n
                  | otherwise = s

applyTokReplace :: String -> String -> String
                -> FileContents -> Maybe FileContents
applyTokReplace t o n (c,_) =
    case mapM (try_tok_internal t (packString o) (packString n)) $
         readAntiMemo c of
    Nothing -> Nothing
    Just c' -> Just (return $ map concatPS c', Nothing)
\end{code}

What makes the token replace patch special is the fact that a token replace
can be merged with almost any ordinary hunk, giving exactly what you would
want.  For example, you might want to change the patch type {\tt
TokReplace} to {\tt TokenReplace} (if you decided that saving two
characters of space was stupid).  If you did this using hunks, it would
modify every line where {\tt TokReplace} occurred, and quite likely provoke
a conflict with another patch modifying those lines.  On the other hand, if
you did is using a token replace patch, the only change that it could
conflict with would be if someone else had used the token ``{\tt
TokenReplace}'' in their patch rather than TokReplace---and that actually
would be a real conflict!

%\section{Outputting interesting and useful information}

%Just being able to manipulate patches and trees is not enough.  We also
%want to be able to view the patches and files.  This requires another set
%of functions, closely related to the patch application functions, which
%will give us the necessary information to browse the changes we have made.
%It is \emph{not} the Patch module's responsibility to add any sort of
%markup or formatting, but simply to provide the information necessary for an
%external module to do the formatting.

\begin{code}
data LineMark = AddedLine PatchInfo | RemovedLine PatchInfo
              | AddedRemovedLine PatchInfo PatchInfo | None
                deriving (Show)
type MarkedUpFile = [(PackedString, LineMark)]
empty_markedup_file :: MarkedUpFile
empty_markedup_file = [(nilPS, None)]

markup_file :: PatchInfo -> Patch
            -> (FilePath, MarkedUpFile) -> (FilePath, MarkedUpFile)
markup_file n (NamedP _ _ p') (f, mk) = markup_file n p' (f, mk)
markup_file n p (f, mk) | is_merger p =
    markup_file n (merger_equivalent p) (f, mk)
markup_file _ (Merger _ _ _ _ _ _) _ = impossible
markup_file _ (ComP []) (f, mk) = (f, mk)
markup_file n (ComP (p:ps)) (f, mk) = markup_file n (ComP ps) $
                                      markup_file n p (f, mk)
markup_file _ (Split []) (f, mk) = (f, mk)
markup_file n (Split (p:ps)) (f, mk) = markup_file n (Split ps) $
                                       markup_file n p (f, mk)
markup_file _ (FP _ AddFile) (f, mk) = (f, mk)
markup_file _ (FP _ RmFile) (f, mk) = (f, mk)
markup_file n (FP f' (Hunk line old new)) (f, mk)
    | fn2fp f' /= f = (f, mk)
    | otherwise = (f, markup_hunk n line
                   (readAntiMemo old) (readAntiMemo new) mk)
markup_file name (FP f' (TokReplace t o n)) (f, mk)
    | fn2fp f' /= f = (f, mk)
    | otherwise = (f, markup_tok name t o n mk)
markup_file _ (DP _ _) (f, mk) = (f, mk)
markup_file _ (Move d d') (f, mk) = (fn2fp $ movedirfilename d d' (fp2fn f), mk)
markup_file _ (ChangePref _ _ _) (f,mk) = (f,mk)
markup_file n (FP f' (Binary _ _)) (f,mk)
    | fn2fp f' == f = (f,(packString "Binary file", AddedLine n):mk)
    | otherwise = (f,mk)

markup_hunk :: PatchInfo -> Int -> [PackedString] -> [PackedString]
            -> MarkedUpFile -> MarkedUpFile
markup_hunk n l old new ((sf, RemovedLine pi):mk) =
    (sf, RemovedLine pi) : markup_hunk n l old new mk
markup_hunk n l old new ((sf, AddedRemovedLine po pn):mk) =
    (sf, AddedRemovedLine po pn) : markup_hunk n l old new mk

markup_hunk name 1 old (n:ns) mk =
    (n, AddedLine name) : markup_hunk name 1 old ns mk
markup_hunk n 1 (o:os) [] ((sf, None):mk)
    | o == sf = (sf, RemovedLine n) : markup_hunk n 1 os [] mk
    | otherwise = [(packString "Error in patch application", AddedLine n)]
markup_hunk n 1 (o:os) [] ((sf, AddedLine nold):mk)
    | o == sf = (sf, AddedRemovedLine nold n) : markup_hunk n 1 os [] mk
    | otherwise = [(packString "Error in patch application", AddedLine n)]
markup_hunk _ 1 [] [] mk = mk

markup_hunk n l old new ((sf, AddedLine pi):mk)
    | l > 1 = (sf, AddedLine pi) : markup_hunk n (l-1) old new mk
    | l < 1 = (sf, AddedLine pi) : markup_hunk n (l-1) old new mk
markup_hunk n l old new ((sf, None):mk)
    | l > 1 = (sf, None) : markup_hunk n (l-1) old new mk
    | l < 1 = (sf, None) : markup_hunk n (l-1) old new mk

markup_hunk _ _ _ _ [] = []

markup_hunk _ _ _ _ mk = (packString "Error: ",None) : mk

markup_tok :: PatchInfo -> String -> String -> String
           -> MarkedUpFile -> MarkedUpFile
markup_tok name t ostr nstr mk = concatMap mt mk
    where o = packString ostr
          n = packString nstr
          mt (sf, AddedLine pi) =
              case concatPS `liftM` try_tok_internal t o n sf of
              Just sf' | sf' == sf -> [(sf, AddedLine pi)]
                       | otherwise -> [(sf, AddedRemovedLine pi name),
                                       (sf', AddedLine name)]
              Nothing ->
                  [(sf, AddedLine pi),
                   (packString "There seems to be an inconsistency...", None),
                   (packString "Please run darcs check.", None)]
          mt mark = [mark]
\end{code}

%files or directories, changed by a patch
%we get it solely from the patch here
%instead of performing patch apply on a population
%we !could! achieve the same by applying a patch to a cleaned population
%and getting modified files and dirs
%but this should be significantly slower when the population grows large
%This could be useful for just presenting a summary of what a patch does
%(especially useful for larger repos)

\begin{code}
patchChanges :: Patch -> [(String,DirMark)]
patchChanges (NamedP _ _ p) = patchChanges p
patchChanges (Move f1 f2) = [(fn2fp f1,MovedFile $ fn2fp f2),
                             (fn2fp f2,MovedFile $ fn2fp f1)]
patchChanges (DP d AddDir) = [(fn2fp d,AddedDir)]
patchChanges (DP d RmDir) = [(fn2fp d,RemovedDir)]
patchChanges (FP f AddFile) = [(fn2fp f,AddedFile)]
patchChanges (FP f RmFile) = [(fn2fp f,RemovedFile)]
patchChanges (FP f _) = [(fn2fp f,ModifiedFile)]
patchChanges (Split ps) = concatMap patchChanges ps
patchChanges (ComP ps) = concatMap patchChanges ps
patchChanges p | is_merger p = patchChanges $ merger_equivalent p
patchChanges (Merger _ _ _ _ _ _) = impossible
patchChanges (ChangePref _ _ _) = []
\end{code}

%apply a patch to a population at a given time

\begin{code}
applyToPop :: PatchInfo -> Patch -> Population -> Population
applyToPop pi patch (Pop _ tree)
 = Pop pi (applyToPopTree patch tree)
   -- ``pi'' is global below!
 where applyToPopTree :: Patch -> PopTree -> PopTree
       applyToPopTree (NamedP _ _ p) tr = applyToPopTree p tr
       applyToPopTree p tr | is_merger p
        = applyToPopTree (merger_equivalent p) tr
       applyToPopTree (Merger _ _ _ _ _ _) _ = impossible
       applyToPopTree (ComP ps) tr =
        foldl (\t p -> applyToPopTree p t) tr ps
       applyToPopTree (Split ps) tr =
        foldl (\t p -> applyToPopTree p t) tr ps
       applyToPopTree p@(FP f AddFile) tr =
           let xxx = splitPS '/' (fn2ps  f) in
               popChange xxx p $ fst $ breakP xxx tr
       applyToPopTree p@(FP f _) tr = popChange (splitPS '/' (fn2ps  f)) p tr
       applyToPopTree p@(DP f AddDir) tr =
           let xxx = splitPS '/' (fn2ps  f) in
               popChange xxx p $ fst $ breakP xxx tr
       applyToPopTree p@(DP d _) tr = popChange (splitPS '/' (fn2ps  d)) p tr
       -- precondition: ``to'' does not exist yet!
       applyToPopTree (Move from to) tr
        = case breakP (splitPS '/' (fn2ps from)) $
               fst $ breakP (splitPS '/' $ fn2ps to) tr of
           (tr',Just ins) ->
               let to' = (splitPS '/' (fn2ps to))
                   ins' = case ins of
                          PopDir i trs -> PopDir (i {nameI = last to',
                                                     modifiedByI = pi,
                                                     modifiedHowI = MovedDir (fn2fp from)})
                                                 trs
                          PopFile i -> PopFile (i {nameI = last to',
                                                   modifiedByI = pi,
                                                   modifiedHowI = MovedFile (fn2fp from)})
                             in insertP to' tr' ins'
           _ -> tr -- ignore the move if ``from'' couldn't be found
       applyToPopTree (ChangePref _ _ _) tr = tr

       -- insert snd arg into fst arg
       insertP :: [PackedString] -> PopTree -> PopTree -> PopTree
       insertP [parent,_] org@(PopDir f trs) tr
        | parent == (nameI f) = PopDir f (tr:trs)
        | otherwise = org
       insertP (n:rest) org@(PopDir f trs) tr
        | (nameI f) == n = PopDir f trs'
        | otherwise = org
          where trs' = map (\o -> insertP rest o tr) trs
       insertP _ org _ = org

       -- change a population according to a patch
       popChange :: [PackedString] -> Patch -> PopTree -> PopTree
       popChange [parent,path] (DP d AddDir) tr@(PopDir f trs)
        | parent == (nameI f) = PopDir f (new:trs)
        | otherwise = tr
              where new = PopDir (Info {nameI = path,
                                        modifiedByI = pi,
                                        modifiedHowI = AddedDir,
                                        createdByI = Just pi,
                                        creationNameI = Just $ fn2ps d}) []
       -- only mark a directory (and contents) as ``deleted'' do not delete it actually
       popChange [path] (DP _ RmDir) tr@(PopDir f trs)
        | path == (nameI f) = PopDir (f {modifiedByI = pi,
                                         modifiedHowI = RemovedDir}) trs'
        | otherwise = tr
          where trs' = map markDel trs -- recursively ``delete'' the contents

       popChange [parent,path] (FP d AddFile) tr@(PopDir f trs)
        | parent == (nameI f) = PopDir f (new:trs)
        | otherwise = tr
              where new = PopFile (Info {nameI = path,
                                         modifiedByI = pi,
                                         modifiedHowI = AddedFile,
                                         createdByI = Just pi,
                                         creationNameI = Just $ fn2ps d})
       popChange [path] (FP _ RmFile) tr@(PopFile f)
        | path == (nameI f) = PopFile (f {modifiedByI = pi,
                                         modifiedHowI = RemovedFile})
        | otherwise = tr
       popChange [path] (FP _ _) (PopFile f)
        | path == (nameI f)
           = PopFile (f {modifiedByI = pi,
                         modifiedHowI = if modifiedHowI f == AddedFile && modifiedByI f == pi
                                        then AddedFile
                                        else ModifiedFile})
       popChange (n:rest) p tr@(PopDir f trs)
        | (nameI f) == n = PopDir f (map (popChange rest p) trs)
        | otherwise = tr
       popChange _ _ tr = tr
       markDel (PopDir f trs) = PopDir (f {modifiedByI = pi,
                                           modifiedHowI = RemovedDir}) trs'
                where trs' = map markDel trs
       markDel (PopFile f) = PopFile (f {modifiedByI = pi,
                                         modifiedHowI = RemovedFile})

-- break a poptree fst: org tree with subtree removed,
--                 snd: removed subtree
breakP :: [PackedString] -> PopTree -> (PopTree,Maybe PopTree)
breakP [parent,path] tr@(PopDir f trees)
 | parent == (nameI f) = case findRem path trees of
                         Just (trees',tree') -> (PopDir f trees',Just tree')
                         _ -> (tr,Nothing)
 | otherwise = (tr,Nothing)
 where findRem _ [] = Nothing
       findRem the_path (d:trs)
        | the_path == pname d = Just (trs,d)
        | otherwise = do (trs',d') <- findRem the_path trs
                         return (d:trs',d')
breakP (n:rest) tr@(PopDir f trs)
 | (nameI f) == n = case catMaybes inss of
                    [ins] -> (PopDir f trs', Just ins)
                    [] -> (tr,Nothing)
                    _ -> error "breakP: more than one break"
 | otherwise = (tr,Nothing)
   where (trs',inss) = unzip (map (breakP rest) trs)
breakP _ tr = (tr,Nothing)

pname :: PopTree -> PackedString
pname (PopDir i _) = nameI i
pname (PopFile i) = nameI i
\end{code}

