#!/usr/bin/perl -T
#
# darcsrv - the darcs repository viewer
#
# Copyright (c) 2004 Will Glozer
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

#
# This program calls darcs (or its own subroutines) to generate XML
# which is rendered into HTML via XSLT.  It is capable of displaying
# the files in a repository, various patch histories, annotations, etc.
#

require strict;

use CGI qw( :standard );
use CGI::Util;
use File::Basename;
use File::stat;
use File::Temp;
use POSIX;

## the following variables can be customized to reflect your system
## configuration by defining them appropriately in the file
## "/etc/cgi.conf".  The syntax accepts equals signs or simply
## blanks separating values from assignments.

$ENV{'PATH'} = read_conf('PATH', $ENV{'PATH'});

# path to executables, or just the executable if they are in $ENV{'PATH'}
$darcs_program    = read_conf("darcs", "darcs");
$xslt_program     = read_conf("xsltproc", "xsltproc");

# directory containing repositories
$repository_root  = read_conf("reposdir", "/var/www");

# XSLT template locations
$template_root = read_conf("xslt_dir", "/usr/local/share/darcs/xslt");

$xslt_annotate = "$template_root/annotate.xslt";
$xslt_browse   = "$template_root/browse.xslt";
$xslt_patches  = "$template_root/patches.xslt";

$xslt_errors   = "$template_root/errors.xslt";

# CSS stylesheet that XSLT templates refer to.  This is a HTTP request
# path, not a local file system path. The default will magically read
# /etc/darcs/styles.css
$stylesheet    = read_conf("stylesheet", "/cgi-bin/darcs.cgi/styles.css");

# XML source for the error pages
$xml_errors = "$template_root/errors.xml";

## end customization

# ----------------------------------------------------------------------

# read a value from the cgi.conf file.

sub read_conf {
  my $flag, $val;
  if ($#_ == 0) {
    ($flag) = @_;
    $val = "";
  } else {
    ($flag, $val) = @_;
  }
  if (open(CGI_CONF, "/etc/darcs/cgi.conf")) {
    while (<CGI_CONF>) {
      if (/^\s*$flag\s*=\s*(\S+)/) {
        $val = $1;
      } elsif (/^\s*$flag\s+(\S+)/) {
        $val = $1;
      }
    }
    close(CGI_CONF);
  }
  return $val;
}

# transform and output `xml' with stylesheet file `xslt'
sub transform {
    my ($xml, $xslt, $args) = @_;

    print "Content-type: text/html\r\n\r\n";
    print `$xslt_program $args $xslt $xml`;
}

# create a temporary XML document with a root element and the repository
# path, if any.
sub make_xml {
    my ($repo, $target) = @_;
    my ($fh, $fname) = File::Temp::tempfile();
    
    # eliminate any slashes in the repository name
    $repo =~ s|/||g;

    printf $fh qq(<darcs repository="$repo" target="%s">\n), \
        $target ? "$repo/$target" : $repo;

    print $fh qq(<path>\n);
    print $fh qq(<directory full-path="/$repo/">$repo</directory>\n);

    if ($target) {
        my ($full_path, $file) = "/$repo/";

        # extract the file portion of a target
        if ($target !~ '/$') {
            my ($index) = rindex($target, '/');
            if ($index == -1) {
                $file = $target;
                $target = "";
            } else {
                $file = substr($target, $index + 1);
                $target = substr($target, 0, $index);
            }
        }

        foreach $path (split('/', $target)) {
            $full_path .= "$path/";
            print $fh qq(<directory full-path="$full_path">$path</directory>\n);
        }

        print $fh qq(<file full-path="$full_path$file">$file</file>\n) if $file;
    }
    print $fh qq(</path>\n\n);
    
    return ($fh, $fname);
}

# finish XML output
sub finish_xml {
    my ($fh) = @_;
    print $fh "\n</darcs>\n";
}

# run a darcs command and return a file handle for the output
sub darcs {
    my ($repo, $cmd, $args, $target) = @_;
    my ($fh, $darcs_out) = make_xml($repo, $target);

    # quote target only if set, otherwise darcs will get a bad param
    $target = "'$target'" if $target;

    chdir "$repository_root/$repo";
    open(DARCS, "$darcs_program '$cmd' --xml-output $args $target|");
    while (<DARCS>) {
        print $fh $_;
    }    

    finish_xml($fh);
    return ($fh, $darcs_out);
}

# get a directory listing as XML output
sub dir_listing {
    my ($repo, $dir) = @_;
    my ($fh, $dir_out) = make_xml($repo, $dir);
        
    print $fh "<files>\n";
    foreach $file (glob("$repository_root/$repo/_darcs/current/$dir/*")) {
        my $secs  = stat($file)->mtime;
        my $mtime = localtime($secs);
        my $ts = POSIX::strftime("%Y%m%d%H%M%S", gmtime $secs);
        
        my ($name, $type);

         if (-d $file) {
             ($name, $type) = (basename($file) . '/', 'directory');
         } else {
             ($name, $type) = (basename($file), 'file');            
         }
         printf $fh qq(  <$type name="$name" modified="$mtime" ts="$ts" />\n);
    }
    print $fh "</files>\n";

    finish_xml($fh);
    return ($fh, $dir_out);
}

# safely extract a parameter from the http request.  This applies a regexp
# to the parameter which should group only the appropriate parameter value
sub safe_param {
    my ($param, $regex, $default) = @_;
    my $value = CGI::Util::unescape(param($param));
    return ($value =~ $regex) ? $1 : $default;
}

# respond to a CGI request
sub respond {
    # untaint script_name, reasonable to expect only \w, -, /, and . in the name
    my $script_name = CGI::Util::unescape(script_name());
    $script_name =~ qr~^([\w/.\-]+)$~ or die qq(bad script_name "$script_name");
    $script_name = $1;

    my ($path) = CGI::Util::unescape(path_info());
    # don't allow ./ or ../ in paths
    $path =~ s|[.]+/||g;

    # don't allow any shell meta characters in paths
    $path =~ qr@^([^\\!\$\^&*()\[\]{}<>~`|';"?\r\n]+)$@ or \
        die qq(bad path_info "$path");
    $path = $1;

    # check whether we're asking for styles.css
    if ($path eq '/styles.css') {
        open (STYLES_CSS, "/etc/darcs/styles.css") or
            die "couldn't find styles.css"; # FIXME should give default styles.css?
        print "Content-type: text/css\r\n\r\n";
        while (<STYLES_CSS>) {
          print $_;
        }
        close (STYLES_CSS);
        return;
    }

    # determine if request is for the repository root or a contained file
    my ($file, $repo, $dir, $working_dir) = fileparse($path);
    if ($repo =~ qr|^(/[^/]+)/(.+)|) {
        ($repo, $dir) = ($1, $2);
        $working_dir = "$1/$2";
    } elsif ($repo eq '/') {
        ($repo, $dir) = ("/$file", "");
        $working_dir = $repo;
    } else {
        $working_dir = $repo;
    }

    # untaint simple parameters, which can only have chars matching \w+
    my $cmd  = safe_param('c', '^(\w+)$', 'browse');
    my $sort = safe_param('s', '^(\w+)$', '');

    # set the xslt processing arguments
    my $xslt_args = qq {
        --stringparam cgi-program '$script_name'
        --stringparam sort-by '$sort'
        --stringparam stylesheet '$stylesheet'
    };
    $xslt_args =~ s/\s+/ /gm;
    
    # make sure the repository exists
    if (!-d "$repository_root$repo/_darcs") {
        $xslt_args .= " --stringparam error-type 'invalid-repository' ";    
        print "Status: 404 Invalid repository\r\n";
        transform($xml_errors, $xslt_errors, $xslt_args);
        return;
    }
    
    # untaint patches and tags. Tags can have arbitrary values, so
    # never pass these unquoted, on pain of pain!
    my $patch = safe_param('p', '^([\w\-.]+)$');
    my $tag   = safe_param('t', '^(.+)$');    
        
    my $darcs_args = "";
    $darcs_args .= " --match 'hash $patch' " if $patch;
    $darcs_args .= " -t '$tag' " if $tag;    

    my ($fn, $fname);

    # process the requested command
    if ($cmd eq 'browse') {
        ($fh, $fname) = dir_listing($repo, $dir);
        transform($fname, $xslt_browse, $xslt_args);
    } elsif ($cmd eq 'patches') {
        ($fh, $fname) = darcs($repo, "changes", $darcs_args, "$dir$file");
        transform($fname, $xslt_patches, $xslt_args);       
    } elsif ($cmd eq 'annotate') {
        $darcs_args .= " --summary ";
        ($fh, $fname) = darcs($repo, "annotate", $darcs_args, "$dir$file");
        transform($fname, $xslt_annotate, $xslt_args);       
    } else {
        $xslt_args .= " --stringparam error-type 'invalid-command' ";
        print "Status: 400 Invalid command\r\n";
        transform($xml_errors, $xslt_errors, $xslt_args);
    }
    File::Temp::unlink0($fn, $fname);
}

# CGI entry point
respond();
