%  Copyright (C) 2003-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs unrevert}\label{unrevert}
\begin{code}
module Unrevert ( unrevert ) where
import System ( ExitCode(..), exitWith )
import Monad ( when, liftM )
import List ( elem )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( Verbose ),
                        verbose, ignoretimes, working_repo_dir,
                      )
import Directory ( removeFile )
import Repository ( PatchSet, read_repo, am_in_repo,
                    read_pending, write_pending, sift_for_pending,
                  )
import Patch ( join_patches, apply_to_slurpy, readPatchPS )
import PatchInfo ( PatchInfo, patchinfo, readPatchInfoPS )
import SlurpDirectory ( slurp, slurp_write_dirty )
import FastPackedString ( PackedString, readFilePS )
import Lock ( withLock, )
import Pull ( merge_with_us_and_pending )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution )
import DarcsUtils ( catchall, putStrLnError )
#include "impossible.h"
\end{code}
\begin{code}
unrevert_description :: String
unrevert_description =
 "Undo the last revert operation.\n"
\end{code}

\options{unrevert}

\haskell{unrevert_help}
\begin{code}
unrevert_help :: String
unrevert_help =
 "Unrevert is used to undo the results of a revert command.\n"
\end{code}
While this is only guaranteed to work properly if you haven't made any
changes since the revert was performed, it makes a best effort to merge the
unreversion with any changes you have since made.  In fact, unrevert should
even work if you've recorded changes since reverting.
\begin{code}
unrevert :: DarcsCommand
unrevert = DarcsCommand {command_name = "unrevert",
                         command_help = unrevert_help,
                         command_description = unrevert_description,
                         command_extra_args = 0,
                         command_extra_arg_help = [],
                         command_command = unrevert_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [verbose, ignoretimes,
                                                 working_repo_dir]}
\end{code}
\begin{code}
unrevert_cmd :: [DarcsFlag] -> [String] -> IO ()
unrevert_cmd opts [] = withLock "./_darcs/lock" $ do
  us <- read_repo "."
  them <- unrevert_patch_bundle
  case get_common_and_uncommon (us, them) of
    (_, us', them') -> do
      (_, work_patch) <- merge_with_us_and_pending opts
                         (map (fromJust.snd) $ reverse $ head us',
                          map (fromJust.snd) $ reverse $ head them')
      pw_resolved <- join_patches `liftM` standard_resolution work_patch
      working <- slurp "."
      when (Verbose `elem` opts) $ putStr "Unreverting...\n"
      case apply_to_slurpy pw_resolved working of
          Nothing -> do putStrLnError "Error unreverting..."
                        exitWith $ ExitFailure 1
          Just work' -> do
              slurp_write_dirty work'
              pending <- read_pending
              write_pending $ sift_for_pending $
                case pending of
                Nothing -> pw_resolved
                Just pend -> join_patches [pend, pw_resolved]
              removeFile "_darcs/patches/unrevert"
              removeFile "_darcs/patches/unrevert_context"
              putStrLn "Finished unreverting..."
              exitWith ExitSuccess
unrevert_cmd _ _ = impossible
\end{code}

\begin{code}
unrevert_patch_bundle :: IO PatchSet
unrevert_patch_bundle = do
  c <- readFilePS "_darcs/patches/unrevert_context"
       `catchall` fail "There's nothing to unrevert!"
  pf <- readFilePS "_darcs/patches/unrevert"
        `catchall` fail "There's nothing to unrevert!"
  return [(patchinfo "now" "unreversion" "me" [], fst `liftM` readPatchPS pf)
          : zip (get_context c) nothings]

nothings :: [Maybe a]
nothings = Nothing : nothings
get_context :: PackedString -> [PatchInfo]
get_context ps =
    case readPatchInfoPS ps of
    Just (pinfo,r') ->
        case get_context r' of
        pis -> pinfo:pis
    Nothing -> []
\end{code}

