%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs pull}
\begin{code}
module Pull ( pull, merge_with_us_and_pending, save_patches,
            ) where
import System ( ExitCode(..), exitWith )
import Monad ( when, liftM )
import List ( elem, nub, sort )
import Maybe ( catMaybes )
import Control.Exception ( block )

import DarcsCommands ( DarcsCommand(..) )
import DarcsArguments ( DarcsFlag( AnyOrder, Test, Verbose, DryRun ),
                        want_external_merge, nocompress, ignoretimes,
                        no_deps, use_external_merge,
                        match_several,
                        all_gui_interactive,
                        verbose, test, dry_run,
                      )
import Repository ( slurp_recorded, slurp_pending, add_to_inventory,
                    write_patch, get_unrecorded, read_repo, am_in_repo,
                    is_repo, sift_for_pending, write_pending, read_pending,
                    sync_repo,
                  )
import Patch ( Patch, join_patches, merge, patch2patchinfo,
               unjoin_patches, apply_to_slurpy, list_touched_files,
               invert,
             )
import PatchInfo ( PatchInfo, human_friendly )
import SlurpDirectory ( slurp_write_dirty, wait_a_moment, co_slurp )
import RepoPrefs ( lastrepo, set_lastrepo, get_preflist )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution, external_resolution )
import Lock ( withLock )
import SelectChanges ( with_selected_changes )
import DarcsUtils ( putStrError, putStrLnError, withCurrentDirectory )
import Test ( test_slurpy )
#include "impossible.h"
\end{code}
\begin{code}
pull_description :: String
pull_description =
 "Pull patches from another repo."
\end{code}

\options{pull}

\haskell{pull_help}
\begin{code}
pull_help :: String
pull_help =
 "Pull is used to bring changes made in another repo into the current repo\n"++
 "(that is, the one that is the current directory).  Pull allows you to bring\n"++
 "over all or some of the patches that are in that repo but not in the\n"++
 "current one.  Pull accepts an argument, which is the URL from which to\n"++
 "pull, and when called without an argument, pull will use the repository\n"++
 "from which you have most recently either pushed or pulled.\n"
\end{code}
\begin{code}
pull :: DarcsCommand
pull = DarcsCommand {command_name = "pull",
                     command_help = pull_help,
                     command_description = pull_description,
                     command_extra_args = 1,
                     command_extra_arg_help = ["[REPOSITORY]"],
                     command_command = pull_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = get_preflist "repos",
                     command_argdefaults = lastrepo,
                     command_darcsoptions = [verbose,
                                             match_several,
                                             all_gui_interactive,
                                             use_external_merge,nocompress,
                                             test, dry_run,
                                             ignoretimes,no_deps]}
\end{code}
\begin{code}
pull_cmd :: [DarcsFlag] -> [String] -> IO ()
pull_cmd opts [repodir] = withLock "./_darcs/lock" $ do
  am_verbose <- return $ Verbose `elem` opts
  repovalid <- is_repo repodir
  when (not repovalid) $ fail $ "bad repo directory: "++repodir
  old_lastrepo <- lastrepo []
  set_lastrepo repodir
  when (old_lastrepo == [repodir]) $ putStr $ "Pulling from "++repodir++"...\n"
  them <- read_repo repodir
  us <- read_repo "."
  case get_common_and_uncommon (us, them) of
    (_, us', them') -> do
     when am_verbose $ putStr $
         "We have the following new (to them) patches:\n"++
         (unlines $ map (human_friendly.fst) $ head us')
     when am_verbose $ putStr $
         "They have the following patches to pull:\n"++
         (unlines $ map (human_friendly.fst) $ head them')
     when (them' == [[]]) $ do putStr "No remote changes to pull in!\n"
                               exitWith ExitSuccess
     s <- slurp_recorded "."
     with_selected_changes "pull" opts s (map fromJustPatch $ reverse $ head them') $
      \ (_,to_be_pulled) -> do
      when (DryRun `elem` opts) $ do
          putStr $
               "Would pull the following changes:\n"++
               (unlines $ map (human_friendly.fromJust.patch2patchinfo) to_be_pulled)
          putStr "Making no changes:  this is a dry run.\n"
          exitWith ExitSuccess
      when (to_be_pulled == []) $ do
          putStr "You don't want to pull any patches, and that's fine with me!\n"
          exitWith ExitSuccess
      when am_verbose $ putStr "Getting and merging the following patches:\n"
      when am_verbose $ putStr $ format_patches_inventory to_be_pulled
      (pc,pw) <- merge_with_us_and_pending opts
                 (map fromJustPatch $ reverse $ head us', to_be_pulled)
      standard_resolved_pw <- standard_resolution pw
      case nub $ sort $ list_touched_files $
           join_patches $ tail standard_resolved_pw of
        [] -> return ()
        cfs -> do putStr $ "We have conflicts in the following files:\n"
                  putStrLn $ unwords cfs
      recorded <- slurp_recorded "."
      recorded_with_pending <- slurp_pending "."
      working <- co_slurp recorded_with_pending "."
      pw_resolved <-
          case want_external_merge opts of
          Nothing -> return $ join_patches standard_resolved_pw
          Just c -> do pend <- get_unrecorded (AnyOrder:opts)
                       join_patches `liftM` external_resolution c working
                            (join_patches $ (++catMaybes [pend]) $
                             map fromJustPatch $ reverse $ head us')
                            (join_patches to_be_pulled) pw
      when am_verbose $ putStr "Applying patches to the local directories...\n"
      case apply_to_slurpy pc recorded of
        Nothing -> do putStrLnError "Error applying patch to recorded!"
                      putStrError $ "The patch was:\n"++ show pc
                      exitWith $ ExitFailure 1
        Just rec' ->
          case apply_to_slurpy pw_resolved working of
          Nothing -> do putStrLnError "Error applying patch to working dir."
                        putStrLnError $ "Current patch is:\n"++show pc
                        putStrLnError $ "Working patch is:\n"++show pw_resolved
                        exitWith $ ExitFailure 1
          Just work' -> do
              when (Test `elem` opts) $
                   do recb <- slurp_recorded "."
                      testproblem <- test_slurpy opts $
                                     fromJust $ apply_to_slurpy pc recb
                      when (testproblem /= ExitSuccess) $ do
                          putStrLnError "Error in test..."
                          exitWith $ ExitFailure 1
              save_patches opts $ unjoin_patches pc
              mp <- read_pending
              block $ do
                  withCurrentDirectory "_darcs/current" $ do
                      slurp_write_dirty rec'
                      wait_a_moment -- so work will be more recent than rec
                      return ()
                  sequence $ map (add_to_inventory ".".fromJust.patch2patchinfo)
                           to_be_pulled
                  slurp_write_dirty work'
                  when (mp /= Nothing) $ write_pending $ sift_for_pending $
                                         join_patches [invert pc, pw_resolved]
              sync_repo
              putStr $ "Finished pulling.\n"
pull_cmd _ _ = impossible
format_patches_inventory :: [Patch] -> String
format_patches_inventory ps =
    unlines $ map (show.fromJust.patch2patchinfo) ps

fromJustPatch :: (PatchInfo, Maybe Patch) -> Patch
fromJustPatch (pinfo, Nothing)
    = error $ "Error reading patch:\n" ++ human_friendly pinfo
fromJustPatch (_, Just p) = p
\end{code}

You can use an external interactive merge tool to resolve conflicts via the
flag \verb!--external-merge!.  For more details see
section~\ref{resolution}.

The \verb!--patch-name! argument can be used to specify a regexp, which
should be of the extended type used by \verb!egrep!.  If this option is
used, only patches which match this regexp (along with their dependencies)
are considered.  Similarly, \verb!--tag-name! can be used along with a
regexp to pull all patches which are in versions with a matching tag.

If you give a \verb!--patch-name! argument, darcs will silently pull along
any other patches upon which the patches which match the patch-name depend.
So \verb!--patch-name bugfix! mean ``pull all the patches with `bugfix' in
their name, along with any patches they require.''  If you really only want
the patches with `bugfix' in their name, you should use the
\verb!--no-deps! option, which is only useful in combination with
\verb!--patch-name!, and makes darcs only pull in those matching patches
which have no dependencies (apart from other matching patches).

If you specify the \verb!--test! option, pull will run the test (if a test
exists) on a scratch copy of the repo contents prior to actually performing
the pull.  If the test fails, the pull will be aborted.

\begin{code}
save_patches :: [DarcsFlag] -> Maybe [Patch] -> IO ()
save_patches _ (Just []) = return ()
save_patches _ Nothing = return ()
save_patches opts (Just (p:ps)) = do write_patch opts p
                                     save_patches opts $ Just ps
\end{code}

\begin{code}
merge_with_us_and_pending :: [DarcsFlag] -> ([Patch],[Patch]) ->
                             IO (Patch, Patch)
merge_with_us_and_pending opts (us,them) =
  case (join_patches us, join_patches them) of
  (usp, themp) ->
      case merge (themp, usp) of
      Nothing -> fail "There was a bug in merging... giving up!"
      Just (themp',_) -> do
         putStr "So far so good... finished merging.\n"
         past_pending <- merge_with_pending opts themp'
         return (themp', past_pending)
merge_with_pending :: [DarcsFlag] -> Patch -> IO Patch
merge_with_pending opts p = do
  pend <- get_unrecorded (AnyOrder:opts) -- we don't care if it looks pretty...
  case pend of
    Nothing -> return p
    Just pendp ->
      case merge (p,pendp) of
      Nothing -> fail "Bug in merging with pending..."
      Just (p',_) -> return p'
\end{code}

