/*
 * Copyright 2004-2007 J. Dahl and L. Vandenberghe.
 *
 * This file is part of CVXOPT version 0.9.
 *
 * CVXOPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * CVXOPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "cvxopt.h"
#include "misc.h"
#include <fftw3.h>

PyDoc_STRVAR(fftw__doc__,
    "Interface to the FFTW3 library.\n");

extern void zscal_(int *n, complex *alpha, complex *x, int *incx);
extern void dscal_(int *n, double *alpha, double *x, int *incx);
        
static char doc_dft[] = 
    "DFT of a matrix,  X := dft(X)\n\n" 
    "PURPOSE\n"
    "Computes the DFT of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'z'.";


static PyObject *dft(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  char *kwlist[] = {"X", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O", kwlist, &X)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == COMPLEX))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'z'");

  int m = X->nrows, n = X->ncols; 
  if (m == 0) return Py_BuildValue("");

  fftw_plan p = fftw_plan_many_dft(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      FFTW_FORWARD, FFTW_ESTIMATE);

  fftw_execute(p);   
  fftw_destroy_plan(p);  
  return Py_BuildValue("");
}

static char doc_idft[] = 
    "IDFT of a matrix,  X := idft(X)\n\n" 
    "PURPOSE\n"
    "Computes the inverse DFT of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'z'.";

static PyObject *idft(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  char *kwlist[] = {"X", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O", kwlist, &X)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == COMPLEX))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'z'");

  int m = X->nrows, n = X->ncols; 
  if (m == 0) return Py_BuildValue("");

  fftw_plan p = fftw_plan_many_dft(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      FFTW_BACKWARD, FFTW_ESTIMATE);
  
  fftw_execute(p); 
  
  number a;
  a.z = 1.0/m;
  int mn = m*n, ix = 1;
  zscal_(&mn, &a.z, MAT_BUFZ(X), &ix);
  
  fftw_destroy_plan(p);
  return Py_BuildValue("");  
}

static char doc_dct[] = 
    "DCT of a matrix.\n"
    "X := dct(X, type=2)\n\n" 
    "PURPOSE\n"
    "Computes the DCT of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'd'.\n\n"
    "type      integer from 1 to 4; chooses either DCT-I, DCT-II, \n"
    "          DCT-III or DCT-IV.";

static PyObject *dct(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  int type = 2;
  char *kwlist[] = {"X", "type", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O|i", kwlist, &X, &type)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == DOUBLE))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'd'");
   
  int m = X->nrows, n = X->ncols;
  if (m == 0) return Py_BuildValue("");

  fftw_r2r_kind kind;
  switch(type) {
  case 1: 
    kind = FFTW_REDFT00; 
    if (m <= 1) PY_ERR(PyExc_ValueError, "m must be greater than 1 for DCT-I");
    break;
  case 2: kind = FFTW_REDFT10; break;
  case 3: kind = FFTW_REDFT01; break;
  case 4: kind = FFTW_REDFT11; break;
  default: PY_ERR(PyExc_ValueError, "type must be between 1 and 4");
  }
  fftw_plan p = fftw_plan_many_r2r(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      &kind, FFTW_ESTIMATE);

  fftw_execute(p);   
  fftw_destroy_plan(p);  
  return Py_BuildValue("");
}

static char doc_idct[] = 
    "IDCT of a matrix.\n"
    "X := idct(X, type=2)\n\n" 
    "PURPOSE\n"
    "Computes the IDCT of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'd'.\n\n"
    "type      integer from 1 to 4; chooses the inverse transform for\n"
    "          either DCT-I, DCT-II, DCT-III or DCT-IV.";

static PyObject *idct(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  int type = 2;
  char *kwlist[] = {"X", "type", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O|i", kwlist, &X, &type)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == DOUBLE))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'd'");
   
  int m = X->nrows, n = X->ncols;
  if (m == 0) return Py_BuildValue("");

  fftw_r2r_kind kind;
  switch(type) {
  case 1: kind = FFTW_REDFT00; 
    if (m <= 1) PY_ERR(PyExc_ValueError, "m must be greater than 1 for DCT-I");
    break;
  case 2: kind = FFTW_REDFT01; break;
  case 3: kind = FFTW_REDFT10; break;
  case 4: kind = FFTW_REDFT11; break;
  default: PY_ERR(PyExc_ValueError, "type must be between 1 and 4");
  }
  fftw_plan p = fftw_plan_many_r2r(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      &kind, FFTW_ESTIMATE);

  fftw_execute(p);   
  
  double a = 1.0/(type == 1 ? MAX(1,2*(m-1)) : 2*m);
  int mn = m*n, ix = 1;
  dscal_(&mn, &a, MAT_BUFD(X), &ix);

  fftw_destroy_plan(p);  
  return Py_BuildValue("");
}

static char doc_dst[] = 
    "DST of a matrix.\n"
    "X := dst(X, type=1)\n\n" 
    "PURPOSE\n"
    "Computes the DST of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'd'.\n\n"
    "type      integer from 1 to 4; chooses either DST-I, DST-II, \n"
    "          DST-III or DST-IV.";

static PyObject *dst(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  int type = 1;
  char *kwlist[] = {"X", "type", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O|i", kwlist, &X, &type)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == DOUBLE))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'd'");
   
  int m = X->nrows, n = X->ncols;
  if (m == 0) return Py_BuildValue("");

  fftw_r2r_kind kind;
  switch(type) {
  case 1: kind = FFTW_RODFT00; break;
  case 2: kind = FFTW_RODFT10; break;
  case 3: kind = FFTW_RODFT01; break;
  case 4: kind = FFTW_RODFT11; break;
  default: PY_ERR(PyExc_ValueError, "type must be between 1 and 4");
  }
  fftw_plan p = fftw_plan_many_r2r(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      &kind, FFTW_ESTIMATE);

  fftw_execute(p);   
  fftw_destroy_plan(p);  
  return Py_BuildValue("");
}

static char doc_idst[] = 
    "IDST of a matrix.\n"
    "X := idst(X, type=1)\n\n" 
    "PURPOSE\n"
    "Computes the IDST of a dense matrix X column by column.\n\n"
    "ARGUMENTS\n"
    "X         A dense matrix of typecode 'd'.\n\n"
    "type      integer from 1 to 4; chooses the inverse transform for\n"
    "          either DST-I, DST-II, DST-III or DST-IV.";

static PyObject *idst(PyObject *self, PyObject *args, PyObject *kwrds)
{
  matrix *X;
  int type = 1;
  char *kwlist[] = {"X", "type", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, kwrds, "O|i", kwlist, &X, &type)) 
    return NULL;

  if (!(Matrix_Check(X) && MAT_ID(X) == DOUBLE))     
    PY_ERR(PyExc_ValueError, "X must be a dense matrix with type 'd'");
   
  int m = X->nrows, n = X->ncols;
  if (m == 0) return Py_BuildValue("");

  fftw_r2r_kind kind;
  switch(type) {
  case 1: kind = FFTW_RODFT00; break;
  case 2: kind = FFTW_RODFT01; break;
  case 3: kind = FFTW_RODFT10; break;
  case 4: kind = FFTW_RODFT11; break;
  default: PY_ERR(PyExc_ValueError, "type must be between 1 and 4");
  }
  fftw_plan p = fftw_plan_many_r2r(1, &m, n, 
      X->buffer, &m, 1, m,
      X->buffer, &m, 1, m,
      &kind, FFTW_ESTIMATE);

  fftw_execute(p);   
  
  double a = 1.0/(type == 1 ? MAX(1,2*(m+1)) : 2*m);
  int mn = m*n, ix = 1;
  dscal_(&mn, &a, MAT_BUFD(X), &ix);

  fftw_destroy_plan(p);  
  return Py_BuildValue("");
}

static PyMethodDef fftw_functions[] = {
    {"dft", (PyCFunction) dft, METH_VARARGS|METH_KEYWORDS, doc_dft},
    {"idft", (PyCFunction) idft, METH_VARARGS|METH_KEYWORDS, doc_idft},
    {"dct", (PyCFunction) dct, METH_VARARGS|METH_KEYWORDS, doc_dct},
    {"idct", (PyCFunction) idct, METH_VARARGS|METH_KEYWORDS, doc_idct},
    {"dst", (PyCFunction) dst, METH_VARARGS|METH_KEYWORDS, doc_dst},
    {"idst", (PyCFunction) idst, METH_VARARGS|METH_KEYWORDS, doc_idst},
    {NULL}  /* Sentinel */
};


PyMODINIT_FUNC initfftw(void)
{
  PyObject *m;

  m = Py_InitModule3("cvxopt.fftw", fftw_functions, fftw__doc__);

  if (import_cvxopt() < 0) return;
}
