/***************************************************************************
                          blop.cpp  -  description
                             -------------------
    begin                : Thu Jul 20 2000
    copyright            : (C) 2000 by Immi
    email                : cuyo@pcpool.mathematik.uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <cstdlib>

#include "cuyointl.h"
#include "blop.h"
#include "leveldaten.h"
#include "variable.h"
#include "code.h"
#include "blopgitter.h" // nur um zukunftsZuweisung() aufrufen zu knnen...
#include "knoten.h"

#include "cuyo.h" // Um Punkte senden zu knnen
#include "spielfeld.h"







/** Erzeugt einen uninitialisierten Blop (der noch nicht verwendet
    werden kann, aber dafr auch keine Fehler hervorruft, wenn irgend
    was anderes nicht initialisiert ist). Das Uninitialisiert sein
    eines Blops erkennt man daran, dass mDaten 0 ist. */
Blop::Blop(): mFreilaufend(true), mBesitzer(0),
  mX(keine_koord), mY(keine_koord),
  mDaten(0), mDatenAlt(0),
  mZeitNummerDatenAlt(-1),
  mZeitNummerInitStapel(-1),
  /* Uninitialisierte Bildstapel erzeugen. Sonst wird versucht, auf
     noch uninitialisierte Felder von ld zuzugreifen. */
  mBild(false), mBildAlt(false),
  mWillInitEvent(false)
{
}



/** Constructor... s ist SortenNr. */
Blop::Blop(int s):
  mKettenGroesse(1),
  mAmPlatzen(0),
  mFreilaufend(true), mBesitzer(0), mX(keine_koord), mY(keine_koord),
  mDatenLaenge(ld->mLevelKnoten->getDatenLaenge()),
  mZeitNummerDatenAlt(-1),
  mZeitNummerInitStapel(-1),
  mWillInitEvent(false)
{
  mDaten = new int[mDatenLaenge];
  mDatenAlt = new int[mDatenLaenge];
  /* Zustandsdaten auf Default-Werte initialisieren */
  for (int i = 0; i < mDatenLaenge; i++)
    mDaten[i] = ld->mLevelKnoten->getDefaultWert(i);

  mDaten[spezvar_kind] = s;
  
  /* In Zukunft soll das mal nicht mehr mglich sein:
  if (gGleichZeit) {
    fprintf(stderr, "Neuer Blop in Gleichzeit\n");
  }
  */
  
  setWillInitEvent(true);
}




Blop::Blop(const Blop & b): mFreilaufend(true), mBesitzer(0),
  mDatenLaenge(ld->mLevelKnoten->getDatenLaenge()),
  mZeitNummerDatenAlt(-1),
  mZeitNummerInitStapel(-1),
  mWillInitEvent(false)
{
  CASSERT(b.mDaten);
  CASSERT(b.mDatenLaenge == ld->mLevelKnoten->getDatenLaenge());
  mDaten = new int[mDatenLaenge];
  mDatenAlt = new int[mDatenLaenge];
  kopiere(b);
}

Blop::~Blop() {
  setWillInitEvent(false);
  if (mDaten)
    delete[] mDaten;
  if (mDatenAlt)
    delete[] mDatenAlt;
}


/** Tut, was man erwartet. */
Blop & Blop::operator=(const Blop & b) {
  CASSERT(b.mDaten);
  CASSERT(b.mDatenLaenge == ld->mLevelKnoten->getDatenLaenge());
    
  if (!mDaten || mDatenLaenge != ld->mLevelKnoten->getDatenLaenge()) {
    /* Wir wechseln wohl grad in einen neuen Level. Zumindest
       ndert sich unsere Datenlnge. Oder wir waren noch berhaupt
       nicht initialisiert. */
    if (mDaten)
      delete[] mDaten;
    if (mDatenAlt)
      delete[] mDatenAlt;
    mDatenLaenge = ld->mLevelKnoten->getDatenLaenge();
    mDaten = new int[mDatenLaenge];
    mDatenAlt = new int[mDatenLaenge];
  }

  kopiere(b);
  return *this;
}


/** kopiert die Daten von b. Erwartet, dass die Datenlngen schon
    bereinstimmen. */
void Blop::kopiere(const Blop & b) {
  /* Daten kopieren */
  for (int i = 0; i < mDatenLaenge; i++)
    mDaten[i] = b.mDaten[i];
    
  mZeitNummerDatenAlt = b.mZeitNummerDatenAlt;
  if (gGleichZeit && mZeitNummerDatenAlt == gAktuelleZeitNummerDatenAlt) {
    /* mDatenAlt[] wird grad benutzt. Da mssen wir das wohl
       mitkopieren. */
    for (int i = 0; i < mDatenLaenge; i++)
      mDatenAlt[i] = b.mDatenAlt[i];
  }
  
  mZeitNummerInitStapel = b.mZeitNummerInitStapel;
  if (mZeitNummerInitStapel < gAktuelleZeitNummerInitStapel) {
    /* Bildstapel kopieren; aber nur, wenn er nicht sowieso
       frisch initialisiert werden muss. */
    mBild = b.mBild;
  }

  /* BildAlt sollte nach einem Kopieren nicht mehr bentigt werden;
     also brauchen wir es nicht zu kopieren. Aber vielleicht ist
     unser mBildAlt noch von einem vorigen Level, und die Stapelgre
     stimmt noch nicht. Deshalb neuen Bildstapel erzeugen.
     Der neue Stapel hat brigens mAnz = -1, so dass - falls
     ber neumalen entschieden wird - auf jeden Fall neu gemalt wird. */
  mBildAlt = BildStapel();


  mKettenGroesse = b.mKettenGroesse;
  mAmPlatzen = b.mAmPlatzen;

  setWillInitEvent(b.mWillInitEvent);
}



/** Setzt Besitzer und Besitzer-Informationen. Braucht nur am Anfang einmal
    aufgerufen zu werden. Muss auch fr den Global-Blop aufgerufen werden;
    sonst darf kein Code ausgefhrt werden. */
void Blop::setBesitzer(BlopBesitzer * bes /*= 0*/, bool re /*=false*/,
		       int x /*= keine_koord*/,
		       int y /*= keine_koord*/) {
  mFreilaufend = false;
  mBesitzer = bes;
  mRechterSpieler = re;
  mX = x;
  mY = y;
}


/** Setzt die Version-Spezvar des Blops. Wird fr Grasblops am Anfang
    aufgerufen. */
void Blop::setVersion(int v) {
  mDaten[spezvar_version] = v;
}






/** malt den Blop; xx und yy sind in Pixeln angegeben;
    der Hintergrund wird vorher gelscht. */
void Blop::malen(QPainter & p, int xx, int yy) const {
  /* Ggf. gelazyten initStapel()-Aufruf nachholen */
  braucheInitStapel();

  try { // Um die Fehlermeldung zu verbessern...

    CASSERT(mDaten);
    
    //if (mX == 0 && mY == 19)
    //  mBild.print();

    /* Der Bildstapel soll sich geflligst selbst malen */
    mBild.malen(p, xx, yy);

  } catch (Fehler fe) {
    throw Fehler(toString() + ", during drawing:\n" + fe.getText());
  }

} // malen








/** liefert die Art zurck */
int Blop::getArt() const {
  CASSERT(mDaten);
  if (mDaten[spezvar_kind] >= 0)
    return blopart_farbe;
  else
    return mDaten[spezvar_kind];
}


/** liefert die Farbe zurck, aber nur, wenns wirklich ein farbiges
    Blop ist (sonst wird keine_farbe zurckgeliefert) */
int Blop::getFarbe() const {
  CASSERT(mDaten);
  if (mDaten[spezvar_kind] >= 0)
    return mDaten[spezvar_kind];
  else
    return keine_farbe;
}

/** liefert die Version zurck. */
int Blop::getVersion() const {
  CASSERT(mDaten);
  return mDaten[spezvar_version];
}

/** liefert true, wenn der Blop am platzen ist */
bool Blop::getAmPlatzen() const {
  CASSERT(mDaten);
  return mAmPlatzen != 0;
}




/** Fhrt die ganzen Animationen durch (Codeanimationen und platzen).
    Sollte innerhalb einer Gleichzeit aufgerufen werden. */
void Blop::animiere() {
  CASSERT(gGleichZeit);
  CASSERT(!mFreilaufend);
  /* Wenn ein Blop auf ein Init-Event wartet, darf noch nix anderes
     aufgerufen werden. */
  CASSERT(!mWillInitEvent);

  /* Ggf. gelazyten initStapel()-Aufruf nachholen. Man knnte genau
     untersuchen, wann whrend animiere() dieser Aufruf bentigt wird,
     aber irgend wann wird er sowieso in jedem Spielschritt bentigt,
     also macht es keinen Sinn, sich Mhe zu geben. */
  braucheInitStapel();
  
  try { // Um ggf. die Fehlermeldung zu verbessern...

    CASSERT(mDaten);

    /* Erst mal ein paar Initialisierungen. Muss auch gemacht werden,
       wenn der Blop nix von animieren wei, weil der nderungswunsch
       initialisiert werden soll. Und vielleicht wollen ja Nachbarn
       was mit diesem Blop tun. */
    initSchritt();

    
    /* Normale (sorteneigene) Animation */

    /* Wenn die Sorte gewechselt hat, mssen wir evtl. noch ein busy-Reset
       verschicken. */
    if (mDaten[spezvar_kind] != mDaten[spezvar_kind_beim_letzten_draw_aufruf]
        && mDaten[spezvar_kind_beim_letzten_draw_aufruf] != -1) {
       Code * alt_co = ld->mSorten[
                           mDaten[spezvar_kind_beim_letzten_draw_aufruf]
			 ]->getEventCode(event_draw);
       if (alt_co)
         alt_co->busyReset(*this);
    }
    mDaten[spezvar_kind_beim_letzten_draw_aufruf] = mDaten[spezvar_kind];

    
    /* Prfen, ob berhaupt Mal-Code existiert. Das muss bei der
       nix-Sorte nicht sein. */
    Code * mc = getSorte()->getEventCode(event_draw);
    if (mc) {    
      /* Bei diesem eval werden die Bildchen neu in mBild eingefgt...
         Das ist beim Global-Blop aber nicht erlaubt. */
      mMalenErlaubt = mDaten[spezvar_kind] != blopart_global;
      mc->eval(*this);
    }


    /* Debug-Ausgaben schreiben... */
    mBild.setDebugOut(mDaten[spezvar_out1], mDaten[spezvar_out2]);



    /* Animation des Platzens */
    
    /* Zunchst: Leere Blops platzen nicht. (Wenn ein Blop whrend eines
       Cual-Programms mal ganz kurz das leere Blop ist, dann normalerweise
       so kurz, dass die nachfolgenden Zeilen es nicht merken. */
    if (mDaten[spezvar_kind] == blopart_keins)
      mAmPlatzen = 0;
    
    if (mAmPlatzen > 0) {
      mAmPlatzen++;
      if (mAmPlatzen > platzbild_anzahl) {
        /* Letztes Platz-Bildchen. Im nchsten Schritt in
           nichts verwandeln. */
        mDaten[spezvar_kind] = blopart_keins;
	/* Wenn der Blop sich als Gras verhlt, dann soll er das nicht
	   mehr tun, nachdem er geplatzt ist. */
	mDaten[spezvar_goalblob] = 0;
	mAmPlatzen = 0;
      }
    } // if Blop ist am platzen
    
  } catch (Fehler fe) {
    throw Fehler(toString() + ", during animation:\n" + fe.getText());
  }
}


 
/** Liefert true, wenn der Blop sich seit dem letzten Aufruf von
    takeUpdaten() verndert hat und deshalb neu gezeichnet werden muss.
    Liefert auf jeden Fall true, wenn der Blop zwischendrin kopiert wurde
    (mit = oder copy-Constructor). */
bool Blop::takeUpdaten() {
  CASSERT(mDaten);

  /* Ggf. gelazyten initStapel()-Aufruf nachholen */
  braucheInitStapel();

  if (mBild == mBildAlt)
    return false;
  mBildAlt = mBild;
  return true;
}



/** Fhrt den Code des angegebenen Events aus (falls Code zu diesem
    Event existiert. Sollte innerhalb einer Gleichzeit aufgerufen
    werden.
    Die Event-Nummern sind in sorte.h definiert. */
void Blop::execEvent(int evtnr) {

  try { // Um die Fehlermeldung zu verbessern...

    CASSERT(gGleichZeit);

    CASSERT(!mFreilaufend);
    /* Wenn ein Blop auf ein Init-Event wartet, darf noch nix anderes
       aufgerufen werden. (mWillInitEvent wird _vor_ der Event-Versendung
       auf false gesetzt.) */
    CASSERT(!mWillInitEvent);

    CASSERT(mDaten);

    Code * ec = getSorte()->getEventCode(evtnr);
    if (ec) {
      /* Whrend Events sollen Mal-Versuche zu throw fhren. */
      mMalenErlaubt = false;
      ec->eval(*this);
    }

  } catch (Fehler fe) {
    throw Fehler(_("%s, during %s event:\n%s"),
              toString().data(), cEventNamen[evtnr], fe.getText().data());
  }
}



/** Startet den Platzvorgang. Sollte nicht fr leere Blops aufgerufen
    werden. */
void Blop::lassPlatzen() {
  CASSERT(mDaten);
  if (mDaten[spezvar_kind] == blopart_keins)
    throw Fehler(_("Empty Blops are not allowed to explode."));
  if (mAmPlatzen == 0)
    mAmPlatzen = 1;
}


/** Teilt einem Farbblop die (neue) Gre seiner Kette mit. */
void Blop::setKettenGroesse(int anz) {
  CASSERT(mDaten);
  CASSERT(mDaten[spezvar_kind] >= 0);

  /* Kettengre abspeichern, fr den Fall, dass sie als spezconst
     abgefragt wird */
  mKettenGroesse = anz;
}




/***** Funktionen, die nix verndern *****/




/** Liefert true, wenn wir uns mit b verbinden wollen. Auch hier drfen
    wir allerdings nur die alten Var-Werte von b lesen. Vermutlich liefert
    es das beste Ergebnis, wenn wir von uns auch den alten Wert nehmen. */
bool Blop::verbindetMit(const Blop & b) const {
  CASSERT(mDaten);
  return getVariableVergangenheit(spezvar_kind) ==
       b.getVariableVergangenheit(spezvar_kind);
}


/** Liest die aktuelle Sorte dieses Blops aus den Leveldaten aus. */
Sorte * Blop::getSorte() const {
  return ld->mSorten[mDaten[spezvar_kind]];
}

/** Fragt beim Besitzer an, in welche Richtungen dieser Blop
    verbunden werden kann und liefert das zurck. */
int Blop::getVerbindungen() const {
  CASSERT(mDaten);
  if (mBesitzer) {
    return mBesitzer->getBesitzVerbindungen(mX, mY);
  } else
    return verbindung_solo;
}

/** liefert true, wenn sich der Blop auch mit dem angegebenen
    Rand verbindet */
bool Blop::verbindetMitRand(int seite) const {
  CASSERT(mDaten);
  return getSorte()->getVerbindetMitRand(seite);
}

/** Liefert zurck, wie viele Punkte dieser Stein zur Kettengre
    beitrgt (normalerweise 1). */
int Blop::getKettenBeitrag() const {
  CASSERT(mDaten);
  CASSERT(mDaten[spezvar_kind] >= 0);
  if (getSorte()->getKetteZaehltNurInnen()) {
    return (getVerbindungen() & verbindung_alle4) == verbindung_alle4;
  } else
    return 1;
}


/** Liefert einen String der Art "Blop Drachen at x,y".
    Fr Fehlermeldungen. */
__String Blop::toString() const {
  __String ret1, ret2;
  ret1.sprintf(_("Blop %s"), getSorte()->getName().data());

  if (mY == keine_koord) {
    ret2.sprintf(_("%s"), ret1.data());
  } else {
    ret2.sprintf(_("%s at %d,%d"), ret1.data(), mX, mY);
  }
  return ret2;
}


/** Wird vom Code aufgerufen, wenn es Punkte geben soll. */
void Blop::bekommPunkte(int pt) const {
  if (getArt() == blopart_global)
    throw Fehler(_("bonus() does not work in the global blob."));
  Cuyo::gCuyo->neuePunkte(mRechterSpieler, pt);
}

/** Wird vom Code aufgerufen, wenn ein Message angezeigt werden soll. */
void Blop::zeigMessage(__String mess) const {
  if (getArt() == blopart_global)
    throw Fehler(_("message() does not work in the global blob."));
  Cuyo::gCuyo->getSpielfeld(mRechterSpieler)->setMessage(mess);
}



/*************************************************************/


/* Macht alle Initialisierungen, die vor jedem Schritt gemacht
   werden. */
void Blop::initSchritt() {
  /* Erst mal die alten Variablen-Werte merken...
     (Eigentlich wre es ja sauberer, mDaten nicht direkt zu ndern,
     sondern setVariable() zu verwenden. Das wrde dann selbst
     merkeAlteVarWerte() aufrufen. Aber egal.) */
  merkeAlteVarWerte();
  
  mDaten[spezvar_file] = 0;
  mDaten[spezvar_pos] = 0;
  mDaten[spezvar_quarter] = viertel_alle;
  mDaten[spezvar_out1] = spezvar_out_nichts;
  mDaten[spezvar_out2] = spezvar_out_nichts;
}



/* Default-Werte werden hier bentigt... */
extern int spezconst_default[spezconst_anz];


int Blop::getSpezConst(int vnr) const {
  CASSERT(mDaten);
  switch (vnr) {
    case spezconst_connect:
      return getVerbindungen();
     case spezconst_size:
       return mKettenGroesse;
     case spezconst_loc_x:
       /* Wenn die *Y*-Koordinate keine_koord ist, gehre ich
          vermutlich dem Fall. */
       if (mY == keine_koord) break;
       return ld->mSpiegeln ? grx - 1 - mX : mX;
     case spezconst_loc_y:
       if (mY == keine_koord) break;
       return ld->mSpiegeln ? gry - 1 - mY : mY;
     case spezconst_loc_p:
       /* Hier sollte noch eine Fehlermeldung ausgegeben werden, wenn man
          das vom Global-Blop aus abfragen will. */
       return mRechterSpieler ? 2 : 1;
     case spezconst_players:
       return ld->mSpielerZahl;
     case spezconst_exploding:
       return mAmPlatzen;
  };

  /* Nicht selbst bearbeitet? Dann nach oben weitergeben. */
  if (mBesitzer) {
    
    /* Wenn der Besitzer der Fall ist, will der wissen, welcher der
       beiden Blops ich bin... */
    if (mY == keine_koord && mX == 1)
      vnr += spezconst_bin_1;
    
    int r = mBesitzer->getSpezConst(vnr);
    if (r != spezconst_defaultwert)
      return r;
  }

  /* Wir haben keinen Besitzer oder der Besitzer weigert sich, uns 
     was mitzuteilen? Dann Default-Wert. */
  return spezconst_default[vnr];
}


/** High-Level: Wird benutzt, wenn eine Variable im cual-Programm steht.
    Kmmert sich auch um all das @()-Zeug und die Zeitverzgerung. */
int Blop::getVariable(const Variable & v) const {
  switch (v.getOrt()) {
  case variable_lokal:
    return getVariable(v.getNummer());
    
  case variable_global:
    return gGlobalBlop.getVariableVergangenheit(v.getNummer());
    
  case variable_relativ: {
    int xx = v.getX(mX, mY);
    int yy = v.getY(mX, mY);

    if (mBesitzer && mBesitzer->koordOK(xx, yy))
      return mBesitzer->getFeld(xx, yy).
               getVariableVergangenheit(v.getNummer());
    else {
      /* An den angegebenen Koordinaten relativ zu diesem Blop
         gibt's grad nix. Dann default zurckliefern. */
      return v.getDefaultWert();
    }
  }

  case variable_fall:
    return mBesitzer->getFall(v.getWelcherFall())->getVariable(v.getNummer());

  default:
    CASSERT(false);
  }
  return 0; // nur, um keine Warnung zu kriegen.
}




/** High-Level: Wird benutzt, wenn eine Variable im cual-Programm steht.
    Kmmert sich auch um all das @()-Zeug und die Zeitverzgerung.
    Und um irgend welche Range-Checks u. . */
void Blop::setVariable(const Variable & v, int wert, int op) {
  /* Da will doch nicht jemand etwa eine Spezconst ndern?! */
  if (v.getNummer() < 0)
    throw Fehler(_("Variable \"%s\" is read-only."),
                 v.toString().data());
  /* Range-Check bei Spez-Vars */
  /* file, pos und qu werden erst beim Malen gecheckt. version braucht
     gar nicht gecheckt zu werden. out1 und out2 auch nicht. */
  switch (v.getNummer()) {
    case spezvar_kind:
      if (wert < blopart_min_cual || wert >= ld->mAnzFarben)
        throw Fehler(_("Variable kind (=%d) out of range (allowed: %d - %d)"),
	             wert, blopart_min_cual, ld->mAnzFarben - 1);
      break;
  }
  
  
  switch (v.getOrt()) {
  case variable_lokal:
    setVariable(v.getNummer(), wert, op);
    break;
  case variable_global:
    gGlobalBlop.setVariableZukunft(v.getNummer(), wert, op);
    break;
  case variable_relativ: {
    int xx = v.getX(mX, mY);
    int yy = v.getY(mX, mY);
    if (mBesitzer && mBesitzer->koordOK(xx, yy)) {
      mBesitzer->getFeld(xx, yy).
        setVariableZukunft(v.getNummer(), wert, op);
    }
    break;
  }
  case variable_fall:
    mBesitzer->getFall(v.getWelcherFall())
      ->setVariable(v.getNummer(), wert, op);
    break;
  default:
    CASSERT(false);
  }
}



/** Low-Level; wird von den High-Level-Funktionen aufgerufen und vom
    Cual-Programm bei internen Variablen.
    Achtung: Fremdblops sollten *immer* die zeitverschobenen Versionen
    benutzen. */
int Blop::getVariable(int vnr) const {
  CASSERT(mDatenLaenge == ld->mLevelKnoten->getDatenLaenge());
  
  if (vnr < 0)
    return getSpezConst(vnr);
  
  CASSERT(vnr < ld->mLevelKnoten->getDatenLaenge());
  return mDaten[vnr];
}




void Blop::setVariable(int vnr, int wert, int op) {
  merkeAlteVarWerte();
  setVariableIntern(vnr, wert, op);
}


/** Noch low-Levler: Speichert alte Werte nicht ab, wie sich das
    innerhalb einer Gleichzeit gehren wrde. */
void Blop::setVariableIntern(int vnr, int wert, int op) {
  CASSERT(mDatenLaenge == ld->mLevelKnoten->getDatenLaenge());
  CASSERT(vnr >= 0 && vnr < ld->mLevelKnoten->getDatenLaenge());
  
  switch (op) {
    case set_code: mDaten[vnr] = wert; break;
    case add_code: mDaten[vnr] += wert; break;
    case sub_code: mDaten[vnr] -= wert; break;
    case mul_code: mDaten[vnr] *= wert; break;
    case div_code: mDaten[vnr] /= wert; break;
    case mod_code: mDaten[vnr] %= wert; break;
    default:
      throw iFehler(_("Internal error in Blop::setVariable() (op = %d)"),
                   op);
  }
}







/** Zeitverschobener Variablenzugriff: Fremdblops sollten immer diese
    Routinen verwenden. */
/** Liefert den Wert der Variable zum Anfang der Gleichzeit zurck. */
int Blop::getVariableVergangenheit(int vnr) const {
  CASSERT(mDatenLaenge == ld->mLevelKnoten->getDatenLaenge());
  
  if (vnr < 0)
    return getSpezConst(vnr);
  
  CASSERT(vnr < ld->mLevelKnoten->getDatenLaenge());
  
  /* Wenn unsere Variablen in dieser Gleichzeit noch nicht gendert wurden,
     wurden sie auch noch nicht nach mDatenAlt kopiert. Dann den Wert aus
     mDaten[] holen. */
  if (mZeitNummerDatenAlt == gAktuelleZeitNummerDatenAlt)
    return mDatenAlt[vnr];
  else
    return mDaten[vnr];
}

/** Setzt die Variable am Ende der Gleichzeit. */
void Blop::setVariableZukunft(int vnr, int wert, int op) {
  CASSERT(gGleichZeit);

  if (gZZAnz >= (int) gZZ.size()) {
    /* Nicht genug Speicher im ZZ-Array? Dann vergrern. */
    gZZ.resize(gZZ.size() * 2 + 16);
  }
  
  gZZ[gZZAnz++] = tZZ(this, vnr, wert, op);
}







/** Speichert, falls ntig, die Variablenwerte in mDatenAlt, fr
    zeitverzgerten Variablenzugriff. Falls ntig bedeutet: Falls
    sie in dieser Gleichzeit noch nicht gespeichert wurden.
    Wird von set[Bool]Variable(vnr) aufgerufen, bevor eine Variable
    gendert wird. */
void Blop::merkeAlteVarWerte() {
  CASSERT(gGleichZeit);
  if (mZeitNummerDatenAlt < gAktuelleZeitNummerDatenAlt) {
    for (int i = 0; i < mDatenLaenge; i++)
      mDatenAlt[i] = mDaten[i];
    mZeitNummerDatenAlt = gAktuelleZeitNummerDatenAlt;
  }
}



/** Schaut, ob noch ein Lazy-Evaluation-initStapel()-Aufruf aussteht
    und fhrt ihn ggf. aus. (Siehe lazyInitStapel().)
    Ist zwar eigentlich nicht wirklich const, wird aber in Situationen
    aufgerufen, wo man sich eigentlich const fhlen mchte. */
void Blop::braucheInitStapel() const {
  if (mZeitNummerInitStapel < gAktuelleZeitNummerInitStapel) {
    ((BildStapel &) mBild).initStapel(mAmPlatzen);
    (long &) mZeitNummerInitStapel = gAktuelleZeitNummerInitStapel;
  }
}



/** Speichert das aktuelle Bild (d. h. aus den spezvar file und pos)
    in die Mal-Liste */
void Blop::speichereBild() {
  CASSERT(mDaten);

  /* Ggf. gelazyten initStapel()-Aufruf nachholen. Eigentlich unntig,
     da das schon am Anfang von animiere() passiert sein sollte. Ich
     fhl' mich aber wohler. */
  braucheInitStapel();

  if (!mMalenErlaubt)
    throw Fehler(_("Drawing is not allowed at the moment."));
  mBild.speichereBild(getSorte(), mDaten[spezvar_file], mDaten[spezvar_pos],
     mDaten[spezvar_quarter]);
}

/** Speichert das aktuelle Bild (d. h. aus den spezvar file und pos)
    in die Mal-Liste von einem anderen Blop, und zwar so, dass es
    erst zum Schluss gemalt wird. */
void Blop::speichereBildRelativ(int xx, int yy) {
  CASSERT(mDaten);
  if (!mMalenErlaubt)
    throw Fehler(_("Drawing is not allowed at the moment."));

  /* Die Koordinatenberechnung ist aus Variable::getX / getY abgeschrieben
     und sollte irgendwo zentral sein. */
  int x = mX + xx;
  int y = mY + (ld->mSpiegeln ? -yy : yy)
	              - (ld->mSechseck && (mX & 1) && (xx & 1));

  if (mBesitzer && mBesitzer->koordOK(x, y)) {
    Blop & b = mBesitzer->getFeld(x, y);
    /* Ggf. gelazyten initStapel()-Aufruf nachholen. An dieser Stelle
       ist das wichtig, weil der bemalte Blop evtl. noch nicht animiert()
       wurde. */
    b.braucheInitStapel();

    /* letztes true sagt, dass das Bild erst am Ende gemalt werden soll. */
    b.mBild.speichereBild(
          getSorte(), mDaten[spezvar_file], mDaten[spezvar_pos],
          mDaten[spezvar_quarter], true);
  }
}



/********** Init-Event-Verwaltung **********/

/** Setzt, ob dieser Blop ein Init-Event will. */
void Blop::setWillInitEvent(bool wie) {
  if (mWillInitEvent == wie)
    return;
  mWillInitEvent = wie;
  
  if (mWillInitEvent) {
    mWIEPos = gWIEListe.size();
    gWIEListe.resize(mWIEPos + 1);
    gWIEListe[mWIEPos] = this;
    
  } else {
    gWIEListe[mWIEPos] = 0;
  }
}


QArray<Blop *> Blop::gWIEListe;

/** Ruft alle noch fehlenden Init-Events aus. */
void Blop::sendeFehlendeInitEvents() {
  beginGleichzeitig();
  for (int i = 0; i < (int) gWIEListe.size(); i++)
    if (gWIEListe[i]) {
      gWIEListe[i]->mWillInitEvent = false;
      gWIEListe[i]->execEvent(event_init);
    }
  gWIEListe.resize(0);
  endGleichzeitig();
}




/********** Statisches Zeug **********/

bool Blop::gGleichZeit = false;
long Blop::gAktuelleZeitNummerDatenAlt = 0;
long Blop::gAktuelleZeitNummerInitStapel = 0;
/** Liste Zukunfts-Zuweisungen, die sich whrend einer Gleichzeit
    ansammeln. */
QArray<Blop::tZZ> Blop::gZZ;
int Blop::gZZAnz;
//bool Blop::gInitEventsAutomatisch = true;



/** Wenn es fr den Cual-Programmierer so aussehen soll, als wrden Dinge
    gleichzeitig passieren (d. h. @ greift zeitverzgert zu), dann sollte
    man erst beginGleichzeitig() aufrufen, dann die ganzen Blop-Programm-
    aufrufe, und dann endGleichzeitig().
    */
void Blop::beginGleichzeitig() {
  CASSERT(!gGleichZeit);
  gGleichZeit = true;
  
  gAktuelleZeitNummerDatenAlt++;
  
  /* Bis jetzt gibt's noch keine Zukunftszuweisungen: */
  gZZAnz = 0;
}


/** Siehe beginGleichzeitig(). */
void Blop::endGleichzeitig() {
  CASSERT(gGleichZeit);
  gGleichZeit = false;
  
  /* Zukunftszuweisungen durchfhren */
  for (int i = 0; i < gZZAnz; i++) {
    const tZZ & zz = gZZ[i];
    zz.mBlop->setVariableIntern(zz.mVNr, zz.mWert, zz.mOperation);
  }
}



/** Bricht eine Gleichzeit einfach ab. Wird beim Auftreten von Fehlern
    aufgerufen, und zwar vom Constructor von Fehler(). */
void Blop::abbruchGleichzeitig() {
  gGleichZeit = false;
}


/** Tut so, als wrde es initStapel() fr alle Blops aufrufen (d. h. die
    Grafiken lschen. In Wirklichkeit passiert das mit Lazy-Evaluation,
    d. h. erst dann, wenn's wirklich gebraucht wird.
    Gebraucht wird's natrlich, wenn ein Blop animiert wird. Aber auch,
    wenn ein Nachbarblop etwas auf diesen Blop malt. */
void Blop::lazyInitStapel() {
  /* Wenn ein Blop sieht, dass seine mZeitNummerInitStapel-Variable
     kleiner ist als gAktuelleZeitNummerInitStapel, ruft er initStapel()
     auf. */
  gAktuelleZeitNummerInitStapel++;
}


/** Stellt ein, ob Blops */
// void Blop::setInitEventsAutomatisch(bool iea) {
//   gInitEventsAutomatisch = iea;
// }



/***** Fr globales Animationszeug *****/


/** Der Blop, der ld->mGlobalCode ausfhrt. Wird am Ende von
    LevelDaten::ladLevel() initialisiert. */
Blop Blop::gGlobalBlop;


