
/*
 * Copyright (c) 2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: cursel.m,v 1.111 2002/06/15 19:19:30 stes Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include <setjmp.h>
#include <string.h>
#include <signal.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include <Object.h>
#include "cursel.h"
#include "yacc.h"

#include <curses.h>
#include <menu.h>
#include <form.h>

#include <unistd.h> /* getcwd */

#include <ordcltn.h>
#include <dictnary.h>
#include <ocstring.h>
#include "var.h"
#include "item.h"
#include "slk.h"
#include "field.h"
#include "descrip.h"
#include "frame.h"
#include "menufram.h"
#include "cmdmenu.h"
#include "txtframe.h"
#include "formfram.h"
#include "backq.h"
#include "singleq.h"
#include "doubleq.h"
#include "process.h"
#include "coroute.h"

/* Global Settings */

int DISPLAYH;
int DISPLAYW;
int needcheckworld;
jmp_buf checkworldbuf;
int running;
int colorpairs[CP_MAX];
WINDOW *banwin;
WINDOW *msgwin;
static int slksection;

int sfwrite(int fd,char *s)
{
  if (s) {
    int ok,len;
    len = strlen(s);
    ok  = write(2,s,len);
    assert(ok == len);
  }
}

id g_slk;
id g_columns; /* introductory frame */
id g_rows; /* introductory frame */
id g_text; /* text of introductory frame */
id g_alignment; /* of introductory frame */
id g_title; /* title of introductory frame */
id g_bancol; /* center or integer, column in banner */
id g_banner; /* banner text */
id g_banner_text; /* banner text color*/
id g_working; /* working test */
id g_active_border; /* color of frame border when current */
id g_active_title_bar; /* color of title background when current */
id g_active_title_text; /* color of title text when current */
id g_active_banner_text; /* color of text in banner */
id g_highlight_bar; /* color menu selector bar */
id g_highlight_bar_text; /* color foreground highlight_bar */
id g_inactive_border; /* color frame non-current */
id g_inactive_title_bar; /* color of title background non-current */
id g_inactive_title_text; /* color of title text when non-current */
id g_screen; /* color of screen (screen background) */ 
id g_slk_bar; /* color of background of slk_text */
id g_slk_text; /* color of screen-labeled function key text */
id g_window_text; /* color of text in a frame */  
id g_interrupt; /* boolean, whether actions can be interrupted */
id g_nobang; /* if set, no shell escapes */
id g_oninterrupt; /* action on interrupt */
id g_permanentmsg; /* message line text */
id g_slk_layout; /* 4-4 or 3-2-3 */
id g_toggle; /* always, integer, or never */

char *transientmsg;
id s_center; /* symbol center */
BOOL u_interrupt;
id u_oninterrupt;

void siginthandler(int signo)
{
  if (u_interrupt) {
    if (u_oninterrupt) {
      id c=expandstr(u_oninterrupt);
      if (c) evalcmd(c);
    } else {
      showtransientmessage("Operation interrupted!");
    }
  }
}

int isnobang(void)
{
  return g_nobang != nil && strcasecmp([g_nobang str],"true") == 0;
}

void clearbanner(void)
{
  int i;
  for(i=0;i<DISPLAYW;i++) mvwaddch(banwin,0,i,' ');
}

void showbanner(char *msg)
{
  int i,n,len;
  len = strlen(msg);
  if (g_bancol == nil || !strcasecmp([g_bancol str],"center")) {
    n = (DISPLAYW - len)/2;if (n<0) n = 0; 
  } else {
    n = atoi([g_bancol str]); 
  }
#if HAVE_COLOR
  if (has_colors()) wcolor_set(banwin,colorpairs[CP_BANNER],NULL);
#endif
  for(i=n;i<DISPLAYW && i<n+len;i++) mvwaddch(banwin,0,i,*msg++);
}

int findcolor(char *s)
{
  assert(s);
  if (!strcasecmp(s,"black")) return COLOR_BLACK;
  if (!strcasecmp(s,"white")) return COLOR_WHITE;
  if (!strcasecmp(s,"red")) return COLOR_RED;
  if (!strcasecmp(s,"green")) return COLOR_GREEN;
  if (!strcasecmp(s,"yellow")) return COLOR_YELLOW;
  if (!strcasecmp(s,"blue")) return COLOR_BLUE;
  if (!strcasecmp(s,"magenta")) return COLOR_MAGENTA;
  if (!strcasecmp(s,"cyan")) return COLOR_CYAN;
  assert(0);
}

void defcolorpair(int cp,id fg,id bg)
{
  static int cpcount = 1;
  if (fg == nil && bg == nil) {
    colorpairs[cp] = colorpairs[CP_TEXT];
  } else {
    int fgc,bgc;
    assert(cpcount < COLOR_PAIRS);
    colorpairs[cp] = cpcount;
    fgc=(fg)?findcolor([fg str]):COLOR_WHITE;
    bgc=(bg)?findcolor([bg str]):COLOR_BLACK;
    init_pair(cpcount,fgc,bgc);
    cpcount++;
  }
}

void dobanner(void)
{
  if (g_banner) {
    clearbanner();
    showbanner([g_banner str]);
  }
}

void showworking(BOOL flag,char *msg)
{
  int i,n;
  n = DISPLAYW - strlen(msg);if (n<0) n = 0; 
#if HAVE_COLOR
  if (has_colors()) wcolor_set(banwin,colorpairs[CP_BANNER],NULL);
#endif
  for(i=n;i<DISPLAYW;i++) mvwaddch(banwin,0,i,(flag)?*msg++:' ');
}

void doworking(BOOL flag)
{
  char *s;
  s = (g_working)?[g_working str]:"Working...";
  showworking(flag,s);
}

void refreshbanner(void)
{
  wrefresh(banwin);
}

void showtransientmessage(char *s)
{
  transientmsg=s;
  drawmessage();
}

void drawmessage(void)
{
  id t;
  char *s = "";
  if (g_permanentmsg) s = [g_permanentmsg str];
  if (activeframe!=nil && ((t=[activeframe framemsg]))) { s = [t str]; }
  if (transientmsg) s = transientmsg;
  showmessage(s);
  wrefresh(msgwin);
}

void showmessage(char *s)
{
  if (s) {
   int i,n;
   n=(s)?strlen(s):0;
   dbg("showmsg: %s\n",s);
#if HAVE_COLOR
   if (has_colors()) wcolor_set(msgwin,colorpairs[CP_TEXT],NULL);
#endif
   if (s) mvwaddstr(msgwin,0,0,s);
   for(i=n;i<DISPLAYW;i++) mvwaddch(msgwin,0,i,' ');
   wnoutrefresh(msgwin);
  }
}

id expandstr(id c)
{
  if (c) {
    int i,n;
    id s = [String new];
    for(i=0,n=[c size];i<n;i++) [s concat:[c at:i]];
    return s;
  } else {
    return nil;
  }
} 

int expandint(id c,int def)
{
  id s = expandstr(c); return (s)?atoi([s str]):def;
} 

BOOL expandbool(id c,BOOL def)
{
  id s = expandstr(c); return (s)?strcasecmp([s str],"true")==0:def;
} 

id chomp(id buffer)
{
  int i,n;
  char *s;
  s = [buffer str];
  n = [buffer size]; 
  i = n;
  while (i--) {
    if (s[i] == '\n') break;
  }
  return [String chars:s count:(i)?i:n];
}

id strfromfd(id buffer,int fd)
{
  int n;
  char buf[BUFSIZ+1];
  while ((n = read(fd,buf,BUFSIZ))) {
    buf[n]='\0';[buffer concatSTR:buf]; 
  }
  return buffer;
}

id lookdesc(id cltn,char *s)
{
  int i,n;
  static int cache;

  n = [cltn size];

  for(i=cache;i<n;i++) {
    id d = [cltn at:i];
    /* case insensitive as original FMLI */
    if (!strcasecmp([d str],s)) {cache=i;return [d value];}
    /* don't search below start of items */
    if (!strcasecmp([d str],"name")) break;
  }

  for(i=0;i<n && i<cache;i++) {
    id d = [cltn at:i];
    /* case insensitive as original FMLI */
    if (!strcasecmp([d str],s)) {cache=i;return [d value];}
    /* don't search below start of items */
    if (!strcasecmp([d str],"name")) return nil;
  }

  return nil; 
}

id mkintdesc(int s)
{
  return [[OrdCltn new] add:[String sprintf:"%i"]];
}

id mkdesc(char *s)
{
  return [[OrdCltn new] add:[String str:s]];
}

id expanddesc(id cltn,char *s)
{
  id d = lookdesc(cltn,s); return (d)?expandstr(d):nil;
}

id ffields(id cltn)
{
  int i,n;
  id field,fields;

  slksection = 0;
  fields = [OrdCltn new];

  for(i=0,field=nil,n=[cltn size];i<n;i++) {
    id d = [cltn at:i];
    if (!strcasecmp([d str],"name")) {
      if (field) [fields add:field];
      field = [[Field new] name:[d value]];
      slksection=i;
    }
    if (!strcasecmp([d str],"button")) {field=nil;break;}
    if (!strcasecmp([d str],"nrow")) [field nrow:[d value]];
    if (!strcasecmp([d str],"ncol")) [field ncol:[d value]];
    if (!strcasecmp([d str],"frow")) [field frow:[d value]];
    if (!strcasecmp([d str],"fcol")) [field fcol:[d value]];
    if (!strcasecmp([d str],"rows")) [field rows:[d value]];
    if (!strcasecmp([d str],"columns")) [field columns:[d value]];
    if (!strcasecmp([d str],"rmenu")) [field rmenu:[d value]];
    if (!strcasecmp([d str],"valid")) [field valid:[d value]];
    if (!strcasecmp([d str],"menuonly")) [field menuonly:[d value]];
    if (!strcasecmp([d str],"invalidmsg")) [field invalidmsg:[d value]];
    if (!strcasecmp([d str],"lininfo")) [field lininfo:[d value]];
    if (!strcasecmp([d str],"value")) [field value:[d value]];
    if (!strcasecmp([d str],"inactive")) [field inactive:[d value]];
  }

  [fields add:field];
  return fields;
}


id mitems(id cltn)
{
  int i,n;
  id item,items;

  slksection = 0;
  items = [OrdCltn new];

  for(i=0,item=nil,n=[cltn size];i<n;i++) {
    id d = [cltn at:i];
    if (!strcasecmp([d str],"name")) {
      slksection=i;
      if (item) [items add:item];
      item=[[Item new] name:[d value]];
    }
    if (!strcasecmp([d str],"button")) { item=nil;break; }
    if (!strcasecmp([d str],"action")) [item action:[d value]];
    if (!strcasecmp([d str],"description")) [item description:[d value]];
    if (!strcasecmp([d str],"itemmsg")) [item itemmsg:[d value]];
    if (!strcasecmp([d str],"show")) [item show:[d value]];
    if (!strcasecmp([d str],"selected")) [item selected:[d value]];
    if (!strcasecmp([d str],"lininfo")) [item lininfo:[d value]];
    if (!strcasecmp([d str],"inactive")) [item inactive:[d value]];
    if (!strcasecmp([d str],"interrupt")) [item interrupt:[d value]];
    if (!strcasecmp([d str],"oninterrupt")) [item oninterrupt:[d value]];
  }

  if (item) [items add:item];
  return items;
}

static id mkslk(id name,id button,id action,id interrupt,id oninterrupt)
{
  id slk = [Slk new];
  [slk name:name];
  [slk button:button];
  [slk action:action];
  [slk interrupt:interrupt];
  [slk oninterrupt:oninterrupt];
  return slk;
}

id scrlabkeys(id cltn)
{
  int i,n;
  id slks;
  id name,action,button,interrupt,oninterrupt;

  slks = [OrdCltn new];
  name=nil;button=nil;action=nil;interrupt=nil;oninterrupt=nil;

  for(i=slksection,n=[cltn size];i<n;i++) {
    id d = [cltn at:i];
    if (!strcasecmp([d str],"name")) {
      if (name!=nil && button!=nil) {
        [slks add:mkslk(name,button,action,interrupt,oninterrupt)];
      }
      name=nil;button=nil;action=nil;interrupt=nil;oninterrupt=nil;
      name=expandstr([d value]);
    }
    if (!strcasecmp([d str],"button")) button = expandstr([d value]);
    if (!strcasecmp([d str],"action")) action = [d value];
    if (!strcasecmp([d str],"interrupt")) interrupt = [d value];
    if (!strcasecmp([d str],"oninterrupt")) oninterrupt = [d value];
  }

  if (name!=nil && button!=nil) {
    [slks add:mkslk(name,button,action,interrupt,oninterrupt)];
  }

  return slks;
}

id findframe(char *name)
{
  id f;
  char *s;
  int i,n;
  for (i=0,n=[framelist size];i<n;i++) {
    f = [framelist at:i];
    s = [[f filename] str];
    if (s!=NULL && strcmp(s,name) == 0) return f;
  }
  return nil;
}

extern id yyres;
extern int yyparse();
extern int mylineno;
extern void yyrestart(FILE*);

id pathaliases;

id openinitfile(char *name)
{
  FILE *f;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    return nil;
  }

  g_columns=lookdesc(yyres,"columns");
  g_rows=lookdesc(yyres,"rows");
  g_text=lookdesc(yyres,"text");
  g_alignment=lookdesc(yyres,"alignment");
  g_title=lookdesc(yyres,"title");
  g_bancol=expanddesc(yyres,"bancol");
  g_banner=expanddesc(yyres,"banner");
  g_banner_text=expanddesc(yyres,"banner_text");
  g_working=expanddesc(yyres,"working");
  g_active_border=expanddesc(yyres,"active_border");
  g_active_title_bar=expanddesc(yyres,"active_title_bar");
  g_active_title_text=expanddesc(yyres,"active_title_text");
  g_active_banner_text=expanddesc(yyres,"active_banner_text");
  g_highlight_bar=expanddesc(yyres,"highlight_bar");
  g_highlight_bar_text=expanddesc(yyres,"highlight_bar_text");
  g_inactive_border=expanddesc(yyres,"inactive_border");
  g_inactive_title_bar=expanddesc(yyres,"inactive_title_bar");
  g_inactive_title_text=expanddesc(yyres,"inactive_title_text");
  g_screen=expanddesc(yyres,"screen");
  g_slk_bar=expanddesc(yyres,"slk_bar");
  g_slk_text=expanddesc(yyres,"slk_text");
  g_window_text=expanddesc(yyres,"window_text");
  g_interrupt=expanddesc(yyres,"interrupt");
  g_nobang=expanddesc(yyres,"nobang");
  g_oninterrupt=expanddesc(yyres,"oninterrupt");
  g_permanentmsg=expanddesc(yyres,"permanentmsg");
  g_slk_layout=expanddesc(yyres,"slk_layout");
  g_toggle=expanddesc(yyres,"toggle");
  g_slk=scrlabkeys(yyres);

  return yyres;
}

id opencmdfile(char *name)
{
  FILE *f;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    return nil;
  }

  [cmdmenu setmenu:lookdesc(yyres,"menu")];
  [cmdmenu setrows:lookdesc(yyres,"rows")];
  [cmdmenu setcolumns:lookdesc(yyres,"columns")];
  [cmdmenu setbegrow:lookdesc(yyres,"begrow")];
  [cmdmenu setbegcol:lookdesc(yyres,"begcol")];
  [cmdmenu additems:mitems(yyres)];

  return cmdmenu;
}

id openaliasfile(char *name)
{
  int i,n;
  FILE *f;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    return nil;
  }

  pathaliases = [Dictionary new];
  for(i=0,n=[yyres size];i<n;i++) {
    id e,d = [yyres at:i];
    e = expandstr([d value]);
    [pathaliases atKeySTR:[d str] put:e];
    dbg("defining %s as %s\n",[d str],[e str]);
  }

  return pathaliases;
}

id openmenu(char *name)
{
  int s;
  FILE *f;
  id frame;


  s=running;running=0;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    running=s;
    return nil;
  }

  frame = [Menu new];
  [frame setfilename:[String str:name]];

  [frame setmenu:lookdesc(yyres,"menu")];
  [frame setrows:lookdesc(yyres,"rows")];
  [frame setcolumns:lookdesc(yyres,"columns")];
  [frame setreread:lookdesc(yyres,"reread")];
  [frame setbegrow:lookdesc(yyres,"begrow")];
  [frame setbegcol:lookdesc(yyres,"begcol")];
  [frame setframemsg:lookdesc(yyres,"framemsg")];
  [frame setinit:lookdesc(yyres,"init")];
  [frame setdone:lookdesc(yyres,"done")];
  [frame setclose:lookdesc(yyres,"close")];
  [frame sethelp:lookdesc(yyres,"help")];
  [frame setlifetime:lookdesc(yyres,"lifetime")];
  [frame setmultiselect:lookdesc(yyres,"multiselect")];
  [frame setcyclic:lookdesc(yyres,"cyclic")];
  [frame setinterrupt:lookdesc(yyres,"interrupt")];
  [frame setoninterrupt:lookdesc(yyres,"oninterrupt")];
  [frame items:mitems(yyres)];
  [frame setscrlabkeys:scrlabkeys(yyres)];
  running=s;
  return frame;
}

id openform(char *name)
{
  int s;
  FILE *f;
  id frame;


  s = running;running = 0;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    running=s;
    return nil;
  }

  frame = [Form new];
  [frame setfilename:[String str:name]];

  [frame setform:lookdesc(yyres,"form")];
  [frame setframemsg:lookdesc(yyres,"framemsg")];
  [frame setbegrow:lookdesc(yyres,"begrow")];
  [frame setbegcol:lookdesc(yyres,"begcol")];
  [frame setinit:lookdesc(yyres,"init")];
  [frame setdone:lookdesc(yyres,"done")];
  [frame setclose:lookdesc(yyres,"close")];
  [frame setreread:lookdesc(yyres,"reread")];
  [frame setinterrupt:lookdesc(yyres,"interrupt")];
  [frame setoninterrupt:lookdesc(yyres,"oninterrupt")];
  [frame setfields:ffields(yyres)];
  [frame setscrlabkeys:scrlabkeys(yyres)];

  running=s;
  return frame;
}

id opentext(char *name)
{
  int s;
  FILE *f;
  id frame;


  s = running;running = 0;

  if (f=fopen(name,"r")) {
    mylineno=1;
    yyrestart(f);
    yyparse();
    fclose(f);
  } else {
    running=s;
    return nil;
  }

  frame = [Textframe new];
  [frame setfilename:[String str:name]];

  [frame setedit:lookdesc(yyres,"edit")];
  [frame settitle:lookdesc(yyres,"title")];
  [frame setdone:lookdesc(yyres,"done")];
  [frame setclose:lookdesc(yyres,"close")];
  [frame settext:lookdesc(yyres,"text")];
  [frame setheader:lookdesc(yyres,"header")];
  [frame setalignment:lookdesc(yyres,"alignment")];
  [frame setwrap:lookdesc(yyres,"wrap")];
  [frame setrows:lookdesc(yyres,"rows")];
  [frame setcolumns:lookdesc(yyres,"columns")];
  [frame setbegrow:lookdesc(yyres,"begrow")];
  [frame setbegcol:lookdesc(yyres,"begcol")];
  [frame setframemsg:lookdesc(yyres,"framemsg")];
  [frame setinterrupt:lookdesc(yyres,"interrupt")];
  [frame setoninterrupt:lookdesc(yyres,"oninterrupt")];
  [frame sethelp:lookdesc(yyres,"help")];
  [frame setscrlabkeys:scrlabkeys(yyres)];

  running = s;
  return frame;
}

id openfile(char *name)
{
  char *bname = strrchr(name,'/');
  if (bname) bname++;else bname = name;
  if (strncmp(bname,"Menu.",5)==0) return openmenu(name);
  if (strncmp(bname,"Form.",5)==0) return openform(name);
  if (strncmp(bname,"Text.",5)==0) return opentext(name);
  fatal("file %s not recognized (use naming rules)\n",name);
}

void fatal(char* fmt,...)
{
  va_list ap; 
  fflush(0);
  endwin();
  va_start(ap,fmt);
  vfprintf(stderr,fmt,ap);
  va_end(ap);
  exit(1);
}

int mailchecksec(void)
{
  char *s = getenv("MAILCHECK"); return (s)?atoi(s):60;
}

void alarmhandler(int signo)
{
  stopvsig();
  longjmp(checkworldbuf,1);
}

void die(int signo)
{
  fflush(0);
  endwin();
  printf("Received Signal %i\n",signo);
  exit(1);
}

void checkworld(void)
{
  int i,n;
  needcheckworld=0;
  for(i=0,n=[framelist size];i<n;i++) {
    id f = [framelist at:i];
    if ([f needsreread]) [f update];
  }
  transientmsg=NULL;
}

void displayall(void)
{
  int i,n;
  for(i=0,n=[framelist size];i<n;i++) {
    id f = [framelist at:i];
    if (f != activeframe) [f displayifneeded];
  }
  [activeframe displayifneeded];
  drawmessage();
  doupdate();
}

void redrawall(void)
{
  dobanner();
  [framelist elementsPerform:@selector(needsdisplay)];
  [activeframe needsdisplay];
  [activeframe setslk];
  displayall();
}

void introductory(void)
{
  id frame = [Textframe new];

  [frame settitle:g_title];
  [frame settext:g_text];
  [frame setalignment:g_alignment];
  [frame setrows:g_rows];
  [frame setcolumns:g_columns];
  [frame setbegrow:s_center];
  [frame setbegcol:s_center];

  clear();
  dobanner();
  [frame activate];
  [frame display];
  refresh();
  sleep(1);
}

void defcolors(void)
{
  if (has_colors()) {
    start_color();
    defcolorpair(CP_TEXT,g_window_text,g_screen);
    defcolorpair(CP_BANNER,g_banner_text,g_screen);
    defcolorpair(CP_SLK,g_slk_bar,g_slk_text);
    defcolorpair(CP_MENU,g_highlight_bar_text,g_highlight_bar);
    defcolorpair(CP_FRAME,g_inactive_border,g_screen);
    defcolorpair(CP_TITLE,g_inactive_title_bar,g_inactive_title_text);
    defcolorpair(CP_CURFRAME,g_active_border,g_screen);
    defcolorpair(CP_CURTITLE,g_active_title_bar,g_active_title_text);
    [Var define:"HAS_COLORS" as:[String str:"true"] env:YES];
  } else {
    [Var define:"HAS_COLORS" as:[String str:"false"] env:YES];
  }
}

void exportdisp(void)
{
  [Var define:"DISPLAYW" as:[String sprintf:"%i",DISPLAYW] env:YES];
  [Var define:"DISPLAYH" as:[String sprintf:"%i",DISPLAYH] env:YES];
}

#ifndef NO_RESIZE
int needtoresize;
 
void doresize(void)
{
  struct winsize size;
  if (ioctl(0,TIOCGWINSZ,&size) == 0) {
     resizeterm(size.ws_row,size.ws_col);
     wresize(stdscr,size.ws_row,size.ws_col);
     DISPLAYW=COLS;
     DISPLAYH=LINES-2; /* message */
     exportdisp();
     redrawall();
  }
  needtoresize = 0;
}
#endif

#ifndef NO_RESIZE
void winchanged(int signo)
{
  needtoresize++;
  signal(SIGWINCH,winchanged);
  doresize(); /* potentially unsafe */
}
#endif

void setup(void)
{
  int ok;
  int fmt;

  signal(SIGHUP,die);
  signal(SIGTERM,die);
  signal(SIGINT,siginthandler);
  signal(SIGALRM,alarmhandler); /* BLOCKED unless waiting for key input */
  signal(SIGUSR2,alarmhandler); 
  alarm(mailchecksec());

#ifndef NO_RESIZE
  signal(SIGWINCH,winchanged);
#endif

  fmt = 0;maxfunkey = 8;
  if (g_slk_layout) {
    if (!strcmp([g_slk_layout str],"3-2-3")) {fmt=0;maxfunkey=8;}
    if (!strcmp([g_slk_layout str],"4-4"))   {fmt=1;maxfunkey=8;}
    if (!strcmp([g_slk_layout str],"4-4-4")) {fmt=3;maxfunkey=12;}
  }
  slk_init(fmt);

  initscr();
  defcolors();
  if (has_colors()) wbkgdset(stdscr,COLOR_PAIR(colorpairs[CP_TEXT])|' ');

  DISPLAYW=COLS;
  DISPLAYH=LINES-2; /* message */
  exportdisp();

  [Var define:"VPID" as:[String sprintf:"%d",getpid()] env:YES];

  banwin=subwin(stdscr,1,DISPLAYW,0,0);
  msgwin=subwin(stdscr,1,DISPLAYW,DISPLAYH,0);

  nonl();
  cbreak();
  noecho();
  ok = keypad(stdscr,TRUE);
  assert (ERR != ok);
  running++;
}

void run(void)
{
  int i,n;
  [[framelist lastElement] activate];
  redrawall();
  [activeframe home]; /* go to first selectable */

  while (activeframe) {
    if (needcheckworld || setjmp(checkworldbuf)) {
       checkworld();
       alarm(mailchecksec());
    }
#ifndef NO_RESIZE
    if (needtoresize) doresize();
#endif
    displayall();

    [activeframe getevent];   

  }

  endwin();
}

void usage(void)
{
  fflush(0);
  endwin();
  printf("UX:cursel " VERSION " : ERROR: usage:\n");
  printf("\n");
  printf("cursel [-i init-file] [-c command-file] [-a alias-file] frame-definition-file ..\n");
  printf("\n");
  printf("UX: cursel: INFO: CURSEL is a tool for providing a user interface\n");
  printf("to your application.\n");
  exit(0);
}

int main(int argc,char **argv)
{
  cmdmenu=[CmdMenu new];
  frmmgmtmenu=[CmdMenu newfrmmgmtmenu];
  framelist=[OrdCltn new];
  s_center = [[OrdCltn new] add:[String str:"center"]];

  argc--;argv++;
  if (!argc) usage();
  

  while (argc--) {
    id w;
    char *a = *argv++;

    if (!strcmp(a,"-i")) {
      if (!argc) usage();
      if (!openinitfile(*argv)) {
        fatal("ERROR: Can't open initialization file: \"%s\"\n",*argv);
      }
      argc--;argv++;
      continue;
    }

    if (!strcmp(a,"-c")) {
      if (!argc) usage();
      if (!opencmdfile(*argv)) {
        fatal("ERROR: Can't open command file: \"%s\"\n",*argv);
      }
      argc--;argv++;
      continue;
    }

    if (!strcmp(a,"-a")) {
      if (!argc) usage();
      if (!openaliasfile(*argv)) {
        fatal("ERROR: Can't open alias file: \"%s\"\n",*argv);
      }
      argc--;argv++;
      continue;
    }

    if (!running) {
      setup(); 
      if (g_title) introductory();
    }

    w = openfile(a); 
    if (!w) {
      fatal("ERROR: Can't open frame definition file: \"%s\"\n",a);
    }
    if ([w mustopenframe]) {
      [framelist add:curframe=[w create]];
      [w renumber];
    }
  }

  run(); 
}

int forkshell(char *shell)
{
  int pid,status;
  pid = fork();
  if (pid == -1) {
    return -1;
  }
  if (pid == 0) {
     char *argv[2];
     argv[0] = (shell)?shell:BINSH;
     argv[1] = NULL;
     signal(SIGINT,SIG_DFL);
     [Frame unblockalarm];
     execv(argv[0],argv);
     exit(127);
  }
  do {
    if (waitpid(pid,&status,0) == -1) {
       if (errno != EINTR) return -1;
    } else {
       return status;
    }
  } while (1);
  return 0;
}

void dounixsystem(void)
{
  char buf[PATH_MAX+1];
  if (isnobang()) { beep();return; }

  running=0;endwin();
  system(USRBINCLEAR);
  printf("To return, type 'exit' or control-d\n");
  printf("You are in %s\n",getcwd(buf,PATH_MAX));
  fflush(NULL);
  forkshell(getenv("SHELL"));
  printf("\n");
  printf("Press ENTER to continue");
  fflush(NULL);
  getchar();
  setup();
  redrawall();
}

void doopencommand(id c)
{
  char *t;
  id args;
  int what = C_NONE;
  int i,j,n = [c size];
  if (n <= 1) return;
  j = 1;
  t = [[c at:j] str];
  if (!strcasecmp(t,"menu")) { what = C_MENU;j++; } 
  if (!strcasecmp(t,"form")) { what = C_FORM;j++; } 
  if (!strcasecmp(t,"text")) { what = C_TEXT;j++; } 
  if (j == 2) {
    if (n <= 2) return;
    t = [[c at:j] str];
  }
  args = [OrdCltn new];
  for(i=j+1,n=[c size];i<n;i++) {
    [args add:[c at:i]];
  }
  openframe(t,what,args);
}

id splitwords(id c)
{
  int i,n;
  id token = [String new];
  id tokens = [OrdCltn new];
  for(i=0,n=[c size];i<n;i++) {
    id w = [c at:i];
    if ([w isKindOf:(id)[Doublequote class]] || [w isKindOf:(id)[Singlequote class]]) {
      [token concat:w];
    } else {
      int j,k;
      char *s;
      s = [w str];
      k = strlen(s);
      for(j=0;j<k;j++) {
        if (s[j] == ' ' || s[j] == '\t') {
          [tokens add:token];
          token = [String new];
        } else {
          [token at:[token size] insert:s+j count:1];
        }
      }
    }
  }
  if ([token size]) [tokens add:token];
  return tokens;
}

int doruncommand(id tokens)
{
  int ok;
  int pid;
  char *p;
  int i,j,n;
  BOOL silent = NO;
  BOOL eprompt = NO;
  BOOL noprompt = NO;

  n = [tokens size];
  if (n <= 1) return 0;

  i = 2;
  if ((p=[[tokens at:1] str])) {
    if (!strcmp(p,"-e")) eprompt++;
    else if (!strcmp(p,"-s")) silent++;
    else if (!strcmp(p,"-n")) noprompt++;
    else i = 1;
  }

  if (i > n) return 0; 
  
  signal(SIGINT,SIG_IGN);
  if (!silent) { endwin();system(USRBINCLEAR); fflush(NULL); }

  dbg("run [");for(j=i;j<n;j++) { dbg("%s;",[[tokens at:j] str]); } dbg("]\n");

  pid = fork();
  assert(pid != -1);
 
  if (pid) {
     int status;
     waitpid(pid,&status,0);
     if (WIFEXITED(status)) {
       ok = WEXITSTATUS(status);
     } else {
       ok = -1;
     }
  } else {
     char **argv;

     argv = (char**)malloc(sizeof(char*) * (n+1));
     for(j=0;j<n;j++) {
       argv[j] = [[tokens at:j] str];
     }
     argv[n]=NULL;

     [Frame unblockalarm];
     signal(SIGINT,(u_interrupt)?SIG_DFL:SIG_IGN);

     ok = execvp(argv[i],argv+i);
     if (ok == -1) {
      sfwrite(2,"cursel: ");
      sfwrite(2,argv[0]);
      sfwrite(2,": ");
      sfwrite(2,strerror(errno));
     }
     _exit(0);
  }

  [activeframe setsiginthandler];

  if (!silent) {
    if (!noprompt) {
      if (eprompt == 0 || (eprompt != 0 && ok != 0)) {
        printf("\n");
        printf("Press ENTER to continue");
        fflush(NULL);
        getchar();
      }
    }
    setup(); 
    redrawall();
  }

  return ok;
}

