# curves.rb, copyright (c) 2006 by Vincent Fourmond: 
# The class describing a curve to be plotted.
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/utils'
require 'CTioga/boundaries'
require 'CTioga/elements'

module CTioga

  Version::register_svn_info('$Revision: 760 $', '$Date: 2008-03-02 02:32:05 +0100 (Sun, 02 Mar 2008) $')

  # This structure holds the different properties to be transmitted to
  # a Curve object for appropriate drawing. 

  class CurveStyle
    ELEMENTS = [
                :color, :marker, :marker_color,
                :line_style, :legend, :linewidth,
                :interpolate, :marker_scale, 
                :error_bar_color, :drawing_order,
                :transparency,   # Stroke transparency
                :marker_transparency, 
                :error_bars_transparency,
                # Now, fill information
                :fill_type,     # not false/nil => there will be
                # some filling. See Curve2D#draw_fill for more info
                :fill_color, :fill_transparency,
                :hist_type,      # See below
                :hist_left,     # the position of the left part of the step
                :hist_right,     # the position of the right part of the step
               ]
    attr_writer *ELEMENTS
    
    # Special accessors: they implement the :"=>stuff" redirection. Beware
    # of circular redirections !!!
    for el in ELEMENTS
      eval <<"EOE"
def #{el.to_s}
  if @#{el.to_s}.is_a?(Symbol) && @#{el.to_s}.to_s =~ /^=>(.*)/
    new_sym = $1.to_sym
    if new_sym == :#{el.to_s}
      return nil
    end
    return self[new_sym]
  else
    return @#{el.to_s}
  end
end
EOE
    end

    # The order of the plotting operations. Default is 0
    # (path then markers then error_bars)
    DrawingOrder = [
                    [:path, :markers, :error_bars],
                    [:path, :error_bars, :markers],
                    [:markers, :path, :error_bars],
                    [:markers, :error_bars, :path ],
                    [:error_bars, :path, :markers],
                    [:error_bars,  :markers, :path]
                   ]

    Defaults = {
      :marker_scale => 0.5,
      :error_bar_color => [0.0,0.0,0.0],
      :drawing_order => 0,
      :transparency => false,
      :fill_type => false,      # No fill by default.
      :fill_color => :'=>color', # Defaults to the same as color
      :hist_type => false, # old style by default...
      :hist_left => 0.0,       # joint lines by default
      :hist_right => 1.0,       # joint lines by default
    }

    # A hash to deal with fill types, to be used as an argument for
    # the Utils::interpret_arg function.
    FillTypeArguments = {
      /no(ne)?/ => false,
      /y[_-]ax(is)?/ => :to_y_axis,
      /bottom/ => :to_bottom,
      /top/ => :to_top,
      /old-styke/ => :old_style, # essentially for histograms
    }
    
    # Creates a CurveStyle element. There are several ways to
    # initialize:
    # * no arguments (_args_ empty): a CurveStyle object is created
    #   with all its values set to nil
    # * at least one argument: arguments are taken as values in the
    #   order given by the ELEMENTS array. In this case, arguments
    #   not present default to the value given in the Defaults
    #   hash. This behaviour is mainly to keep old things working.
    # * The new and best way to do it is to feed it a hash, in which
    #   case elements not present in the hash get the values in
    #   Default.
    def initialize(*args)
      if args.length == 0
        return                  # We don't set anything in this case
      end
      
      if (h = args[0]).is_a? Hash
        for el in ELEMENTS
          if h.key?(el)
            self[el] = h[el]
          elsif Defaults.key?(el)
            self[el] = Defaults[el]
          end
        end
      else
        # If there is at least one argument, we consider them as
        # values in the order of ELEMENTS
        for el in ELEMENTS
          if args.length > 0
            self[el] =  args.shift
          else
            if Defaults.key?(el)
              self[el] = Defaults[el]
            end
          end
        end
      end
      # This is not the place where to do this. It should be implemented
      # as part of the attribute accessors.
#       # Now, if there is any entry with a value of a symbol starting with
#       # => , its value is replaced by the value for the pointed elements.
#       # That is =>color means 'replace by the color'. Be careful however
#       # with circular dependencies !!!
#       for el in ELEMENTS
#         if self[el].is_a?(Symbol) && self[el].to_s =~ /^=>(.*)/
#           self[el] = self[$1.to_sym]
#         end
#       end
    end
    
    # Overrides the style entries with the ones found in _other_.
    def override!(other)
      for iv in ELEMENTS
        if other.instance_variables.include?("@" + iv.to_s)
          # We use instance_variables.include? rather
          # than instance_variable_defined?, as the latter
          # is present only in recent versions of Ruby 1.8
          send(iv.to_s + "=", other.send(iv))
        end
      end
    end

    # Returns a hash suitable to pass on to save_legend_info, or return
    # false if there was no legend specified in the style.
    def legend_info
      if legend 
        legend_info = {
          'text' => legend,
          'marker' => marker,
          'marker_color' => marker_color,
          'marker_scale' => marker_scale, 
        }
        if color && line_style
          legend_info['line_color'] = color
          legend_info['line_type'] = line_style
        else                             # Line drawing is disabled.
          legend_info["line_width"] = -1 
        end
        return legend_info
      else
        return false
      end
    end


    # Remove attributes. Can be used to 'unset' current default.
    def delete(*vars)
      for iv in vars
        begin
          remove_instance_variable('@' + iv.to_s)
        rescue NameError
        end
      end
    end

    # Hash-like accessor:
    def [](elem)
      if ELEMENTS.include?(elem)
        return self.send(elem)
      else
        raise NameError, "Unkown element of CurveStyle: #{elem}"
      end
    end

    def []=(elem, value)
      if ELEMENTS.include?(elem)
        self.send(elem.to_s + '=', value)
      else
        raise NameError, "Unkown element of CurveStyle: #{elem}"
      end
    end

    # A function to be used to output legend pictograms separately.
    # It takes all the place available in the current context of the
    # FigureMaker object t
    def output_legend_pictogram(t)
      t.context do
        # output line
        if color && line_style
          t.line_color = color
          t.line_width = linewidth if linewidth
          #           t.line_cap = dict['line_cap']
          t.line_type = line_style
          t.stroke_line(0.0, 0.5, 1.0, 0.5)
        end
        if marker
          t.line_type = "none"
          t.show_marker( 'x' => 0.5,
                         'y' => 0.5,
                         'marker' => marker,
                         'color' => marker_color,
                         'scale' => marker_scale,
                         'alignment' => ALIGNED_AT_MIDHEIGHT,
                         'justification' => CENTERED)
        end
      end
    end

  end

  # The class Curve2D stores both the data and the way it
  # should be plotted, such as it's legend, it's color, it's line
  # style and so on...
  class Curve2D  < TiogaElement

    # for Dvectors:
    include Dobjects

    # The CurveStyle object representing the curve's style.
    attr_reader :style

    # The underlying Function object:
    attr_reader :function

    def need_style?
      return true
    end

    def set_style(style)
      @style = style.dup
    end

    def initialize(style = nil, data = nil)
      # style is a CurveStyle object
      set_style(style) if style
      set_data(data) if data
      @line_cap = nil
    end

    def has_legend?
      if @style.legend
        return true
      else
        return false
      end
    end

    # Sets the data for the curve
    def set_data(ar)
      @function = ar
    end

    # This function returns the index of a point
    # from the curve according to the given _spec_, a string.
    # * if _spec_ is a number >= 1, it represents the index of the point
    #   in the Function object.
    # * if _spec_ is a number <1, it represents the relative position of the
    #   point in the object (it is then multiplied by the size to get
    #   the actual index).
    # * if _spec_ is of the form 'x,y', the closest point belonging to the
    #   function is taken. Not implemented yet.
    def parse_position(spec)
      if spec =~ /(.+),(.+)/
        raise "The (x,y) point position is not implemented yet"
      else
        val = Float(spec)
        if val < 1 
          index = (@function.size * val).round
        else
          index = val.round
        end
        index
      end
    end

    # Returns the tangent of the curve at the given point, that is a vector
    # parallel to it. _dir_ specifies if it is a left tangent or a right
    # tangent or an average of both. Nothing else than the latter is
    # implemented for now.
    def tangent(index, dir = :both)
      before = @function.point(index - 1)
      point = @function.point(index)
      after = @function.point(index + 1)
      raise "Point invalid" unless point
      tangent = Dvector[0,0]
      if index > 0
        tangent += (point - before)
      end
      if index < (@function.size - 1)
        tangent += (after - point)
      end
      return tangent
    end

    # This function returns the bouding box of the specified graphes
    # No margin adjustment is done here, as it can lead to very uneven
    # graphs
    def get_boundaries
      top = @function.y.max
      bottom = @function.y.min
      left = @function.x.min
      right = @function.x.max

      width = (left == right) ? 1 : right - left
      height = (top == bottom) ? 1 : top - bottom

      return [left,right,top,bottom]
    end

    # Computes the outmost boundaries of the given boundaries.
    # Any NaN in here will happily get ignored.
    def Curve2D.compute_boundaries(bounds)
      left = Dvector.new
      right = Dvector.new
      top = Dvector.new
      bottom = Dvector.new
      bounds.each do |a|
        left.push(a[0])
        right.push(a[1])
        top.push(a[2])
        bottom.push(a[3])
      end
      return [left.min, right.max, top.max, bottom.min]
    end

    # Creates a path for the given curve. This should be defined
    # with care, as it will be used for instance for region coloring
    # and stroking. The function should only append to the current
    # path, not attempt to create a new path or empty what was done
    # before.
    def make_path(t)
      bnds = Utils::Boundaries.new(parent.effective_bounds)
      if @style.interpolate
        for f in @function.split_monotonic
          new_f = f.bound_values(*bnds.real_bounds)
          t.append_interpolant_to_path(f.make_interpolant)
        end
      else
        f = @function.bound_values(*bnds.real_bounds)
        t.append_points_to_path(f.x, f.y)
      end
    end

    # Draw the path
    def draw_path(t)
      t.line_width = @style.linewidth if @style.linewidth
      if @style.color && @style.line_style
        t.line_type = @style.line_style
        t.stroke_transparency = @style.transparency || 0
        t.stroke_color = @style.color
        if @line_cap
          t.line_cap = @line_cap
        end
        make_path(t)
        t.stroke
      end
    end

    def draw_markers(t)
      xs = @function.x
      ys = @function.y

      if @style.marker
        t.line_type = [[], 0]   # Always solid for striking markers
        t.stroke_transparency = @style.marker_transparency || 0
        t.fill_transparency = @style.marker_transparency || 0
        t.show_marker('Xs' => xs, 'Ys' => ys,
                      'marker' => @style.marker, 
                      'scale' => @style.marker_scale, 
                      'color' => @style.marker_color)
      end
    end

    # Returns a y value suitable for fills/histograms or other kinds of
    # stuff based on a specification:
    # * false/nil: returns nil
    # * to_y_axis: y = 0
    # * to_bottom:  the bottom of the plot
    # * to_top: the top of the plot
    # * "y = ....":  the given value.
    def y_value(spec)
      case spec
      when false, nil, :old_style
        return false
      when Float                # If that is already a Float, fine !
        return spec
      when :to_y_axis
        return 0.0
      when :to_bottom
        return parent.effective_bounds[3] # bottom
      when :to_top
        return parent.effective_bounds[2] # top
      when /y\s*=\s*(.*)/
        return Float($1)
      else
        warn "Y value #{spec} not understood"
        return false                  # We don't have anything to do, then.
      end
    end

    # A function to close the path created by make_path.
    # Overridden in the histogram code.
    def close_path(t, y)
      t.append_point_to_path(@function.x.last, y)
      t.append_point_to_path(@function.x.first, y)
      t.close_path
    end

    # Draws the filled region according to the :fill_type element
    # of the style pseudo-hash. It can be:
    def draw_fill(t)
      y = y_value(@style.fill_type)
      return unless y

      t.fill_transparency = @style.fill_transparency || 0
      # Now is the tricky part. To do the actual fill, we first make a
      # path according to the make_path function.
      make_path(t)

      # Then we add two line segments that go from the end to the
      # beginning.
      close_path(t, y)

      # Now the path is ready. Just strike -- or, rather, fill !
      t.fill_color = @style.fill_color
      t.fill
    end

    def plot(t = nil)
      debug "Plotting curve #{inspect}"
      t.context do

        # The fill is always first
        draw_fill(t)

        for op in CurveStyle::DrawingOrder[@style[:drawing_order]]
          self.send("draw_#{op}".to_sym, t)
        end
      end
    end

    # The function that plots error bars.
    def draw_error_bars(t = nil)
      # We first check that we actually need to do anything 
      if @function.errors.key?(:xmin) or @function.errors.key?(:ymin)
        error_bar = {}                 # Just create it once, anyway
        error_bar['color'] = @style.error_bar_color
        errors = @function.errors # So we won't have to worry
        # some data should be shared
        @function.errors[:x].each_index do |i|
          error_bar['x'] = errors[:x][i]
          if errors.key?(:xmin)
            error_bar['dx_plus'] = errors[:xmax][i] - errors[:x][i]
            error_bar['dx_minus'] = errors[:x][i] - errors[:xmin][i]
          else
            %w(dx_plus dx_minus).each do |el|
              error_bar.delete(el)
            end
            error_bar['dx'] = 0
          end
          error_bar['y'] = errors[:y][i]
          if errors.key?(:ymin)
            error_bar['dy_plus'] = errors[:ymax][i] - errors[:y][i]
            error_bar['dy_minus'] = errors[:y][i] - errors[:ymin][i]
          else
            %w(dy_plus dy_minus).each do |el|
              error_bar.delete(el)
            end
            error_bar['dy'] = 0
          end
          t.stroke_transparency = @style.error_bars_transparency || 0
          t.show_error_bars(error_bar)
        end
      end
    end

    # plot is the 'real_do' method.
    alias :do :plot    
  end

  # The basic class to create histograms.
  class Histogram2D < Curve2D

    def initialize(*args)
      super
      @line_cap = LINE_CAP_BUTT
    end

    # Creates a path for the given curve. This should be defined
    # with care, as it will be used for instance for region coloring
    # and stroking. The function should only append to the current
    # path, not attempt to create a new path or empty what was done
    # before.
    def make_path(t)
      # vectors to store the resulting path
      x_res = Dvector.new
      y_res = Dvector.new
      x_first = 2 * @function.x[0] - @function.x[1]
      y_first = 2 * @function.y[0] - @function.y[1]
      n = @function.size - 1
      x_last = 2 * @function.x[n] - @function.x[n-1]
      y_last = 2 * @function.y[n] - @function.y[n-1]
      
      t.make_steps('xs' => @function.x, 'ys' => @function.y,
                   'dest_xs' => x_res, 'dest_ys' => y_res,
                   'x_first' => x_first, 'y_first' => y_first,
                   'x_last' =>  x_last,  'y_last' => y_last)
      y_base = y_value(@style.hist_type)
      if y_base
        x_prev = y_prev = nil
        # We remove outer elements, not needed.
        x_res.shift
        y_res.shift
        x_res.pop
        y_res.pop
        for x,y in Function.new(x_res, y_res)
          if x_prev
            # We create a path according to the specs
            x_left = (x_prev + (x - x_prev) * @style.hist_left)
            x_right = (x_prev + (x - x_prev) * @style.hist_right)
            x_p = Dvector[x_left, x_right, x_right]
            y_p = Dvector[y,y,y_base]
            t.move_to_point(x_left,y_base)
            t.append_points_to_path(x_p,y_p)
            y_prev = x_prev = nil
          else
            # Accumulate to get a full step. 
            x_prev = x
            y_prev = y
          end
        end
      else
        t.append_points_to_path(x_res, y_res)
      end
    end

    # In the case of histograms with a defined level, we ignore the
    # fill_type setting for the y value, we use the same level.
    def close_path(t,y)
      if y_value(@style.hist_type)
        t.close_path
      else
        super
      end
    end

  end
end
