# multitext.rb : A simple backend to deal with multiple basic text files.
# Copyright (C) 2006 Jean-Julien Fleck
# Directly inspired from Vincent Fourmond's text.rb

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA



require 'SciYAG/Backends/text'
require 'Dobjects/Dvector'
require 'Dobjects/Function'

module SciYAG

  module Backends

    class MultiTextBackend < TextBackend

      include Dobjects
      
      describe 'multitext', 'MultiText format', <<EOD
This backend can read multiple text files in a format close to the one 
understood by gnuplot and the like. Allows to plot one column form one file 
with respect to another column of another file.
EOD
      
      inherit_parameters :skip
      
      def initialize
        @dummy = nil
        @current = nil            
        # @current is an array of Dvectors holding the contents of the most
        # recently read files, so that there is no need to read it again.
        @skip = 0
        @included_modules = []    # to make sure we give them to
        # compute_formula
        super()
      end

      def extend(mod)
        super
        @included_modules << mod
      end

      # This is called by the architecture to get the data. It splits
      # the set name into filename@cols, reads the file if necessary and
      # calls get_data
      def query_xy_data(set)
        cols = []
        files= []
        nb_match = -1
        new_set = set.gsub(/\[(.*?)@(.*?)\]/) do |match|
          cols.push($2)
          files.push($1)
          nb_match += 1
          "column[#{nb_match}]"
        end
        @current = Hash.new
        files.uniq.each do |file|
          @current[file] = Dvector.fancy_read(file, nil, 
                                              'index_col'  => true, 
                                              'skip_first' => @skip) 
        end
        return Function.new(*get_data(cols,files,new_set))
      end

      # Reads the data using the columns specification, provided that
      # the appropriate files have already been loaded into @current. For now
      # no single sanity check.
      def get_data(cols,files,set)
        vectors = []
        cols.each_index do |i|
          vectors.push(@current[files[i]][cols[i].to_i])
        end
        x_formula, y_formula = set.split(':')      
        mods = @included_modules
        return [Dvector.compute_formula(x_formula,vectors,mods),
                Dvector.compute_formula(y_formula,vectors,mods)]
      end

    end

  end
end
