# tioga_primitives.rb, copyright (c) 2006 by Vincent Fourmond: 
# Support for direct inclusion of graphics primitives in
# the graphs.
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'CTioga/debug'
require 'CTioga/log'
require 'MetaBuilder/parameters'
require 'shellwords'

module CTioga

  # A class to parse the format of graphics primitives, which is something
  # like:
  #  text: 12,34 "nice text" angle=35
  #
  class TiogaPrimitiveMaker

    extend Debug
    include Log

    # Internal structure to represent the syntax of a primitive call

    # The symbol for the funcall
    attr_accessor :symbol
    # An array of [name, types] for compulsory arguments
    attr_accessor :compulsory  
    # A hash of the optional arguments
    attr_accessor :optional 

    # Creates a TiogaPrimitiveMaker object, from the specifications
    def initialize(symb, comp, opt)
      @symbol = symb
      @compulsory = comp
      @optional = opt
    end

    # This function is fed with an array of Strings. The example in
    # TiogaPrimitiveMaker would look like
    #  ["12,34", "nice text", "angle=35"]
    # It returns the hash that can be fed to the appropriate
    # Tioga primitive. It is fed the plot
    def parse_args(args)
      ret = {}
      for name, type_spec in @compulsory
        type_spec ||= {:type => :string} # Absent means string
        type = MetaBuilder::ParameterType.get_type(type_spec)
        val = type.string_to_type(args.shift)
        ret[name] = val
      end
      # We now parse the rest of the arguments:
      for opt in args
        if opt =~ /^([^=]+)=(.*)/
          name = $1
          arg = $2
          type_spec = @optional[name] || {:type => :string}
          type = MetaBuilder::ParameterType.get_type(type_spec)
          val = type.string_to_type(arg)
          ret[name] = val
        else
          warn "Malformed optional argument #{opt}"
        end
      end
      return ret
    end

    # Turns a graphics spefication into a Tioga Funcall.
    def make_funcall(args, plotmaker)
      dict = parse_args(args)
      return TiogaFuncall.new(@symbol, dict) 
    end

    # A few predefined types to make it all more readable -
    # and easier to maintain !!
    FloatArray = {:type => :array, :subtype => :float}
    Point = FloatArray
    Boolean = {:type => :boolean}
    Marker = {                  # Damn useful !!
      :type => :array, 
      :subtype => {:type => :integer, :namespace => Tioga::MarkerConstants},
      :namespace => Tioga::MarkerConstants
    }
    Number = {:type => :float}
    Color = {
      :type => :array, :subtype => :float, 
      :namespace => Tioga::ColorConstants
    }
    # Very bad definition, but, well...
    LineStyle = {:type => :array, :subtype => :integer,
      :namespace => Tioga::FigureConstants,
    }
    
    # Available primitives:
    PRIMITIVES = {
      "text" => self.new(:show_text, 
                         [ 
                          ['point', Point],
                          ['text']
                         ],
                         {
                           'angle' => Number,
                           'color' => Color,
                           'scale' => Number,
                         }),
      "arrow" => self.new(:show_arrow, 
                          [ 
                           ['tail', Point ],
                           ['head', Point ],
                          ],
                          {
                            'head_marker' => Marker,
                            'tail_marker' => Marker,
                            'head_scale'  => Number,
                            'tail_scale'  => Number,
                            'line_width'  => Number,
                            'line_style'  => LineStyle,
                            'color'       => Color,
                          }),
      "marker" => self.new(:show_marker, 
                           [ 
                            ['point', Point ],
                            ['marker', Marker ],
                           ],
                           {
                             'color'       => Color,
                             'angle'       => Number,
                             'scale'       => Number,
                           }),
    }
                       
    
    # Parse a specification such as
    #  text: 12,34 "nice text" angle=35
    # plotmaker is a pointer to the plotmaker instance.
    def self.parse_spec(spec, plotmaker)
      spec =~ /^([^:]+):(.*)/
      name = $1
      args = Shellwords.shellwords($2)
      ret = PRIMITIVES[name].make_funcall(args, plotmaker)
      debug "draw: #{name} -> #{ret.inspect}"
      ret
    end

    # Returns a small descriptive text about the currently known
    # graphics primitives. If _details_ is on, more details are shown
    # (not implemented yet).
    def self.introspect(details = true)
      str = ""
      for name, spec in PRIMITIVES
        str += "\t#{name}: " +
          spec.compulsory.map do |a|
          a[0]
        end. join(' ') +
          if details
            "\n\t\toptions: #{spec.optional.keys.join(',')}\n\t\t"
          else
            " [options]\t "
          end + "see Tioga function #{spec.symbol}\n"
      end
      return str
    end
  end

  class TangentSemiPrimitive < TiogaPrimitiveMaker
    TiogaPrimitiveMaker::PRIMITIVES["tangent"] = 
      self.new(:show_arrow,
               [
                ['spec', :string],
               ],
               {                
                 'xextent'  => FloatArray,
                 'yextent'  => FloatArray,
               }.merge(TiogaPrimitiveMaker::PRIMITIVES['arrow'].optional)
               # We automatically add stuff from the arrow specs,
               # as they share a lot of keys...
               )
    
    def make_funcall(args, plotmaker)
      dict = parse_args(args)
      curve = plotmaker.last_curve
      raise 'The tangent drawing command needs to be specified *after* the curve it applies to' unless curve
      # We now need to transform
      index = curve.parse_position(dict["spec"])
      dict.delete('spec')
      tangent = plotmaker.last_curve.tangent(index)
      point = curve.function.point(index)

      # We are now preparing the coordinates of the arrow:
      # * if markonly is on, it doesn't matter much
      # * if xextent, we take the array as X extents in either
      #   direction
      # * the same obviously applies for yextent.
      # Only one spec can be used at a time
      if dict['xextent']
        fact = dict['xextent'].shift/tangent[0]
        dict['head'] = point + (tangent * fact)
        fact = (dict['xextent'].shift || 0.0)/tangent[0]
        dict['tail'] = point - (tangent * fact)
      elsif dict['yextent']
        fact = dict['yextent'].shift/tangent[1]
        dict['head'] = point + (tangent * fact)
        fact = (dict['yextent'].shift || 0.0)/tangent[1]
        dict['tail'] = point - (tangent * fact)
      else
        dict['line_width'] = 0
        dict['head'] = point
        dict['tail'] = point - tangent
      end
      # Remove unnecessary keys...
      %w(spec xextent yextent).map {|k| dict.delete(k)}

      # We setup other defaults than the usual ones, from
      # the current curve.
      dict['color'] ||= curve.style.color # color from curve
      dict['line_width'] ||= curve.style.linewidth
      # TODO: add linestyle when that is possible

      dict['tail_marker'] ||= 'None' # No tail by default.

      # And we return the Funcall...
      return TiogaFuncall.new(@symbol, dict) 
    end
  end

end
