# plotmaker.rb, copyright (c) 2006 by Vincent Fourmond: 
# The main class for making plots
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).



# Very important for command-line parsing
require 'optparse'
require 'Tioga/tioga'
require 'Tioga/Utils'

# Information about style
require 'CTioga/styles'
# Moving arrays
# require 'CTioga/movingarrays'
# And, most important of all, elements
require 'CTioga/elements'
require 'CTioga/curves'
require 'CTioga/tioga_primitives'

# The debugging facility
require 'CTioga/debug'
require 'CTioga/log'

# The backends
require 'CTioga/backends'

# Support for themes
require 'CTioga/themes'




# for interpreting the CTIOGA environment variable
require 'shellwords'
require 'MetaBuilder/metabuilder'


module CTioga
  


  # The regular expression saying that what we see on the command-line
  # means "use default value".
  DEFAULT_RE = /^\s*(auto|default)\s*$/i
  
  # The regular expression saying that what we see on the command-line
  # means "disable".
  DISABLE_RE = /^\s*(no(ne)?|off)\s*$/i

  # The regular expression saying that what we see on the command-line
  # means "true".
  TRUE_RE = /^\s*(true|yes|on)\s*$/i

  # A small function to help quoting a string for inclusion
  # in a pdfTeX primitive.

  def self.pdftex_quote_string(str)
    return str.gsub(/([%#])|([()])|([{}~_^])|\\/) do 
      if $1
        "\\#{$1}"
      elsif $2                  # Quoting (), as they can be quite nasty !!
        "\\string\\#{$2}"
      elsif $3
        "\\string#{$3}"
      else                      # Quoting \
        "\\string\\\\"
      end
    end
  end


  # This class is responsible for reading the command-line, via it's parse
  # function, and to actually turn it into nice Tioga commands to make
  # even nicer graphes.
  class PlotMaker

    # PlotMaker is now handled (at least partially) by MetaBuilder.
    # That should save a lot of code.

    include MetaBuilder::DescriptionInclude
    extend  MetaBuilder::DescriptionExtend
    
    describe 'test', "A test class", <<EOD
A class to test visually the effects of different stuff
EOD

    

    include SciYAG
    include Tioga

    include Debug
    include Log

    # For dimension conversion and TeX quoting.
    include Tioga::Utils

    # For the backend handling:
    include CTioga::Backends

    # Support for Themes:
    include Themes


    # these are basically the attributes which are modified directly
    # on the command-line
    attr_writer :fig_name, :cleanup, :legend

    # The command-line parser
    attr_reader :parser

    # The last Curve object on the stack
    attr_reader :last_curve

    # Whether we are trying to do real-size PDF or not. When set, it
    # is the size of the PDF
    attr_accessor :real_size

    # The current object
    attr_accessor :current_object

    def initialize
      # The first thing to do is to setup logging, as you really
      # want to be able to communicate with users, don't you ?
      init_logger


      @args = []                # Holding the command-line
      @parser = OptionParser.new


      initialize_themes

      # Whether the plots should be interpolated.
      @interpolate = false 

      @line_width = nil

      @fig_name = "Plot"

      # Initialize the backend structure:
      init_backend_structure

      # Initialize the debugging options:
#      init_debug

      # The multiplication factor for the plots.
      # Nothing happens by default.
      @x_factor = nil
      @y_factor = nil

      # if the axes are log scale
      @x_log = false
      @y_log = false

      # now, the elements for the structure of the plot:
      @root_object = SubPlot.new
      
      # the current object is the one to which we'll add plot elements.
      @current_object = @root_object

      # @legend is holding the text attached to the current item being
      # plotted; it has to be cleared manually after every item processed
      @legend = nil

      # general purpose variables:
      @cleanup = false
      @viewer = false

      # Some options we could set on the command line
      @init_funcalls = []

      # Whether to provide default legends. On by default
      @autolegends = true

      @command_line = ""

      @real_size = "12cmx12cm"

      # If set, should be the size of the TeX font
      
      @tex_fontsize = nil

      # The frame sides for the setup_real_size function
      @frame_sides = [0.1,0.9,0.9,0.1]

      # If set, we create separate legend informations:
      @separate_legend = false

      # The array holding the styles.
      @separate_legend_styles = []
      # The size of the produced PDF in PDF points.
      @separate_legend_width = 12
      @separate_legend_height = 8

      # Whether to mark the command-line in the PDF file
      @mark = false             # better be off, kind of painful sometimes

      # The LaTeX preamble:
      @preamble = ""

      # The last curve used:
      @last_curve = nil

      prepare_option_parser
    end





    # This function reads a configuration file and executes it's
    # statements in the scope of a local module, which is then
    # included in this instance, and in the backend's instances.
    def read_config_file(file)
      f = File.open(file)
      info "Reading config file #{file}"
      lines = f.readlines
      lines << "\nend\n"
      lines.unshift "module CTiogaRC\n"
      code = lines.join
      eval code
      extend CTiogaRC
      for b_e in backends.values
        b_e.extend CTiogaRC
      end
      # This is for the compute_formula function.
      Dvector.extend CTiogaRC
    end
    
    CONFIG_FILE_NAME = ".ctiogarc"

    # Looks for a configuration file, either in the current directory
    # or in the HOME directory.
    def lookup_config_file
      if File.readable? CONFIG_FILE_NAME
        return CONFIG_FILE_NAME
      end

      if ENV.has_key?('HOME')
        home_rc = File.join(ENV['HOME'], CONFIG_FILE_NAME)
        if File.readable?(home_rc)
          return home_rc
        end
      end
      return nil
    end

    # sets up the FigureMaker object to use for real size. Uses the
    # @real_size instance variable as a source for the size
    def setup_real_size(t)
      # Get the width and height from @real_size
      sizes = @real_size.split("x").collect {|s| 
        tex_dimension_to_bp(s)
      }

      t.def_enter_page_function {
        t.page_setup(*sizes)
        t.set_frame_sides(*@frame_sides) 
      }

      # Setting label and title scale to 1
      t.title_scale = 1
      t.xlabel_scale = 1
      t.ylabel_scale = 1
    end

    # Reads a configuration file if one is found.
    def read_config
      if lookup_config_file
        read_config_file(lookup_config_file)
      end
    end


    # Sets the value of a string hosted by @current_object
    def current_set_string(name)
      return proc {|s|
        if s
          @current_object.send(name,s)
        else
          @current_object.send(name,nil)
        end
      }
    end

    def evaluate_float(s)
      return Float(eval(s))
    end

    def bool_arg(str)
      if str =~ TRUE_RE || str.nil?
        true
      else
        false
      end
    end

    def cleanup=(arg)
      @cleanup = bool_arg(arg)
    end

    # Push a function call onto the stack of the current element
    def add_elem_funcall(sym, *args)
      @current_object.add_funcall(TiogaFuncall.new(sym, *args))
    end

    # Push a function call onto the stack of things we should do
    # just after creating the FigureMakerobject
    def add_init_funcall(sym, *args)
      @init_funcalls << TiogaFuncall.new(sym, *args)
    end

    # Forwards the boundary settings to the current object.
    def set_bounds(which, val)
      method = ("bound_" + which + "=").to_sym
      @current_object.send(method, val)
    end

    def set_range(a,b,str)
      first,last = str.split(/\s*:\s*/)
      first = first.to_f if first
      last = last.to_f if last
      set_bounds(a,first)
      set_bounds(b,last)
    end

    # Returns an array suitable for use with set_frame_margins based on
    # some text. The array is duplicated until it has at least four elements.
    def margins_from_text(txt)
      ary = txt.split(/\s*,\s*/).map {|s| s.to_f}
      while ary.length < 4
        ary += ary
      end
      return ary
    end

    # In the order: left, right, bottom, top, to suit Tioga's
    # default positioning of the text along the axis.
    def set_frame_margins(val)
      @frame_sides = [val[0], 1.0 - val[1], 1.0 - val[3], val[2]]
    end

    # A hash to help choose which elements we need to modify in the
    # legends spec hash.
    Legend_Specs_Sides = {
      'right' => ['plot_right_margin', 'legend_left_margin'],
      'left' => ['plot_left_margin', 'legend_right_margin'],
      'top' => ['plot_top_margin', 'legend_bottom_margin'],
      'bottom' => ['plot_bottom_margin', 'legend_top_margin'],
    }

    # This function sets up the bounds for plots/legend based on
    # a _side_ and _size_ attribute. Without any arguments,
    # just return an empty
    def prepare_legend_specs(side = false, size = 0.0, delta = 0.0 )
      specs = {}
      # Resetting specs to 0
      for a in %w(top bottom left right)
        specs["legend_#{a}_margin"] = 0.0
        specs["plot_#{a}_margin"] = 0.0
      end
      if side
        if Legend_Specs_Sides.key?(side.downcase)
          a,b = *Legend_Specs_Sides[side.downcase]
          specs[a] = size
          specs[b] = 1 - size + delta
        else
          error "Legend side #{side} unknown"
        end
      end
      return specs
    end


    # A function that transforms an inset specification into
    # margins.
    def inset_margins(spec)
      spec =~ /(.*),(.*):([^x]*)(?:x(.*))?/
      x = $1.to_f; y = $2.to_f; w = $3.to_f
      h = ($4 || $3).to_f
      margins = [x - w/2, 1 - (x + w/2), 1 - (y+h/2), y - h/2]
      return margins
    end


    # This function prepares the parser by giving it the arguments and
    # some small help text going along with them. This function will
    # unfortunately get really large with time, but there's nothing
    # to do about it.
    def prepare_option_parser
      theme_prepare_parser(@parser)

      @parser.separator "\nLaTeX options"
      @parser.on("--use PACKAGE",
                 "Adds PACKAGE to the LaTeX preamble") do |w|
        @preamble += "\\usepackage{#{w}}\n"
      end

      @parser.on("--preamble STRING",
                 "Adds STRING to the LaTeX preamble") do |w|
        @preamble += "#{w}\n"
      end


      @parser.on("--[no-]histogram",
                 "All the next curves will be drawn", "as histograms") do |w|
        @histogram = w
      end



#       # Sets:
#       for arg in [[@colors, "color"],
#                   [@markers, "marker"],
#                   [@markers_colors, "marker-color"],
#                   [@linestyle, "line-style"]]
#         @parser.on("--#{arg[1]}-set SET", 
#                    "Choose the current set for #{arg[1]} ",
#                    "(#{arg[0].available_sets.join(' ')})",
#                    moving_array_set(arg[0]))
#       end



      @parser.separator "\nGlobal look:"

      @parser.on("--background COLOR",
                 "Sets the background color for plots") do |c|
        color = CTioga.get_tioga_color(c)
        @current_object.add_elem(TiogaFuncall.new(:fill_color=, color))
        @current_object.add_elem(TiogaFuncall.new(:fill_frame))
      end

      @parser.on("--aspect-ratio [RATIO]",
                 "Sets the aspect ratio (defaults to 1)") do |a|
        a = 1.0 unless a
        a = a.to_f
        if a <= 0.0
          warn "Aspect ratio #{a} not valid, ignored" 
        else
          add_elem_funcall(:set_aspect_ratio, a)
        end
      end
      @parser.on("--golden-ratio",
                 "Sets the aspect ratio to the golden number") do |a|
        add_elem_funcall(:set_aspect_ratio, 1.61803398874989)
      end
      @parser.on("--xrange RANGE",
                 "X plotting range") do |a|
        set_range('left','right', a)
      end
      @parser.on("--yrange RANGE",
                 "y plotting range") do |a|
        set_range('bottom','top', a)
      end
      @parser.on("--margin MARGIN",
                 "Sets the margin around data", "(left,right,top,bottom",
                 "in fractions of the corresponding size)") do |v|
        current_object.plot_margins = margins_from_text(v)
      end

      @parser.on("--rescale FACTOR",
                 "Scales everything by a given factor",
                 "(useful for subplots)") do
        |fact|
        add_elem_funcall(:rescale, fact.to_f)
      end


      @parser.separator "\nVarious axis stuff:"
      @parser.on("--xfact FACT",
                 "Multiply all x values by FACT") do |f|
        @x_factor = evaluate_float(f)
      end
      @parser.on("--yfact FACT",
                 "Multiply all y values by FACT") do |f|
        @y_factor = evaluate_float(f)
      end
      @parser.on("--[no-]xlog",
                 "Uses logarithmic scale for X axis") do |v|
        @x_log = v
        add_elem_funcall(:xaxis_log_values=, v)
      end
      @parser.on("--[no-]ylog",
                 "Uses logarithmic scale for Y axis") do |v|
        @y_log = v
        add_elem_funcall(:yaxis_log_values=, v)
      end
      @parser.on("--comma",
                 "Uses a comma for the decimal separator") do 
        @decimal_separator = ','
      end
      @parser.on("--decimal SEP",
                 "Uses SEP for the decimal separator") do |s|
        @decimal_separator = s
      end


      @parser.separator "\nSubfigures, subplots, regions..."
      @parser.on("--subplot",
                 "Start a subplot") do
        plot = SubPlot.new(:subplot, current_object)
        @current_object.add_elem(plot)
        self.current_object = plot
      end
      @parser.on("--inset SPECS",
                 "Creates an inset with the given",
                 "specifications: " +
                 "x,y:w[xh]",
                 "(x,y: position of the center, w width and",
                 "h height if different than width).", 
                 "All numbers " +
                 "are between 0 and 1" ) do |a|
        plot = SubPlot.new(:subplot, current_object)
        @current_object.add_elem(plot)
        self.current_object = plot
        margins = inset_margins(a)
        debug "inset margins: #{margins.inspect}"
        current_object.frame_margins = margins
      end
      @parser.on("--subframes FRAMES",
                 "Setup the frames relative to the parent", 
                 "for the current " +
                 "subplot",
                 "(distance from left,right,top,bottom)") do |v|
        current_object.frame_margins = margins_from_text(v)
      end

      @parser.on("--region",
                 "Start a colored region") do
        plot = Region.new(current_object)
        @current_object.add_elem(plot)
        self.current_object = plot
      end

      @parser.on("--end",
                 "Ends the last subplot or region") do 
        if current_object.parent
          self.current_object = current_object.parent
        else
          warn "--end while in top level"
        end
      end
      

      @parser.separator "\nOptions for real size output:"
      # Real size
      @parser.on("-r", "--[no-]real-size [SIZE]",
                 "Tries to produce a PDF file suitable",
                 "for inclusion " +
                 "with \\includegraphics") do |spec|
        @real_size = spec
      end
      @parser.on("--frame-margins VAL",
                 "The proportion of space to leave on", "each side " +
                 "for the text display" ) do |val|
        set_frame_margins(margins_from_text(val))
      end


      @parser.separator "\nLegends and texts:"

      @parser.on("-x","--xlabel [LABEL]",
                 "Label of the x axis",
                 current_set_string(:x_label=))
      @parser.on("-y","--ylabel [LABEL]",
                 "Label of the y axis",
                 current_set_string(:y_label=))
      @parser.on('-t', "--title [TITLE]",
                 "Sets the title of the plot",
                 current_set_string(:title=))
      @parser.on("-l","--[no-]legend [LEGEND]",
                 "Sets the legend of the next curve.") do |l|
        override_style.legend = l
      end

      @parser.on("--[no-]auto-legend", 
                 "Whether to automatically add legends","to curves") do |l|
        @autolegends = l
      end

      @parser.on('-N', "--no-legends", 
                 "Alias for --no-auto-legend") do
        @autolegends = false
      end

      ## Default TeX fontsize
      @parser.on("--fontsize [NB]",
                 "Default TeX fontsize in points") do |size|
        @tex_fontsize = size
      end

      @parser.on("--tick-label-scale SCALE",
                 "Sets the scale of the text for tick labels") do |l|
        add_elem_funcall(:xaxis_numeric_label_scale=,Float(l))
        add_elem_funcall(:yaxis_numeric_label_scale=,Float(l))
      end


      @parser.on("--[no-]separate-legends",
                 "If set, PDF files are produced that contain",
                 "what the legend pictogram would look like") do |l|
        @separate_legends = l
        if l
          # Switches off autolegends by default
          @autolegends = false
        end
      end

      @parser.on("--legend-pos SPEC", 
                 "Sets the legend position and its size.",
                 "SPEC looks like pos,size[,delta]") do |a|
        w,s,d = a.split /\s*,\s*/
        debug "Legend position #{w.inspect} #{s.inspect}"
        specs = prepare_legend_specs(w, Float(s), d.to_f)
        debug "Specs: #{specs.inspect}"
        current_object.legend_specs = specs
      end

      @parser.on("--legend-inside SPEC", 
                 "Makes a legend inside the plot.",
                 "See --inset for the format of SPEC") do |a|
        margins = inset_margins(a)
        t = %w(legend_left_margin legend_right_margin 
          legend_top_margin legend_bottom_margin).zip(margins)
        specs = prepare_legend_specs
        t.inject(specs) { |s,o| s[o[0]] = o[1]; s}
        
        debug "Specs: #{specs.inspect}"
        current_object.legend_specs = specs
      end


      @parser.separator "\nGraphic primitives:"
      @parser.on("-d","--draw SPEC",
                 "Draw a graphic primitive") do |spec|
        add_elems_to_current(TiogaPrimitiveMaker.parse_spec(spec, self))
      end
      @parser.on("-d","--draw-help",
                 "Display what is known about graphic", 
                 "primitives and exit") do
        puts
        puts "Currently known graphics primitives are"
        puts
        puts TiogaPrimitiveMaker.introspect
        exit
      end

      prepare_backend_options(@parser)
                                              
      @parser.separator "\nHousekeeping and miscellaneous options:"
      @parser.on("--xpdf", 
                 "Runs xpdf to show the plot obtained") do 
        @viewer = "xpdf"
      end

      @parser.on("--open", 
                 "Runs open to show the plot obtained") do 
        @viewer = "open"
      end

      @parser.on("--[no-]viewer [VIEWER]", 
                 "Runs VIEWER to show the plot obtained --",
                 "or doesn't " +
                 "show up the viewer at all") do |x|
        @viewer = x
      end
      
      @parser.on("--[no-]cleanup", 
                 "Removes all the accessory files produced") do |x|
        @cleanup = x
      end

      @parser.on("--tex-cleanup", 
                 "Removes all files produced except those",
                 "necessary " +
                 "for inclusion in a TeX document") do 
        @tex_cleanup = true
      end

      @parser.on("--clean-all", 
                 "Removes all files produced -- better use",
                 "along with " +
                 "--xpdf or --open") do 
        @clean_all = true
      end
      
      @parser.on("--save-dir DIR", 
                 "Sets the directory for output") do |x|
        add_init_funcall(:save_dir=, x)
      end

      
      @parser.on("--include FILE",
                 "Imports the file's functions into ctioga's",
                 "namespace " +
                 "so that they can be used", "by backends") do |file|
        read_config_file(file)
      end

      @parser.on("-n", "--name NAME",
                 "Base name") do |name|
        @fig_name = name
      end

      @parser.on("--display-commandline",
                 "Adds the command line used to make",
                 "to graph on the " +
                 "bottom of the graph") do 
        add_elem_funcall(:show_marker, 
                         {
                           "x" => 0.5,
                           "y" => -0.2,
                           "text" => @command_line,
                           "scale" => 0.6,
                           "font" => Tioga::MarkerConstants::Courier,
                         }
                         )
      end

      @parser.on("--[no-]mark", 
                 "Fills the 'creator' field of the " ,
                 "produced PDF file with the command-line",
                 "when on. ") do |v|
        @mark = v
      end

      logger_options(@parser)
      @parser.on("--debug-patterns", 
                 "Produces an alignement-checking PDF file") do |v|
        @debug_patterns = v
      end

      @parser.on("--eps", 
                 "Attempt to produce an EPS file,",
                 "using pdftops, LaTeX and dvips.",
                 "The PDF output is kept in any case") do |v|
        @eps = v
      end
      @parser.on("--args FILE",
                 "Reads argument from FILE, *one per",
                 "line* and then resumes normal processing") do |v|
        File.open(v) do |f|
          unshift_cmdline_args(*f.readlines.collect {|s| s.chomp})
        end
      end

      @parser.on_tail("-h", "--help", "Show this message") do
        puts @parser
        puts 
        puts "Please note that all the options can be abbreviated"
        puts "as long as they are still unique"
        exit
      end
    end

    def unshift_cmdline_args(*args)
      @args.unshift(*args)
    end
    
    # the functions for plot structure...
    def subplot
      new_object = SubPlot.new(:subplot,@current_object)
      @current_object.add_elem(new_object)
      @current_object = new_object
    end

    def subfigure
      new_object = SubPlot.new(:subfigure,@current_object)
      @current_object.add_elem(new_object)
      @current_object = new_object
    end

    def end
      @current_object = @current_object.parent if @current_object.parent
    end

    def add_elems_to_current(*elems)
      for el in elems
        @current_object.add_elem(el)
      end
    end

    # This function is called whenever PlotMaker needs to process one
    # data set. It's name is given in parameter here.
    def process_data_set(set)
      # First, we get the dataset:
      begin
        data = xy_data_set(set)
      rescue Exception => ex
        error "Problem when processing set #{set} " +
          "(backend #{backend.class}): #{ex.message}"
        debug "Error backtrace: #{ex.backtrace.join "\n"}"
        warn "Simply ignoring set #{set}"
        return                  # Simply return.
      end
        
      # Scale plots if necessary:
      data.mul!(:x,@x_factor) if @x_factor
      data.mul!(:y,@y_factor) if @y_factor

      # Implement a logarithmic scale
      data.safe_log10!(:x) if @x_log
      data.safe_log10!(:y) if @y_log

      data.strip_nan

      # Then we create a curve object and give it style
      if @histogram
        curve = Histogram2D.new(get_current_style(set)) 
      else
        curve = Curve2D.new(get_current_style(set)) 
      end
      if @separate_legends
        @separate_legend_styles << curve.style
      end
      curve.set_data(data)

      # Then we add ot to the current object:
      add_elems_to_current(curve)
      @last_curve = curve       # And we update the last_curve pointer.
    end

    def frames_str(t)
      return "top : #{t.frame_top} bottom: #{t.frame_bottom} " +
        "left : #{t.frame_left} right: #{t.frame_right}"
    end

    
    # Provides the name of an output file, given its suffix and
    # optionnaly it base name (defaults to @figure_name).
    def output_filename(suffix, base = @fig_name) 
      if @figmaker.save_dir
        File.join(t.save_dir,"#{base}#{suffix}")
      else
        "#{base}#{suffix}"
      end
    end

    # Runs the plots, using the arguments given in the command-line
    def run(args)

      # A little trick to allow in-place modification of the
      # command-line from an option.
      @args += args

      # Runs the command to set command-line defaults. Ignored silently
      # in the files given on the command-line.

      quoted_args = args.collect do |s|
        CTioga.shell_quote_string(s)
      end.join ' '

      @command_line = "#{File.basename($0)} #{quoted_args}"

      puts <<"EOBANNER"
This is ctioga version #{CTioga.version}, copyright (C) 2006 Vincent Fourmond
ctioga comes with absolutely NO WARRANTY. This is free software, you are 
welcome to redistribute it under certain conditions. See the COPYING file
in the original tarball for details. You are also welcome to contribute if
you like it, see in the manual page where to ask.
EOBANNER

      # Read the CTIOGA environment variable 
      if ENV.key? "CTIOGA"
        ctioga_env = Shellwords.shellwords(ENV["CTIOGA"])
        @parser.parse(ctioga_env)
      end

      if respond_to? :ctioga_defaults
        send :ctioga_defaults
      end

      @parser.order!(@args) do |spec|
        # We use Backend#expand to leave room for the backend to
        # interpret the text as many different sets.
        sets = expand_spec(spec)
        info "Expanding #{spec} to #{sets.join(', ')}"
        for set in sets
          process_data_set(set)
        end
      end
      
      # now, we plot the stuffs... if there is actually
      # something to be plotted ;-) !

      t = FigureMaker.new
      @figmaker = t             # Boaf...

      # Setting the TeX fontsize if default is not adequate

      t.tex_fontsize = @tex_fontsize if @tex_fontsize

      # add hyperref preamble to have nice PDF comments:
      if @mark
        # We should use the \pdfinfo pdftex primitive, as
        # this will allow to have less surprises...
        t.tex_preview_preamble += 
          "\n\\pdfinfo {\n  /Title (" +
          CTioga.pdftex_quote_string(@command_line) +
          ")\n}\n"
      end

      if @debug
        debug_figmaker(t)
      end

      # Now, we must use real_size stuff !! (default 12cmx12cm)
      info "Producing PDF file of real size #{@real_size}"
      setup_real_size(t) 

      if @debug_patterns
        debug_patterns(t)
      end

      return unless @root_object.number > 0 

      # Asking the theme to do something.
      theme_bod_hook(t)

      # We put the command line used in the .tex file:
      t.tex_preview_preamble += 
        "% Produced by ctioga, with the command-line\n" +
        "% #{@command_line}\n"
      
      if @decimal_separator
        t.tex_preamble += <<'EOD'.gsub(',',@decimal_separator) # Neat !
\def\frenchsep#1{\frenchsepma#1\endoffrenchsep} 
\def\fseat#1{\frenchsepma}
\def\frenchsepma{\futurelet\next\frenchsepmw}
\def\frenchsepmw{\ifx\next\endoffrenchsep\let\endoffrenchsep=\relax%
\else\if\next.\ifmmode\mathord,\else,\fi%
\else\next\fi\expandafter
\fseat\fi}
EOD
        t.yaxis_numeric_label_tex = t.
          yaxis_numeric_label_tex.gsub('#1','\frenchsep{#1}')
        t.xaxis_numeric_label_tex = t.
          xaxis_numeric_label_tex.gsub('#1','\frenchsep{#1}')
      end

      # Add custom preamble:
      t.tex_preview_preamble += @preamble

      

      # Cleanup code
      # off by default, as this could cause problems for
      # legend pictures generation/EPS generation.
      # The files will get cleaned up anyway.
      t.autocleanup = false

      # If the initialization function was found, we call it
      if respond_to? :ctioga_init
        send :ctioga_init, t
      end

      for funcall in @init_funcalls
        funcall.do(t)
      end

      t.def_figure(@fig_name) {
        @root_object.do(t)
      }

      debug "Contents of the main FigureMaker object: " +
        figmaker_options(t)

      info "Generating file '#{@fig_name}.pdf'"
      t.make_preview_pdf(t.figure_index(@fig_name))

      if @eps
        info "Attempting to create an EPS file, watch out error messages"
        # First, we convert the _figure.pdf into eps with pdftops
        cmd = "pdftops -eps #{output_filename('_figure.pdf')}"
        debug "Running #{cmd}"
        system cmd
        f = output_filename('.tex')
        new = output_filename('.new.tex')
        debug "Modifying #{f}"
        orig = open(f)
        out = open(new, "w")
        for l in orig
          l.gsub!('pdftex,','') # \usepackage[pdftex,...]...
          l.gsub!('_figure.pdf', '_figure.eps')
          out.print l
        end
        orig.close
        out.close

        cmd = "latex #{new}"
        debug "Running #{cmd}"
        system cmd

        cmd = "dvips -o #{output_filename('.eps')} -t unknown -T " +
          "#{@real_size.gsub(/x/,',')} #{output_filename('.new.dvi')}"
        debug "Running #{cmd}"
        system cmd
        
      end
      
      # Generating the separate legends if requested
      if not @separate_legend_styles.empty?
        index = 0
        info "Creating separate lengends"
        for style in @separate_legend_styles 
          name = sprintf "%s_legend_%02d", @fig_name, index
          index += 1
          t.def_figure(name) {
            t.set_device_pagesize(@separate_legend_width * 10,
                                  @separate_legend_height * 10)
            t.set_frame_sides(0,1,1,0) 
            t.update_bbox(0,0)
            t.update_bbox(1,1)
            style.output_legend_pictogram(t)
          }
          # create_figure takes a number. It is part of Tioga's internals.
          t.create_figure_temp_files(t.figure_index(name))
          

          # Some post_processing to rename the files appropriately
          # and delete the unwanted _figure.txt file
          base_name = if t.save_dir
                        File.join(t.save_dir,name)
                      else
                        name
                      end
          # Remove the _figure.txt files
          begin
            File.rename(base_name + "_figure.pdf", base_name + ".pdf")
            if @eps             # Create also EPS files:
              system "pdftops -eps #{base_name}.pdf"
            end
            File.delete(base_name + ".tex")
            File.delete(base_name + "_figure.txt")
          rescue
            warn "Files should have been deleted, but were not found"
          end
        end
      end

      if @cleanup
        files_to_remove = %w(.tex .out .aux .log _figure.pdf _figure.txt) +
          %w(.new.tex .new.aux .new.dvi .new.log _figure.eps) # EPS-related files
      elsif @tex_cleanup
        files_to_remove = %w(.tex .out .aux .log .pdf)
      elsif @clean_all
        files_to_remove = %w(.tex .out .aux .log .pdf _figure.pdf _figure.txt) +
          %w(.new.tex .new.aux .new.dvi .new.log .eps _figure.eps) # EPS-related files

      end
      if @viewer 
        # we display the output file:
        info "Starting the viewer #{@viewer} as requested"
        pdf_file = if t.save_dir
                     File.join(t.save_dir,@fig_name+".pdf")
                   else
                     @fig_name+".pdf"
                   end
        system("#{@viewer} #{pdf_file}")
      end
      
      if files_to_remove
        files_to_remove.collect! {|s| 
          output_filename(s)
        }
        files_to_remove << "pdflatex.log" 
        info "Cleaning up temporary files"
        debug "Cleaning #{files_to_remove.join ' '}"
        files_to_remove.each {|f| begin
                                    File.delete(f)
                                  rescue
                                    debug "File absent #{f}, not deleted" 
                                  end
        }
      end
    end

  end

  # Takes a string a returns a quoted version that should be able
  # to go through shell expansion. For now, it won't work with
  # strings containing ', but that's already a good help.
  def CTioga.shell_quote_string(str)
    if str =~ /[\s"*$()\[\]{}']/
      if str =~ /'/
        a = str.gsub(/(["$\\])/) { "\\#$1" }
        return "\"#{a}\""
      else 
        return "'#{str}'"
      end
    else
      return str
    end
  end
  
end

