# backends.rb, copyright (c) 2006 by Vincent Fourmond: 
# The module handling the communication with backends. 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

# The backend structure...
require 'SciYAG/backends'
require 'MetaBuilder/factory'

module CTioga

  # The position of the URL, used for getting the version
  SVN_URL = '$HeadURL: svn+ssh://fourmond@rubyforge.org/var/svn/sciyag/releases/ctioga-1.4/lib/backends.rb $'

  CTIOGA_VERSION = if SVN_URL =~ /releases\/ctioga-([^\/]+)/
                     $1
                   else
                     "SVN version"
                   end

  # The current version of the program.
  def CTioga.version
    return CTIOGA_VERSION
  end
  

  # This module handles all manipulations of backends.
  module Backends

    include SciYAG
    
    # Initializes the backend structure.
    def init_backend_structure
      @backend_factory = 
        MetaBuilder::Factories::UniqueFactory.new(Backend, 'text')
      SciYAG::Backend.logger = self.logger
    end

    def backend
      @backend_factory.current
    end

    def backends
      @backend_factory.instances
    end

    def set_backend(str)
      @backend_factory.current = str
    end


    # Push a filter onto the current backend
    def current_push_filter(filter)
      backend.push_xy_filter(filter)
    end

    # Fills an OptionParser with the apppropriate stuff to
    # deal with backends:
    def prepare_backend_options(op)
      op.separator "\nHow to select backends, and their options:"
      @backend_factory.option_parser_factory(op) 

      op.separator "\nFilters to apply onto data:"
      for filter_desc in Filter.description_list
        filter_desc.parser_instantiate_option(op,
                                              self,
                                              :current_push_filter)
      end
      op.on("--filter-pop", "Removes the last filter pushed " +
                 "onto the current backend") do
        backend.pop_xy_filter
      end

      op.on("--filter-clear", "Removes all filters applying " +
                 "to the current backend") do
        backend.clear_xy_filters
      end
    end

    # Interprets the +set+ with the current backend and returns the data
    def xy_data_set(set)
      return backend.xy_data(set) 
    end

    # Expands the current set according to the current's backend rules
    def expand_spec(spec)
      backend.expand_sets(spec)
    end

  end
end
