# parameter.rb : A class to describe a parameter
# Copyright (C) 2006 Vincent Fourmond
 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

require 'forwardable'

# The MetaBuilder module contains the classes necessary to create classes
# with a simple meta-information system that lets you save/export the
# state of an object (under certain assumptions), query parameters in
# different ways (command-line, Qt, Tk and Gtk widgets, and so on...).
module MetaBuilder

  # A class that handles a parameter type. It has to be subclassed to
  # actually provide a parameter. The subclasses must provide the
  # following:
  #
  # * a #string_to_type function to convert from string to the type;
  # * a #type_to_string to convert back from type to string
  # * an instance #type_name that returns a really small description
  #   of the type, to be used for instance to name command-line parameters.
  # * a #type_name statement that registers the current class to the
  #   ParameterType system.
  #
  # Moerover, it is a good idea to reimplement the
  # #qt4_create_input_widget method; the default implementation works,
  # but you probably wish it would look better.
  #
  # Types are implemented using hashes: this way, additionnal
  # parameters can easily be added. The hash *must* have a :type key
  # that will be interpreted by the children of ParameterType. Examples:
  #
  #  { :type => :integer}
  #  { :type => :file, :filter => "Text Files (*.txt)}
  #
  # And so on. You definitely should document your type and it's
  # attributes properly, if you ever want that someone uses it.
  #
  # The list of currently recognised types is here:
  # 
  # <tt>:integer</tt> ::   ParameterTypes::IntegerParameter
  # <tt>:float</tt> ::     ParameterTypes::FloatParameter
  # <tt>:string</tt> ::    ParameterTypes::StringParameter
  # <tt>:file</tt> ::      ParameterTypes::FileParameter
  # <tt>:boolean</tt> ::   ParameterTypes::BooleanParameter
  # <tt>:list</tt> ::      ParameterTypes::ListParameter
  #
  # Additionally to the parameters the given type is requiring, you can
  # pass some other kind of information using this hash, such as option
  # parser short argument, aliases, and so on. This has nothing to do
  # with type conversion, but it is the best place where to put this kind
  # of things, in my humble opinion. The currently recognized such additional
  # parameters are:
  # * :option_parser_short: a short option name for option_parser.
  class ParameterType

    # A hash that makes the :type value of the _type_ argument correspond
    # to a ParameterType child
    @@parameter_types = { }

    # A default constructor. It should be safe to use it directly for
    # children, unless something more specific is needed. Any descendent
    # should *always* register _type_ as @type - or, even better, call
    # super.
    def initialize(type)
      if type.is_a?(Symbol)
        type = {:type => type}
      end
      @type = type
    end

    # This class function actually registers the current type
    # to the ParameterType ancestor. _name_ should be a symbol.
    # Moreover, if the second argument is provided, it automatically
    # creates a #type_name instance method returning this value.
    def self.type_name(name, public_name = nil)
      if @@parameter_types.has_key?(name)
        warn "Redefining type #{name} " +
          "from #{@@parameter_types[name]} to #{self}"
      end
      @@parameter_types[name] = self
      self.send(:define_method,:type_name) do
        public_name
      end
    end      

    # This function converts a 'description' of the type wanted
    # into a ParameterType child. Used by the Parameter system.
    # As a special treat, a lone symbol is converted into
    #  {:type => :symbol}
    def self.get_param_type(type)
      if type.is_a?(Symbol)
        type = {:type => type}
      end
      raise "The type argument must be a Hash" unless type.is_a?(Hash)
      begin
        return @@parameter_types.fetch(type[:type])
      rescue
        raise "Type #{type[:type]} unknown to the parameter system"
      end
    end

    # Returns a ParameterType child instance suitable for conversion
    # of the given type specification
    def self.get_type(type)
      return get_param_type(type).new(type)
    end

    # Runs the string_to_type conversion hook
    def stt_run_hook(val)
      if @type.key?(:stt_hook)
        return @type[:stt_hook].call(val)
      else
        val
      end
    end

    # This function converts the given string to the appropriate type. It is a
    # wrapper around the #string_to_type_internal function
    # that can take advantage of a few general features. It is recommanded
    # to define a #string_to_type_internal function rather to redefine
    # #string_to_type
    def string_to_type(string)
      # First makes a constant lookup.
      if @type.key?(:namespace)
        begin
          return stt_run_hook(lookup_const(string))
        rescue
        end
      end
      return stt_run_hook(string_to_type_internal(string))
    end

    # This function does the exact opposite of the #string_to_type one.
    # It defaults to using the to_s methods of the parameter. Be careful:
    # it is absolutely important that for any valid type,
    #
    #   string_to_type(type_to_string(type)) == type
    def type_to_string(type)
      return type_to_string_internal(type)
    end

    # The internal function for converting type to a string. Used by
    # #type_to_string, children should only reimplement this function
    # and leave #type_to_string
    def type_to_string_internal(type)
      return type.to_s
    end

    # Looks for the value as a constant specified in the :namespace
    # modules. Raises InvalidInput if not found.
    def lookup_const(str)
      for mod in [@type[:namespace]].flatten
        begin
          return mod.const_get(str)
        rescue
          # Nothing, we still look up
        end
      end
      raise InvalidInput, "Constant #{str} not found"
    end

    # Returns a type name suitable for displaying, for instance, in
    # an option parser, or inside a dialog box, and so on. Has to
    # be one word (not to confuse the option parser, for instance);
    # it is better if it is lowercase.
    def type_name
      return 'notype'
    end

    # Creates on option for the OptionParser _parser_. The _args_
    # will be fed directly to OptionParser#on. The _block_ is called with
    # the value in the target type.
    def option_parser_raw(parser, *args, &block)
      b = block                 # For safe-keeping.
      c = proc do |str|
        b.call(string_to_type(str))
      end
      parser.on(*args, &c)
    end

    # Creates an option for the OptionParser _parser_. The block is fed
    # with the converted value. The default implementation should be
    # fine for most classes, but this still leaves the room for
    # reimplementation if necessary. The parameters are:
    # 
    # * _parser_: the OptionParser;
    # * _name_: the name of the option;
    # * _desc_: it description,
    # * _block_: the block used to set the data.
    def option_parser_option(parser, name, desc, &block)
      args = [option_parser_long_option(name), desc]
      if @type.has_key?(:option_parser_short)
        args.unshift(@type[:option_parser_short])
      end
      option_parser_raw(parser, *args, &block)
    end

    # Returns a value to be fed to OptionParser#on as a 'long' option.
    # It is separated from the rest to allow easy redefinition
    # (in special cases). _name_ is the name of the option.
    def option_parser_long_option(name)
      return "--#{name} #{type_name.upcase}"
    end

    # An exception that must be raised when ParamterType#string_to_type
    # is given incorrect input.
    class IncorrectInput < Exception
    end
  end

  # A parameter describes a way of storing some information into an
  # instance of an object. No type checking is done on the target object
  # when the actual reading/writing is done. However, the type checking
  # is done upstream by the Description system.
  #
  # A Parameter consists of several things:
  #
  # * a #name, to identify it in a unique fashion;
  # * a #type, used to convert to and from String;
  # * some explanative text, used to inform the user: #long_name and
  #   #description
  # * two symbols that are used to gain read and write access of the
  #   parameter on the target object.
  #
  class Parameter 

    # To forward string_to_type and type_to_string to the @param_type
    extend Forwardable

    # The short name of the parameter
    attr_accessor :name
    
    # The long name of the parameter, to be translated
    attr_accessor :long_name
    
    # The function names that should be used to set the symbol and
    # retrieve it's current value. The corresponding functions should
    # read or return a string, and writer(reader) should be a noop.
    attr_accessor :reader_symbol, :writer_symbol
    
    # The (text) description of the parameter
    attr_accessor :description
    
    # The way the type was specified to the system
    attr_accessor :type_spec
    
    # The actual ParameterType subclass instance in charge of handling
    # the conversion from string.
    attr_accessor :param_type


    # (Vincent Fourmond 5/12/2006): this will be moved to ParameterType, as
    # I think it has nothing to do here.

    # The attributes hash is meant to hold various convenient
    # information which are not directly linked to type conversion,
    # such as an alternative for labels, shortcuts for command-line options
    # and the like. Not compulsory. The currently understood values are:
    # * :option_parser_short: a short option for option_parser.
    #     attr_accessor :attributes
    
    def initialize(name, writer_symbol,
                   reader_symbol,
                   long_name, 
                   type,
                   description, attributes = {})
      @name = name
      @writer_symbol = writer_symbol
      @reader_symbol = reader_symbol
      @description = description
      @long_name = long_name
      @type_spec = type
      type_class = ParameterType.get_param_type(@type_spec)
      raise "Parameter type not recognized: #{@type_spec}" unless type_class
      
      @param_type = type_class.new(@type_spec)
      @attributes = attributes
    end


    # Two methods forwarded to the ParameterType object.
    def_delegators :@param_type, :string_to_type, :type_to_string


    # Sets directly the target parameter, without type conversion
    def set_raw(target, val)
      target.send(@writer_symbol, val) 
    end

    
    # Uses the #writer_symbol of the _target_ to set the value of the
    # parameter to the one converted from the String _str_
    def set(target, str)
      set_raw(target, string_to_type(str)) 
    end
    

    # Aquires the value from the backend, and returns it in the
    # form of a string
    def get(target)
      return type_to_string(get_raw(target))
    end

    # Aquires the value from the backend, and returns it.
    def get_raw(target)
      target.send(@reader_symbol)
    end

    # Creates an option in the OptionParser _parser_, with the name
    # _name_, that will change the settings of the object _instance_
    def option_parser_option(parser, name, instance)
      obj = instance # copy needed to avoid problems with
      # overwritten local variables
      block = proc do |value|
        set_raw(obj, value)
      end

      # The use of #attributes is already deprecated. Don't use !!!
#       if attributes.has_key?(:option_parser_short)
#         @param_type.option_parser_raw(parser, attributes[:option_parser_short],
#                                       @param_type.option_parser_long_option(name), 
#                                       description, &block) 
#       else
      @param_type.option_parser_option(parser, name, description, &block) 
#       end
    end
    
  end

end
