# text.rb : A simple backend to deal with basic text files.
# Copyright (C) 2006 Vincent Fourmond

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA



require 'SciYAG/Backends/backend.rb'
require 'Dobjects/Dvector'
require 'Dobjects/Function'

module SciYAG

  # A module for easy use of NaN in operations
  module NaN
    NaN = 0.0/0.0
    def nan
      return NaN
    end
  end

  class TextBackend < Backend

    # A constant holding a relation extension -> command to
    # decompress (to be fed to sprintf with the filename as argument)
    UNCOMPRESSORS = {
      ".gz" => "gunzip -c %s",
      ".bz2" => "bunzip2 -c %s",
    }

    include Dobjects
    
    describe 'text', 'Text format', <<EOD
This backend can read text files in a format close to the one understood
by gnuplot and the like.
EOD

    # Inherit the baseline handling, can be useful !
    inherit_parameters :base_line
      
    param_accessor :skip, 'skip', "Skip lines", {:type => :integer}, 
    "Number of lines to be skipped at the beginning of the file"

    param_accessor :default_column_spec, 'col', 
    "Default column specification", {:type => :string}, 
    "Which columns to use when the @1:2 syntax is not used"

    param_accessor :separator, 'separator', "Data columns separator", 
    {:type => :string_or_regexp}, 
    "The columns separator. Defaults to /\s+/"

#     param_accessor :select, 'select', "Select lines", {:type => :string},
#     "Skips line where the code returns false"
    
    def initialize
      @dummy = nil
      @current = nil   
      # Current is the name of the last file used. Necessary for '' specs.
      @current_data = nil       # The data of the last file used.
      @cache = {}               # A cache file_name -> data
      @skip = 0
      @included_modules = [NaN]    # to make sure we give them to
      # Dvector.compute_formula
      @default_column_spec = "1:2"

      @separator = /\s+/

      super()
    end

    def extend(mod)
      super
      @included_modules << mod
    end

    # Reads data from a file. If needed, extract the file from the set
    # specification.
    def read_file(file)         
      if file =~ /(.*)@.*/
        file = $1
      end
      name = file               # As file will be modified.
      if ! @cache.key?(file)    # Read the file if it is not cached.
        if file == "-"
          file = $stdin
        elsif file =~ /(.*?)\|\s*$/
          file = IO.popen($1)
        elsif not File.readable? file
          # Try to find a compressed version
          for ext,method in UNCOMPRESSORS
            if File.readable? "#{file}#{ext}"
              file = IO.popen(method % "#{file}#{ext}")
              info "Using compressed file #{name}#{ext} in stead of #{name}"
              break 
            end
          end
        else 
          for ext, method in UNCOMPRESSORS
            if file =~ /#{ext}$/ 
              file = IO.popen(method % file)
              info "Taking file #{name} as a compressed file"
              break
            end
          end
        end
        @cache[name] = Dvector.fancy_read(file, nil, 
                                          'index_col' => true,
                                          'skip_first' => @skip,
                                          'sep' => @separator
                                          )
      end
      return @cache[name]
    end


    # This is called by the architecture to get the data. It splits
    # the set name into filename@cols, reads the file if necessary and
    # calls get_data
    def query_xy_data(set)
      if set =~ /(.*)@(.*)/
        col_spec = $2
        file = $1
      else
        col_spec = @default_column_spec
        file = set
      end
      if file.length > 0
        @current_data = read_file(file)
        @current = file
      end
      x,y,err = get_data(col_spec)
      return [Function.new(x,y),err]
    end

    # Reads the data using the columns specification, provided that
    # the appropriate fle has already been loaded into @current. For now
    # no single sanity check.
    def get_data(col_spec)
      # First, we must split the column specification into what
      # I would call target specifications. A target is in the form
      # of stuff=spec, where stuff can be basically anything. x=, y=,
      # yerr= are implied for the first specifications. 

      defaults = [:x,:y,:yerr]
      specifications = {}       # A hash value spec => column spec
      col_spec.split(/:/).each do |spec|
        d = defaults.shift
        if spec =~ /^\s*(\w+)\s*=(.*)/
          spec = $2
          d = $1.to_sym
        end
        specifications[d] = spec
      end
      
      debug "spec #{col_spec} becomes #{specifications.inspect}"

      values = {}
      if col_spec =~ /\$/       # There is a formula in the specification
        for key,spec in specifications
          values[key] = Dvector.
            compute_formula(spec.gsub(/\$(\d+)/, 'column[\1]'), 
                            @current_data,
                            @included_modules)
        end
      else
        for key,spec in specifications
          values[key] = @current_data[spec.to_i].dup
        end
      end
      errors = compute_error_bars(values)
      # Now, we're left with a hash...
      return [values[:x],values[:y], errors]
    end

    # Turns a target => values specification into something usable as
    # error bars, that is :xmin, :xmax and the like hashes. The rules
    # are the following:
    # * ?min/?max are passed on directly;
    # * ?e(abs) are transformed into ?min = ? - ?eabs, ?max = ? + ?eabs
    # * ?eu(p/?ed(own) are transformed respectively into ? +/- ?...
    # * ?er(el) become ?min = ?*(1 - ?erel, ?max = ?(1 + ?erel)
    # * ?erup/?erdown follow the same pattern...
    def compute_error_bars(values)
      target = {}
      for key in values.keys
        case key.to_s
        when /^[xy]$/
          target[key] = values[key].dup # Just to make sure.
        when /^(.)e(a(bs?)?)?$/
          target["#{$1}min".to_sym] = values[$1.to_sym] - values[key]
          target["#{$1}max".to_sym] = values[$1.to_sym] + values[key]
        when /^(.)eu(p)?$/
          target["#{$1}max".to_sym] = values[$1.to_sym] + values[key]
        when /^(.)ed(o(wn?)?)?$/
          target["#{$1}min".to_sym] = values[$1.to_sym] - values[key]
        when /^(.)er(el?)?$/
          target["#{$1}min".to_sym] = values[$1.to_sym] * 
            (values[key].neg + 1)
          target["#{$1}max".to_sym] = values[$1.to_sym] * 
            (values[key] + 1)
        when /^(.)erd(o(wn?)?)?$/
          target["#{$1}min".to_sym] = values[$1.to_sym] * 
            (values[key].neg + 1)
        when /^(.)erup?$/
          target["#{$1}max".to_sym] = values[$1.to_sym] * 
            (values[key] + 1)
        else
          warn "Somehow, the target specification #{key} " +
            "didn't make it through"
        end
      end
      return target
    end

    # Expands specifications into few sets. This function will separate the
    # set into a file spec and a col spec. Within the col spec, the 2##6
    # keyword is used to expand to 2,3,4,5,6. 2## followed by a non-digit
    # expands to 2,...,last column in the file. For now, the expansions
    # stops on the first occurence found, and the second form doesn't
    # work yet. But soon...
    def expand_sets(spec)
      if m = /(\d+)##(\D|$)/.match(spec)
        a = m[1].to_i 
        trail = m[2]
        b = read_file(spec)
        b = (b.length - 1) 
        ret = []
        a.upto(b) do |i|
          ret << m.pre_match + i.to_s + trail + m.post_match
        end
        return ret
      else
        return super
      end
    end

  end

end
