/*
 *  methods for encryption/security mechanisms for cryptmount
 *  $Revision: 222 $, $Date: 2008-10-03 12:31:16 +0100 (Fri, 03 Oct 2008) $
 *  (C)Copyright 2005-2008, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_NANOSLEEP
#  include <time.h>
#endif
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/times.h>

#include <linux/major.h>

#include "armour.h"
#include "cryptmount.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


#ifndef AS_MODULE


/* List of all available key-managers (to be constructed later): */
static keymanager_t *keymgrs = NULL;


/*
 *  ==== Raw key-management routines ====
 */

static int kmraw_init_algs(void)
{
    return 0;
}


static int kmraw_free_algs(void)
{
    return 0;
}


static void kmraw_mk_default(keyinfo_t *keyinfo)
{
    if (keyinfo == NULL) return;

    if (keyinfo->digestalg == NULL) {
        keyinfo->digestalg = cm_strdup("none");
    }

    if (keyinfo->cipheralg == NULL) {
        keyinfo->cipheralg = cm_strdup("none");
    }
}


static int kmraw_is_compat(const keyinfo_t *keyinfo, FILE *fp_key)
{
    if (keyinfo->format != NULL) {
        return (strcmp(keyinfo->format, "raw") == 0);
    } else {
        if (keyinfo->cipheralg != NULL) {
            return (strcmp(keyinfo->cipheralg, "none") == 0);
        }
    }

    return 0;
}


static unsigned kmraw_get_properties(const keyinfo_t *keyinf)
{
    return 0;
}


static int kmraw_get_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt,
            unsigned char **key, int *keylen, FILE *fp_key,
            km_overrides_t *overrides)
    /* Extract key from unencrypted (plain) file */
{   enum { BUFFSZ=512 };
    char buff[BUFFSZ];
    size_t len, lmt;
    int eflag=ERR_NOERROR;

    *key = NULL; *keylen = 0;

    if (fp_key == NULL) {
        eflag = ERR_BADFILE;
        goto bail_out;
    }


    /* Read data directly from keyfile: */
    for (;;) {
        lmt = (keyinfo->maxlen > 0 && (*keylen + BUFFSZ) > keyinfo->maxlen
                ? (size_t)(keyinfo->maxlen - *keylen) : (size_t)BUFFSZ);
        len = fread((void*)buff, (size_t)1, (size_t)lmt, fp_key);
        if (len == 0) break;

        /* Copy new block of data onto end of current key: */
        *key = (unsigned char*)sec_realloc((void*)*key, (size_t)(*keylen+len));
        memcpy((void*)(*key + *keylen), (const void*)buff, len);
        *keylen += len;
    }

    if (ferror(fp_key) != 0) {
        fprintf(stderr, _("Key-extraction failed for \"%s\"\n"),
                keyinfo->filename);
        /* This is a trivial case of decryption failure: */
        eflag = ERR_BADDECRYPT;
    }

  bail_out:

    return eflag;
}


static int kmraw_put_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt,
            const unsigned char *key, const int keylen, FILE *fp_key,
            const km_overrides_t *overrides)
    /* Store key in unencrypted (plain) file */
{   int eflag=ERR_NOERROR;

    /* Write data directly into keyfile: */
    if (fwrite((const void*)key, (size_t)keylen, (size_t)1, fp_key) == 0
      || ferror(fp_key) != 0) {
        fprintf(stderr, _("Key-writing failed for \"%s\"\n"),
                keyinfo->filename);
        eflag = ERR_BADENCRYPT;
    }

    return eflag;
}


static void *kmraw_md_prepare(void)
{   cm_sha1_ctxt_t *mdcontext;

    mdcontext = cm_sha1_init();

    return (void*)mdcontext;
}


static void kmraw_md_block(void *state, unsigned char *buff, size_t len)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    /* Apply built-in SHA1 hash to data-block: */
    cm_sha1_block(ctxt, buff, len);
}


static void kmraw_md_final(void *state, unsigned char **mdval, size_t *mdlen)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_final(ctxt, mdval, mdlen);
}


static void kmraw_md_release(void *state)
{   cm_sha1_ctxt_t *ctxt=(cm_sha1_ctxt_t*)state;

    cm_sha1_free(ctxt);
}


static keymanager_t keymgr_raw = {
    "raw", 0,       kmraw_init_algs, kmraw_free_algs,
                    kmraw_mk_default, kmraw_is_compat,
                    kmraw_get_properties,
                    kmraw_get_key, kmraw_put_key,
                    kmraw_md_prepare, kmraw_md_block,
                    kmraw_md_final, kmraw_md_release,
    NULL
#ifdef TESTING
    , NULL, NULL, 0
#endif
};

#endif  /* !AS_MODULE */



/*
 *  ==== abstract key-management interfaces ====
 */

#ifndef AS_MODULE

static keymanager_t **add_keymgr(keymanager_t **ptr, keymanager_t *km)
    /* add key-manager interface definition(s) to list of available managers */
{
    if (km == NULL) return ptr;

    *ptr = km;

    /* allow for addend itself being a list of key-managers: */
    for (;;) {
#ifdef TESTING
        if (km->install_testctxt != NULL) km->install_testctxt(test_ctxtptr);
#endif
        if (km->next == NULL) break;
        km = km->next;
    }

    return &km->next;
}


static void build_keymgrs(void)
    /* construct list of available key-managers */
{   keymanager_t **ptr;

    if (keymgrs != NULL) return;    /* already initialized */

    ptr = &keymgrs;
    ptr = add_keymgr(ptr, kmblti_gethandle());
    ptr = add_keymgr(ptr, kmgcry_gethandle());
    ptr = add_keymgr(ptr, kmluks_gethandle());
    ptr = add_keymgr(ptr, &keymgr_raw);
}


const char **get_keymgr_list(void)
    /* construct list of key-manager names */
{   keymanager_t *km;
    const char **arr=NULL;
    int i, cnt;

    build_keymgrs();
    for (km=keymgrs,cnt=0; km!=NULL; km=km->next,++cnt);

    arr = (const char**)malloc((size_t)((cnt+1)*sizeof(const char*)));
    for (i=0,km=keymgrs; i<cnt; km=km->next,++i) {
        arr[i] = km->ident;
    }
    arr[i] = NULL;

    return arr;
}


keymanager_t *get_keymanager(const struct keyinfo *keyinfo, FILE *fp_key)
    /* Search for keymanager that supports given key */
{   keymanager_t *km;

    if (keyinfo == NULL) return NULL;
    build_keymgrs();

    for (km=keymgrs; km!=NULL; km=km->next) {
        if (fp_key != NULL) (void)fseek(fp_key, 0L, SEEK_SET);
        if (km->is_compat(keyinfo, fp_key)) {
            return init_keymanager(km);
        }
    }

    return NULL;
}


keymanager_t *init_keymanager(keymanager_t *km)
{
    if (km != NULL) {
        if (!km->initialized) {
            km->init_algs();    /* FIXME - check return status */
            km->initialized = 1;
        }
    }

    return km;
}


int free_keymanagers(void)
{   keymanager_t *km;

    for (km=keymgrs; km!=NULL; km=km->next) {
        if (km->initialized) {
            km->free_algs();
            km->initialized = 0;
        }
    }

    return 0;
}

#endif  /* !AS_MODULE */


int cm_get_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt, unsigned char **key, int *keylen,
            km_overrides_t *overrides)
{   keymanager_t *km;
    FILE *fp=NULL;
    unsigned attempts=0;
    int eflag=ERR_NOERROR;

    if (keyinfo == NULL || keyinfo->filename == NULL) {
        fprintf(stderr, _("Missing key-file for target \"%s\"\n"), ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    fp = fopen(keyinfo->filename, "rb");
    if (fp == NULL) {
        fprintf(stderr,
            _("Failed to open keyfile \"%s\" for target \"%s\"\n"),
            keyinfo->filename, ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    if ((km = get_keymanager(keyinfo, fp)) != NULL) {
        do {
            (void)fseek(fp, 0L, SEEK_SET);
            eflag = km->get_key(ident, keyinfo, pw_ctxt, key, keylen,
                                fp, overrides);
            if (eflag == ERR_BADDECRYPT) sleep(1);
        } while (++attempts < keyinfo->retries && eflag == ERR_BADDECRYPT);
    } else {
        fprintf(stderr,
                _("Unrecognized key format (%s) for target \"%s\"\n"),
                (keyinfo->format != NULL ? keyinfo->format : "NULL"), ident);
        eflag = ERR_BADKEYFORMAT;
    }

  bail_out:
    if (fp != NULL) fclose(fp);

    return eflag;
}


int cm_put_key(const char *ident, const keyinfo_t *keyinfo,
            const km_pw_context_t *pw_ctxt,
            const unsigned char *key, const int keylen, FILE *fp_key,
            const km_overrides_t *overrides)
{   keymanager_t *km;
    int eflag = ERR_NOERROR;

    if (fp_key == NULL) {
        fprintf(stderr, _("Missing output stream for target \"%s\"\n"), ident);
        return ERR_BADFILE;
    }

    if ((km = get_keymanager(keyinfo, NULL)) != NULL) {
        eflag = km->put_key(ident, keyinfo, pw_ctxt, key, keylen,
                            fp_key, overrides);
    } else {
        fprintf(stderr,
                _("Unrecognized key format (%s) for target \"%s\"\n"),
                (keyinfo->format != NULL ? keyinfo->format : "NULL"), ident);
        eflag = ERR_BADKEYFORMAT;
    }

    return eflag;
}


unsigned cm_get_keyproperties(const keyinfo_t *keyinfo)
    /* Extract information about key, e.g. whether encrypted */
{   keymanager_t *km;
    unsigned props;

    if ((km = get_keymanager(keyinfo, NULL)) != NULL) {
        props = km->get_properties(keyinfo);
    } else {
        /* Safest to assume that key shouldn't be overwritten: */
        props = KM_PROP_FIXEDLOC | KM_PROP_FORMATTED;
    }

    return props;
}


km_overrides_t *alloc_overrides()
{   km_overrides_t *overrides;

    overrides = (km_overrides_t*)malloc(sizeof(km_overrides_t));

    overrides->mask = 0;
    overrides->cipher = NULL;
    overrides->km_data = NULL;

    return overrides;
}

void merge_overrides(const cment_t *ent, const km_overrides_t *overrides,
                    cment_t *merged)
    /* Combine parameters suggested by key-manager with cment_t */
{   int64_t delta;

    if (ent == NULL || overrides == NULL) return;

    memcpy((void*)merged, (const void*)ent, sizeof(cment_t));

    if ((overrides->mask & KM_OR_START) != 0) {
        delta = (overrides->start - ent->start);
        if (delta >= 0) {
            merged->start = overrides->start;
            if (merged->length >= 0) merged->length -= delta;
        }
    }

    if ((overrides->mask & KM_OR_CIPHER) != 0) {
        merged->cipher = overrides->cipher;
    }

    if ((overrides->mask & KM_OR_IVOFFSET) != 0) {
        merged->ivoffset = overrides->ivoffset;
    }
}


void free_overrides(km_overrides_t *overrides)
{
    if (overrides == NULL) return;

    if (overrides->cipher != NULL) free((void*)overrides->cipher);
    if (overrides->km_data != NULL) free((void*)overrides->km_data);

    free((void*)overrides);
}


#ifdef TESTING

int km_test_managers(void)
{   keymanager_t *km;
    int flg = 0;

    build_keymgrs();

    for (km=keymgrs; km!=NULL; km=km->next) {
        if (km->run_tests != NULL) flg |= km->run_tests();
    }

    return flg;
}


int km_test_keyrw(void)
    /* test key read-writing (creation/extraction) */
{   enum { MAXKEY=256 };
    keymanager_t *km;
    keyinfo_t keyinfo;
    int i, keylen=0, keylen1;
    char str[256];
    unsigned char key0[MAXKEY],*key1=NULL;
    FILE *fp;
    km_overrides_t *overrides=NULL;
    extern cm_testinfo_t *test_ctxtptr;

    CM_TEST_START("key read-write");

    build_keymgrs();
    for (km=keymgrs; km!=NULL; km=km->next) {
        if ((km->test_flags & CM_READONLY) != 0) continue;

        keyinfo.format = NULL;
        keyinfo.filename = "NOWHERE";
        keyinfo.digestalg = NULL;
        keyinfo.cipheralg = NULL;
        keyinfo.maxlen = -1;
        km->mk_default(&keyinfo);
        km->init_algs();

        for (keylen=1; keylen<=MAXKEY; keylen<<=2) {
            sprintf(str, "Key read-write, %s, keylen=%d",
                km->ident, keylen);
            CM_TEST_IDENT(str);

            /* Generate (low-entropy) key: */
            for (i=0; i<keylen; ++i) {
                key0[i] = (i * 0x9d) ^ ((keylen * (unsigned long)km) % 253);
            }

            /* Write key to file: */
            fp = tmpfile();
            if (fp == NULL) CM_TEST_ABORT(context);
            overrides = alloc_overrides();
            i = km->put_key("TEST-OUT", &keyinfo, NULL,
                            key0, keylen, fp, overrides);
            CM_ASSERT_EQUAL(ERR_NOERROR, i);

            key1 = NULL; keylen1 = -keylen;

            /* Try reading key back from file: */
            rewind(fp);
            i = km->get_key("TEST-IN", &keyinfo, NULL,
                            &key1, &keylen1, fp, overrides);
            CM_ASSERT_EQUAL(ERR_NOERROR, i);
            CM_ASSERT_EQUAL(keylen, keylen1);
            CM_ASSERT_DIFFERENT(key0, key1);
            CM_ASSERT_DIFFERENT(NULL, key1);
            for (i=0; i<keylen; ++i) {
                CM_ASSERT_EQUAL(key0[i], key1[i]);
            }

            free_overrides(overrides);
            fclose(fp);
            sec_free((void*)key1);
        }

        km->free_algs();
        free((void*)keyinfo.cipheralg);
        free((void*)keyinfo.digestalg);
    }
    CM_ASSERT_DIFFERENT(keylen, 0);

    CM_TEST_OK(context);
}

#endif  /* TESTING */



/*
 *  ==== miscellaneous routines ====
 */

size_t mk_key_string(const unsigned char *key, const size_t keylen, char *buff)
    /* Create text version of crypto key */
{   size_t i;

    for (i=0; i<keylen; ++i) {
        sprintf(buff+2*i, "%02x", (unsigned)(key[i]));
    }

    return (2 * keylen);
}


int sycheck_directory(const char *dirname)
    /* Check that permissions on directory are suitably restrictive */
{   struct stat sdir;

    /* Get information about directory (if present): */
    errno = 0;
    if (stat(dirname,&sdir) != 0) {
        if (errno == ENOENT) return ERR_NOERROR;
        fprintf(stderr, "Cannot open \"%s\"\n", dirname);
        return ERR_INSECURE;
    }

    /* Check file/directory ownerships: */
    if (sdir.st_uid != (uid_t)0) {
        fprintf(stderr, "\"%s\" must be owned by root\n", dirname);
        return ERR_INSECURE;
    }

    /* Check that directory isn't globally writable: */
    if (!S_ISDIR(sdir.st_mode) || (sdir.st_mode & S_IWOTH) != 0) {
        fprintf(stderr, "Lax permissions on \"%s\"\n", dirname);
        return ERR_INSECURE;
    }

    return ERR_NOERROR;
}


int sycheck_cmtab(const char *cmtab)
    /* Check that permissions on ${sysconfdir}/cryptmount/cmtab are sensible */
{   struct stat sfile;
    char *dirname=NULL;
    int pos, eflag=ERR_NOERROR;

    /* Extract directory name from cmtab filename: */
    pos = (int)strlen(cmtab);
    dirname = (char*)malloc((size_t)(pos + 1));
    for ( ; pos>0 && cmtab[pos-1] != '/'; --pos) dirname[pos] = '\0';
    while (--pos >= 0) dirname[pos] = cmtab[pos];
    eflag = sycheck_directory(dirname);
    if (eflag != ERR_NOERROR) goto bail_out;

    if (stat(cmtab,&sfile) != 0) {
        fprintf(stderr, "Cannot open \"%s\"\n", cmtab);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

    /* Check file ownerships: */
    if (sfile.st_uid != (uid_t)0) {
        fprintf(stderr, "\"%s\" must be owned by root\n", cmtab);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

    /* Check that file isn't globally writable: */
    if (!S_ISREG(sfile.st_mode) || (sfile.st_mode & S_IWOTH) != 0) {
        fprintf(stderr, "Lax permissions on \"%s\"\n", cmtab);
        eflag = ERR_INSECURE;
        goto bail_out;
    }

  bail_out:

    if (dirname != NULL) free((void*)dirname);

    return eflag;
}


static int sy_path(const char *path)
    /* Check whether pathname is considered secure */
{
    if (path == NULL) return ERR_NOERROR;
    if (path[0] == '/') return ERR_NOERROR;

    return ERR_INSECURE;
}

int sycheck_target(const cment_t *cment)
    /* Check that paths within target-specification are sensible */
{   int eflag=ERR_NOERROR;

    if (cment == NULL) return 0;

    eflag |= sy_path(cment->dev);
    eflag |= sy_path(cment->dir);
    eflag |= sy_path(cment->key.filename);

    if (eflag != ERR_NOERROR) {
        fprintf(stderr, "Specification for target \"%s\" contains non-absolute pathname\n", cment->ident);
    }

    return eflag;
}



/*
 *  ==== mutex-locking on configuration directory ====
 */

int cm_mutex_lock(void)
    /* Try to acquire lock on configuration directory (via symlink marker) */
{   char *fname=NULL, ident[64];
    int tries=10, eflag=ERR_BADMUTEX;
#if HAVE_NANOSLEEP
    int ticks;
    struct timespec delay;
#endif

    (void)cm_path(&fname, "_lock_");
    snprintf(ident, sizeof(ident), "%u-%u",
                (unsigned)getpid(), (unsigned)getuid());

    while (tries-->0) {
        errno = 0;
        if (symlink(ident, fname) == 0) {
            /* Lock acquired */
            eflag = 0; break;
        } else {
            if (errno == EEXIST) {
                /* Try again later */
#if HAVE_NANOSLEEP
                ticks = (53 * tries + 97 * (long)&tries) % 11;
                delay.tv_sec = (ticks / 10);
                delay.tv_nsec = (ticks % 10) * 1000L * 1000L * 1000L;
                nanosleep(&delay, NULL);
#else
                sleep(1);
#endif
            }
            else break;     /* failed to make link for more peculiar reason */
        }
    }

    if (eflag != 0) {
        fprintf(stderr, "Failed to create lock-file \"%s\" (errno=%d)\n",
                fname, errno);
    }

    free((void*)fname);

    return eflag;
}

int cm_mutex_unlock(void)
    /* Release lock on configuration directory */
{   char *fname=NULL;
    struct stat sbuff;
    int eflag=0;

    (void)cm_path(&fname, "_lock_");

    if (lstat(fname, &sbuff) != 0
      || !S_ISLNK(sbuff.st_mode)
      || unlink(fname) != 0) {
        fprintf(stderr, "Failed to remove lock-file \"%s\"\n", fname);
        eflag = ERR_BADMUTEX;
    }

    free((void*)fname);

    return eflag;
}

/* vim: set ts=4 sw=4 et: */

/*
 *  (C)Copyright 2005-2008, RW Penney
 */
