// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2003 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// font.cpp
////////////////////////////////////////////////////////////////////////

#include "font.h"

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Font
// DESCRIPTION: Load a font from a data file. The file is expected to be
//              open already.
// PARAMETERS : file - SDL_RWops data file descriptor
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Font::Font( SDL_RWops *file ) {
  struct {
    unsigned char w;
    unsigned char h;
    unsigned char chars;
    unsigned char flags;
  } fontinfo;

  SDL_RWread( file, (char *)&fontinfo, sizeof(fontinfo), 1 );
  width = fontinfo.w;
  height = fontinfo.h;
  chars = fontinfo.chars;
  flags = fontinfo.flags;

  surface = new Surface;
  surface->LoadImageData( file );

  // init font colors
  SDL_Color *fg = &surface->s_surface->format->palette->colors[1];
  col = Color( fg->r, fg->g, fg->b );

  surface->DisplayFormat();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::TextWidth
// DESCRIPTION: Calculate the length of a string in pixels.
// PARAMETERS : str - string to get the length of
// RETURNS    : length of str in pixels
////////////////////////////////////////////////////////////////////////

unsigned short Font::TextWidth( const char *str ) const {
  if ( !str ) return 0;

  unsigned short maxlen = 0, len = 0;
  for ( int i = 0; str[i] != '\0'; ++i ) {
    if ( str[i] == '\n' ) {
      if ( len > maxlen ) maxlen = len;
      len = 0;
    } else len += CharWidth( str[i] );
  }
  return (len > maxlen ? len : maxlen);
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::TextHeight
// DESCRIPTION: Calculate the height of a string in pixels.
// PARAMETERS : str     - string to get the height of; A new line is
//                        started whenever the length exceeds the
//                        available line width or a newline character
//                        is encountered
//              linew   - available line width in pixels
//              spacing - vertical spacing between lines
// RETURNS    : height of str in pixels
////////////////////////////////////////////////////////////////////////

unsigned short Font::TextHeight( const char *str, unsigned short linew,
                                 unsigned short spacing ) const {
  unsigned short lines = 1;
  int linelen = 0, lastspace = 0, lastspacewidth = 0, linewidth = 0, charwidth;

  for ( const char *ptr = str; *ptr != '\0'; ++ptr ) {
    ++linelen;
    charwidth = CharWidth( *ptr );
    if ( *ptr == ' ' ) {
      lastspace = linelen;
      lastspacewidth = 0;
    } else lastspacewidth += charwidth;

    if ( (linewidth + charwidth <= linew) && (*ptr != '\n') ) {
      linewidth += charwidth;
    } else {
      ++lines;
      if ( *ptr == '\n' ) {
        linelen = 0;
        linewidth = 0;
      } else if ( lastspace > 0 ) {	// break line at last space
        linelen -= lastspace;
        linewidth = lastspacewidth;
      } else {				// break line in the middle of a word
        linelen = 1;
        linewidth = charwidth;
      }
      lastspace = 0;
      lastspacewidth = 0;
    }
  }

  return lines * (Height() + spacing);
}


////////////////////////////////////////////////////////////////////////
// NAME       : Font::SetColor
// DESCRIPTION: Set the foreground color of the font.
// PARAMETERS : fcol - new font color
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::SetColor( const Color &fcol ) {
  if ( col != fcol ) {
    col = fcol;
    surface->Colorize( 0, 0, surface->Width(), surface->Height(), fcol );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Write
// DESCRIPTION: Print a string to a surface.
// PARAMETERS : str  - string
//              dest - destination surface
//              x    - left edge of string
//              y    - top edge of string
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::Write( const char *str, Surface *dest, short x, short y ) const {
  while ( *str ) {
    Write( *str++, dest, x, y );
    x += width;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Write
// DESCRIPTION: Print a string to a surface with clipping.
// PARAMETERS : str  - string
//              dest - destination surface
//              x    - left edge of string
//              y    - top edge of string
//              clip - clipping rectangle
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::Write( const char *str, Surface *dest, short x, short y,
                  const Rect &clip ) const {
  if ( (y + height < clip.y) || (y >= clip.y + clip.h) ||
       (x >= clip.x + clip.w) ) return;

  Rect oldclip;
  dest->GetClipRect( oldclip );
  dest->SetClipRect( clip );
  Write( str, dest, x, y );
  dest->SetClipRect( oldclip );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Write
// DESCRIPTION: Print a colored string to a surface.
// PARAMETERS : str   - string
//              dest  - destination surface
//              x     - left edge of string
//              y     - top edge of string
//              color - string color
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::Write( const char *str, Surface *dest, short x, short y,
                  const Color &color ) {
  Color old = col;
  SetColor( color );
  Write( str, dest, x, y );
  SetColor( old );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Write
// DESCRIPTION: Print a character to a surface.
// PARAMETERS : c    - character
//              dest - destination surface
//              x    - left edge of destination area
//              y    - top edge of destination area
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::Write( char c, Surface *dest, short x, short y ) const {
  // convert letters to printables or filter out unavailable characters
  if ( (c < '!') || (c > 'z') ) return;
  else if ( (flags & FONT_CAPS_ONLY) && (c >= 'a') && (c <= 'z') ) c = c - 'a' + 'A' - '!';
  else c -= '!';

  if ( c < chars ) {
    Rect src( c * width, 0, width, height );
    surface->Blit( dest, src, x, y );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Write
// DESCRIPTION: Print a colored character to a surface.
// PARAMETERS : c     - character
//              dest  - destination surface
//              x     - left edge of destination area
//              y     - top edge of destination area
//              color - character color
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Font::Write( char c, Surface *dest, short x, short y,
                  const Color &color ) {
  Color old = col;
  SetColor( color );
  Write( c, dest, x, y );
  SetColor( old );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Font::Allowed
// DESCRIPTION: Check whether a character exists in this font.
// PARAMETERS : c - character
// RETURNS    : true if character is available, false otherwise
////////////////////////////////////////////////////////////////////////

bool Font::Allowed( unsigned short c ) const {
  if ( (c != ' ') && ((c < '!') || (c > 'z')) ) return false;
  return true;
}

