// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// mission.cpp
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "mission.h"
#include "fileio.h"
#include "globals.h"

const char *event_labels[] = {
      "Message", "Mining", "Score", "Next Map",
      "Create Unit", "Manipulate Event", "Research", 0 };
const char *etrigger_labels[] = {
      "Turn", "Unit Destruction", "Shop Owner", "Unit Owner",
      "Unit Position", 0 };

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::Mission
// DESCRIPTION: Create a new mission.
// PARAMETERS : size - map size
//              ts   - tile set to be used
//              us   - unit set to be used
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Mission::Mission( const Point &size, TerrainSet *ts, UnitSet *us ) :
                  p1(PLAYER_ONE), p2(PLAYER_TWO) {
  flags = 0;
  level_info = -1;
  next_map = -1;
  msg_num = 0;
  msg_len = 0;
  password[0] = '\0';
  SetTitle( "Unknown" );

  unit_set = us;
  terrain_set = ts;

  map.SetUnitSet( unit_set );
  map.SetTerrainSet( terrain_set );
  map.SetSize( size );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::~Mission
// DESCRIPTION: Destructor
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Mission::~Mission( void ) {
  delete unit_set;
  delete terrain_set;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::Load
// DESCRIPTION: Load a game from a mission file.
// PARAMETERS : filename - mission file name
// RETURNS    : 0 on success, non-zero otherwise
////////////////////////////////////////////////////////////////////////

int Mission::Load( const char *filename ) {
  int rc = -1;
  SDL_RWops *file = SDL_RWFromFile( filename, "rb" );
  if ( !file ) return -1;

  // read game info
  unsigned char version;
  unsigned long id = SDL_ReadLE32( file );
  SDL_RWread( file, &version, 1, 1 );
  if ( (version == FILE_VERSION) && (id == FID_MISSION) ) {
    unsigned short len, i;
    unsigned char buf;
    char *uset, *tset;

    // set last filename
    last_file_name = file_part( filename );
    i = last_file_name.rfind( '.' );
    last_file_name.erase( i );

    flags = SDL_ReadLE16( file );
    i = SDL_ReadLE16( file );       // turn number - unused

    if ( flags & GI_SAVEFILE )      // refuse to load saved games
      goto out;
    else if ( flags & GI_PASSWORD ) {
      SDL_RWread( file, password, 1, 8 );
      crypt( password );
    } else password[0] = '\0';

    SDL_RWread( file, &level_info, 1, 1 );
    SDL_RWread( file, &next_map, 1, 1 );
    SDL_RWread( file, &buf, 1, 1 );     // current player - unused
    SDL_RWread( file, &buf, 1, 1 );     // turn phase - unused

    len = SDL_ReadLE16( file );        // load name of unit set
    uset = new char [len + 1];
    SDL_RWread( file, uset, 1, len );
    uset[len] = '\0';

    len = SDL_ReadLE16( file );        // load name of terrain set
    tset = new char [len + 1];
    SDL_RWread( file, tset, 1, len );
    tset[len] = '\0';

    map.Load( file );                  // load map

    unit_set = new UnitSet;
    rc = unit_set->Load( uset );
    delete [] uset;
    if ( rc ) {
      fprintf( stderr, "Error: Unit set not available.\n" );
      goto out;
    }

    terrain_set = new TerrainSet;
    rc = terrain_set->Load( tset );
    delete [] tset;
    if ( rc ) {
      fprintf( stderr, "Error: Terrain set not available.\n" );
      goto out;
    }

    map.SetUnitSet( unit_set );
    map.SetTerrainSet( terrain_set );

    p1.Load( file );
    p2.Load( file );

    len = SDL_ReadLE16( file );         // load buildings
    for ( i = 0; i < len; ++i ) {
      Building *b = new Building();
      b->Load( file );
      buildings.AddTail( b );
      map.SetBuilding( b, b->Position() );
    }

    len = SDL_ReadLE16( file );         // load units
    for ( i = 0; i < len; ++i ) {
      unsigned char tid;
      SDL_RWread( file, &tid, 1, 1 );

      Unit *u = new Unit( file, unit_set->GetUnitInfo(tid) );
      if ( u ) {
        units.AddTail( u );
        if ( !map.GetMapObject(u->Position()) )
          map.SetUnit( u, u->Position() );
        else u->SetFlags( U_SHELTERED );
      }
    }

    len = SDL_ReadLE16( file );         // load combats

    // combat actions - only present in saved games

    len = SDL_ReadLE16( file );         // load events
    for ( i = 0; i < len; ++i ) {
      Event *e = new Event();
      e->Load( file );
      events.AddTail( e );
    }

    // load text messages
    msg_num = SDL_ReadLE16( file );             // number of messages
    msg_len = SDL_ReadLE32( file );
    if ( msg_num ) {                            // load messages
      char *ptr, *msgs = new char[msg_len];

      SDL_RWread( file, msgs, 1, msg_len );

      i = 0;
      ptr = msgs;
      while ( i < msg_num ) {
        crypt( ptr );
        messages.AddTail( new Message( ptr ) );
        ++i;

        while ( *ptr != '\0' ) ++ptr;
        ++ptr;
      }
      delete [] msgs;
    }

    len = SDL_ReadLE16( file );

    // history - only present in saved games

    rc = 0;
  }

out: 
  SDL_RWclose( file );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::Save
// DESCRIPTION: Save the mission to a file.
// PARAMETERS : filename - data file name
// RETURNS    : 0 on successful write, non-zero otherwise
////////////////////////////////////////////////////////////////////////

int Mission::Save( const char *filename ) {
  SDL_RWops *file = SDL_RWFromFile( filename, "wb" );
  if ( !file ) {
    fprintf( stderr, "Failed to save mission!" );
    return -1;
  }

  unsigned char version = FILE_VERSION, current_player = PLAYER_ONE,
                turn_phase = TURN_START;
  unsigned short len, num;

  // set last filename
  last_file_name = file_part( filename );
  num = last_file_name.rfind( '.' );
  last_file_name.erase( num );

  // save game info
  SDL_WriteLE32( file, FID_MISSION );
  SDL_RWwrite( file, &version, 1, 1 );

  SDL_WriteLE16( file, flags );
  SDL_WriteLE16( file, 1 );        // turn

  if ( flags & GI_PASSWORD ) {
    crypt( password );
    SDL_RWwrite( file, password, 1, 8 );
    crypt( password );
  }

  SDL_RWwrite( file, &level_info, 1, 1 );
  SDL_RWwrite( file, &next_map, 1, 1 );
  SDL_RWwrite( file, &current_player, 1, 1 );
  SDL_RWwrite( file, &turn_phase, 1, 1 );

  len = unit_set->GetName().length();            // save mission set info
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, unit_set->GetName().c_str(), 1, len );
  len = terrain_set->GetName().length();
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, terrain_set->GetName().c_str(), 1, len );

  map.Save( file );            // save map

  p1.Save( file );              // save player data
  p2.Save( file );

  num = buildings.CountNodes();                 // save buildings
  SDL_WriteLE16( file, num );
  for ( Building *b = static_cast<Building *>( buildings.Head() );
        b; b = static_cast<Building *>( b->Next() ) ) b->Save( file );

  // save transports; basically, transports are not much different
  // from other units but we MUST make sure that transports having
  // other units on board are loaded before those units; we need to
  // make two passes through the list, first saving all unsheltered
  // units (which are possibly transports carrying other units), and
  // all sheltered units in the second run
  num = units.CountNodes();
  SDL_WriteLE16( file, num ); 
  Unit *u;
  for ( u = static_cast<Unit *>( units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {    // make sure transports are
    if ( !u->IsSheltered() ) u->Save( file );          // stored before carried units
  }

  for ( u = static_cast<Unit *>( units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {
    if ( u->IsSheltered() ) u->Save( file );
  }

  num = 0;                                             // save combat data
  SDL_WriteLE16( file, num );

  num = events.CountNodes();                           // save events
  SDL_WriteLE16( file, num );
  for ( Event *e = static_cast<Event *>( events.Head() );
        e; e = static_cast<Event *>( e->Next() ) )
    e->Save( file );

  SDL_WriteLE16( file, msg_num );                     // save text messages
  SDL_WriteLE32( file, msg_len );
  for ( Message *m = static_cast<Message *>( messages.Head() );
        m; m = static_cast<Message *>( m->Next() ) ) m->Save( file );

  SDL_WriteLE16( file, 0 );                           // turn history

  SDL_RWclose( file );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::Export
// DESCRIPTION: Save the mission data to a plain text file.
// PARAMETERS : filename - save file name
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Mission::Export( const string &filename ) const {
  ofstream file( filename.c_str() );
  if ( !file ) return -1;

  string title( file_part(filename) );
  int suffix = title.rfind( '.' );
  title.erase( suffix );

  file << "[mission]\n";
  file << "title = " << title << '\n';
  file << "mapwidth = " << map.Width() << '\n';
  file << "mapheight = " << map.Height() << '\n';
  file << "info = " << (short)level_info << '\n';
  file << "nextmap = " << (short)next_map << '\n';
  file << "players = " << (((flags & GI_AI) != 0) ? '1' : '2') << '\n';
  if ( password[0] != '\0' )
    file << "password = " << password << '\n';
  file << "tileset = " << terrain_set->GetName() << '\n';
  file << "unitset = " << unit_set->GetName() << "\n\n";

  map.Export( file );
  p1.Export( file );
  p2.Export( file );

  Unit *u;
  for ( u = static_cast<Unit *>(units.Head()); u;
        u = static_cast<Unit *>(u->Next()) )
    if ( !u->IsSheltered() ) u->Export( file );

  for ( u = static_cast<Unit *>(units.Head()); u;
        u = static_cast<Unit *>(u->Next()) )
    if ( u->IsSheltered() ) u->Export( file );

  for ( Building *b = static_cast<Building *>(buildings.Head()); b;
        b = static_cast<Building *>(b->Next()) )
    b->Export( file, unit_set );

  for ( Event *e = static_cast<Event *>(events.Head()); e;
        e = static_cast<Event *>(e->Next()) )
    e->Export( file, unit_set );

  if ( messages.CountNodes() > 0 ) {
    file << "[messages]";

    for ( Message *m = static_cast<Message *>(messages.Head()); m;
          m = static_cast<Message *>(m->Next()) ) {
      file << '\n';
      m->Export( file );
      file << '%';
    }
    file << "%\n\n";
  }

  file.close();
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : Mission::CreateUnit
// DESCRIPTION: Create a new unit.
// PARAMETERS : type - unit type info
//              pid  - identifier of cotrolling player
//              pos  - position on map
// RETURNS    : created unit or NULL on error
////////////////////////////////////////////////////////////////////////

Unit *Mission::CreateUnit( const UnitType *type, unsigned char pid, const Point &pos ) {
  Unit *u = new Unit( type, pid, GetUnitID(), pos );
  if ( u ) {
    units.AddTail( u );

    if ( !map.GetMapObject( pos ) ) map.SetUnit( u, pos );
    else u->SetFlags( U_SHELTERED );
  }
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::DeleteUnit
// DESCRIPTION: Delete a unit, and for transporters also all freight.
// PARAMETERS : u - unit to remove from mission
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Mission::DeleteUnit( Unit *u ) {
  u->Remove();

  if ( !u->IsSheltered() ) {
    map.SetUnit( NULL, u->Position() );

    if ( u->IsTransport() ) {
      Unit *next, *f = static_cast<Unit *>( units.Head() );
      while ( f ) {
        next = static_cast<Unit *>( f->Next() );

        if ( f->Position() == u->Position() ) {
          f->Remove();
          delete f;
        }

        f = next;
      }
    }
  }

  delete u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetUnitByID
// DESCRIPTION: Find the unit corresponding to the given identifier.
// PARAMETERS : id - identifier of the unit to be searched for
// RETURNS    : pointer to the unit, or NULL if no unit with that ID
//              exists
////////////////////////////////////////////////////////////////////////

Unit *Mission::GetUnitByID( unsigned short id ) const {
  Unit *u = static_cast<Unit *>( units.Head() );
  while ( u ) {
    if ( u->ID() == id ) return u;
    u = static_cast<Unit *>( u->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetUnitID
// DESCRIPTION: Get a unique identifier for a new unit.
// PARAMETERS : -
// RETURNS    : unique unit ID
////////////////////////////////////////////////////////////////////////

unsigned short Mission::GetUnitID( void ) const {
  unsigned short id = 0;
  Unit *u = static_cast<Unit *>( units.Head() );

  while ( u ) {
    if ( u->ID() == id ) {
      ++id;
      u = static_cast<Unit *>( units.Head() );
    } else u = static_cast<Unit *>( u->Next() );
  }

  return id;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::CreateBuilding
// DESCRIPTION: Create a new building.
// PARAMETERS : pid - identifier of cotrolling player
//              pos - position on map
// RETURNS    : created building or NULL on error
////////////////////////////////////////////////////////////////////////

Building *Mission::CreateBuilding( unsigned char pid, const Point &pos ) {
  Building *b = new Building( pos, GetBuildingID(), pid, "Depot" );
  if ( b ) {
    buildings.AddTail( b );
    map.SetBuilding( b, pos );
  }
  return b;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::DeleteBuilding
// DESCRIPTION: Delete a building and all units inside.
// PARAMETERS : b - building to remove from mission
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Mission::DeleteBuilding( Building *b ) {
  map.SetBuilding( NULL, b->Position() );
  b->Remove();

  Unit *next, *f = static_cast<Unit *>( units.Head() );
  while ( f ) {
    next = static_cast<Unit *>( f->Next() );

    if ( f->Position() == b->Position() ) {
      f->Remove();
      delete f;
    }

    f = next;
  }

  delete b;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetBuildingByID
// DESCRIPTION: Find the shop corresponding to the given identifier.
// PARAMETERS : id - identifier of the shop to be searched for
// RETURNS    : pointer to the shop, or NULL if no building with that ID
//              exists
////////////////////////////////////////////////////////////////////////

Building *Mission::GetBuildingByID( unsigned short id ) const {
  Building *b = static_cast<Building *>( buildings.Head() );
  while ( b ) {
    if ( b->ID() == id ) return b;
    b = static_cast<Building *>( b->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetBuildingID
// DESCRIPTION: Get a unique identifier for a new building.
// PARAMETERS : -
// RETURNS    : unique building ID
////////////////////////////////////////////////////////////////////////

unsigned short Mission::GetBuildingID( void ) const {
  unsigned short id = 0;
  Building *b = static_cast<Building *>( buildings.Head() );

  while ( b ) {
    if ( b->ID() == id ) {
      ++id;
      b = static_cast<Building *>( buildings.Head() );
    } else b = static_cast<Building *>( b->Next() );
  }

  return id;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::CreateEvent
// DESCRIPTION: Create a new event. The event is initialized with some
//              default values (which may be invalid in some cases).
// PARAMETERS : type    - event type
//              trigger - event trigger type
// RETURNS    : created event or NULL on error
////////////////////////////////////////////////////////////////////////

Event *Mission::CreateEvent( unsigned char type, unsigned char trigger ) {
  Event *e = new Event( GetEventID(), type, trigger );
  if ( e ) events.AddTail( e );
  return e;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::DeleteEvent
// DESCRIPTION: Delete an event.
// PARAMETERS : e - event to remove from mission
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Mission::DeleteEvent( Event *e ) {
  e->Remove();
  delete e;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetEventByID
// DESCRIPTION: Find the event corresponding to the given identifier.
// PARAMETERS : id - identifier of the event to be searched for
// RETURNS    : pointer to the event, or NULL if no event with that ID
//              exists
////////////////////////////////////////////////////////////////////////

Event *Mission::GetEventByID( unsigned short id ) const {
  Event *e = static_cast<Event *>( events.Head() );
  while ( e ) {
    if ( e->ID() == id ) return e;
    e = static_cast<Event *>( e->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetEventID
// DESCRIPTION: Get a unique identifier for a new event.
// PARAMETERS : -
// RETURNS    : unique event ID
////////////////////////////////////////////////////////////////////////

unsigned char Mission::GetEventID( void ) const {
  unsigned char id = 0;
  Event *e = static_cast<Event *>( events.Head() );

  while ( e ) {
    if ( e->ID() == id ) {
      ++id;
      e = static_cast<Event *>( events.Head() );
    } else e = static_cast<Event *>( e->Next() );
  }

  return id;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::GetMessage
// DESCRIPTION: Get a message from the catalog.
// PARAMETERS : id - message identifier of the requested message
// RETURNS    : pointer to requested message if successful, 0 otherwise
////////////////////////////////////////////////////////////////////////

const char *Mission::GetMessage( short id ) const {
  if ( id >= 0 ) {
    Message *msg = static_cast<Message *>(messages.GetNode(id));
    if ( msg ) return msg->GetText().c_str();
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::SetPassword
// DESCRIPTION: Set the map access code for campaign missions.
// PARAMETERS : pass - password (may be NULL)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Mission::SetPassword( const char *pass ) {

  for ( int i = 0; i < 8; ++i ) password[i] = '\0';

  if ( pass ) {
    strncpy( password, pass, 7 );
    flags |= GI_PASSWORD;
  } else flags &= ~GI_PASSWORD;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::SetNumPlayers
// DESCRIPTION: Set the intended number of human players for this map.
//              This is only a hint for the player.
// PARAMETERS : num - number of intended players (1 or 2)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Mission::SetNumPlayers( unsigned char num ) {
  if ( num == 1 ) flags |= GI_AI;
  else flags &= ~GI_AI;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::StorageLeft
// DESCRIPTION: Get the amount of free space in a transport, which can
//              be used to store crystals or units. Crystals require 1
//              storage unit per 10 crystals, units need Unit::Weight()
//              each.
// PARAMETERS : -
// RETURNS    : number of unused storage units; for non-transport units
//              returns 0
////////////////////////////////////////////////////////////////////////

unsigned short Mission::StorageLeft( Unit &u ) const {
  unsigned short space;

  if ( u.IsTransport() ) {
    space = u.Type()->Slots() - ((u.Crystals() + 9) / 10);

    if ( !u.IsSheltered() ) {
      for ( Unit *walk = static_cast<Unit *>(units.Head()); walk;
            walk = static_cast<Unit *>(walk->Next()) ) {
        if ( (u.Position() == walk->Position()) && walk->IsSheltered() )
          space -= walk->Weight();
      }
    }
  } else space = 0;
  return space;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::Validate
// DESCRIPTION: Check the mission for (logical) errors. This will catch
//              mistakes concerning settings which can not directly be
//              enforced by CoMET (e.g. you could create an event
//              targetting a unit and delete that unit afterwards).
// PARAMETERS : errors - string object; for each error found, a short
//                       message will be appended which can afterwards
//                       be presented to the user
// RETURNS    : number of errors and warnings detected
////////////////////////////////////////////////////////////////////////

unsigned short Mission::Validate( string &errors ) {
  unsigned short errs = 0;
  char buf[256];

  Player *pls[2] = { &p1, &p2 };
  for ( int i = 0; i < 2; ++i ) {
    if ( pls[i]->Briefing() == -1 ) {
      ++errs;
      sprintf( buf, "Warning: No briefing set for %s.\n\n", pls[i]->Name() );
      errors.append( buf );
    } else if ( !GetMessage(pls[i]->Briefing()) ) {
      ++errs;
      sprintf( buf, "Error: Invalid briefing %d set for %s.\n\n",
               pls[i]->Briefing(), pls[i]->Name() );
      errors.append( buf );
    }
  }

  if ( level_info == -1 ) {
    ++errs;
    errors.append( "Warning: No level info message configured.\n\n" );
  } else if ( !GetMessage(level_info) ) {
    ++errs;
    sprintf( buf, "Error: Invalid level info message %d configured.\n\n",
             level_info );
    errors.append( buf );
  }

  errs += ValidateMap( errors );

  for ( Node *n = events.Head(); n; n = n->Next() )
    errs += ValidateEvent( *static_cast<Event *>(n), errors );

  sprintf( buf, "Validation summary\n"
                "------------------\n"
                "Detected %d errors or warnings.", errs );
  errors.append( buf );

  return errs;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::ValidateMap
// DESCRIPTION: Check the map for errors.
// PARAMETERS : errors - error string
// RETURNS    : number of errors and warnings detected
////////////////////////////////////////////////////////////////////////

unsigned short Mission::ValidateMap( string &errors ) const {
  unsigned short errs = 0;
  char buf[512];
  Point p;

  for ( p.y = 0; p.y < map.Height(); ++p.y ) {
    for ( p.x = 0; p.x < map.Width(); ++p.x ) {

      if ( map.IsBuilding( p ) && !map.GetBuilding( p ) ) {
        ++errs;
        sprintf( buf, "Error: Tile at %d/%d denotes a shop "
                      "entrance, but no shop has been created.\n\n",
                 p.x, p.y );
        errors.append( buf );
      }

      Unit *u = map.GetUnit( p );
      if ( u && !u->IsSheltered() &&
           ((map.TerrainTypes(p) & u->Terrain()) == 0) ) {
        ++errs;
        sprintf( buf, "Warning: '%s' (%d) at %d/%d "
                      "is walking on invalid terrain.\n\n",
                 u->Name(), u->ID(), p.x, p.y );
        errors.append( buf );
      }
    }
  }
  return errs;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Mission::ValidateEvent
// DESCRIPTION: Check the event for errors.
// PARAMETERS : errors - error string
// RETURNS    : number of errors and warnings detected
////////////////////////////////////////////////////////////////////////

unsigned short Mission::ValidateEvent( Event &e, string &errors ) const {
  unsigned short errs = 0;
  char buf[512];

  if ( (e.Title() != -1) && !GetMessage(e.Title()) ) {
    ++errs;
    sprintf( buf, "Error: Invalid title %d for %s (%d) event.\n\n",
             e.Title(), e.Name(), e.ID() );
    errors.append( buf );
  }

  if ( (e.Message() != -1) && !GetMessage(e.Message()) ) {
    ++errs;
    sprintf( buf, "Error: Invalid message %d for %s (%d) event.\n\n",
             e.Message(), e.Name(), e.ID() );
    errors.append( buf );
  }


  switch ( e.Trigger() ) {
  case ETRIGGER_HAVE_BUILDING:
    if ( (e.GetTData(0) == -1) || !GetBuildingByID(e.GetTData(0)) ) {
      ++errs;
      sprintf( buf, "Error: Trigger of %s (%d) event targets "
                    "non-existing shop with ID %d.\n\n",
               e.Name(), e.ID(), e.GetTData(0) );
      errors.append( buf );
    }
    break;

  case ETRIGGER_HAVE_UNIT:
  case ETRIGGER_UNIT_POSITION:
    if ( (e.GetTData(0) == -1) || !GetUnitByID(e.GetTData(0)) ) {
      ++errs;
      sprintf( buf, "Error: Trigger of %s (%d) event targets "
                    "non-existing unit with ID %d.\n\n",
               e.Name(), e.ID(), e.GetTData(0) );
      errors.append( buf );
    }
    break;

  case ETRIGGER_TURN:
    break;

  case ETRIGGER_UNIT_DESTROYED:
    if ( e.GetTData(0) != -1 ) {
      Unit *u = GetUnitByID( e.GetTData(0) );

      if ( !u ) {
        ++errs;
        sprintf( buf, "Error: Trigger of %s (%d) event targets "
                      "non-existing unit with ID %d.\n\n",
                 e.Name(), e.ID(), e.GetTData(0) );
        errors.append( buf );
      } else if ( u->Owner() != e.GetTData(1) ) {
        ++errs;
        sprintf( buf, "Warning: Trigger of %s (%d) event had "
                      "incorrect owner of %s (%d). Fixed.\n\n",
                 e.Name(), e.ID(), u->Name(), u->ID() );
        errors.append( buf );
        e.SetTData( 1, u->Owner() );
      }
    }
    break;

  default:
    ++errs;
    sprintf( buf, "Warning: Unknown event trigger %s for %s (%d) event.\n\n",
             e.TriggerName(), e.Name(), e.ID() );
    errors.append( buf );
  }


  switch ( e.Type() ) {
  case EVENT_CREATE_UNIT:
  case EVENT_MINING:
  case EVENT_RESEARCH: {
    int i = (e.Type() == EVENT_CREATE_UNIT) ? 1 : 0;
    if ( (e.GetData(i) == -1) || !GetBuildingByID(e.GetData(i)) ) {
      ++errs;
      sprintf( buf, "Error: %s (%d) event targets "
                    "non-existing shop with ID %d.\n\n",
               e.Name(), e.ID(), e.GetData(i) );
      errors.append( buf );
    }
    break; }

  case EVENT_MANIPULATE_EVENT:
    if ( (e.GetData(0) == -1) || !GetEventByID(e.GetData(0)) ) {
      ++errs;
      sprintf( buf, "Error: %s (%d) event targets "
                    "non-existing event with ID %d.\n\n",
               e.Name(), e.ID(), e.GetData(0) );
      errors.append( buf );
    }
    break;

  case EVENT_NEXT_MAP:
    if ( (e.GetData(0) != -1) && !GetMessage(e.GetData(0)) ) {
    ++errs;
    sprintf( buf, "Error: Invalid mission file name %d for %s (%d) event.\n\n",
             e.GetData(0), e.Name(), e.ID() );
    errors.append( buf );
  }

  case EVENT_MESSAGE:
  case EVENT_SCORE:
    break;

  default:
    ++errs;
    sprintf( buf, "Warning: Unknown event type %s for event ID %d.\n\n",
             e.Name(), e.ID() );
    errors.append( buf );
  }

  return errs;
}


////////////////////////////////////////////////////////////////////////
// NAME       : Player::Player
// DESCRIPTION: Create a new player.
// PARAMETERS : id - player identifier (PLAYER_ONE or PLAYER_TWO)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Player::Player( unsigned char id ) {
  p_id = id;
  p_level = 0;
  p_success = 0;
  p_briefing = -1;
  sprintf( p_name, "Player %d", id + 1 );
  for ( int i = 0; i < 8; ++i ) p_password[i] = '\0';
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Load
// DESCRIPTION: Load player information from a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Player::Load( SDL_RWops *file ) {
  SDL_RWread( file, &p_id, 1, 1 );
  SDL_RWread( file, &p_level, 1, 1 );
  SDL_RWread( file, &p_success, 1, 1 );
  SDL_RWread( file, &p_briefing, 1, 1 );
  SDL_RWread( file, p_name, 1, 16 );
  SDL_RWread( file, p_password, 1, 8 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Save
// DESCRIPTION: Save player information to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, non-zero on error
////////////////////////////////////////////////////////////////////////

int Player::Save( SDL_RWops *file ) const {
  SDL_RWwrite( file, &p_id, 1, 1 );
  SDL_RWwrite( file, &p_level, 1, 1 );
  SDL_RWwrite( file, &p_success, 1, 1 );
  SDL_RWwrite( file, &p_briefing, 1, 1 );
  SDL_RWwrite( file, p_name, 1, 16 );
  SDL_RWwrite( file, p_password, 1, 8 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Player::Export
// DESCRIPTION: Save the player to a plain text file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Player::Export( ofstream &file ) const {
  file << "[player]\n";
  file << "name = " << p_name << '\n';
  file << "briefing = " << (short)p_briefing << "\n\n";
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Event
// DESCRIPTION: Create a new event.
// PARAMETERS : id       - unique event identifier
//              type     - event type
//              trigger  - event trigger type
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

Event::Event( unsigned char id, unsigned char type, unsigned char trigger ) {
  e_id = id;
  e_type = type;
  e_trigger = trigger;
  e_depend = -1;
  e_title = -1;
  e_message = -1;
  e_flags = 0;
  e_player = PLAYER_ONE;

  // set event data defaults
  switch ( e_type ) {
  case EVENT_MESSAGE:
    e_data[0] = e_data[1] = e_data[2] = 0;
    break;
  case EVENT_MINING:
    e_data[0] = -1;        // change mining in shop -1
    e_data[1] = 0;         // set crystals to this amount
    e_data[2] = 0;         // set storage to absolute amount
    break;
  case EVENT_SCORE:
    e_data[0] = 100;       // award 100 points
    e_data[1] = e_data[2] = 0;
    break;
  case EVENT_NEXT_MAP:
    e_data[0] = -1;       // prepare map with this name
    e_data[1] = e_data[2] = 0;
    break;
  case EVENT_CREATE_UNIT:
    e_data[0] = 0;        // build first unit type in set
    e_data[1] = -1;       // build in shop with identifier -1
    e_data[2] = 0;
    break;
  case EVENT_RESEARCH:
    e_data[0] = -1;      // make unit available in shop -1
    e_data[1] = 0;       // research first unit in set
    e_data[2] = 0;
    break;
  case EVENT_MANIPULATE_EVENT:
    e_data[0] = -1;      // modify event -1
    e_data[1] = 0;       // flags affected
    e_data[2] = 0;       // set selected flags
    break;
  }

  switch ( e_trigger ) {
  case ETRIGGER_TURN:
    e_tdata[0] = 0;     // activate on first turn
    e_tdata[1] = e_tdata[2] = 0;
    break;
  case ETRIGGER_UNIT_DESTROYED:
    e_tdata[0] = -1;   // destroy all enemy units
    e_tdata[1] = PLAYER_ONE;
    e_tdata[2] = 0;
    break;
  case ETRIGGER_HAVE_BUILDING:
  case ETRIGGER_HAVE_UNIT:
    e_tdata[0] = -1;          // check shop/unit -1
    e_tdata[1] = PLAYER_ONE;  // owned by first player
    e_tdata[2] = -1;          // activate any turn if condition is true
    break;
  case ETRIGGER_UNIT_POSITION:
    e_tdata[0] = -1;   // require no specific unit
    e_tdata[1] = 0;    // x coordinate of target hex
    e_tdata[2] = 0;    // y coordinate of target hex
    break;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Load
// DESCRIPTION: Load an event from a file.
// PARAMETERS : file    - file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Event::Load( SDL_RWops *file ) {
  int i;

  SDL_RWread( file, &e_id, 1, 1 );
  SDL_RWread( file, &e_type, 1, 1 );
  SDL_RWread( file, &e_trigger, 1, 1 );
  SDL_RWread( file, &e_depend, 1, 1 );
  for ( i = 0; i < 3; ++i ) e_tdata[i] = SDL_ReadLE16( file );
  for ( i = 0; i < 3; ++i ) e_data[i] = SDL_ReadLE16( file );
  e_title = SDL_ReadLE16( file );
  e_message = SDL_ReadLE16( file );
  e_flags = SDL_ReadLE16( file );
  SDL_RWread( file, &e_player, 1, 1 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Save
// DESCRIPTION: Save the event data to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Event::Save( SDL_RWops *file ) const {
  int i;

  SDL_RWwrite( file, &e_id, 1, 1 );
  SDL_RWwrite( file, &e_type, 1, 1 );
  SDL_RWwrite( file, &e_trigger, 1, 1 );
  SDL_RWwrite( file, &e_depend, 1, 1 );
  for ( i = 0; i < 3; ++i ) SDL_WriteLE16( file, e_tdata[i] );
  for ( i = 0; i < 3; ++i ) SDL_WriteLE16( file, e_data[i] );
  SDL_WriteLE16( file, e_title );
  SDL_WriteLE16( file, e_message );
  SDL_WriteLE16( file, e_flags );
  SDL_RWwrite( file, &e_player, 1, 1 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Export
// DESCRIPTION: Save the event to a plain text file.
// PARAMETERS : file - data file descriptor
//              uset - current unit set
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Event::Export( ofstream &file, const UnitSet *uset ) const {
  const char *type[] = { "message", "mining", "score",
        "nextmap", "createunit", "manipulateevent", "research" };

  file << "[event]\n";
  file << "id = " << (short)e_id << '\n';
  file << "type = " << type[e_type-1] << '\n';
  file << "player = " << (e_player + 1) << '\n';
  if ( e_depend != -1 )  file << "depend = " << (short)e_depend << '\n';
  if ( e_title != -1 )   file << "title = " << e_title << '\n';
  if ( e_message != -1 ) file << "message = " << e_message << '\n';
  if ( e_flags != 0 )    file << "flags = " << e_flags << '\n';

  file << "trigger = ";
  switch ( e_trigger ) {
  case ETRIGGER_TURN:
    file << "turn\n";
    file << "tturn = " << e_tdata[0] << '\n';
    break;
  case ETRIGGER_UNIT_DESTROYED:
    file << "unitdestroyed\n";
    file << "tunit = " << e_tdata[0] << '\n';
    file << "towner = " << e_tdata[1] + 1 << '\n';
    break;
  case ETRIGGER_HAVE_BUILDING:
    file << "havebuilding\n";
    file << "tbuilding = " << e_tdata[0] << '\n';
    file << "towner = " << e_tdata[1] + 1 << '\n';
    file << "tturn = " << e_tdata[2] << '\n';
    break;
  case ETRIGGER_HAVE_UNIT:
    file << "haveunit\n";
    file << "tunit = " << e_tdata[0] << '\n';
    file << "towner = " << e_tdata[1] + 1 << '\n';
    file << "tturn = " << e_tdata[2] << '\n';
    break;
  case ETRIGGER_UNIT_POSITION:
    file << "unitposition\n";
    file << "tunit = " << e_tdata[0] << '\n';
    file << "txpos = " << e_tdata[1] << '\n';
    file << "typos = " << e_tdata[2] << '\n';
    break;
  }

  switch ( e_type ) {
  case EVENT_MINING:
    file << "building = " << e_data[0] << '\n';
    file << "crystals = " << e_data[1] << '\n';
    file << "action = " << e_data[2] << '\n';
    break;
  case EVENT_RESEARCH:
    file << "building = " << e_data[0] << '\n';
    file << "unit = " << uset->GetUnitInfo(e_data[1])->Name() << '\n';
    break;
  case EVENT_SCORE:
    file << "success = " << e_data[0] << '\n';
    break;
  case EVENT_CREATE_UNIT:
    file << "unit = " << uset->GetUnitInfo(e_data[0])->Name() << '\n';
    file << "building = " << e_data[1] << '\n';
    break;
  case EVENT_NEXT_MAP:
    file << "map = " << e_data[0] << '\n';
    break;
  case EVENT_MANIPULATE_EVENT:
    file << "event = " << e_data[0] << '\n';
    file << "flags = " << e_data[1] << '\n';
    file << "action = " << e_data[3] << '\n';
    break;
  }

  file << '\n';
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::Name
// DESCRIPTION: Get the name of the event (or rather, the event type).
// PARAMETERS : -
// RETURNS    : event type string
////////////////////////////////////////////////////////////////////////

const char *Event::Name( void ) const {
  return event_labels[Type()-1];
}

////////////////////////////////////////////////////////////////////////
// NAME       : Event::TriggerName
// DESCRIPTION: Get the name of the event trigger type.
// PARAMETERS : -
// RETURNS    : event trigger type string
////////////////////////////////////////////////////////////////////////

const char *Event::TriggerName( void ) const {
  return etrigger_labels[Trigger()];
}

////////////////////////////////////////////////////////////////////////
// NAME       : Message::Save
// DESCRIPTION: Save the message to a file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Message::Save( SDL_RWops *file ) const {
  char *crypted = new char[text.length() + 1];
  strcpy( crypted, text.c_str() );
  crypt( crypted );

  SDL_RWwrite( file, crypted, 1, text.length() + 1 );
  delete [] crypted;
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Message::Export
// DESCRIPTION: Save the message to a plain text file.
// PARAMETERS : file - data file descriptor
// RETURNS    : 0 on success, -1 on error
////////////////////////////////////////////////////////////////////////

int Message::Export( ofstream &file ) const {
  file << text << '\n';
  return 0;
}

