// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// edwidget.cpp
////////////////////////////////////////////////////////////////////////

#include <stdlib.h>

#include "edwidget.h"
#include "misc.h"

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::NumberWidget
// DESCRIPTION: Create a new NumberWidget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              number - initial value to display
//              min    - minimum allowed value
//              max    - maximum allowed value
//              flags  - widget flags (see widget.h for details)
//              title  - widget title
//              window - widget parent window
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

NumberWidget::NumberWidget( short id, short x, short y, unsigned short w, unsigned short h,
            short number, short min, short max, unsigned short flags,
            const char *title, Window *window ) :
    StringWidget( id, x, y, w, h, itoa(number, numbuf), strlen(itoa(max,numbuf2)),
                  flags, title, window ) {
  num = number;
  minval = min;
  maxval = max;
  SetValidator( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetNumber
// DESCRIPTION: Fill the widget with a value.
// PARAMETERS : number - new widget value
//              upd    - whether to update the display (default is true)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetNumber( short number, bool upd /* = true */ ) {
  num = MIN( MAX( minval, number ), maxval );

  SetString( itoa( num, numbuf ), upd );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::ValidateKey
// DESCRIPTION: Only accept numbers for input.
// PARAMETERS : str - string currently entered in widget (not used)
//              key - char to be entered
//              pos - position at which to enter the char
// RETURNS    : TRUE if key is accepted, FALSE if refused
////////////////////////////////////////////////////////////////////////

bool NumberWidget::ValidateKey( const char *str, unsigned short key,
                                unsigned short pos ) const {
  return ((key >= '0') && (key <= '9'));
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetMin
// DESCRIPTION: Set a new minimum value.
// PARAMETERS : min - new minimum
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetMin( short min ) {
  minval = min;

  if ( Number() < min ) SetNumber( min );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetMax
// DESCRIPTION: Set a new maximum value.
// PARAMETERS : max - new maximum
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetMax( short max ) {
  maxval = max;

  if ( Number() > max ) SetNumber( max );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::Release
// DESCRIPTION: When the widget is released make sure the number which
//              was entered is inside the requested boundaries and
//              adjust it if necessary.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::Release( void ) {
  const char *str = String();
  short n = (str ? atoi(str) : 0);
  num = n;

  if ( num < minval ) num = minval;
  else if ( num > maxval ) num = maxval;

  StringWidget::Release();

  SetString( itoa( num, numbuf ) );
}

#define GFX_CYCLE_W 7
#define GFX_CYCLE_H 12
#define GFX_CYCLE_X 138
#define GFX_CYCLE_Y 46

////////////////////////////////////////////////////////////////////////
// NAME       : CycleWidget::CycleWidget
// DESCRIPTION: Create a new cycle widget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget relative to window border
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              flags  - see widget.h for details (default title
//                       placement is WIDGET_ALIGN_LEFT)
//              title  - widget title (may be NULL)
//              defval - default value, starting with 0
//              labels - array of available choices, terminated by a
//                       NULL-string
//              window - widget parent window
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

CycleWidget::CycleWidget( short id, short x, short y, unsigned short w,
             unsigned short h, unsigned short flags, const char *title,
             unsigned short defval, const char *labels[], Window *window ) :
     ButtonWidget( id, x, y, w, h, flags, title, window ),
     maxval(0), choices(labels) {

  if ( !(flags & (WIDGET_ALIGN_CENTER|WIDGET_ALIGN_LEFT|
                  WIDGET_ALIGN_RIGHT|WIDGET_ALIGN_TOP)) )
    this->flags |= WIDGET_ALIGN_LEFT;

  image[0] = Image( window->GetView()->GetSystemIcons(),
                    GFX_CYCLE_X, GFX_CYCLE_Y,
                    GFX_CYCLE_W, GFX_CYCLE_H );

  for ( int i = 1; labels[i]; ++i ) ++maxval;
  val = MIN( defval, maxval );
}

////////////////////////////////////////////////////////////////////////
// NAME       : CycleWidget::Draw
// DESCRIPTION: Draw the cycle widget.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void CycleWidget::Draw( void ) {
  ButtonWidget::Draw();

  short off = (Clicked() ? 1 : 0);

  surface->FillRect( x + w - image[0].Width() - 8 + off, y + 4 + off, 1, h - 8, Color(CF_COLOR_SHADOW) );
  surface->FillRect( x + w - image[0].Width() - 7 + off, y + 4 + off, 1, h - 8, Color(CF_COLOR_HIGHLIGHT) );

  image[0].Draw( surface, x + w - image[0].Width() - 4 + off, y + (h - image[0].Height())/2 + off );

  // draw current label
  font->Write( choices[val], surface,
               x + (w - image[0].Width() - 10 - font->TextWidth(choices[val]))/2 + off,
               y + (h - font->Height()) / 2 + off, *this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : CycleWidget::Release
// DESCRIPTION: Update the button label and redraw the widget.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void CycleWidget::Release( void ) {
  val = (val + 1) % (maxval + 1);

  ButtonWidget::Release();
}

////////////////////////////////////////////////////////////////////////
// NAME       : CycleWidget::SetValue
// DESCRIPTION: Set the cycle value.
// PARAMETERS : value - new value
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void CycleWidget::SetValue( unsigned short value ) {
  val = value % (maxval + 1);
  Draw();
  Show();
}

